

package BondLib "Modelica library for Bond Graph modeling"
  extends Modelica.Icons.Library;

  annotation (
    Window(
      x=0.01,
      y=0.55,
      width=0.4,
      height=0.4,
      library=1,
      autolayout=1),
    Documentation(info="<html>
The <b>BondLib</b> library is designed as a graphical library for modeling physical systems using the bond graph metaphor.  This library has been created by Prof. Fran&ccedil;ois Cellier and his students.<p>

When modeling systems in this fashion, it is common to hide many connections behind the bondgraphic connectors.  To this end, one first places neighboring connected elements next to each other <br>
<img src=\"../Images/BG_Fig1.png\"> <br>
making the connection between the connectors.  The two models are then shifted together, until the two connectors are in the same position on the screen <br>
<img src=\"../Images/BG_Fig2.png\"> <br>
thereby hiding the connection from view. <p>

Bond graphs are described in terms of four basic variables: <p>

<ol>
<li> efforts (<b>e</b>), <br>
<li> flows (<b>f</b>), <br>
<li> generalized momentums (<b>p</b>), and <br>
<li> generalized positions (<b>q</b>).
</ol> <p>

The four variables are related to each other in the following fashion: <br>
<img src=\"../Images/BG_Fig3.png\"> <p>

Power is always the product of effort and flow: <br>
<b>P = e*f</b> <p>

The three basic OnePort elements of the bond graph methodology are: <p>

<ol>
<li> resistors (<b>R</b>): <b>e = R(f)</b>, <br>
<li> capacitors (<b>C</b>): <b>e = C(q)</b>, and <br>
<li> inductors (<b>I</b>): <b>f = I(p)</b>,
</ol> <p>

where <b>R</b>, <b>C</b>, and <b>I</b> can be arbitrarily non-linear functions operating in the first and third quadrants only. <p>

Bond graphs are domain independent.  The four basic variables of a number of physical domains are summarized below: <br>
<img src=\"../Images/BG_Fig4.png\"> <p>

<hr> <p>

<b>References:</b> <p>

<ol>
<li> Cellier, F.E. (1991), <i>Continuous System Modeling</i>, Springer-Verlag, New York, ISBN: 0-387-97502-0, 755p.
<li> Cellier, F.E. (1992), \"<a href=\"http://www.inf.ethz.ch/~fcellier/Publikationen/HierarchBG.pdf\">Hierarchical Non-linear
     Bond Graphs: A Unified Methodology for Modeling Complex Physical Systems</a>,\" <i>Simulation</i>, <b>58</b>(4),
     pp. 230-248.
<li> Cellier, F.E. (1995), \"Bond Graphs: The Right Choice for Educating Students in Modeling Continuous-time Physical
     Systems,\" <i>Simulation</i>, <b>64</b>(3), pp. 154-159.
<li> Cellier, F.E. and R.T. McBride (2003), \"<a href=\"http://www.inf.ethz.ch/~fcellier/Publikationen/icbgm03_bglib.pdf\">Object-oriented
     Modeling of Complex Physical Systems Using the Dymola Bond-graph Library</a>,\" <i>Proc. ICBGM'03,
     6<sup>th</sup> SCS Intl. Conf. on Bond Graph Modeling and Simulation</i>, Orlando, Florida, pp. 157-162.
<li> Cellier, F.E. and A. Nebot (2005), \"<a href=\"http://www.inf.ethz.ch/~fcellier/Publikationen/modelica2005_cellier.pdf\">The
     Modelica Bond Graph Library</a>,\" <i>Proc. 4<sup>th</sup> Modelica Conference</i>, Hamburg, Germany.
</ol> <p>

<hr> <p>

<b>Main Author:</b> <p>

Prof. Dr. Fran&ccedil;ois E. Cellier <br>
Institut f&uuml;r Computational Science <br>
ETH Z&uuml;rich <br>
ETH Zentrum CAB G82.1 <br>
CH-8092 Z&uuml;rich <br>
Switzerland <p>

Phone: +41(44)632-7474 <br>
Fax: +41(44)632-1374 <br>
Email: <a href=\"mailto:FCellier@Inf.ETHZ.CH\"> FCellier@Inf.ETHZ.CH</a> <br>
URL: <a href=\"http://www.inf.ethz.ch/~fcellier/\"> http://www.inf.ethz.ch/~fcellier/</a> <p>

<hr> <p>

<b>Release Notes:</b> <p>

<ul>
<li>Version 1.0: February 10, 2005 <br>
<li>Version 1.1: June 21, 2005 <br>
<li>Version 1.2: November 21, 2005 <br>
<li>Version 1.3: December 13, 2006 <br>
<li>Version 2.0: July 18, 2007 <br>
<li>Version 2.1: August 2, 2007 <br>
<li>Version 2.2: August 30, 2007 <br>
<li>Version 2.3: September 24, 2007
</ul> <p>

<hr> <p>

<b>Copyright (C) 2005-2007, Fran&ccedil;ois E. Cellier.</b> <p>

<i>The ModelicaAdditions.BondLib package is <b>free</b> software;
it can be redistributed and/or modified
under the terms of the <b>Modelica license</b>, see the license conditions
and the accompanying <b>disclaimer</b> in the documentation of package
Modelica in file \"Modelica/package.mo\".</i> <p>

<hr> <p>
</html>
"), Diagram(
      Rectangle(extent=[-100,100; 100,-100],style(
          color=0,
          thickness=2,
          fillColor=9)),
      Text(
        extent=[-60,68; 52,46],
        style(
          color=0,
          thickness=2,
          fillColor=9,
          fillPattern=1),
        string="The bond graph"),
      Text(
        extent=[-73.9,15.05; 68,-14],
        style(
          color=0,
          thickness=2,
          fillColor=9,
          fillPattern=1),
        string="domain independent"),
      Text(
        extent=[-55.8,34.1; 54,4],
        style(
          color=0,
          thickness=2,
          fillColor=9,
          fillPattern=1),
        string="offers a uniform, "),
      Text(
        extent=[-47.7,-8.85; 38,-26],
        style(
          color=0,
          thickness=2,
          fillColor=9,
          fillPattern=1),
        string="interface to"),
      Text(
        extent=[-79.6,-19.8; 82,-50],
        style(
          color=0,
          thickness=2,
          fillColor=9,
          fillPattern=1),
        string="physical system modeling."),
      Text(
        extent=[-45.9,51.05; 40,24],
        style(
          color=0,
          thickness=2,
          fillColor=9,
          fillPattern=1),
        string="methodology")),
    uses(Modelica(version="2.2.1")),
    version="2.3",
    conversion(from(version="", script="ConvertFromBondLib_.mos"),
        noneFromVersion="2.0"));

  package UsersGuide "User's Guide"
    annotation (DocumentationClass=true, Documentation(info="<html>
<h3><font color=\"#008000\" size=5>Users Guide of Package BondLib</font></h3>
<p>
Library <b>BondLib</b> is a <b>free</b> Modelica package providing components to model multi-energy physical systems in a convenient domain-independent format. This package contains the <b>users guide</b> to the library, and has the following content:
</p>
<ol>
<li><a href=\"Modelica://BondLib.UsersGuide.Overview\">Overview of Library</a>
       gives an overview of the library.
<li><a href=\"Modelica://BondLib.UsersGuide.Versions\">Release Notes</a>
       offers a historic perspective on the development of this library.
</ol>
</html>
"));
    package Overview "Overview of Library"

      annotation (Documentation(info="<html>
This overview contains the following sections: <p>

<ol>
<li><a href=\"Modelica://BondLib.UsersGuide.Overview.Introduction\">An introduction to bond graph modeling</a>
<li><a href=\"Modelica://BondLib.UsersGuide.Overview.Connectors\">Bond graph connectors</a>
<li><a href=\"Modelica://BondLib.UsersGuide.Overview.Bonds\">Causal and a-causal bonds</a>
<li><a href=\"Modelica://BondLib.UsersGuide.Overview.Basic\">Basic bond graph elements</a>
<li><a href=\"Modelica://BondLib.UsersGuide.Overview.Advanced\">Advanced bond graph elements</a>
<li><a href=\"Modelica://BondLib.UsersGuide.Overview.Wrapping\">Wrapping bond graphs</a>
<li><a href=\"Modelica://BondLib.UsersGuide.Overview.Electrical\">The electrical library</a>
<li><a href=\"Modelica://BondLib.UsersGuide.Overview.Mechanical\">The mechanical libraries</a>
<li><a href=\"Modelica://BondLib.UsersGuide.Overview.Thermal\">The thermal library</a>
<li><a href=\"Modelica://BondLib.UsersGuide.Overview.Encapsulating\">Encapsulating bond graphs</a>
<li><a href=\"Modelica://BondLib.UsersGuide.Overview.Hydraulic\">The hydraulic library</a>
<li><a href=\"Modelica://BondLib.UsersGuide.Overview.SystemDynamics\">The System Dynamics library</a>
<li><a href=\"Modelica://BondLib.UsersGuide.Overview.Examples\">Examples</a>
<li><a href=\"Modelica://BondLib.UsersGuide.Overview.Literature\">Literature</a>
</ol>
</html>
"));
      class Introduction "An introduction to bond graph modeling"

        annotation (Documentation(info="<html>
Bond graphs offer a generic domain-independent means to graphically modeling physical systems in an object-oriented fashion. <p>

All physical systems are characterized by the fact that they observe a number of conservation principles.  Prime among those is the conservation of energy.  When modeling a physical system, it is important for the modeler to be aware of these conservation principles, and ensure that the models constructed adhere to them. <p>

Yet, if the modeling formalism doesn't guarantee that the conservation principles are being observed, i.e., if it is up to the modeler to verify that his or her models adhere to them, these principles can get easily violated, because modelers (just like all other people) are constantly under time pressure, and usually won't take the time to validate their models, once they have been constructed and compile/simulate without generating error messages. <p>

It may therefore be a good idea to embrace a modeling formalism that checks the conservation principles implicitly for the modeler.  For example, if a modeler wishes to model an electrical capacitor, he or she knows that: <p>

<b>i = C*<font color=red>der</font>(u)</b> <p>

Yet, this model may fail, when the capacitance <b>C</b> happens to be a function of time.  It is better to ask oneself, what happens physically in the capacitor. <p>

We know that the capacitor stores <i>electrical charge</i>, <b>Q</b>.  The electrical charge is the integral of the current.  Thus, when current flows into the capacitor, electrical charge builds up.  The charge is a physical quantity having mass.  It thus needs to be conserved.  Thus, the charge, <b>Q</b>, rather than the voltage, <b>u</b>, is a much better choice for a state variable, since voltage doesn't need to be conserved. <p>

The voltage across the capacitor is proportional to the charge stored in the capacitor: <p>

<b>u = C*Q</b> <p>

What happens to the energy?  The power, <b>P</b>, flowing into the capacitor is the product of current and voltage: <p>

<b>P = u*i</b> <p>

The energy, <b>E</b>, stored in the capacitor is the integral of the power.  It also needs to be conserved.  Yet, we only need one state variable, thus energy doesn't get conserved automatically.  We must ensure explicitly that this happens. <p>

Bond graphs ensure that no energy is ever lost or (worse!) generated out of thin air in interconnections between models.  They do this by explicitly modeling the power flow through a physical system. <p>

Power, in all physical systems, can always be expressed as a product of two adjugate variables, or possibly in the most complex cases, as the sum of products of adjugate variables. <p>

Bond graphs work with these adjugate variables as the basic modeling variables: the <i>effort</i>, <b>e</b>, and the <i>flow</i>, <b>f</b>. Power is always the product of effort and flow: <p>

<b>P = e*f</b> <p>

What these adjugate variables represent, depends on the physical nature of the model.  For example in electrical systems, the effort would be the <i>voltage</i>, whereas the flow would be the <i>current</i>.  In mechanical translational systems, the effort would be the <i>force</i>, whereas the flow would be the <i>velocity</i>. <p>

A power flow is represented by a harpoon, with which the two adjugate variables are being associated.  The hook of the harpoon points always to the \"left\" (relative to the direction, in which the harpoon points), and the effort variable is written on the side of the harpoon, whereas the flow variable is written on the opposite side: <p>

<img src=\"../Images/BG_Fig5.png\"> <p>

The direction of the harpoon is the direction of positive power flow. <p>

Evidently, since <i>efforts</i> and <i>flows</i> can represent different physical properties, no strong type checking is possible in bond graph modeling.  All variables need to be declared as <b><font color=red>Real</font></b> rather than <b><font color=red>Modelica.SIunits.Voltage</font></b>, for example.  This is a price that the modeler must be willing to pay, if he or she wishes to use bond graphs. <p>

Sometimes, as in the case of the capacitor, it is useful to also consider the integrals of the two adjugate variables, as these are in fact natural state variables.  Since bond graphs were invented by a mechanical engineer, the integral of the effort is called the <i>generalized momentum</i>, <b>p</b>, whereas the integral of the flow is called the <i>generalized position</i>, <b>q</b>, since the integral of <i>force</i> is a <i>momentum</i>, whereas the integral of <i>velocity</i> is a position. <p>

The below table lists the four basic bond graph variables, <b>e</b>, <b>f</b>, <b>p</b>, and <b>q</b>, for a number of different energy domains: <p>

<img src=\"../Images/BG_Fig4.png\"> <p>

The bond graph model of a non-linear capacitor would thus take the form: <p>

<img src=\"../Images/BG_Fig6.png\"> <p>

where <b><font color=red>ModPassiveOnePort</font></b> is one of the standard bond graph interfaces offered in the <b><font color=\"#008000\">Interfaces</font</b> sub-library of the bond graph library.  It represents a modulated one-port, where the modulation signal (a real-valued input) is given the name <b>s</b>.  The variables <b>e</b> and <b>f</b> don't need to be declared either, as they are already declared inside the (partial) interface model.
</html>
"));
      end Introduction;

      class Connectors "Bond graph connectors"

        annotation (Documentation(info="<html>
As bonds carry two physical variables, the <i>effort</i>, <b>e</b> and the <i>flow</i>, <b>f</i>, the bond graph connectors need to carry at least these two variables.  Yet, also the direction of positive power flow is explicitly encoded in the bond models, and thereby also in the connectors.  To this end, a (non-physical) directional variable, <b>d</b>, is associated with each bond: <p>

<img src=\"../Images/BG_Fig7.png\"> <p>

<b>d</b> assumes a value of <b>-1</b> at the connector, from which the bond emanates, and a value of <b>+1</b> at the  connector, where it ends. <p>

All bond graph variables are declared as <i>potential variables</i>, i.e., rather than making use of the <i>flow variables</i> of Modelica, the bond graph library encodes the direction of flow explicitly using the <b>d</b> variable. <p>

The bond graph connector is a grey dot.  Its definition is given as follows: <p>

<img src=\"../Images/BG_Fig8.png\"> <p>

The corresponding bond model has been encoded as: <p>

<img src=\"../Images/BG_Fig9.png\"> <p>

It passes the physical variables along, like an electrical wire, and it defines the directional variable, <b>d</b>, at the two ends. <p>

In the bond graph methodology, there are two types of nodes, called <i>junctions</i>, the <b>0-junction</b>, which corresponds to a regular Modelica node, and the <b>1-junction</b>, which offers the dual functionality. <p>

<img src=\"../Images/BG_Fig10.png\"> <p>

The flows into a 0-junction add up to zero, whereas the efforts around a 0-junction are the same.  Correspondingly, the efforts into a 1-junction add up to zero, whereas the flows around it are the same.  This, by the way, is the reason, why the bond graph library doesn't make use of the Modelica flow variables, since Modelica offers only one type of nodes. <p>

For example, the 1-junction with three bond connectors is defined as: <p>

<img src=\"../Images/BG_Fig11.png\"> <p>

It inherits the effort and flow vectors from the (partial) <b><font color=red>ThreePortOne</font></b> interface model: <p>

<img src=\"../Images/BG_Fig12.png\"> <p>

which makes use of the directional <b>d</b> variables to decide on the sign in summing the efforts (or flows) into the junction.
</html>
"));
      end Connectors;

      class Bonds "Causal and a-causal bonds"

        annotation (Documentation(info="<html>
Modelica is perfectly capable of deciding on its own the computational causality of all equations.  Thus, models can be generated in the form of implicitly defined differential and algebraic equation (DAE) systems.  If we use the a-causal bonds, as presented so far, this is precisely, what we get. <p>

However, it may be useful to write models in their causal form, as the computational causality information may offer additional insight into the system. <p>

Since bonds carry two physical variables, we need to generate two equations to compute their values.  It turns out that, in every case, one of the two variables is computed at each end of the bond.  We can denote the side, where the flow variable is computed, by marking it with a vertical stroke, the so-called <i>causality stroke</i>. <p>

To this end, the bond graph library offers two additional types of bonds: <p>

<img src=\"../Images/BG_Fig13.png\"> <p>

In this case, the equations can be read out of the bond graph in their causal forms: <p>

<b>B1.e = B2.e</b> <br>
<b>B3.e = B2.e</b> <br>
<b>B2.f = B1.f - B3.f</b> <p>

for the 0-junction, and: <p>

<b>B4.f = B5.f</b> <br>
<b>B6.f = B5.f</b> <br>
<b>B5.e = B4.e - B6.e</b> <p>

for the 1-junction, respectively.  Since the 0-junction has only one flow equation, there must be exactly one causality stroke at the junction.  Similarly, since the 1-junction has only one effort equation, there must be exactly one causality stroke away from the junction. <p>

Causal and a-causal bonds can be arbitrarily mixed.  We recommend to use causal bonds as much as possible.
</html>
"));
      end Bonds;

      class Basic "Basic bond graph elements"

        annotation (Documentation(info="<html>
The four basic bond graph variables can be related in the following way: <p>

<img src=\"../Images/BG_Fig3.png\"> <p>

<i>Resistors</i> relate efforts and flows to each other in a static fashion, i.e.: <p>

<b>e = R(f)</b> <p>

Similarly, <i>capacitors</i> relate efforts and generalized positions to each other, i.e.: <p>

<b>e = C(q)</b> <p>

Finally, <i>inductors</i> relate flows and generalized momentums to each other, i.e.: <p>

<b>f = I(p)</b> <p>

All of these relationships must be defined in the first and thrid quadrants only for the element to be physical. There cannot exist any other relationships in physics; thus, all bond graphs can be written in terms of the three variables <b>R</b>, <b>C</b>, and <b>I</b>. <p>

In addition, energy can be transformed, either from one domain to another, or within a domain.  The transformation of free energy can follow two different patterns: <p>

<b>e1 = m*e2</b> <br>
<b>e1*f1 = e2*f2</b> <p>

or: <p>

<b>e1 = r*f2</b> <br>
<b>e1*f1 = e2*f2</b> <p>

We call the former relationship a <i>transformer</i>, <b>TF</b>, whereas the latter relationship is called a <i>gyrator</i>, <b>GY</i>.  The second equation in both cases ensures that no power is either lost or generated by the transformation. <p>

In their causal forms, transformers and gyrators can be written as follows: <p>

<img src=\"../Images/BG_Fig14.png\"> <p>

Thus, both the transformers and the gyrators have two possible causalities.  <i>Transformers</i> have always one causality stroke at the transformer and the other away from it, whereas <i>gyrators</i> have either both causality strokes at the gyrator, or both away from it. <p>

Finally, we need two types of sources, the effort source, <b>Se</b>, and the flow source, <b>Sf</b>: <p>

<img src=\"../Images/BG_Fig15.png\"> <p>

The causality strokes of the two sources are fixed. <p>

As we haven't discussed the causality strokes for the three basic modeling elements yet, let us cover that topic now: <p>

<img src=\"../Images/BG_Fig16.png\"> <p>

Both capacitors and inductors have <i>preferred causalities</i>, so-called <i>integral causalities</i>, whereas the causality of the resistors is free. <p>

That is all there is.  These elements exhaust all of physics, and puritans among the bond graph practitioners would argue that no other elements should be offered. <p>

If the preferred causality of a bond graph can be assigned in a unique way, the bond graph has neither algebraic loops nor structural singularities, i.e., can be described by an index-0 ordinary differential equation (ODE) system.  If the modeler has free choice in assigning causality strokes, the model contains one or several algebraic loops.  If a conflict exists on either capacitors or inductors, the model consists of a higher-index DAE system.
</html>
"));
      end Basic;

      class Advanced "Advanced bond graph elements"

        annotation (Documentation(info="<html>
For practical purposes, it makes sense to add more elements to the list. <p>

On the one hand, we may wish to offer different types of sources, such as sinusoidal and trapezoidal sources, to make the tool more practical for simulation.  Similarly, we may wish to offer different types of non-linear resistors, e.g. for hydraulic systems or thermal systems.  These are not really new elements, only specializations of the elements introduced previously. <p>

It may also make sense to offer sensor (detector) elements.  We may wish to sense any one of the four basic variables.  An effort detector, <b>De</b>, receives the effort from the model.  It computes the flow, by setting it to zero.  Thus, an effort sensor is in fact the same as a zero flow source: <p>

<img src=\"../Images/BG_Fig17.png\"> <p>

with the model: <p>

<img src=\"../Images/BG_Fig18.png\"> <p>

A position sensor: <p>

<img src=\"../Images/BG_Fig19.png\"> <p>

on the other hand, has the model: <p>

<img src=\"../Images/BG_Fig20.png\"> <p>

i.e., a position sensor is a macro element that is itself described by a bond graph.  A position sensor is a flow sensor followed by an integrator. <p>

Finally, we ought to discuss the switching elements.  The bond graph switch: <p>

<img src=\"../Images/BG_Fig21.png\"> <p>

has a Boolean input signal using the convention that, if the input signal has a value of <i>true</i>, the switch is <i>open</i>, i.e., there is zero flow.  On the other hand, if the Boolean input signal assumes a value of <i>false</i>, the switch is <i>closed</i>, i.e., there is zero effort. <p>

Switch elements are strange, because they change the location of their causality stroke as a function of the value of the Boolean input signal.  Thus, switch elements must always be attached to a-causal bonds. <p>

Furthermore, if the causality of the bond connecting to the switch is fixed from the outside, an ideal switch will lead to a division by zero in one of the two switch positions.  Thus, ideal switch elements must always be placed inside algebraic loops. <p>

To avoid this pitfall, the library offers also a leaking switch element, <b>Sw2</i>, that contains a parasitic resistor in the closed state, and a parasitic conductance in the open state. <p>

The library also offers special switching elements, such as electrical diodes.
</html>
"));
      end Advanced;

      class Wrapping "Wrapping bond graphs"

        annotation (Documentation(info="<html>
Although bond graphs can be used to describe arbitrarily complex physical systems, they offer a fairly low-level interface and are therefore not always easily readable.  Furthermore, different modelers may prefer different modeling formalisms, because they are more familiar with them. <p>

Since bond graphs offer a low-level interface, it is always possible to wrap bond graphs inside other modeling metaphors.  For example, the electrical resistor model: <p>

<img src=\"../Images/BG_Fig22.png\"> <p>

can easily be reimplemented as follows: <p>

<img src=\"../Images/BG_Fig23.png\"> <p>

The <b>El2BG</b> and <b>BG2El</b> blocks are wrappers that convert the electrical connectors to bond graph connectors, and vice versa. <p>

<font color=red><b>BondLib</b></font> features wrapped <font color=blue><b>electrical (analog)</b></font>, <font color=blue><b>mechanical 1D (translational, rotational)</b></font>, and <font color=blue><b>thermal 1D (heat transfer)</b></font> libraries that are analogous to the corresponding sub-libraries of the <font color=red><b>Modelica standard library</b></font>. The wrapped electrical analog library contains a complete wrapped <font color=blue><b>Spice implementation</b></font> for DC analysis, AC analysis, transient analysis, and thermal analysis. <p>

Wrapped <font color=blue><b>mechanical 2D and 3D libraries</b></font> are also available,  but they were included in the <font color=red><b>MultiBondLib</b></font> library, as they are based on multi-bond graphs rather than on regular bond graphs. <p>

The wrapped (internal) bond graph variables are not visible from the outside.  They were <b>protected</b> in order to make the wrapped models look externally more similar to the corresponding models of the <font color=red><b>Modelica standard library</b></font>. <p>

Also in the case of capacitors and inductors, Modelica's <b>stateSelect</b> feature has been used to make the wrapped models use the same state variables as the corresponding non-wrapped models of the <font color=red><b>Modelica standard library</b></font>, i.e. <b>C.v</b> instead of <b>C.C1.e</b> in the case of a capacitor. <p>

A Boolean flag <b>enforceStates</b> is by default set to <b>true</b>, but can be set to <b>false</b> if the internal bond graph variables are to be used as state variables.
</html>
"));
      end Wrapping;

      class Electrical "The electrical library"

        annotation (Documentation(info="<html>
We made use of the wrapping technique to re-implement large portions of the standard electrical analog library of Modelica using bond graph technology.  These models have the same touch and feel as the corresponding models contained in the standard library.  They are interchangeable and combinable in an arbitrary fashion. <p>

The new electrical analog library is a subset of the standard library, because not all elements have been recoded yet.  However, it is also a superset, because additional elements have been provided in some places. <p>

Especially, the new library contains <font color=red size=5><b><i>ModelicaSpice</i></b></font>, a Spice-style implementation of semiconductor models, including the four classes of bipolar transistors (the laterally and vertically diffused NPN and PNP transistors with all of the technological parameters that Spice offers), the two classes of MOSFETs (NMOS and PMOS), the two classes of JFETs (NJF and PJF), and an accurate model of diodes.  Just as in Spice, each of these models contains several alternative descriptions that can be selected by choosing the appropriate modeling level.
</html>
"));
      end Electrical;

      class Mechanical "The mechanical libraries"

        annotation (Documentation(info="<html>
We made use of the wrapping technique to re-implement large portions of the translational and rotational 1D mechanical libraries of the Modelica standard library using bond graph technology.  These models have the same touch and feel as the corresponding models contained in the standard library.  However, they are not combinable with the corresponding models of the Modelica standard library, because the translational and rotational connectors of the wrapped libraries are different from the corresponding connectors of the standard library. Since bond graphs operate on forces (torques) and velocities (angular velocities), it was important to include velocity information in the connectors, whereas the corresponding connectors of the standard library carry only positional information.
</html>"));
      end Mechanical;

      class Thermal "The thermal library"

        annotation (Documentation(info="<html>
We made use of the wrapping technique to re-implement large portions of the heat transfer library of the Modelica standard library using bond graph technology.  These models have the same touch and feel as the corresponding models contained in the standard library.  They are interchangeable and combinable in an arbitrary fashion. <p>

Not all models of the wrapped library are identical to the corresponding models of the standard library.  Especially the <b>conductance</b> and <b>convection</b> models behave differently, because the additional generated heat (entropy) by these resistors has been taken into account in the wrapped library, whereas it is ignored in the corresponding models of the standard library.
</html>"));
      end Thermal;

      class Encapsulating "Encapsulating bond graphs"

        annotation (Documentation(info="<html>
<i>Encapsulation</i> of bond graph models is a technique similar to <i>wrapping</i>, however, it doesn't require special wrapping models.  The bond graph methodology already offers standard interfaces to mathematical signals.  On the one hand, there are the modulated sources, <b>mSe</b> and <b>mSf</b>, which convert input signals to bond graph signals.  On the other hand, there are the four detector elements, <b>De</b>, <b>Df</b>, <b>Dp</b>, and <b>Dq</b>, that offer the inverse functionality by converting bond graph signals to mathematical signals. <p>

These elements can be used to encapsulate bond graphs inside block diagrams, which is particularly useful for the description of control systems, the individual components of which may still be composed using bond graph technology.
</html>"));
      end Encapsulating;

      class Hydraulic "The hydraulic library"

        annotation (Documentation(info="<html>
The hydraulic library contains a few bond graph models of hydraulic systems, such as a hydraulic motor with a servo valve. <p>

This library was built using the encapsulation technique discussed earlier.  The hydraulic motor is described by a bond graph with two bond graph connectors at the input, and one signal connector at the output: <p>

<img src=\"../Images/BG_Fig24.png\"> <p>

The servo valve, on the other hand, is described by another bond graph with one signal connector at the input, and two bond graph connectors at the output: <p>

<img src=\"../Images/BG_Fig25.png\"> <p>

In this way, the two components can be connected in the form of a block diagram: <p>

<img src=\"../Images/BG_Fig26.png\"> <p>

where they can then be used as part of a control loop: <p>

<img src=\"../Images/BG_Fig27.png\">
</html>
"));
      end Hydraulic;

      class SystemDynamics "The System Dynamics library"
        annotation (Documentation(info="<html>
<font color=red><b>System Dynamics</b></font> is a methodology that was designed to describe <font color=red><b>mass and information flows</b></font> in dynamical systems.  Since there are no <font color=red><b>energy flows</b></font>, bond graphs seem to be the wrong methodology for implementing System Dynamics models. <p>

Yet, masses being transported across a system carry their own internal energy along with them, and it may make sense to enhance the System Dynamics methodology to encompass those internal energy flows as well.  In that context, the <font color=red><b>Levels</b></font> of the System Dynamics methodology turn into <font color=red><b>mass capacitances</b></font> of the bond graph methodology. <p>

<font color=red><b>Mass capacitances</b></font> are non-linear.  Analogous to the thermal capacitances, they are inverse proportional to the effort, which in the case of a mass capacitance happens to be an enthalpy.  We know that from the chemical domain, where we also deal with these types of mass capacitances. <p>

<font color=red><b>Masses</b></font> in System Dynamics are measured in all kinds of units.  These can be numbers of people, money in a bank account, or frustration level in a relationship.  Hence the \"enthalpy\" that accompanies the mass flow can also assume a variety of different measurement units. <p>

<font color=red><b>System Dynamics</b></font> has been designed for coarse-grain modeling, i.e., for modeling at a macroscopic level without delving down into all of the gory details of physics.  <font color=red><b>Bond graphs</b></font> on the other hand have been designed for fine-grain modeling, i.e., for capturing physical knowledge down to the basics. <p>

Hence for modeling System Dynamics using bond graphs, we'll need to introduce a concept of <font color=red><b>sloppy bond graphs</b></font>. <p>

As System Dynamics is a world of its own, the <font color=red><b>SystemDynamics library</b></font> comes with its own <a href=\"Modelica://BondLib.SystemDynamics.UsersGuide\">User's Guide</a>.
</html>"));
      end SystemDynamics;

      class Examples "Examples"

        annotation (Documentation(info="<html>
The library contains an interesting example section: <p>

<ol>
<li> <b>SimpleBondGraph</b> describes a simple bond graph of a small electrical circuit.  The development of this bond graph was explained in [3].
<li> <b>ElectricalCircuit</b> contains a model of the same circuit, this time using the wrapping technology and components of the new electrical analog library contained as part of the bond graph library.
<li><b>OperationalAmplifierCircuit</b> describes a commercial hifi operational amplifier built in bipolar technology with six NPN and six PNP transistors [1].
<li><b>HydraulicControl</b> contains a hydraulic motor control problem, also explained in [2,3], that explains the encapsulation technique.
<li><b>SolarHouse</b> shows the use of bond graphs for thermodynamic modeling.  The example contains a fairly elaborate model of a passive solar heating system implemented in an experimental house in Tucson [5,6].
<li><b>Biosphere </b> shows another example of thermodynamic modeling, describing a model of the thermodynamic budget of the Biosphere 2 structure, located outside Tucson, without air conditioning [3,4].
<li><b>ModelicaSpice</b> is a collection of examples demonstrating the use of <font color=red><b><i>ModelicaSpice</i></b></font> for electronic device modeling and simulation (transient analysis, DC analysis, and AC analysis).
</ol> <p>

<hr> <p>

<b>References:</b> <p>

<ol>
<li> Cellier, F.E. (1991), <i>Continuous System Modeling</i>, Springer-Verlag, New York, ISBN: 0-387-97502-0, 755p.
<li> Cellier, F.E. and R.T. McBride (2003), \"<a href=\"http://www.inf.ethz.ch/~fcellier/Publikationen/icbgm03_bglib.pdf\">Object-oriented
     Modeling of Complex Physical Systems Using the Dymola Bond-graph Library</a>,\" <i>Proc. ICBGM'03,
     6<sup>th</sup> SCS Intl. Conf. on Bond Graph Modeling and Simulation</i>, Orlando, Florida, pp. 157-162.
<li> Cellier, F.E. and A. Nebot (2005), \"<a href=\"http://www.inf.ethz.ch/~fcellier/Publikationen/modelica2005_cellier.pdf\">The
     Modelica Bond Graph Library</a>,\" <i>Proc. 4<sup>th</sup> Modelica Conference</i>, Hamburg, Germany.
<li> Nebot, A., F.E. Cellier, and F. Mugica (1999), \"<a href=\"http://www.inf.ethz.ch/~fcellier/Publikationen/ecoeng_bio2.pdf\">Simulation
     of Heat and Humidity Budget of Biosphere 2 without its Air Conditioning</a>,\" <i>Ecological Engineering</i>,
     <b>13</b>, pp. 333-356.
<li> Weiner, M. (1992), <i>Bond Graph Model of a Passive Solar Heating System</i>, MS Thesis, Dept. of Electr. &amp;
     Comp. Engr., University of Arizona, Tucson, Arizona.
<li> Weiner, M., and F.E. Cellier (1993), \"<a href=\"http://www.inf.ethz.ch/~fcellier/Publikationen/weiner.pdf\">Modeling
     and Simulation of a Solar Energy System by Use of Bond Graphs</a>,\" <i>Proc. 1<sup>st</sup> SCS Intl. Conf. on Bond Graph
     Modeling</i>, San Diego, California, pp.301-306.
</ol>
</html>
"));
      end Examples;

      class Literature "Literature"

        annotation (Documentation(info="<html>
We offer a fairly substantial bibliography of bond graph modeling in the \"<a href=\"http://www.inf.ethz.ch/~fcellier/BondGraphs/bg.html\">Bond Graph Compendium</a>.\"
</html>
"));
      end Literature;
    end Overview;

    package Versions "Release Notes"

      annotation (Documentation(info="<html>
This section explains the major improvements made to the library from one version to the next: <p>

<ol>
<li><a href=\"Modelica://BondLib.UsersGuide.Versions.Version_1_0\">old -> Version 1.0</a>
    (February 10, 2005)

<li><a href=\"Modelica://BondLib.UsersGuide.Versions.Version_1_1\">Version 1.0 -> Version 1.1</a>
    (June 21, 2005)

<li><a href=\"Modelica://BondLib.UsersGuide.Versions.Version_1_2\">Version 1.1 -> Version 1.2</a>
    (November 21, 2005)

<li><a href=\"Modelica://BondLib.UsersGuide.Versions.Version_1_3\">Version 1.2 -> Version 1.3</a>
    (December 13, 2006)

<li><a href=\"Modelica://BondLib.UsersGuide.Versions.Version_2_0\">Version 1.3 -> Version 2.0</a>
    (July 18, 2007)

<li><a href=\"Modelica://BondLib.UsersGuide.Versions.Version_2_1\">Version 2.0 -> Version 2.1</a>
    (August 2, 2007)

<li><a href=\"Modelica://BondLib.UsersGuide.Versions.Version_2_2\">Version 2.1 -> Version 2.2</a>
    (August 30, 2007)

<li><a href=\"Modelica://BondLib.UsersGuide.Versions.Version_2_3\">Version 2.2 -> Version 2.3</a>
    (September 24, 2007)
</ol>
</html>"));
      class Version_1_0 "old -> Version 1.0 (February 10, 2005)"

        annotation (Documentation(info="<html>
<font color=red><b>BondLib 1.0</b></font> constitutes the first official release of BondLib,
a library that has been around in one form or another since 1987.  Whereas the original version
of BondLib (based on the original Dymola syntax) had been purely alphanumeric, a first
graphical version of BondLib, based on Dymola 4.0, was created in 2001.  However, that
version hadn't been documented yet and was strictly meant to be used by my students. <p>

<ul>
<li><font color=red><b>Conversion from Dymola 4.0:</b></font> The library has been
    upgraded to the new graphical interface of Dymola 5.0.</li>

<li><font color=red><b>Documentation:</b></font> Appropriate documentation has been
    added to all modules.</li>

<li><font color=red><b>Electrical library:</b></font> An electrical analog library,
    similar in scope to the corresponding sublibrary of the Modelica standard library
    but based on an internal (wrapped) bond graph representation has been added.</li>

<li><font color=red><b>Examples:</b></font> A set of examples has been added to the
    library for improved documentation.</li>

</ol>
</html>"));
      end Version_1_0;

      class Version_1_1 "Version 1.0 -> Version 1.1 (June 21, 2005)"

        annotation (Documentation(info="<html>
The following improvements were made between Version 1.0 and Version 1.1: <p>

<ol>
<li><font color=red><b>Units:</b></font> Proper measurement units were added to
    the modules of the electrical library and to all examples.  The basic
    bond graph methodology is, of course, application independent, and
    therefore, its modules continue to use generic units (\"Real\").</li> <p>

<li><font color=red><b>User's Guide:</b></font> A user's guide has been
    added introducing the bond graph methodology.</li>
</ol>
</html>"));
      end Version_1_1;

      class Version_1_2 "Version 1.1 -> Version 1.2 (November 21, 2005)"

        annotation (Documentation(info="<html>
The following improvements were made between Version 1.1 and Version 1.2: <p>

<ol>
<li><font color=red><b>ModelicaSpice:</b></font> A new sublibrary has been
    added to the electrical analog sublibrary of BondLib implementing a
    full-fledged version of HSpice for DC analysis, AC analysis, and
    transient analysis.</li>
</ol>
</html>"));
      end Version_1_2;

      class Version_1_3 "Version 1.2 -> Version 1.3 (December 13, 2006)"

        annotation (Documentation(info="<html>
The following improvements were made between Version 1.2 and Version 1.3: <p>

<ol>
<li><font color=red><b>Sensor models:</b></font> The <b>Dp</b> and <b>Dq</b>
    models were enhanced by the possibility of specifying non-zero initial
    conditions.</li> <p>

<li><font color=red><b>Electrical library:</b></font> The internal variables
    and models of wrapped models were <b>protected</b> to make them invisible
    from the outside.  In this way, the models of the standard library and those
    of the wrapped bond graph library look much more similar to each other in the
    <b>Simulation window</b>, and the simulation doesn't store unnecessarily
    internal variables that are of no interest to the modeler. Dymola's
    <b>stateSelect algorithm</b> is being used to enforce that the electrical voltages
    across capacitors [and the electrical currents through inductors] are being used as
    state variables instead of the (internal) effort variables of the underlying bond
    graph C elements [and the flow variables of the bond graph I elements].</li>
</ol>
</html>"));
      end Version_1_3;

      class Version_2_0 "Version 1.3 -> Version 2.0 (July 18, 2007)"

        annotation (Documentation(info="<html>
The following improvements were made between Version 1.3 and Version 2.0: <p>

<ol>
<li><font color=red><b>Mechanical 1D libraries:</b></font> Two mechanical
    1D libraries for modeling translational and rotational motions were added.
    Although these libraries are similar to the corresponding libraries of the
    Modelica standard library, their connectors are incompatible due to the
    underlying bond graph technology being used.  Rather than only transferring
    positional and force (torque) information, the connectors of the new
    libraries also carry velocity information. Consequently, the models of the
    new libraries cannot be mixed with the models of the corresponding libraries
    that form part of the Modelica standard library.</li> <p>

<li><font color=red><b>Spice MOSFET models:</b></font> The MOSFET models of the
    SPICE library (part of the electrical analog library) still contained errors
    that have meanwhile been corrected by Alfonso Urqua and Carla Martn of the
    Universidad de Educacin a Distancia in Madrid.</li>
</ol>
</html>"));
      end Version_2_0;

      class Version_2_1 "Version 2.0 -> Version 2.1 (August 2, 2007)"

        annotation (Documentation(info="<html>
The following improvements were made between Version 2.0 and Version 2.1: <p>

<ol>
<li><font color=red><b>HeatTransfer library:</b></font> A heat transfer library
    based on bond graph technology was added.  Its connectors are compatible
    with the connectors of the corresponding heat transfer library of the Modelica
    standard library. Consequently, the models of the new library can be
    arbitrarily mixed with the models of the corresponding library that
    forms part of the Modelica standard library.</li> <p>

<li><font color=red><b>Electrical analog library:</b></font> The heat port
    and substrate port of the electrical analog library have been enhanced
    to allow them to remain unconnected from the outside.  An unconnected
    heat port corresponds to the assumption of a constant operating
    temperature of Tnom.  An unconnected substrate port of a bipolar transistor
    corresponds to the assumption of an internal connection between the substrate
    and either the collector or the emitter, depending on whether the transistor
    is laterally or vertically diffused.</li>
</ol>
</html>"));
      end Version_2_1;

      class Version_2_2 "Version 2.1 -> Version 2.2 (August 30, 2007)"

        annotation (Documentation(info="<html>
The following improvements were made between Version 2.1 and Version 2.2: <p>

<ol>
<li><font color=red><b>Electrical analog library:</b></font> All semiconductor
    models were equipped with parameter lists, in which inactive parameters
    (depending on the modeling level of the device) are dimmed to alert the
    users of the fact that these parameters are currently inactive.</li> <p>

<li><font color=red><b>Unit checking:</b></font> The entire library was checked
    for inconsistencies in the use of measurement units.  Warning messages are
    currently still obtained for the MOSFET models, because they include
    variables measured in V^{0.5}, a feature not currently supported by
    the new Dymola unit checking algorithm, and also for the Biosphere
    model, because it isn't built around SI units.</li>
</ol>
</html>"));
      end Version_2_2;

      class Version_2_3 "Version 2.2 -> Version 2.3 (September 24, 2007)"

        annotation (Documentation(info="<html>
The following improvements were made between Version 2.2 and Version 2.3: <p>

<ol>
<li><font color=red><b>System Dynamics library:</b></font> A new System Dynamics
    library has been added.  Its levels and rates have been modeled as
    encapsulated bond graphs.  Since there are no wrapper models used,
    it is the user's responsibility to verify that <b>Levels</b> and
    <b>Rates</b> always toggle.  All <b>Level</b> models end in junctions,
    whereas all <b>Rate</b> models end in bonds.  This is an exciting
    development as it enables us for the first time to model also
    population dynamics and macro-economic models in a systematic
    fashion using bond graphs.</li>
</ol>
</html>"));
      end Version_2_3;
    end Versions;
  end UsersGuide;

  package Interfaces
    "Connectors and partial models for Modelica bond graph library"
    annotation (
      preferedView="info",
      Icon(
        Rectangle(extent=[-100, -100; 80, 50], style(
            color=58,
            fillColor=30,
            fillPattern=1)),
        Polygon(points=[-100, 50; -80, 70; 100, 70; 80, 50; -100, 50], style(
            color=58,
            fillColor=30,
            fillPattern=1)),
        Polygon(points=[100, 70; 100, -80; 80, -100; 80, 50; 100, 70], style(
            color=58,
            fillColor=30,
            fillPattern=1)),
        Text(
          extent=[-85, 35; 65, -85],
          string="Library",
          style(color=58)),
        Text(
          extent=[-120, 122; 120, 73],
          string="%name",
          style(color=1))),
      Window(
        x=0.02,
        y=0.53,
        width=0.4,
        height=0.4,
        library=1,
        autolayout=1),
      Documentation(info="<html>
This package contains the three basic bond graph connectors as well as some partial models that invoke them. <p>

The <b>BondCon</b> connector is the basic bi-directional bondgraphic connector that is used always.  The <b>eBondCon</b> and <b>fBondCon</b> connectors are uni-directional connectors that are only used in the definitions of the two causal bonds.
</html>
"));
    connector BondCon "Bi-directional bondgraphic connector"
      Real e "Bondgraphic effort variable";
      Real f "Bondgraphic flow variable";
      Real d "Directional variable";
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Ellipse(extent=[-60, 60; 60, -60], style(color=8, fillColor=8))),
        Icon(Ellipse(extent=[-60, 60; 60, -60], style(color=8, fillColor=8))),
        Window(
          x=0.45,
          y=0.01,
          width=0.35,
          height=0.49),
        Documentation(info="<html>
This is the basic bi-directional bondgraphic connector that is used everywhere, except in the definitions of the two causal bonds. <p>

The bondgraphic connector references three variables, two of which are physical variables, whereas the third one is a topological variable.  All three are defined as potential variables. <p>

<pre>
Potential variables:

 e:  Bondgraphic effort variable

 f:  Bondgraphic flow variable

 d:  Directional variable indicating the direction of positive power flow
</pre>
</html>
"));
    end BondCon;

    connector eBondCon "Uni-directional bondgraphic connector"
      input Real e "Bondgraphic effort variable";
      output Real f "Bondgraphic flow variable";
      output Real d "Direction variable";
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Ellipse(extent=[-60, 60; 60, -60], style(color=8, fillColor=8)),
              Text(
            extent=[-34, 34; 38, -28],
            string="e",
            style(color=41, fillColor=41))),
        Icon(Ellipse(extent=[-60, 60; 60, -60], style(color=8, fillColor=8)),
            Text(
            extent=[-34, 34; 38, -28],
            string="e",
            style(color=41, fillColor=41))),
        Window(
          x=0.45,
          y=0.01,
          width=0.35,
          height=0.49),
        Documentation(info="<html>
This is one of two uni-directional bondgraphic connectors used in the definitions of the two causal bonds. <p>

The bondgraphic connectors reference three variables, two of which are physical variables, whereas the third one is a topological variable.  All three are defined as potential variables. <p>

<pre>
Potential variables:

 e:  Bondgraphic effort variable (input)

 f:  Bondgraphic flow variable (output)

 d:  Directional variable indicating the direction of positive power flow (output)
</pre>
</html>
"));
    end eBondCon;

    connector fBondCon "Uni-directional bondgraphic connector"
      output Real e "Bondgraphic effort variable";
      input Real f "Bondgraphic flow variable";
      output Real d "Direction variable";
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Ellipse(extent=[-60, 60; 60, -60], style(color=8, fillColor=8)),
              Text(
            extent=[-36, 38; 36, -34],
            string="f",
            style(color=41, fillColor=41))),
        Icon(Ellipse(extent=[-60, 60; 60, -60], style(color=8, fillColor=8)),
            Text(
            extent=[-36, 38; 36, -34],
            string="f",
            style(color=41, fillColor=41))),
        Window(
          x=0.45,
          y=0.01,
          width=0.35,
          height=0.49),
        Documentation(info="<html>
This is one of two uni-directional bondgraphic connectors used in the definitions of the two causal bonds. <p>

The bondgraphic connectors reference three variables, two of which are physical variables, whereas the third one is a topological variable.  All three are defined as potential variables. <p>

<pre>
Potential variables:

 e:  Bondgraphic effort variable (output)

 f:  Bondgraphic flow variable (input)

 d:  Directional variable indicating the direction of positive power flow (output)
</pre>
</html>
"));
    end fBondCon;

    partial model PassiveOnePort
      "Partial model invoking one bondgraphic connector"
      Real e "Bondgraphic effort";
      Real f "Bondgraphic flow";
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram,
        Icon,
        Documentation(info="<html>
This is a partial model invoking a single bondgraphic connector.  It is used to describe passive OnePort elements.<p>

<pre>
Potential variables:

 e:  Bondgraphic effort variable

 f:  Bondgraphic flow variable, normalized positive for flows into the model
</pre>
</html>
"));
      BondLib.Interfaces.BondCon BondCon1 "Bond graph connector"
                                          annotation (extent=[-110, -10; -90,
            10]);
    equation
      e = BondCon1.e;
      f = BondCon1.d*BondCon1.f;
    end PassiveOnePort;

    partial model ModPassiveOnePort
      "Partial model invoking one bondgraphic connector and one signal connector"

      Real e "Bondgraphic effort";
      Real f "Bondgraphic flow";
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Window(
          x=0.45,
          y=0.01,
          width=0.35,
          height=0.49),
        Documentation(info="<html>
This is a partial model invoking a single bondgraphic connector and a signal connector.  It is used to describe passive modulated OnePort elements.<p>

<pre>
Potential variables:

 e:  Bondgraphic effort variable

 f:  Bondgraphic flow variable, normalized positive for flows into the model

 s:  Real-valued signal input used for modulation
</pre>
</html>
"),     Diagram);
      BondLib.Interfaces.BondCon BondCon1 "Bond graph connector"
                                          annotation (extent=[-110, -10; -90,
            10]);
      Modelica.Blocks.Interfaces.RealInput s "Signal input"
                                                annotation (extent=[100, -10;
            80, 10]);
    equation
      e = BondCon1.e;
      f = BondCon1.d*BondCon1.f;
    end ModPassiveOnePort;

    partial model ActiveOnePort
      "Partial model invoking one bondgraphic connector"
      Real e "Bondgraphic effort";
      Real f "Bondgraphic flow";
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram,
        Icon,
        Documentation(info="<html>
This is a partial model invoking a single bondgraphic connector.  It is used to describe active OnePort elements.<p>

<pre>
Potential variables:

 e:  Bondgraphic effort variable

 f:  Bondgraphic flow variable, normalized positive for flows out of the model
</pre>
</html>
"));
      BondLib.Interfaces.BondCon BondCon1 "Bond graph connector"
                                          annotation (extent=[-110, -10; -90,
            10]);
    equation
      e = BondCon1.e;
      f = -BondCon1.d*BondCon1.f;
    end ActiveOnePort;

    partial model ModActiveOnePort
      Real e "Bondgraphic effort";
      Real f "Bondgraphic flow";
      annotation (Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]), Window(
          x=0.45,
          y=0.01,
          width=0.35,
          height=0.49),
        Diagram);
      BondLib.Interfaces.BondCon BondCon1 "Bond graph connector"
                                          annotation (extent=[-110, -10; -90,
            10]);
      Modelica.Blocks.Interfaces.RealInput s "Signal input"
                                                annotation (extent=[100, -10;
            80, 10]);
    equation
      e = BondCon1.e;
      f = -BondCon1.d*BondCon1.f;
    end ModActiveOnePort;

    partial model TwoPort "Partial model invoking two bondgraphic connectors"
      Real e1 "Bondgraphic primary effort";
      Real f1 "Bondgraphic primary flow";
      Real e2 "Bondgraphic secondary effort";
      Real f2 "Bondgraphic secondary flow";
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram,
        Icon(Line(points=[42, -44; 68, -50; 42, -56], style(color=0)), Line(
              points=[-70, -50; 68, -50], style(color=0))),
        Documentation(info="<html>
This is a partial model invoking two bondgraphic connectors.  It is used to describe directed TwoPort elements.  Positive flow is from inflow to outflow.  The direction of positive flow is indicated by an arrow. <p>

<pre>
Potential variables:

 e1:  Bondgraphic effort variable of inflow

 f1:  Bondgraphic flow variable of inflow, normalized positive for flows into the model

 e2:  Bondgraphic effort variable of outflow

 f2:  Bondgraphic flow variable of outflow, normalized positive for flows out of the model
</pre>

</html>
"));
      BondLib.Interfaces.BondCon BondCon1 "Left bond graph connector"
                                          annotation (extent=[-110, -10; -90,
            10]);
      BondLib.Interfaces.BondCon BondCon2 "Right bond graph connector"
                                          annotation (extent=[90, -10; 110, 10]);
    equation
      e1 = BondCon1.e;
      f1 = BondCon1.d*BondCon1.f;
      e2 = BondCon2.e;
      f2 = -BondCon2.d*BondCon2.f;
    end TwoPort;

    partial model TwoPort2 "Partial model invoking two bondgraphic connectors"
      Real e1 "Bondgraphic primary effort";
      Real f1 "Bondgraphic primary flow";
      Real e2 "Bondgraphic secondary effort";
      Real f2 "Bondgraphic secondary flow";
      BondLib.Interfaces.BondCon BondCon1 "First bond graph connector"
                                          annotation (extent=[-110, -10; -90,
            10]);
      BondLib.Interfaces.BondCon BondCon2 "Second bond graph connector"
                                          annotation (extent=[90, -10; 110, 10]);
    equation
      e1 = BondCon1.e;
      f1 = BondCon1.d*BondCon1.f;
      e2 = BondCon2.e;
      f2 = BondCon2.d*BondCon2.f;
      annotation (Documentation(info="<html>
This is a partial model invoking two bondgraphic connectors.  It is used to describe symmetric TwoPort elements.  Both inflow and outflow are assumed positive into the model. <p>

<pre>
Potential variables:

 e1:  Bondgraphic effort variable of inflow

 f1:  Bondgraphic flow variable of inflow, normalized positive for flows into the model

 e2:  Bondgraphic effort variable of outflow

 f2:  Bondgraphic flow variable of outflow, normalized positive for flows into the model
</pre>

</html>
"), Diagram);
    end TwoPort2;

    partial model TwoPortZero
      "Partial model invoking two bondgraphic connectors"
      Real e1 "Bondgraphic primary effort";
      Real f1 "Bondgraphic primary flow";
      Real e2 "Bondgraphic secondary effort";
      Real f2 "Bondgraphic secondary flow";
      annotation (Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]), Documentation(info="<html>
This is a partial model invoking two bondgraphic connectors.  It is used only for the description of the TwoPort 0-junction. <p>

<pre>
Potential variables:

 e1:  Bondgraphic effort variable of first connector

 f1:  Bondgraphic flow variable of first connector, normalized positive for flows into the model

 e2:  Bondgraphic effort variable of second connector

 f2:  Bondgraphic flow variable of second connector, normalized positive for flows into the model
</pre>

</html>
"),     Diagram);
      BondLib.Interfaces.BondCon BondCon1 "First bond graph connector"
                                          annotation (extent=[-110, -10; -90,
            10]);
      BondLib.Interfaces.BondCon BondCon2 "Second bond graph connector"
                                          annotation (extent=[90, -10; 110, 10]);
    equation
      e1 = BondCon1.e;
      f1 = BondCon1.d*BondCon1.f;
      e2 = BondCon2.e;
      f2 = BondCon2.d*BondCon2.f;
    end TwoPortZero;

    partial model TwoPortOne
      "Partial model invoking two bondgraphic connectors"
      Real e1 "Bondgraphic primary effort";
      Real f1 "Bondgraphic primary flow";
      Real e2 "Bondgraphic secondary effort";
      Real f2 "Bondgraphic secondary flow";
      annotation (Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]), Documentation(info="<html>
This is a partial model invoking two bondgraphic connectors.  It is used only for the description of the TwoPort 1-junction. <p>

<pre>
Potential variables:

 e1:  Bondgraphic effort variable of first connector, normalized positive for flows into the model

 f1:  Bondgraphic flow variable of first connector

 e2:  Bondgraphic effort variable of second connector, normalized positive for flows into the model

 f2:  Bondgraphic flow variable of second connector
</pre>

</html>
"),     Diagram);
      BondLib.Interfaces.BondCon BondCon1 "First bond graph connector"
                                          annotation (extent=[-110, -10; -90,
            10]);
      BondLib.Interfaces.BondCon BondCon2 "Second bond graph connector"
                                          annotation (extent=[90, -10; 110, 10]);
    equation
      e1 = BondCon1.d*BondCon1.e;
      f1 = BondCon1.f;
      e2 = BondCon2.d*BondCon2.e;
      f2 = BondCon2.f;
    end TwoPortOne;

    partial model ModTwoPort
      "Partial model invoking two bondgraphic connectors and one signal connector"

      Real e1 "Bondgraphic primary effort";
      Real f1 "Bondgraphic primary flow";
      Real e2 "Bondgraphic secondary effort";
      Real f2 "Bondgraphic secondary flow";
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Window(
          x=0.45,
          y=0.01,
          width=0.35,
          height=0.49),
        Diagram,
        Icon(Line(points=[-70, -50; 68, -50], style(color=0)), Line(points=[42,
                  -44; 68, -50; 42, -56], style(color=0))),
        Documentation(info="<html>
This is a partial model invoking two bondgraphic connectors and a signal connector.  It is used to describe modulated directed TwoPort elements.  Positive flow is from inflow to outflow.  The direction of positive flow is indicated by an arrow. <p>

<pre>
Potential variables:

 e1:  Bondgraphic effort variable of inflow

 f1:  Bondgraphic flow variable of inflow, normalized positive for flows into the model

 e2:  Bondgraphic effort variable of outflow

 f2:  Bondgraphic flow variable of outflow, normalized positive for flows out of the model

 s:   Real-valued signal input used for modulation
</pre>

</html>
"));
      BondLib.Interfaces.BondCon BondCon1 "Left bond graph connector"
                                          annotation (extent=[-110, -10; -90,
            10]);
      BondLib.Interfaces.BondCon BondCon2 "Right bond graph connector"
                                          annotation (extent=[90, -10; 110, 10]);
      Modelica.Blocks.Interfaces.RealInput s "Signal input"
                                                annotation (extent=[10, -100; -
            10, -80], rotation=90);
    equation
      e1 = BondCon1.e;
      f1 = BondCon1.d*BondCon1.f;
      e2 = BondCon2.e;
      f2 = -BondCon2.d*BondCon2.f;
    end ModTwoPort;

    partial model ThreePortZero
      "Partial model invoking three bondgraphic connectors"
      Real e[3] "Bondgraphic effort vector";
      Real f[3] "Bondgraphic flow vector";
      annotation (Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]), Documentation(info="<html>
This is a partial model invoking three bondgraphic connectors.  It is used only for the description of the ThreePort 0-junction. <p>

<pre>
Potential variables:

 e[3]:  Vector of bondgraphic effort variables of connectors

 f[3]:  Vector of bondgraphic flow variables of connectors, normalized positive for flows into the model
</pre>

</html>
"),     Diagram);
      BondLib.Interfaces.BondCon BondCon1 "First bond graph connector"
                                          annotation (extent=[-110, -10; -90,
            10]);
      BondLib.Interfaces.BondCon BondCon2 "Second bond graph connector"
                                          annotation (extent=[90, -10; 110, 10]);
      BondLib.Interfaces.BondCon BondCon3 "Third bond graph connector"
                                          annotation (extent=[-10, -110; 10, -
            90]);
    equation
      e[1] = BondCon1.e;
      f[1] = BondCon1.d*BondCon1.f;
      e[2] = BondCon2.e;
      f[2] = BondCon2.d*BondCon2.f;
      e[3] = BondCon3.e;
      f[3] = BondCon3.d*BondCon3.f;
    end ThreePortZero;

    partial model ThreePortOne
      "Partial model invoking three bondgraphic connectors"
      Real e[3] "Bondgraphic effort vector";
      Real f[3] "Bondgraphic flow vector";
      annotation (Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]), Documentation(info="<html>
This is a partial model invoking three bondgraphic connectors.  It is used only for the description of the ThreePort 1-junction. <p>

<pre>
Potential variables:

 e[3]:  Vector of bondgraphic effort variables of connectors, normalized positive for flows into the model

 f[3]:  Vector of bondgraphic flow variables of connectors
</pre>

</html>
"),     Diagram);
      BondLib.Interfaces.BondCon BondCon1 "First bond graph connector"
                                          annotation (extent=[-110, -10; -90,
            10]);
      BondLib.Interfaces.BondCon BondCon2 "Second bond graph connector"
                                          annotation (extent=[90, -10; 110, 10]);
      BondLib.Interfaces.BondCon BondCon3 "Third bond graph connector"
                                          annotation (extent=[-10, -110; 10, -
            90]);
    equation
      e[1] = BondCon1.d*BondCon1.e;
      f[1] = BondCon1.f;
      e[2] = BondCon2.d*BondCon2.e;
      f[2] = BondCon2.f;
      e[3] = BondCon3.d*BondCon3.e;
      f[3] = BondCon3.f;
    end ThreePortOne;

    partial model FourPortZero
      "Partial model invoking four bondgraphic connectors"
      Real e[4] "Bondgraphic effort vector";
      Real f[4] "Bondgraphic flow vector";
      annotation (Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]), Documentation(info="<html>
This is a partial model invoking four bondgraphic connectors.  It is used only for the description of the FourPort 0-junction. <p>

<pre>
Potential variables:

 e[4]:  Vector of bondgraphic effort variables of connectors

 f[4]:  Vector of bondgraphic flow variables of connectors, normalized positive for flows into the model
</pre>

</html>
"),     Diagram);
      BondLib.Interfaces.BondCon BondCon1 "First bond graph connector"
                                          annotation (extent=[-110, -10; -90,
            10]);
      BondLib.Interfaces.BondCon BondCon2 "Second bond graph connector"
                                          annotation (extent=[90, -10; 110, 10]);
      BondLib.Interfaces.BondCon BondCon3 "Third bond graph connector"
                                          annotation (extent=[-10, -110; 10, -
            90]);
      BondLib.Interfaces.BondCon BondCon4 "Fourth bond graph connector"
                                          annotation (extent=[-10, 90; 10, 110]);
    equation
      e[1] = BondCon1.e;
      f[1] = BondCon1.d*BondCon1.f;
      e[2] = BondCon2.e;
      f[2] = BondCon2.d*BondCon2.f;
      e[3] = BondCon3.e;
      f[3] = BondCon3.d*BondCon3.f;
      e[4] = BondCon4.e;
      f[4] = BondCon4.d*BondCon4.f;
    end FourPortZero;

    partial model FourPortZero2
      "Partial model invoking up to four bondgraphic connectors"
      Real e[4] "Bondgraphic effort vector";
      Real f[4] "Bondgraphic flow vector";
      BondCon BondCon1 "First bond graph connector"
                                          annotation (extent=[-110,-10; -90,10]);
      BondCon BondCon2 "Second bond graph connector"
                                          annotation (extent=[90,-10; 110,10]);
      BondCon BondCon3 "Third bond graph connector"
                                          annotation (extent=[-10,-110; 10,-90]);
      BondCon BondCon4 "Fourth bond graph connector"
                                          annotation (extent=[-10,90; 10,110]);
    equation
      e[1] = BondCon1.e;
      f[1] = BondCon1.d*BondCon1.f;
      e[2] = BondCon2.e;
      f[2] = BondCon2.d*BondCon2.f;
      e[3] = BondCon3.e;
      f[3] = BondCon3.d*BondCon3.f;
      e[4] = BondCon4.e;
      f[4] = BondCon4.d*BondCon4.f;

      if cardinality(BondCon1) == 0 then
        BondCon1.d = 1;
        f[1] = 0;
      end if;
      if cardinality(BondCon2) == 0 then
        BondCon2.d = 1;
        f[2] = 0;
      end if;
      if cardinality(BondCon3) == 0 then
        BondCon3.d = 1;
        f[3] = 0;
      end if;
      if cardinality(BondCon4) == 0 then
        BondCon4.d = 1;
        f[4] = 0;
      end if;
      annotation (Documentation(info="<html>
This is a partial model invoking four bondgraphic connectors.  It is used only for the description of a 0-junction with up to four bonds connecting to it. <p>

<pre>
Potential variables:

 e[4]:  Vector of bondgraphic effort variables of connectors

 f[4]:  Vector of bondgraphic flow variables of connectors, normalized positive for flows into the model
</pre>

</html>"), Diagram);
    end FourPortZero2;

    partial model FourPortOne
      "Partial model invoking four bondgraphic connectors"
      Real e[4] "Bondgraphic effort vector";
      Real f[4] "Bondgraphic flow vector";
      annotation (Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]), Documentation(info="<html>
This is a partial model invoking four bondgraphic connectors.  It is used only for the description of the FourPort 1-junction. <p>

<pre>
Potential variables:

 e[4]:  Vector of bondgraphic effort variables of connectors, normalized positive for flows into the model

 f[4]:  Vector of bondgraphic flow variables of connectors
</pre>

</html>
"),     Diagram);
      BondLib.Interfaces.BondCon BondCon1 "First bond graph connector"
                                          annotation (extent=[-110, -10; -90,
            10]);
      BondLib.Interfaces.BondCon BondCon2 "Second bond graph connector"
                                          annotation (extent=[90, -10; 110, 10]);
      BondLib.Interfaces.BondCon BondCon3 "Third bond graph connector"
                                          annotation (extent=[-10, -110; 10, -
            90]);
      BondLib.Interfaces.BondCon BondCon4 "Fourth bond graph connector"
                                          annotation (extent=[-10, 90; 10, 110]);
    equation
      e[1] = BondCon1.d*BondCon1.e;
      f[1] = BondCon1.f;
      e[2] = BondCon2.d*BondCon2.e;
      f[2] = BondCon2.f;
      e[3] = BondCon3.d*BondCon3.e;
      f[3] = BondCon3.f;
      e[4] = BondCon4.d*BondCon4.e;
      f[4] = BondCon4.f;
    end FourPortOne;

    partial model FourPortOne2
      "Partial model invoking up to four bondgraphic connectors"
      Real e[4] "Bondgraphic effort vector";
      Real f[4] "Bondgraphic flow vector";
      BondCon BondCon1 "First bond graph connector"
                                          annotation (extent=[-110,-10; -90,10]);
      BondCon BondCon2 "Second bond graph connector"
                                          annotation (extent=[90,-10; 110,10]);
      BondCon BondCon3 "Third bond graph connector"
                                          annotation (extent=[-10,-110; 10,-90]);
      BondCon BondCon4 "Fourth bond graph connector"
                                          annotation (extent=[-10,90; 10,110]);
    equation
      e[1] = BondCon1.d*BondCon1.e;
      f[1] = BondCon1.f;
      e[2] = BondCon2.d*BondCon2.e;
      f[2] = BondCon2.f;
      e[3] = BondCon3.d*BondCon3.e;
      f[3] = BondCon3.f;
      e[4] = BondCon4.d*BondCon4.e;
      f[4] = BondCon4.f;

      if cardinality(BondCon1) == 0 then
        BondCon1.d = 1;
        e[1] = 0;
      end if;
      if cardinality(BondCon2) == 0 then
        BondCon2.d = 1;
        e[2] = 0;
      end if;
      if cardinality(BondCon3) == 0 then
        BondCon3.d = 1;
        e[3] = 0;
      end if;
      if cardinality(BondCon4) == 0 then
        BondCon4.d = 1;
        e[4] = 0;
      end if;
      annotation (Documentation(info="<html>
This is a partial model invoking four bondgraphic connectors.  It is used only for the description of a 1-junction with up to four bonds connecting to it. <p>

<pre>
Potential variables:

 e[4]:  Vector of bondgraphic effort variables of connectors, normalized positive for flows into the model

 f[4]:  Vector of bondgraphic flow variables of connectors
</pre>

</html>"), Diagram);
    end FourPortOne2;

    partial model FivePortZero
      "Partial model invoking five bondgraphic connectors"
      Real e[5] "Bondgraphic effort vector";
      Real f[5] "Bondgraphic flow vector";
      annotation (Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]), Documentation(info="<html>
This is a partial model invoking five bondgraphic connectors.  It is used only for the description of the FivePort 0-junction. <p>

<pre>
Potential variables:

 e[5]:  Vector of bondgraphic effort variables of connectors

 f[5]:  Vector of bondgraphic flow variables of connectors, normalized positive for flows into the model
</pre>

</html>
"),     Diagram);
      BondLib.Interfaces.BondCon BondCon1 "First bond graph connector"
                                          annotation (extent=[-110, -10; -90,
            10]);
      BondLib.Interfaces.BondCon BondCon2 "Second bond graph connector"
                                          annotation (extent=[90, -10; 110, 10]);
      BondLib.Interfaces.BondCon BondCon3 "Third bond graph connector"
                                          annotation (extent=[-50, -110; -30, -
            90]);
      BondLib.Interfaces.BondCon BondCon4 "Fourth bond graph connector"
                                          annotation (extent=[30, -110; 50, -90]);
      BondLib.Interfaces.BondCon BondCon5 "Fifth bond graph connector"
                                          annotation (extent=[-10, 90; 10, 110]);
    equation
      e[1] = BondCon1.e;
      f[1] = BondCon1.d*BondCon1.f;
      e[2] = BondCon2.e;
      f[2] = BondCon2.d*BondCon2.f;
      e[3] = BondCon3.e;
      f[3] = BondCon3.d*BondCon3.f;
      e[4] = BondCon4.e;
      f[4] = BondCon4.d*BondCon4.f;
      e[5] = BondCon5.e;
      f[5] = BondCon5.d*BondCon5.f;
    end FivePortZero;

    partial model FivePortOne
      "Partial model invoking five bondgraphic connectors"
      Real e[5] "Bondgraphic effort vector";
      Real f[5] "Bondgraphic flow vector";
      annotation (Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]), Documentation(info="<html>
This is a partial model invoking five bondgraphic connectors.  It is used only for the description of the FivePort 1-junction. <p>

<pre>
Potential variables:

 e[5]:  Vector of bondgraphic effort variables of connectors, normalized positive for flows into the model

 f[5]:  Vector of bondgraphic flow variables of connectors
</pre>

</html>
"),     Diagram);
      BondLib.Interfaces.BondCon BondCon1 "First bond graph connector"
                                          annotation (extent=[-110, -10; -90,
            10]);
      BondLib.Interfaces.BondCon BondCon2 "Second bond graph connector"
                                          annotation (extent=[90, -10; 110, 10]);
      BondLib.Interfaces.BondCon BondCon3 "Third bond graph connector"
                                          annotation (extent=[-50, -110; -30, -
            90]);
      BondLib.Interfaces.BondCon BondCon4 "Fourth bond graph connector"
                                          annotation (extent=[30, -110; 50, -90]);
      BondLib.Interfaces.BondCon BondCon5 "Fifth bond graph connector"
                                          annotation (extent=[-10, 90; 10, 110]);
    equation
      e[1] = BondCon1.d*BondCon1.e;
      f[1] = BondCon1.f;
      e[2] = BondCon2.d*BondCon2.e;
      f[2] = BondCon2.f;
      e[3] = BondCon3.d*BondCon3.e;
      f[3] = BondCon3.f;
      e[4] = BondCon4.d*BondCon4.e;
      f[4] = BondCon4.f;
      e[5] = BondCon5.d*BondCon5.e;
      f[5] = BondCon5.f;
    end FivePortOne;

    partial model SixPortZero
      "Partial model invoking six bondgraphic connectors"
      Real e[6] "Bondgraphic effort vector";
      Real f[6] "Bondgraphic flow vector";
      annotation (Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]), Documentation(info="<html>
This is a partial model invoking six bondgraphic connectors.  It is used only for the description of the SixPort 0-junction. <p>

<pre>
Potential variables:

 e[6]:  Vector of bondgraphic effort variables of connectors

 f[6]:  Vector of bondgraphic flow variables of connectors, normalized positive for flows into the model
</pre>

</html>
"),     Diagram);
      BondLib.Interfaces.BondCon BondCon1 "First bond graph connector"
                                          annotation (extent=[-110, -10; -90,
            10]);
      BondLib.Interfaces.BondCon BondCon2 "Second bond graph connector"
                                          annotation (extent=[90, -10; 110, 10]);
      BondLib.Interfaces.BondCon BondCon3 "Third bond graph connector"
                                          annotation (extent=[-50, -110; -30, -
            90]);
      BondLib.Interfaces.BondCon BondCon4 "Fourth bond graph connector"
                                          annotation (extent=[30, -110; 50, -90]);
      BondLib.Interfaces.BondCon BondCon5 "Fifth bond graph connector"
                                          annotation (extent=[-50, 90; -30, 110]);
      BondLib.Interfaces.BondCon BondCon6 "Sixth bond graph connector"
                                          annotation (extent=[30, 90; 50, 110]);
    equation
      e[1] = BondCon1.e;
      f[1] = BondCon1.d*BondCon1.f;
      e[2] = BondCon2.e;
      f[2] = BondCon2.d*BondCon2.f;
      e[3] = BondCon3.e;
      f[3] = BondCon3.d*BondCon3.f;
      e[4] = BondCon4.e;
      f[4] = BondCon4.d*BondCon4.f;
      e[5] = BondCon5.e;
      f[5] = BondCon5.d*BondCon5.f;
      e[6] = BondCon6.e;
      f[6] = BondCon6.d*BondCon6.f;
    end SixPortZero;

    partial model SixPortOne
      "Partial model invoking six bondgraphic connectors"
      Real e[6] "Bondgraphic effort vector";
      Real f[6] "Bondgraphic flow vector";
      annotation (Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]), Documentation(info="<html>
This is a partial model invoking six bondgraphic connectors.  It is used only for the description of the SixPort 1-junction. <p>

<pre>
Potential variables:

 e[6]:  Vector of bondgraphic effort variables of connectors, normalized positive for flows into the model

 f[6]:  Vector of bondgraphic flow variables of connectors
</pre>

</html>
"),     Diagram);
      BondLib.Interfaces.BondCon BondCon1 "First bond graph connector"
                                          annotation (extent=[-110, -10; -90,
            10]);
      BondLib.Interfaces.BondCon BondCon2 "Second bond graph connector"
                                          annotation (extent=[90, -10; 110, 10]);
      BondLib.Interfaces.BondCon BondCon3 "Third bond graph connector"
                                          annotation (extent=[-50, -110; -30, -
            90]);
      BondLib.Interfaces.BondCon BondCon4 "Fourth bond graph connector"
                                          annotation (extent=[30, -110; 50, -90]);
      BondLib.Interfaces.BondCon BondCon5 "Fifth bond graph connector"
                                          annotation (extent=[-50, 90; -30, 110]);
      BondLib.Interfaces.BondCon BondCon6 "Sixth bond graph connector"
                                          annotation (extent=[30, 90; 50, 110]);
    equation
      e[1] = BondCon1.d*BondCon1.e;
      f[1] = BondCon1.f;
      e[2] = BondCon2.d*BondCon2.e;
      f[2] = BondCon2.f;
      e[3] = BondCon3.d*BondCon3.e;
      f[3] = BondCon3.f;
      e[4] = BondCon4.d*BondCon4.e;
      f[4] = BondCon4.f;
      e[5] = BondCon5.d*BondCon5.e;
      f[5] = BondCon5.f;
      e[6] = BondCon6.d*BondCon6.e;
      f[6] = BondCon6.f;
    end SixPortOne;

    partial model EightPortZero
      "Partial model invoking up to eight bondgraphic connectors"
      Real e[8] "Bondgraphic effort vector";
      Real f[8] "Bondgraphic flow vector";
      annotation (
        Coordsys(extent=[-200,-200; 200,200]),
        Diagram,
        Documentation(info="<html>
This is a partial model invoking eight bondgraphic connectors.  It is used only for the description of a 0-junction with up to eight bonds connecting to it. <p>

<pre>
Potential variables:

 e[8]:  Vector of bondgraphic effort variables of connectors

 f[8]:  Vector of bondgraphic flow variables of connectors, normalized positive for flows into the model
</pre>

</html>"));
      BondCon BondCon1 "First bond graph connector"
                       annotation (extent=[-210,-10; -190,10]);
      BondCon BondCon2 "Second bond graph connector"
                       annotation (extent=[-110,90; -90,110]);
      BondCon BondCon3 "Third bond graph connector"
                       annotation (extent=[-10,190; 10,210]);
      BondCon BondCon4 "Fourth bond graph connector"
                       annotation (extent=[90,90; 110,110]);
      BondCon BondCon5 "Fifth bond graph connector"
                       annotation (extent=[190,-10; 210,10]);
      BondCon BondCon6 "Sixth bond graph connector"
                       annotation (extent=[90,-110; 110,-90]);
      BondCon BondCon7 "Seventh bond graph connector"
                       annotation (extent=[-10,-210; 10,-190]);
      BondCon BondCon8 "Eighth bond graph connector"
                       annotation (extent=[-110,-110; -90,-90]);
    equation
      e[1] = BondCon1.e;
      f[1] = BondCon1.d*BondCon1.f;
      e[2] = BondCon2.e;
      f[2] = BondCon2.d*BondCon2.f;
      e[3] = BondCon3.e;
      f[3] = BondCon3.d*BondCon3.f;
      e[4] = BondCon4.e;
      f[4] = BondCon4.d*BondCon4.f;
      e[5] = BondCon5.e;
      f[5] = BondCon5.d*BondCon5.f;
      e[6] = BondCon6.e;
      f[6] = BondCon6.d*BondCon6.f;
      e[7] = BondCon7.e;
      f[7] = BondCon7.d*BondCon7.f;
      e[8] = BondCon8.e;
      f[8] = BondCon8.d*BondCon8.f;

      if cardinality(BondCon1) == 0 then
        BondCon1.d = 1;
        f[1] = 0;
      end if;
      if cardinality(BondCon2) == 0 then
        BondCon2.d = 1;
        f[2] = 0;
      end if;
      if cardinality(BondCon3) == 0 then
        BondCon3.d = 1;
        f[3] = 0;
      end if;
      if cardinality(BondCon4) == 0 then
        BondCon4.d = 1;
        f[4] = 0;
      end if;
      if cardinality(BondCon5) == 0 then
        BondCon5.d = 1;
        f[5] = 0;
      end if;
      if cardinality(BondCon6) == 0 then
        BondCon6.d = 1;
        f[6] = 0;
      end if;
      if cardinality(BondCon7) == 0 then
        BondCon7.d = 1;
        f[7] = 0;
      end if;
      if cardinality(BondCon8) == 0 then
        BondCon8.d = 1;
        f[8] = 0;
      end if;
    end EightPortZero;

    partial model EightPortOne
      "Partial model invoking up to eight bondgraphic connectors"
      Real e[8] "Bondgraphic effort vector";
      Real f[8] "Bondgraphic flow vector";
      annotation (Coordsys(extent=[-200,-200; 200,200]), Documentation(info="<html>
This is a partial model invoking eight bondgraphic connectors.  It is used only for the description of a 1-junction with up to eight bonds connecting to it. <p>

<pre>
Potential variables:

 e[8]:  Vector of bondgraphic effort variables of connectors, normalized positive for flows into the model

 f[8]:  Vector of bondgraphic flow variables of connectors
</pre>

</html>"),
        Diagram);
      BondCon BondCon1 "First bond graph connector"
                       annotation (extent=[-210,-10; -190,10]);
      BondCon BondCon2 "Second bond graph connector"
                       annotation (extent=[-110,90; -90,110]);
      BondCon BondCon3 "Third bond graph connector"
                       annotation (extent=[-10,190; 10,210]);
      BondCon BondCon4 "Fourth bond graph connector"
                       annotation (extent=[90,90; 110,110]);
      BondCon BondCon5 "Fifth bond graph connector"
                       annotation (extent=[190,-10; 210,10]);
      BondCon BondCon6 "Sixth bond graph connector"
                       annotation (extent=[90,-110; 110,-90]);
      BondCon BondCon7 "Seventh bond graph connector"
                       annotation (extent=[-10,-210; 10,-190]);
      BondCon BondCon8 "Eighth bond graph connector"
                       annotation (extent=[-110,-110; -90,-90]);
    equation
      e[1] = BondCon1.d*BondCon1.e;
      f[1] = BondCon1.f;
      e[2] = BondCon2.d*BondCon2.e;
      f[2] = BondCon2.f;
      e[3] = BondCon3.d*BondCon3.e;
      f[3] = BondCon3.f;
      e[4] = BondCon4.d*BondCon4.e;
      f[4] = BondCon4.f;
      e[5] = BondCon5.d*BondCon5.e;
      f[5] = BondCon5.f;
      e[6] = BondCon6.d*BondCon6.e;
      f[6] = BondCon6.f;
      e[7] = BondCon7.d*BondCon7.e;
      f[7] = BondCon7.f;
      e[8] = BondCon8.d*BondCon8.e;
      f[8] = BondCon8.f;

      if cardinality(BondCon1) == 0 then
        BondCon1.d = 1;
        e[1] = 0;
      end if;
      if cardinality(BondCon2) == 0 then
        BondCon2.d = 1;
        e[2] = 0;
      end if;
      if cardinality(BondCon3) == 0 then
        BondCon3.d = 1;
        e[3] = 0;
      end if;
      if cardinality(BondCon4) == 0 then
        BondCon4.d = 1;
        e[4] = 0;
      end if;
      if cardinality(BondCon5) == 0 then
        BondCon5.d = 1;
        e[5] = 0;
      end if;
      if cardinality(BondCon6) == 0 then
        BondCon6.d = 1;
        e[6] = 0;
      end if;
      if cardinality(BondCon7) == 0 then
        BondCon7.d = 1;
        e[7] = 0;
      end if;
      if cardinality(BondCon8) == 0 then
        BondCon8.d = 1;
        e[8] = 0;
      end if;
    end EightPortOne;
  end Interfaces;

  package Bonds "Bonds of Modelica bond graph library"
    annotation (
      preferedView="info",
      Icon(
        Rectangle(extent=[-100, -100; 80, 50], style(fillColor=30, fillPattern=
                1)),
        Polygon(points=[-100, 50; -80, 70; 100, 70; 80, 50; -100, 50], style(
              fillColor=30, fillPattern=1)),
        Polygon(points=[100, 70; 100, -80; 80, -100; 80, 50; 100, 70], style(
              fillColor=30, fillPattern=1)),
        Text(
          extent=[-85, 35; 65, -85],
          string="Library",
          style(color=3)),
        Text(
          extent=[-120, 122; 120, 73],
          string="%name",
          style(color=1))),
      Window(
        x=0.55,
        y=0.46,
        width=0.35,
        height=0.49,
        library=1,
        autolayout=1),
      Documentation(info="<html>
This package contains models of the three bonds of the Modelica bond graph library.  The <b>Bond</b> model describes the a-causal bonds, whereas the <b>eBond</b> and <b>fBond</b> blocks describe the two causal bonds. <p>

Although it is possible to work with a-causal bonds throughout, it is recommended to use causal bonds whenever possible. <p>

Bonds must be connected to either elements or junctions.  It is illegal to connect two bonds in series.
</html>
"));
    model Bond "The a-causal bond model of the bond graph library"
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(
          Line(points=[-80, 0; 80, 0], style(color=0, thickness=2)),
          Line(points=[80, 0; 40, 40], style(color=0, thickness=2)),
          Text(
            extent=[-40, -8; 20, -40],
            string="f",
            style(color=0)),
          Text(
            extent=[-40, 40; 18, 10],
            string="e",
            style(color=0))),
        Icon(
          Line(points=[-80, 0; 80, 0], style(color=0, thickness=2)),
          Line(points=[80, 0; 40, 40], style(color=0, thickness=2)),
          Text(
            extent=[-70, -4; 72, -42],
            string="%name",
            style(color=0))),
        Window(
          x=0.45,
          y=0.01,
          width=0.35,
          height=0.49),
        Documentation(info="<html>
This is the model of the general a-causal bond of the Modelica bond graph library. <p>

The model invokes two general bond connectors.  It does so directly, i.e., without inheriting them from one of the TwoPort partial models. <p>

The Model passes the effort and flow variables from the inflow to the outflow, and it sets the directional variable <b>d</b> to <b>-1</b> at the inflow, and to <b>+1</b> at the outflow.
</html>
"));
      Interfaces.BondCon BondCon1 "Left bond graph connector"
                                  annotation (extent=[-108, -10; -88, 10]);
      Interfaces.BondCon BondCon2 "Right bond graph connector"
                                  annotation (extent=[90, -10; 110, 10]);
    equation
      BondCon2.e = BondCon1.e;
      BondCon2.f = BondCon1.f;
      BondCon1.d = -1;
      BondCon2.d = +1;
    end Bond;

    block eBond "One of two causal bond models of the bond graph library"
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Icon(
          Line(points=[-80, 0; 80, 0], style(color=0, thickness=2)),
          Line(points=[80, 0; 40, 40], style(color=0, thickness=2)),
          Line(points=[-80, 40; -80, -40], style(color=0, thickness=2)),
          Text(
            extent=[-68, -4; 64, -42],
            string="%name",
            style(color=0))),
        Diagram(
          Line(points=[-80, 0; 80, 0], style(color=0, thickness=2)),
          Line(points=[80, 0; 40, 40], style(color=0, thickness=2)),
          Line(points=[-80, 40; -80, -40], style(color=0, thickness=2)),
          Text(
            extent=[-38, 42; 20, 12],
            string="e",
            style(color=0)),
          Text(
            extent=[-40, -8; 20, -40],
            string="f",
            style(color=0)),
          Line(points=[4, -24; 34, -24], style(
              color=0,
              thickness=2,
              arrow=1)),
          Line(points=[-24, 24; -56, 24], style(
              color=0,
              thickness=2,
              arrow=1))),
        Window(
          x=0.45,
          y=0.01,
          width=0.35,
          height=0.49),
        Documentation(info="<html>
This is the model of one of two causal bonds of the Modelica bond graph library. <p>

The model invokes one <b>fBond</b> and one <b>eBond</b> connector.  The <b>fBond</b> connector is used at the inflow, i.e., where the flow variable is being computed, whereas the <b>eBond</b> connector is used at the outflow, i.e., where the effort variable is being computed. Thus, the causality stroke of this causal bond is at the inflow. <p>

The model passes the effort and flow variables from the inflow to the outflow, and it sets the directional variable <b>d</b> to <b>-1</b> at the inflow, and to <b>+1</b> at the outflow.
</html>
"));
      Interfaces.fBondCon fBondCon1 "Left bond graph connector"
                                    annotation (extent=[-110, -10; -90, 10]);
      Interfaces.eBondCon eBondCon1 "Right bond graph connector"
                                    annotation (extent=[90, -10; 110, 10]);
    equation
      fBondCon1.e = eBondCon1.e;
      eBondCon1.f = fBondCon1.f;
      fBondCon1.d = -1;
      eBondCon1.d = +1;
    end eBond;

    block fBond "One of two causal bond models of the bond graph library"
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(
          Line(points=[-80, 0; 80, 0], style(color=0, thickness=2)),
          Text(
            extent=[-40, -8; 20, -40],
            string="f",
            style(color=0)),
          Text(
            extent=[-38, 42; 20, 12],
            string="e",
            style(color=0)),
          Line(points=[80, 0; 40, 40], style(color=0, thickness=2)),
          Line(points=[80, 40; 80, -40], style(color=0, thickness=2)),
          Line(points=[-22, -24; -54, -24], style(
              color=0,
              thickness=2,
              arrow=1)),
          Line(points=[2, 26; 32, 26], style(
              color=0,
              thickness=2,
              arrow=1))),
        Icon(
          Line(points=[-80, 0; 80, 0], style(color=0, thickness=2)),
          Line(points=[80, 0; 40, 40], style(color=0, thickness=2)),
          Line(points=[80, 40; 80, -40], style(color=0, thickness=2)),
          Text(
            extent=[-66, -4; 62, -44],
            string="%name",
            style(color=0))),
        Window(
          x=0.52,
          y=0.42,
          width=0.35,
          height=0.49),
        Documentation(info="<html>
This is the model of one of two causal bonds of the Modelica bond graph library. <p>

The model invokes one <b>eBond</b> and one <b>fBond</b> connector.  The <b>eBond</b> connector is used at the inflow, i.e., where the effort variable is being computed, whereas the <b>fBond</b> connector is used at the outflow, i.e., where the flow variable is being computed. Thus, the causality stroke of this causal bond is at the outflow. <p>

The model passes the effort and flow variables from the inflow to the outflow, and it sets the directional variable <b>d</b> to <b>-1</b> at the inflow, and to <b>+1</b> at the outflow.
</html>
"));
      Interfaces.eBondCon eBondCon1 "Left bond graph connector"
                                    annotation (extent=[-110, -10; -90, 10]);
      Interfaces.fBondCon fBondCon1 "Right bond graph connector"
                                    annotation (extent=[90, -10; 110, 10]);
    equation
      fBondCon1.e = eBondCon1.e;
      eBondCon1.f = fBondCon1.f;
      eBondCon1.d = -1;
      fBondCon1.d = +1;
    end fBond;
  end Bonds;

  package Junctions "Junctions of Modelica bond graph library"
    annotation (
      preferedView="info",
      Icon(
        Rectangle(extent=[-100, -100; 80, 50], style(fillColor=30, fillPattern=
                1)),
        Polygon(points=[-100, 50; -80, 70; 100, 70; 80, 50; -100, 50], style(
              fillColor=30, fillPattern=1)),
        Polygon(points=[100, 70; 100, -80; 80, -100; 80, 50; 100, 70], style(
              fillColor=30, fillPattern=1)),
        Text(
          extent=[-85, 35; 65, -85],
          string="Library",
          style(color=3)),
        Text(
          extent=[-120, 122; 120, 73],
          string="%name",
          style(color=1))),
      Window(
        x=0.49,
        y=0.48,
        width=0.35,
        height=0.49,
        library=1,
        autolayout=1),
      Documentation(info="<html>
This package contains models of the different junctions of the Modelica bond graph library.  The library distinguishes explicitly between junctions with 2, 3, 4, 5, and 6 connectors.  If more than 6 connectors are needed, the junction needs to be split.  Since neighboring same-sex junctions can always be treated logically like a single junction, it is possible to construct junctions with arbitrarily many connectors.<p>

Junctions must be connected to bonds.  It is illegal to connect junctions to either modeling elements or other junctions directly.  If junctions are being split, a bond must be placed between them.
</html>
"));
    model J0p2 "Model of TwoPort 0-junction"
      extends Interfaces.TwoPortZero;
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Text(
            extent=[-60, 80; 60, -80],
            string="0",
            style(color=0))),
        Icon(Text(
            extent=[-60, 80; 60, -80],
            string="0",
            style(color=0))),
        Documentation(info="<html>
This is the model of the TwoPort 0-junction.  It has one causality stroke.
</html>
"));
    equation
      e2 = e1;
      f1 + f2 = 0;
    end J0p2;

    model J0p3 "Model of ThreePort 0-junction"
      extends Interfaces.ThreePortZero;
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Text(
            extent=[-60, 80; 60, -80],
            string="0",
            style(color=0))),
        Icon(Text(
            extent=[-60, 80; 60, -80],
            string="0",
            style(color=0))),
        Documentation(info="<html>
This is the model of the ThreePort 0-junction.  It has one causality stroke.
</html>
"));
    equation
      e[2:3] = e[1:2];
      sum(f) = 0;
    end J0p3;

    model J0p4 "Model of FourPort 0-junction"
      extends Interfaces.FourPortZero;
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Text(
            extent=[-60, 80; 60, -80],
            string="0",
            style(color=0))),
        Icon(Text(
            extent=[-60, 80; 60, -80],
            string="0",
            style(color=0))),
        Documentation(info="<html>
This is the model of the FourPort 0-junction.  It has one causality stroke.
</html>
"));
    equation
      e[2:4] = e[1:3];
      sum(f) = 0;
    end J0p4;

    model J0p5 "Model of FivePort 0-junction"
      extends Interfaces.FivePortZero;
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Text(
            extent=[-60, 80; 60, -80],
            string="0",
            style(color=0))),
        Icon(Text(
            extent=[-60, 80; 60, -80],
            string="0",
            style(color=0))),
        Documentation(info="<html>
This is the model of the FivePort 0-junction.  It has one causality stroke.
</html>
"));
    equation
      e[2:5] = e[1:4];
      sum(f) = 0;
    end J0p5;

    model J0p6 "Model of SixPort 0-junction"
      extends Interfaces.SixPortZero;
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Text(
            extent=[-60, 80; 60, -80],
            string="0",
            style(color=0))),
        Icon(Text(
            extent=[-60, 80; 60, -80],
            string="0",
            style(color=0))),
        Documentation(info="<html>
This is the model of the SixPort 0-junction.  It has one causality stroke.
</html>
"));
    equation
      e[2:6] = e[1:5];
      sum(f) = 0;
    end J0p6;

    model J0 "Model of a standard 0-junction with up to four ports"
      extends Interfaces.FourPortZero2;
      annotation (
        Diagram(Text(
            extent=[-60,80; 60,-80],
            string="0",
            style(color=0))),
        Icon(   Text(
            extent=[-60,80; 60,-80],
            string="0",
            style(color=0))),
        Documentation(info="<html>
This is the model of the generalized FourPort 0-junction.  It has one causality stroke. <p>

Contrary to the regular FourPoint 0-junction (<b>J0p4</b>), this model allows connectors to remain unconnected externally.  Hence the junction can be used for up to four bond connections.
</html>"));
    equation
      e[2:4] = e[1:3];
      sum(f) = 0;
    end J0;

    model J0large "Model of a large 0-junction with up to eight ports"
      extends Interfaces.EightPortZero;
      annotation (
        Icon(
          Line(points=[-102,0; -194,0], style(color=10, rgbcolor={135,135,135})),
          Line(points=[0,100; 0,194], style(color=10, rgbcolor={135,135,135})),
          Line(points=[104,0; 194,0], style(color=10, rgbcolor={135,135,135})),
          Line(points=[0,-104; 0,-194], style(color=10, rgbcolor={135,135,135})),
          Line(points=[-70,70; -96,96], style(color=10, rgbcolor={135,135,135})),
          Line(points=[70,70; 96,96], style(color=10, rgbcolor={135,135,135})),
          Line(points=[70,-70; 96,-96], style(color=10, rgbcolor={135,135,135})),
          Line(points=[-70,-70; -96,-96], style(color=10, rgbcolor={135,135,135})),
          Ellipse(extent=[120,120; -122,-120], style(
              color=10,
              rgbcolor={135,135,135},
              fillColor=7,
              rgbfillColor={255,255,255},
              fillPattern=1)),
             Text(
            extent=[-140,140; 140,-140],
            string="0",
            style(color=0, rgbcolor={0,0,0}))),
        Diagram(
             Text(
            extent=[-140,140; 140,-140],
            string="0",
            style(color=0, rgbcolor={0,0,0}))),
        Documentation(info="<html>
This is the model of the EightPort 0-junction.  It has one causality stroke. <p>

This model allows connectors to remain unconnected externally.  Hence the junction can be used for up to eight bond connections.
</html>"));
    equation
      e[2:8] = e[1:7];
      sum(f) = 0;
    end J0large;

    model J1p2 "Model of TwoPort 1-junction"
      extends Interfaces.TwoPortOne;
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Text(
            extent=[-60, 80; 60, -80],
            string="1",
            style(color=0))),
        Icon(Text(
            extent=[-60, 80; 60, -80],
            string="1",
            style(color=0))),
        Documentation(info="<html>
This is the model of the TwoPort 1-junction.  It has one causality stroke.
</html>
"));
    equation
      f2 = f1;
      e1 + e2 = 0;
    end J1p2;

    model J1p3 "Model of ThreePort 1-junction"
      extends Interfaces.ThreePortOne;
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Text(
            extent=[-60, 80; 60, -80],
            string="1",
            style(color=0))),
        Icon(Text(
            extent=[-60, 80; 60, -80],
            string="1",
            style(color=0))),
        Documentation(info="<html>
This is the model of the ThreePort 1-junction.  It has two causality strokes.
</html>
"));
    equation
      f[2:3] = f[1:2];
      sum(e) = 0;
    end J1p3;

    model J1p4 "Model of FourPort 1-junction"
      extends Interfaces.FourPortOne;
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Text(
            extent=[-60, 80; 60, -80],
            string="1",
            style(color=0))),
        Icon(Text(
            extent=[-60, 80; 60, -80],
            string="1",
            style(color=0))),
        Documentation(info="<html>
This is the model of the FourPort 1-junction.  It has three causality strokes.
</html>
"));
    equation
      f[2:4] = f[1:3];
      sum(e) = 0;
    end J1p4;

    model J1p5 "Model of FivePort 1-junction"
      extends Interfaces.FivePortOne;
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Text(
            extent=[-60, 80; 60, -80],
            string="1",
            style(color=0))),
        Icon(Text(
            extent=[-60, 80; 60, -80],
            string="1",
            style(color=0))),
        Documentation(info="<html>
This is the model of the FivePort 1-junction.  It has four causality strokes.
</html>
"));
    equation
      f[2:5] = f[1:4];
      sum(e) = 0;
    end J1p5;

    model J1p6 "Model of SixPort 1-junction"
      extends Interfaces.SixPortOne;
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Text(
            extent=[-60, 80; 60, -80],
            string="1",
            style(color=0))),
        Icon(Text(
            extent=[-60, 80; 60, -80],
            string="1",
            style(color=0))),
        Documentation(info="<html>
This is the model of the SixPort 1-junction.  It has two causality strokes.
</html>
"));
    equation
      f[2:6] = f[1:5];
      sum(e) = 0;
    end J1p6;

    model J1 "Model of a standard 1-junction with up to four ports"
      extends Interfaces.FourPortZero2;
      annotation (
        Diagram(Text(
            extent=[-60,80; 60,-80],
            string="1",
            style(color=0))),
        Icon(   Text(
            extent=[-60,80; 60,-80],
            string="1",
            style(color=0))),
        Documentation(info="<html>
This is the model of the generalized FourPort 1-junction. <p>

Contrary to the regular FourPoint 1-junction (<b>J1p4</b>), this model allows connectors to remain unconnected externally.  Hence the junction can be used for up to four bond connections.
</html>"));
    equation
      f[2:4] = f[1:3];
      sum(e) = 0;
    end J1;

    model J1large "Model of a large 1-junction with up to eight ports"
      extends Interfaces.EightPortOne;
      annotation (
        Icon(
          Line(points=[-102,0; -194,0], style(color=10, rgbcolor={135,135,135})),
          Line(points=[0,100; 0,194], style(color=10, rgbcolor={135,135,135})),
          Line(points=[104,0; 194,0], style(color=10, rgbcolor={135,135,135})),
          Line(points=[0,-104; 0,-194], style(color=10, rgbcolor={135,135,135})),
          Line(points=[-70,70; -96,96], style(color=10, rgbcolor={135,135,135})),
          Line(points=[70,70; 96,96], style(color=10, rgbcolor={135,135,135})),
          Line(points=[70,-70; 96,-96], style(color=10, rgbcolor={135,135,135})),
          Line(points=[-70,-70; -96,-96], style(color=10, rgbcolor={135,135,135})),
          Ellipse(extent=[120,120; -122,-120], style(
              color=10,
              rgbcolor={135,135,135},
              fillColor=7,
              rgbfillColor={255,255,255},
              fillPattern=1)),
             Text(
            extent=[-140,140; 140,-140],
            style(color=0, rgbcolor={0,0,0}),
            string="1")),
        Diagram(
             Text(
            extent=[-140,140; 140,-140],
            style(color=0, rgbcolor={0,0,0}),
            string="1")),
        Documentation(info="<html>
This is the model of the EightPort 1-junction. <p>

This model allows connectors to remain unconnected externally.  Hence the junction can be used for up to eight bond connections.
</html>"));
    equation
      f[2:8] = f[1:7];
      sum(e) = 0;
    end J1large;
  end Junctions;

  package Passive "Passive elements of Modelica bond graph library"
    annotation (
      preferedView="info",
      Icon(
        Rectangle(extent=[-100, -100; 80, 50], style(fillColor=30, fillPattern=
                1)),
        Polygon(points=[-100, 50; -80, 70; 100, 70; 80, 50; -100, 50], style(
              fillColor=30, fillPattern=1)),
        Polygon(points=[100, 70; 100, -80; 80, -100; 80, 50; 100, 70], style(
              fillColor=30, fillPattern=1)),
        Text(
          extent=[-85, 35; 65, -85],
          string="Library",
          style(color=3)),
        Text(
          extent=[-120, 122; 120, 73],
          string="%name",
          style(color=1))),
      Window(
        x=0.48,
        y=0.48,
        width=0.35,
        height=0.49,
        library=1,
        autolayout=1),
      Documentation(info="<html>
The basic passive modeling elements used in the bond graph methodology are: <p>

<ul>
<li> the resistors (<b>R</b>),
<li> the capacitors (<b>C</b>),
<li> the inductors (<b>I</b>),
<li> the transformers (<b>TF</b>), and
<li> the gyrators (<b>GY</b>).
</ul> <p>

These elements are collected here.
</html>

"));
    model R "The bondgraphic linear resistor element"
      extends Interfaces.PassiveOnePort;
      parameter Real R=1 "Bondgraphic Resistance";
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Text(
            extent=[-60, 80; 60, -80],
            string="R",
            style(color=0, fillColor=0))),
        Icon(
          Text(extent=[-70, -50; 70, -90], string="R=%R"),
          Text(
            extent=[-60, 80; 60, -80],
            string="R",
            style(color=0, fillColor=0)),
          Text(extent=[-70, 90; 70, 50], string="%name")),
        Documentation(info="<html>
The bondgraphic linear resistor is a passive OnePort element.  It inherits the effort and flow variables from the passive OnePort.  The resistance is modeled as a parameter. <p>

The causality of the resistor is free. <p>

<hr>

<pre>
Potential variables:

 e:  Bondgraphic effort variable

 f:  Bondgraphic flow variable, normalized positive for flows into the model


Parameters:

 R:  Resistance (default value = 1)
</pre> <p>

<hr> <p>

Equations: <p>

<b>e = R*f</b>
</html>
"));
    equation
      e = R*f;
    end R;

    model mR "The bondgraphic modulated resistor element"
      extends Interfaces.ModPassiveOnePort;
      Real R "Bondgraphic Resistance";
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Text(
            extent=[-60, 80; 60, -80],
            string="mR",
            style(color=0))),
        Icon(Text(
            extent=[-58,80; 62,-80],
            string="mR",
            style(color=0)), Text(extent=[-70, 80; 70, 40], string="%name")),
        Documentation(info="<html>
The bondgraphic modulated resistor is a modulated passive OnePort element.  It inherits the effort and flow variables as well as the modulating signal from the modulated passive OnePort.  The resistance is modeled as a real-valued variable. <p>

The causality of the resistor is free. <p>

<hr>

<pre>
Potential variables:

 e:  Bondgraphic effort variable

 f:  Bondgraphic flow variable, normalized positive for flows into the model

 R:  Resistance (modulated input signal)
</pre> <p>

<hr> <p>

Equations: <p>

<b>e = R*f</b>
</html>
"));
    equation
      R = s;
      e = R*f;
    end mR;

    model HydroR "The bondgraphic hydraulic resistor element"
      extends Interfaces.PassiveOnePort;
      parameter Real k(unit="m7/kg") = 1 "Hydraulic Valve Constant";
      Real sf "Sign of f";
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Text(
            extent=[-60, 80; 60, -80],
            string="R",
            style(color=0, fillColor=0)), Text(
            extent=[14, 10; 122, -12],
            string="Hydro",
            style(color=0))),
        Icon(
          Text(
            extent=[-60, 80; 60, -80],
            string="R",
            style(color=0, fillColor=0)),
          Text(
            extent=[14, 10; 122, -12],
            string="Hydro",
            style(color=0)),
          Text(extent=[-70, 90; 70, 50], string="%name"),
          Text(extent=[-70, -50; 70, -90], string="k=%k")),
        Documentation(info="<html>
The bondgraphic hydraulic resistor is a passive OnePort element.  It models the turbulent flow through a hydraulic valve.  The model inherits the effort and flow variables from the passive OnePort.  The valve constant is modeled as a parameter. <p>

This non-linear model is formulated in resistive causality.  There exists a second model, <b>HydroG</b>, that models the same device in conductive causality.  It is useful to make use of the model in its correct causality, as Modelica may be unable to automatically turn the causality of this non-linear equation around. <p>

The hydraulic resistor model has a preferred causality away from the element. <p>

<hr>

<pre>
Potential variables:

 e:  Bondgraphic effort variable

 f:  Bondgraphic flow variable, normalized positive for flows into the model


Parameters:

 k:  Valve constant (default value = 1)
</pre> <p>

<hr> <p>

Equations: <p>

<b>e = sign(f)*f<sup>2</sup>/k</b>
</html>
"));
    equation
      sf = noEvent(if f > 0 then 1 else -1);
      e = sf*f*f/k;
    end HydroR;

    model HydromR "The bondgraphic hydraulic modulated resistor element"
      extends Interfaces.ModPassiveOnePort;
      Real k(unit="m7/kg") "Hydraulic Valve Constant";
      Real sf "Sign of f";
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Text(
            extent=[-60, 80; 60, -80],
            string="mR",
            style(color=0)), Text(
            extent=[-48, -26; 60, -48],
            string="Hydro",
            style(color=0))),
        Icon(
          Text(
            extent=[-60, 80; 60, -80],
            string="mR",
            style(color=0)),
          Text(extent=[-70, 90; 70, 50], string="%name"),
          Text(
            extent=[-48, -26; 60, -48],
            string="Hydro",
            style(color=0))),
        Window(
          x=0.45,
          y=0.01,
          width=0.44,
          height=0.65),
        Documentation(info="<html>
The bondgraphic hydraulic modulated resistor is a modulated passive OnePort element.  It models the turbulent flow through a hydraulic valve.  The model inherits the effort and flow variables as well as the modulation signal from the modulated passive OnePort.  The valve constant is modeled as a real-valued variable. <p>

This non-linear model is formulated in resistive causality.  There exists a second model, <b>HydromG</b>, that models the same device in conductive causality.  It is useful to make use of the model in its correct causality, as Modelica may be unable to automatically turn the causality of this non-linear equation around. <p>

The hydraulic resistor model has a preferred causality away from the element. <p>

<hr>

<pre>
Potential variables:

 e:  Bondgraphic effort variable

 f:  Bondgraphic flow variable, normalized positive for flows into the model

 k:  Valve constant (modulated input signal)
</pre> <p>

<hr> <p>

Equations: <p>

<b>e = sign(f)*f<sup>2</sup>/k</b>
</html>
"));
    equation
      sf = noEvent(if f > 0 then 1 else -1);
      k = s;
      e = sf*f*f/k;
    end HydromR;

    model RF "The bondgraphic linear TwoPort resistive field element"
      extends Interfaces.TwoPort2;
      parameter Real R1=1 "Primary Resistance";
      parameter Real R2=1 "Secondary Resistance";
      parameter Real R12=0 "Coupling Resistance";
      annotation (
        Icon(Text(
            extent=[-62, 80; 58, -80],
            style(color=0),
            string="RF"), Text(extent=[-68, 90; 72, 50], string="%name")),
        Diagram(Text(
            extent=[-62, 80; 58, -80],
            style(color=0),
            string="RF")),
        Documentation(info="<html>
The bondgraphic TwoPort resistive field is a symmetric TwoPort element.  It inherits the effort and flow variables from the symmetric TwoPort.  The resistance values are modeled as a set of three parameters. <p>

The TwoPort resistive field has free causality. <p>

<hr>

<pre>
Potential variables:

 e1:   Bondgraphic effort variable of inflow

 f1:   Bondgraphic flow variable of inflow, normalized positive for flows into the model

 e2:   Bondgraphic effort variable of outflow

 f2:   Bondgraphic flow variable of outflow, normalized positive for flows into the model


Parameters:

 R1:   Primary resistance (default value = 1)

 R2:   Secondary resistance (default value = 1)

 R12:  Coupling resistance (default value = 0)
</pre> <p>

<hr> <p>

Equations: <p>

<b>e1 = R1*f1 + R12*f2</b> <br>
<b>e2 = R12*f1 + R2*f2</b>
</html>
"));
    equation
      e1 = R1*f1 + R12*f2;
      e2 = R12*f1 + R2*f2;
    end RF;

    model G "The bondgraphic linear conductor element"
      extends Interfaces.PassiveOnePort;
      parameter Real G=1 "Bondgraphic Conductance";
      annotation (
        Icon(
          Text(extent=[-68,-52; 72,-92], string="G=%G"),
          Text(
            extent=[-58,78; 62,-82],
            style(color=0, fillColor=0),
            string="G"),
          Text(extent=[-68,88; 72,48],   string="%name")),
        Diagram(
          Text(
            extent=[-58,78; 62,-82],
            style(color=0, fillColor=0),
            string="G")),
        Documentation(info="<html>
The bondgraphic linear conductor is a passive OnePort element.  It inherits the effort and flow variables from the passive OnePort.  The conductance is modeled as a parameter. <p>

The causality of the conductor is free. <p>

<hr>

<pre>
Potential variables:

 e:  Bondgraphic effort variable

 f:  Bondgraphic flow variable, normalized positive for flows into the model


Parameters:

 G:  Conductance (default value = 1)
</pre> <p>

<hr> <p>

Equations: <p>

<b>f = G*e</b>
</html>"));
    equation
      f = G*e;
    end G;

    model mG "The bondgraphic modulated rconductor element"
      extends Interfaces.ModPassiveOnePort;
      Real G "Bondgraphic Conductance";
      annotation (
        Icon(Text(
            extent=[-62,78; 58,-82],
            style(color=0),
            string="mG"),    Text(extent=[-70,80; 70,40],   string="%name")),
        Diagram(
             Text(
            extent=[-62,78; 58,-82],
            style(color=0),
            string="mG")),
        Documentation(info="<html>
The bondgraphic modulated conductor is a modulated passive OnePort element.  It inherits the effort and flow variables as well as the modulating signal from the modulated passive OnePort.  The conductance is modeled as a real-valued variable. <p>

The causality of the conductor is free. <p>

<hr>

<pre>
Potential variables:

 e:  Bondgraphic effort variable

 f:  Bondgraphic flow variable, normalized positive for flows into the model

 G:  Conductance (modulated input signal)
</pre> <p>

<hr> <p>

Equations: <p>

<b>f = G*e</b>
</html>"));
    equation
      G = s;
      f = G*e;
    end mG;

    model HydroG "The bondgraphic hydraulic conductor element"
      extends Interfaces.PassiveOnePort;
      parameter Modelica.SIunits.VolumeFlowRate k = 1
        "Hydraulic Valve Constant";
      parameter Modelica.SIunits.Pressure p0 = 1 "Reference pressure";
      Real se "Sign of e";
      Modelica.SIunits.Pressure ae "Absolute value of e";
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Text(
            extent=[-60, 80; 60, -80],
            string="G",
            style(color=0, fillColor=0)), Text(
            extent=[16, 10; 124, -12],
            string="Hydro",
            style(color=0))),
        Icon(
          Text(
            extent=[16, 10; 124, -12],
            string="Hydro",
            style(color=0)),
          Text(
            extent=[-60, 80; 60, -80],
            string="G",
            style(color=0, fillColor=0)),
          Text(extent=[-70, 90; 70, 50], string="%name"),
          Text(extent=[-70, -50; 70, -90], string="k=%k")),
        Window(
          x=0.45,
          y=0.01,
          width=0.35,
          height=0.49),
        Documentation(info="<html>
The bondgraphic hydraulic conductor is a passive OnePort element.  It models the turbulent flow through a hydraulic valve.  The model inherits the effort and flow variables from the passive OnePort.  The valve constant is modeled as a parameter. <p>

This non-linear model is formulated in conductive causality.  There exists a second model, <b>HydroR</b>, that models the same device in resistive causality.  It is useful to make use of the model in its correct causality, as Modelica may be unable to automatically turn the causality of this non-linear equation around. <p>

The hydraulic conductor model has a preferred causality at the element. <p>

<hr>

<pre>
Potential variables:

 e:  Bondgraphic effort variable

 f:  Bondgraphic flow variable, normalized positive for flows into the model


Parameters:

 k:  Valve constant (default value = 1)
</pre> <p>

<hr> <p>

Equations: <p>

<b>f = k*sign(e)*sqrt(abs(e))</b>
</html>
"));
    equation
      se = noEvent(if e > 0 then 1 else -1);
      ae = noEvent(if e > 0 then e else -e);
      f = k*se*sqrt(ae/p0);
    end HydroG;

    model HydromG "The bondgraphic hydraulic modulated conductor element"
      extends Interfaces.ModPassiveOnePort;
      parameter Modelica.SIunits.Pressure p0 = 1 "Reference pressure";
      Modelica.SIunits.VolumeFlowRate k "Hydraulic Valve Constant";
      Real se "Sign of e";
      Modelica.SIunits.Pressure ae "Absolute value of e";
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Text(
            extent=[-60, 80; 60, -80],
            string="mG",
            style(color=0)), Text(
            extent=[-48, -26; 60, -48],
            string="Hydro",
            style(color=0))),
        Icon(
          Text(
            extent=[-60, 80; 60, -80],
            string="mG",
            style(color=0)),
          Text(extent=[-70, 90; 70, 50], string="%name"),
          Text(
            extent=[-48, -26; 60, -48],
            string="Hydro",
            style(color=0))),
        Window(
          x=0.45,
          y=0.01,
          width=0.44,
          height=0.65),
        Documentation(info="<html>
The bondgraphic hydraulic modulated conductor is a modulated passive OnePort element.  It models the turbulent flow through a hydraulic valve.  The model inherits the effort and flow variables as well as the modulation signal from the modulated passive OnePort.  The valve constant is modeled as a real-valued variable. <p>

This non-linear model is formulated in conductive causality.  There exists a second model, <b>HydromR</b>, that models the same device in resistive causality.  It is useful to make use of the model in its correct causality, as Modelica may be unable to automatically turn the causality of this non-linear equation around. <p>

The hydraulic conductor model has a preferred causality at the element. <p>

<hr>

<pre>
Potential variables:

 e:  Bondgraphic effort variable

 f:  Bondgraphic flow variable, normalized positive for flows into the model

 k:  Valve constant (modulated input signal)
</pre> <p>

<hr> <p>

Equations: <p>

<b>f = k*sign(e)*sqrt(abs(e))</b>
</html>
"));
    equation
      se = noEvent(if e > 0 then 1 else -1);
      ae = noEvent(if e > 0 then e else -e);
      k = s;
      f = k*se*sqrt(ae/p0);
    end HydromG;

    model GF "The bondgraphic linear TwoPort conductive field element"
      extends Interfaces.TwoPort2;
      parameter Real G1=1 "Primary Conductance";
      parameter Real G2=1 "Secondary Conductance";
      parameter Real G12=0 "Coupling Conductance";
      annotation (
        Icon(Text(
            extent=[-60, 78; 60, -82],
            style(color=0),
            string="GF"), Text(extent=[-66, 86; 74, 46], string="%name")),
        Diagram(Text(
            extent=[-60, 78; 60, -82],
            style(color=0),
            string="GF")),
        Documentation(info="<html>
The bondgraphic TwoPort conductive field is a symmetric TwoPort element.  It inherits the effort and flow variables from the symmetric TwoPort.  The conductance values are modeled as a set of three parameters. <p>

The TwoPort conductive field has free causality. <p>

<hr>

<pre>
Potential variables:

 e1:   Bondgraphic effort variable of inflow

 f1:   Bondgraphic flow variable of inflow, normalized positive for flows into the model

 e2:   Bondgraphic effort variable of outflow

 f2:   Bondgraphic flow variable of outflow, normalized positive for flows into the model


Parameters:

 G1:   Primary conductance (default value = 1)

 G2:   Secondary conductance (default value = 1)

 G12:  Coupling conductance (default value = 0)
</pre> <p>

<hr> <p>

Equations: <p>

<b>f1 = G1*e1 + G12*e2</b> <br>
<b>f2 = G12*e1 + G2*e2</b>
</html>
"));
    equation
      f1 = G1*e1 + G12*e2;
      f2 = G12*e1 + G2*e2;
    end GF;

    model C "The bondgraphic linear capacitor element"
      extends Interfaces.PassiveOnePort;
      parameter Real C=1 "Bondgraphic Capacitance";
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Text(
            extent=[-60, 80; 60, -80],
            string="C",
            style(color=0, fillColor=0))),
        Icon(
          Text(extent=[-70, 90; 70, 50], string="%name"),
          Text(
            extent=[-60, 80; 60, -80],
            string="C",
            style(color=0, fillColor=0)),
          Text(extent=[-70, -50; 70, -90], string="C=%C")),
        Documentation(info="<html>
The bondgraphic linear capacitor is a passive OnePort element.  It inherits the effort and flow variables from the passive OnePort.  The capacitance is modeled as a parameter. <p>

The capacitor model has a preferred causality away from the element. <p>

<hr>

<pre>
Potential variables:

 e:  Bondgraphic effort variable

 f:  Bondgraphic flow variable, normalized positive for flows into the model


Parameters:

 C:  Capacitance (default value = 1)
</pre> <p>

<hr> <p>

Equations: <p>

<b>f = C*der(e)</b>
</html>
"));
    equation
      f = C*der(e);
    end C;

    model mC "The bondgraphic modulated capacitor element"
      extends Interfaces.ModPassiveOnePort;
      Real C "Bondgraphic Capacitance";
      Real q "Bondgraphic generalized position";
      annotation (
        Icon(Text(
            extent=[-64,78; 56,-82],
            style(color=0),
            string="mC"),    Text(extent=[-70,80; 70,40],   string="%name")),
        Diagram(
             Text(
            extent=[-64,78; 56,-82],
            style(color=0),
            string="mC")),
        Documentation(info="<html>
The bondgraphic modulated capacitor is a modulated passive OnePort element.  It inherits the effort and flow variables as well as the modulating signal from the modulated passive OnePort.  The capacitance is modeled as a real-valued variable. <p>

The capacitor has a preferred causality at the element. <p>

<hr>

<pre>
Potential variables:

 e:  Bondgraphic effort variable

 f:  Bondgraphic flow variable, normalized positive for flows into the model

 q:  Bondgraphic generalized position variable

 C:  Capacitance (modulated input signal)
</pre> <p>

<hr> <p>

Equations: <p>

<b>q = C*e</b> <br>
<b>f = der(q)</b>
</html>"));
    equation
      C = s;
      q = C*e;
      f = der(q);
    end mC;

    model CF "The bondgraphic linear TwoPort capacitive field element"
      extends Interfaces.TwoPort2;
      parameter Real C1=1 "Primary Capacitance";
      parameter Real C2=1 "Secondary Capacitance";
      parameter Real C12=0 "Coupling Capacitance";
      final parameter Real Det=C1*C2 - C12^2;
      final parameter Real Cinv1=C2/Det;
      final parameter Real Cinv2=C1/Det;
      final parameter Real Cinv12=-C12/Det;
      annotation (
        Icon(Text(
            extent=[-60, 78; 60, -82],
            style(color=0),
            string="CF"), Text(extent=[-66, 88; 74, 48], string="%name")),
        Diagram(Text(
            extent=[-60, 78; 60, -82],
            style(color=0),
            string="CF")),
        Documentation(info="<html>
The bondgraphic TwoPort capacitive field is a symmetric TwoPort element.  It inherits the effort and flow variables from the symmetric TwoPort.  The capacitance values are modeled as a set of three parameters.  The symmetric capacitance matrix is manually inverted to speed up the compilation. <p>

The TwoPort capacitive field has preferred causalities at the <b>CF</b> element. <p>

<hr>

<pre>
Potential variables:

 e1:   Bondgraphic effort variable of inflow

 f1:   Bondgraphic flow variable of inflow, normalized positive for flows into the model

 e2:   Bondgraphic effort variable of outflow

 f2:   Bondgraphic flow variable of outflow, normalized positive for flows into the model


Parameters:

 C1:   Primary capacitance (default value = 1)

 C2:   Secondary capacitance (default value = 1)

 C12:  Coupling capacitance (default value = 0)
</pre> <p>

<hr> <p>

Equations: <p>

<b>Det = C1*C2 - C12<sup>2</sup></b> <br>
<b>Cinv1 = C2/Det</b> <br>
<b>Cinv2 = C1/Det</b> <br>
<b>Cinv12 = -C12/Det</b> <p>
<b>der(e1) = Cinv1*f1 + Cinv12*f2</b> <br>
<b>der(e2) = Cinv12*f1 + Cinv2*f2</b>
</html>
"));
    equation
      der(e1) = Cinv1*f1 + Cinv12*f2;
      der(e2) = Cinv12*f1 + Cinv2*f2;
    end CF;

    model I "The bondgraphic linear inductor element"
      extends Interfaces.PassiveOnePort;
      parameter Real I=1 "Bondgraphic Inductance";
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Text(
            extent=[-60, 80; 60, -80],
            string="I",
            style(color=0, fillColor=0))),
        Icon(
          Text(
            extent=[-60, 80; 60, -80],
            string="I",
            style(color=0, fillColor=0)),
          Text(extent=[-70, 90; 70, 50], string="%name"),
          Text(extent=[-70, -50; 70, -90], string="I=%I")),
        Documentation(info="<html>
The bondgraphic linear inductor is a passive OnePort element.  It inherits the effort and flow variables from the passive OnePort.  The inductance is modeled as a parameter. <p>

The inductor model has a preferred causality at the element. <p>

<hr>

<pre>
Potential variables:

 e:  Bondgraphic effort variable

 f:  Bondgraphic flow variable, normalized positive for flows into the model


Parameters:

 I:  Inductance (default value = 1)
</pre> <p>

<hr> <p>

Equations: <p>

<b>e = I*der(f)</b>
</html>
"));
    equation
      e = I*der(f);
    end I;

    model mI "The bondgraphic modulated inductor element"
      extends Interfaces.ModPassiveOnePort;
      Real I "Bondgraphic Inductance";
      Real p "Bondgraphic generalized momentum";
      annotation (
        Icon(Text(
            extent=[-50,48; 48,-50],
            style(color=0),
            string="mI"),    Text(extent=[-70,80; 70,40],   string="%name")),
        Diagram(
             Text(
            extent=[-50,48; 48,-50],
            style(color=0),
            string="mI")),
        Documentation(info="<html>
The bondgraphic modulated inductor is a modulated passive OnePort element.  It inherits the effort and flow variables as well as the modulating signal from the modulated passive OnePort.  The inductance is modeled as a real-valued variable. <p>

The inductor has a preferred causality away from the element. <p>

<hr>

<pre>
Potential variables:

 e:  Bondgraphic effort variable

 f:  Bondgraphic flow variable, normalized positive for flows into the model

 p:  Bondgraphic generalized momentum variable

 I:  inductance (modulated input signal)
</pre> <p>

<hr> <p>

Equations: <p>

<b>p = I*f</b> <br>
<b>e = der(p)</b>
</html>"));
    equation
      I = s;
      p = I*f;
      e = der(p);
    end mI;

    model IF "The bondgraphic linear TwoPort inductive field element"
      extends Interfaces.TwoPort2;
      parameter Real I1=1 "Primary Inductance";
      parameter Real I2=1 "Secondary Inductance";
      parameter Real I12=0 "Coupling Inductance";
      final parameter Real Det=I1*I2 - I12^2;
      final parameter Real Iinv1=I2/Det;
      final parameter Real Iinv2=I1/Det;
      final parameter Real Iinv12=-I12/Det;
      annotation (
        Icon(Text(extent=[-68, 88; 72, 48], string="%name"), Text(
            extent=[-48, 72; 44, -74],
            style(color=0),
            string="IF")),
        Diagram(Text(
            extent=[-48, 72; 44, -74],
            style(color=0),
            string="IF")),
        Documentation(info="<html>
The bondgraphic TwoPort inductive field is a symmetric TwoPort element.  It inherits the effort and flow variables from the symmetric TwoPort.  The inductance values are modeled as a set of three parameters.  The symmetric inductance matrix is manually inverted to speed up the compilation. <p>

The TwoPort inductive field has preferred causalities away from the <b>IF</b> element. <p>

<hr>

<pre>
Potential variables:

 e1:   Bondgraphic effort variable of inflow

 f1:   Bondgraphic flow variable of inflow, normalized positive for flows into the model

 e2:   Bondgraphic effort variable of outflow

 f2:   Bondgraphic flow variable of outflow, normalized positive for flows into the model


Parameters:

 I1:   Primary inductance (default value = 1)

 I2:   Secondary inductance (default value = 1)

 I12:  Coupling inductance (default value = 0)
</pre> <p>

<hr> <p>

Equations: <p>

<b>Det = I1*I2 - I12<sup>2</sup></b> <br>
<b>Iinv1 = I2/Det</b> <br>
<b>Iinv2 = I1/Det</b> <br>
<b>Iinv12 = -I12/Det</b> <p>
<b>der(f1) = Iinv1*e1 + Iinv12*e2</b> <br>
<b>der(f2) = Iinv12*e1 + Iinv2*e2</b>
</html>
"));
    equation
      der(f1) = Iinv1*e1 + Iinv12*e2;
      der(f2) = Iinv12*e1 + Iinv2*e2;
    end IF;

    model TF "The bondgraphic linear transformer element"
      extends Interfaces.TwoPort;
      parameter Real m=1 "Bondgraphic Transformation Constant";
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Text(
            extent=[-60, 80; 60, -80],
            string="TF",
            style(color=0))),
        Icon(
          Text(
            extent=[-60, 80; 60, -80],
            string="TF",
            style(color=0)),
          Text(extent=[-70, 90; 70, 50], string="%name"),
          Text(extent=[-70, -50; 70, -90], string="m=%m")),
        Window(
          x=0.45,
          y=0.01,
          width=0.44,
          height=0.65),
        Documentation(info="<html>
The bondgraphic linear transformer is a directed TwoPort element.  It inherits the effort and flow variables from the directed TwoPort.  The transformation constant is modeled as a parameter. <p>

The bond graph literature is not consistent in the definition of the transformation constant.  In this library, the constant is defined as the amplification of <b>flow</b> from the inflow to the outflow.  Some other bond graph references define the transformation constant as the amplification of <b>effort</b> from the inflow to the outflow. <p>

The transformer model has one causality stroke. <p>

<hr>

<pre>
Potential variables:

 e1:  Bondgraphic effort variable of inflow

 f1:  Bondgraphic flow variable of inflow, normalized positive for flows into the model

 e2:  Bondgraphic effort variable of outflow

 f2:  Bondgraphic flow variable of outflow, normalized positive for flows out of the model


Parameters:

 m:   Transformation constant (default value = 1)
</pre> <p>

<hr> <p>

Equations: <p>

<b>f2 = m*f1</b> <br>
<b>e1 = m*e2</b>
</html>
"));
    equation
      e1 = m*e2;
      f2 = m*f1;
    end TF;

    model mTF "The bondgraphic modulated transformer element"
      extends Interfaces.ModTwoPort;
      Real m "Bondgraphic Transformation Constant";
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Text(
            extent=[-60, 80; 60, -80],
            string="mTF",
            style(color=0))),
        Icon(Text(
            extent=[-60, 80; 60, -80],
            string="mTF",
            style(color=0)), Text(extent=[-70, 90; 70, 50], string="%name")),
        Window(
          x=0.45,
          y=0.01,
          width=0.44,
          height=0.65),
        Documentation(info="<html>
The bondgraphic modulated transformer is a modulated directed TwoPort element.  It inherits the effort and flow variables from the modulated directed TwoPort.  The transformation constant is modeled as a real-valued variable. <p>

The bond graph literature is not consistent in the definition of the transformation constant.  In this library, the constant is defined as the amplification of <b>flow</b> from the inflow to the outflow.  Some other bond graph references define the transformation constant as the amplification of <b>effort</b> from the inflow to the outflow. <p>

The transformer model has one causality stroke. <p>

<hr>

<pre>
Potential variables:

 e1:  Bondgraphic effort variable of inflow

 f1:  Bondgraphic flow variable of inflow, normalized positive for flows into the model

 e2:  Bondgraphic effort variable of outflow

 f2:  Bondgraphic flow variable of outflow, normalized positive for flows out of the model

 m:   Transformation constant (modulation input signal)
</pre> <p>

<hr> <p>

Equations: <p>

<b>f2 = m*f1</b> <br>
<b>e1 = m*e2</b>
</html>
"));
    equation
      m = s;
      e2 = m*e1;
      f1 = m*f2;
    end mTF;

    model GY "The bondgraphic linear gyrator element"
      extends Interfaces.TwoPort;
      parameter Real r=1 "Bondgraphic Gyration Constant";
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Text(
            extent=[-60, 80; 60, -80],
            string="GY",
            style(color=0))),
        Icon(
          Text(
            extent=[-60, 80; 60, -80],
            string="GY",
            style(color=0)),
          Text(extent=[-70, 90; 70, 50], string="%name"),
          Text(extent=[-70, -50; 70, -90], string="r=%r")),
        Window(
          x=0.45,
          y=0.01,
          width=0.44,
          height=0.65),
        Documentation(info="<html>
The bondgraphic linear gyrator is a directed TwoPort element.  It inherits the effort and flow variables from the directed TwoPort.  The gyration constant is modeled as a parameter. <p>

The bond graph literature is not consistent in the definition of the gyration constant.  In this library, the constant is defined as the amplification of <b>flow</b> at the inflow to <b>effort</b> at the outflow.  Some other bond graph references define the gyration constant as the amplification of <b>effort</b> at the inflow to <b>flow</b> at the outflow. <p>

The gyrator model has either zero or two causality strokes. <p>

<hr>

<pre>
Potential variables:

 e1:  Bondgraphic effort variable of inflow

 f1:  Bondgraphic flow variable of inflow, normalized positive for flows into the model

 e2:  Bondgraphic effort variable of outflow

 f2:  Bondgraphic flow variable of outflow, normalized positive for flows out of the model


Parameters:

 r:   Gyration constant (default value = 1)
</pre> <p>

<hr> <p>

Equations: <p>

<b>e2 = r*f1</b> <br>
<b>e1 = r*f2</b>
</html>
"));
    equation
      e1 = r*f2;
      e2 = r*f1;
    end GY;

    model mGY "The bondgraphic modulated gyrator element"
      extends Interfaces.ModTwoPort;
      Real r "Bondgraphic Gyration Constant";
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Text(
            extent=[-60, 80; 60, -80],
            string="mGY",
            style(color=0))),
        Icon(Text(
            extent=[-60, 80; 60, -80],
            string="mGY",
            style(color=0)), Text(extent=[-70, 90; 70, 50], string="%name")),
        Window(
          x=0.45,
          y=0.01,
          width=0.44,
          height=0.65),
        Documentation(info="<html>
The bondgraphic modulated gyrator is a modulated directed TwoPort element.  It inherits the effort and flow variables from the modulated directed TwoPort.  The gyration constant is modeled as a real-valued variable. <p>

The bond graph literature is not consistent in the definition of the gyration constant.  In this library, the constant is defined as the amplification of <b>flow</b> at the inflow to <b>effort</b> at the outflow.  Some other bond graph references define the gyration constant as the amplification of <b>effort</b> at the inflow to <b>flow</b> at the outflow. <p>

The gyrator model has either zero or two causality strokes. <p>

<hr>

<pre>
Potential variables:

 e1:  Bondgraphic effort variable of inflow

 f1:  Bondgraphic flow variable of inflow, normalized positive for flows into the model

 e2:  Bondgraphic effort variable of outflow

 f2:  Bondgraphic flow variable of outflow, normalized positive for flows out of the model

 r:   Gyration constant (modulation input signal)
</pre> <p>

<hr> <p>

Equations: <p>

<b>e2 = r*f1</b> <br>
<b>e1 = r*f2</b>
</html>
"));
    equation
      r = s;
      e1 = r*f2;
      e2 = r*f1;
    end mGY;
  end Passive;

  package Semiconductors "Bondgraphic description of semiconductor elements"
    annotation (preferedView="info",
      Icon(
        Rectangle(extent=[-100,-100; 80,50],   style(fillColor=30, fillPattern=
                1)),
        Polygon(points=[-100,50; -80,70; 100,70; 80,50; -100,50],      style(
              fillColor=30, fillPattern=1)),
        Polygon(points=[100,70; 100,-80; 80,-100; 80,50; 100,70],      style(
              fillColor=30, fillPattern=1)),
        Text(
          extent=[-85,35; 65,-85],
          string="Library",
          style(color=3)),
        Text(
          extent=[-120,122; 120,73],
          string="%name",
          style(color=1))), Documentation(info="<html>
This package contains bond graph models of electronic semiconductors using the standard bond graph interface.
</html>
"));
    model Dreal "Electrical real diode of the Modelica bond graph library"
      extends Interfaces.PassiveOnePort;
      parameter Modelica.SIunits.Current Ids=1.e-6 "Saturation current";
      parameter Modelica.SIunits.Voltage Vt=0.04
        "Voltage equivalent of temperature (kT/qn)";
      parameter Real Maxexp(final min=Modelica.Constants.small) = 15
        "Max. exponent for linear continuation";
      parameter Modelica.SIunits.Resistance R=1.e8 "Parallel Ohmic resistance";
      annotation (
        Icon(
          Text(
            extent=[-88,68; 74,-76],
            string="D",
            style(color=0)),
          Text(
            extent=[26,-18; 88,-52],
            style(color=0, pattern=0),
            string="real"),
          Text(extent=[-76,92; 64,52],   string="%name")),
        Diagram(
          Line(points=[-80,0; 80,0],   style(color=10)),
          Polygon(points=[70,4; 80,0; 70,-4; 70,4],     style(color=10,
                fillColor=10)),
          Line(points=[0,80; 0,-80],   style(
              color=10,
              fillColor=10,
              fillPattern=1)),
          Polygon(points=[-4,70; 0,80; 4,70; -4,70],     style(
              color=10,
              fillColor=10,
              fillPattern=1)),
          Text(
            extent=[4,82; 14,72],
            style(
              color=10,
              fillColor=10,
              fillPattern=1),
            string="f"),
          Text(
            extent=[70,-4; 80,-14],
            style(
              color=10,
              fillColor=10,
              fillPattern=1),
            string="e"),
          Line(points=[-80,-4; -28,-4; -6,-2; 0,0],         style(
              color=0,
              thickness=2,
              fillColor=10,
              fillPattern=1)),
          Text(
            extent=[-66,-6; -36,-14],
            style(
              color=10,
              pattern=3,
              fillColor=10,
              fillPattern=1),
            string="blocking"),
          Text(
            extent=[16,50; 48,42],
            style(
              color=10,
              pattern=3,
              fillColor=10,
              fillPattern=1),
            string="conducting"),
          Line(points=[0,0; 6,10; 12,28; 16,66],            style(
              color=0,
              thickness=2,
              fillColor=10,
              fillPattern=1))),
        Documentation(info="<html>
The real diode element is a passive OnePort element.  It inherits the effort and flow variables from the passive OnePort.  Real diodes are a special class of non-linear resistors. <p>

The real diode model consists of the diode itself and an parallel Ohmic resistance <b>R</b>. The diode formula essentially is: <p>

<b>f = Ids*(exp(e/Vt) - 1)</b> <p>

However, if the exponent <b>e/Vt</b> reaches the limit <b>Maxexp</b>, the diode characterisic is linearly continued to avoid overflow. <p>

The causality of the real diode element is free. <p>

<hr>

<pre>
Potential variables:

 e:       Bondgraphic effort variable

 f:       Bondgraphic flow variable, normalized positive for flows into the model


Parameters:

 Ids:     Saturation current (default value = 1e-6 Amp)

 Vt:      Thermal voltage (default value = 0.04 Volt)

 Maxexp:  Maximum exponent (default value = 15)

 R:       Parallel Ohmic leakage resistance (default value = 1e8 Ohm)
</pre> <p>

<hr> <p>

Equations: <p>

<b>f = if (e/Vt &gt; Maxexp) then Ids*(exp(Maxexp)*(1 + e/Vt - Maxexp) - 1) + e/R else Ids*(exp(e/Vt) - 1) + e/R</b>
</html>
"));
    equation
      f = if (e/Vt > Maxexp) then Ids*(exp(Maxexp)*(1 + e/Vt - Maxexp) - 1) +
        e/R else Ids*(exp(e/Vt) - 1) + e/R;
    end Dreal;

    model NPN "NPN transistor of the Modelica bond graph library"
      parameter Real Bf=50 "Forward beta coefficient";
      parameter Real Br=0.1 "Reverse beta coefficient";
      parameter Modelica.SIunits.Current Is=1.e-16
        "Transport saturation current";
      parameter Modelica.SIunits.InversePotential Vak=0.02
        "Inverse early voltage";
      parameter Modelica.SIunits.Time Tauf=0.12e-9 "Ideal forward transit time";
      parameter Modelica.SIunits.Time Taur=5e-9 "Ideal reverse transit time";
      parameter Modelica.SIunits.Capacitance Ccs=1e-12
        "Collector-substrate capacitance";
      parameter Modelica.SIunits.Capacitance Cje=0.4e-12
        "Base-emitter zero bias depletion capacitance";
      parameter Modelica.SIunits.Capacitance Cjc=0.5e-12
        "Base-collector zero bias depletion capacitance";
      parameter Modelica.SIunits.Voltage Phie=0.8
        "Base-emitter diffusion voltage";
      parameter Real Me=0.4 "Base-emitter gradation exponent";
      parameter Modelica.SIunits.Voltage Phic=0.8
        "Base-collector diffusion voltage";
      parameter Real Mc=0.333 "Base-collector gradation exponent";
      parameter Modelica.SIunits.Conductance Gbc=1e-15
        "Base-collector leakage conductance";
      parameter Modelica.SIunits.Conductance Gbe=1e-15
        "Base-emitter leakage conductance";
      parameter Modelica.SIunits.Voltage Vt=0.02585 "Thermal voltage";
      parameter Real EMin=-100 "if x < EMin, the exp(x) function is linearized";
      parameter Real EMax=40 "if x > EMax, the exp(x) function is linearized";
      Interfaces.BondCon B "Base"
                           annotation (extent=[-110,-10; -90,10]);
      Interfaces.BondCon C "Collector"
                           annotation (extent=[-10,90; 10,110]);
      Interfaces.BondCon E "Emitter"
                           annotation (extent=[-10,-110; 10,-90]);
      annotation (
        Diagram(
          Text(
            extent=[6,-90; 26,-110],
            style(color=9, rgbcolor={149,164,171}),
            string="E"),
          Text(
            extent=[-108,28; -88,8],
            string="B",
            style(color=9, rgbcolor={149,164,171})),
          Text(
            extent=[6,110; 26,90],
            style(color=9, rgbcolor={149,164,171}),
            string="C"),
          Ellipse(extent=[-112,42; -108,38], style(
              color=9,
              rgbcolor={149,164,171},
              fillColor=74,
              rgbfillColor={0,0,185})),
          Ellipse(extent=[-122,-42; -118,-38], style(
              color=9,
              rgbcolor={149,164,171},
              fillColor=74,
              rgbfillColor={0,0,185})),
          Text(
            extent=[-98,48; -88,40],
            string="ibc",
            style(color=9, rgbcolor={149,164,171})),
          Text(
            extent=[-96,-66; -86,-74],
            string="ibc",
            style(color=9, rgbcolor={149,164,171})),
          Text(
            extent=[78,72; 88,64],
            string="ibc",
            style(color=9, rgbcolor={149,164,171})),
          Text(
            extent=[54,72; 64,64],
            style(color=9, rgbcolor={149,164,171}),
            string="ibe"),
          Text(
            extent=[-96,-86; -86,-94],
            style(color=9, rgbcolor={149,164,171}),
            string="ibe"),
          Text(
            extent=[-100,-32; -90,-40],
            style(color=9, rgbcolor={149,164,171}),
            string="ibe"),
          Text(
            extent=[-36,-12; -26,-20],
            style(color=9, rgbcolor={149,164,171}),
            string="vbc"),
          Text(
            extent=[48,58; 58,50],
            style(color=9, rgbcolor={149,164,171}),
            string="vbc"),
          Text(
            extent=[78,8; 88,0],
            style(color=9, rgbcolor={149,164,171}),
            string="IC0"),
          Text(
            extent=[-38,-90; -28,-98],
            style(color=9, rgbcolor={149,164,171}),
            string="IB0")),
        Icon(
          Text(
            extent=[-82,82; 86,-86],
            style(color=0),
            string="NPN"),
          Text(
            extent=[-108,28; -88,8],
            string="B",
            style(color=9, rgbcolor={149,164,171})),
          Text(
            extent=[6,110; 26,90],
            style(color=9, rgbcolor={149,164,171}),
            string="C"),
          Text(
            extent=[6,-90; 26,-110],
            style(color=9, rgbcolor={149,164,171}),
            string="E"),
          Line(points=[0,80; 0,-80], style(
              color=9,
              rgbcolor={149,164,171},
              thickness=2,
              fillColor=74,
              rgbfillColor={0,0,185},
              fillPattern=1)),
          Line(points=[-20,-60; 0,-80], style(
              color=9,
              rgbcolor={149,164,171},
              thickness=2,
              fillColor=74,
              rgbfillColor={0,0,185},
              fillPattern=1)),
          Line(points=[0,-80; 20,-60], style(
              color=9,
              rgbcolor={149,164,171},
              thickness=2,
              fillColor=74,
              rgbfillColor={0,0,185},
              fillPattern=1))),
        Documentation(info="<html>
The NPN element of the bond graph library implements the Ebers-Moll model of a NPN bipolar transistor.  The substrate is not shown on the icon layer, as it is assumed grounded. The transistor is assumed to be vertically diffused, as the substrate is connected to the collector, rather than to the base. <p>

The NPN bipolar transistor is a directed ThreePort.  The direction of positive power flow is assumed into the model at the base, <b>B</b>, and at the collector, <b>C</b>, whereas it is assumed out of the model at the emitter, <b>E</b>. <p>

The causality of the NPN model is relatively free.  However, if there is a causality stroke placed at the collector from the outside, the substrate capacitor will assume differential causality. <p>

<hr>

<pre>
Parameters:

 Bf:    Forward beta coefficient (default value = 50)

 Br:    Reverse beta coefficient (default value = 0.1)

 Is:    Transport saturation current (default value = 1e-16 Amp)

 Vak:   Inverse early voltage (default value = 0.02 /Volt)

 Tauf:  Ideal forward transit time (default value = 0.12e-9 sec)

 Taur:  Ideal reverse transit time (default value = 5e-9 sec)

 Ccs:   Collector-substrate capacitance (default value = 1e-12 F)

 Cje:   Base-emitter zero bias depletion capacitance (default value = 0.4e-12 F)

 Cjc:   Base-collector zero bias depletion capacitance (default value = 0.5e-12 F)

 Phie:  Base-emitter diffusion voltage (default value = 0.8 Volt)

 Me:    Base-emitter gradation exponent (default value = 0.4)

 Phic:  Base-collector diffusion voltage (default value = 0.8 Volt)

 Mc:    Base-collector gradation exponent (default value = 0.333)

 Gbc:   Base-collector leakage conductance (default value = 1e-15 Mho)

 Gbe:   Base-emitter leakage conductance (default value = 1e-15 Mho)

 Vt:    Thermal voltage (default value = 0.02585 Volt)

 EMin:  Minimum exponent (default value = -100)

 EMax:  Maximum exponent (default value = 40)
</pre> <p>

<hr> <p>

References: <p>

Vlach, J. and K. Singal (1983), <I>Computer methods for circuit analysis and design</I>, Van Nostrand Reinhold, New York,
on page 317 ff.
</html>
"));
      Junctions.J0p3 J0p3_1 annotation (extent=[-10,-10; 10,10], rotation=-90);
      Bonds.Bond B1 annotation (extent=[-10,50; 10,70], rotation=90);
      Bonds.Bond B2 annotation (extent=[-10,10; 10,30], rotation=90);
      Bonds.Bond B3 annotation (extent=[-10,-30; 10,-10], rotation=-90);
      Bonds.Bond B4 annotation (extent=[-10,-70; 10,-50], rotation=-90);
      Junctions.J1p3 J1p3_1 annotation (extent=[-10,30; 10,50], rotation=-90);
      Junctions.J0p4 J0p4_1 annotation (extent=[-10,70; 10,90]);
      Junctions.J1p3 J1p3_2 annotation (extent=[-10,-50; 10,-30], rotation=-90);
      Junctions.J0p3 J0p3_2 annotation (extent=[10,-90; -10,-70], rotation=-90);
      Junctions.J1p3 J1p3_3 annotation (extent=[40,-10; 20,10], rotation=-90);
      Bonds.Bond B5 annotation (extent=[20,10; 40,30], rotation=-90);
      Bonds.Bond B6 annotation (extent=[20,-30; 40,-10], rotation=-90);
      Bonds.Bond B7 annotation (extent=[-30,70; -10,90], rotation=180);
      Passive.C C1(C=Ccs) annotation (extent=[-30,70; -50,90]);
      Bonds.eBond B8 annotation (extent=[40,-10; 60,10], rotation=180);
      Sources.mSf Ic0 annotation (extent=[60,-10; 80,10]);
      Junctions.J0p3 J0p3_3 annotation (extent=[-50,50; -30,30], rotation=-180);
      Bonds.eBond B9 annotation (extent=[-30,30; -10,50], rotation=180);
      Bonds.fBond B10 annotation (extent=[-50,10; -30,30], rotation=-90);
      Sensors.De Vbc annotation (extent=[-50,-10; -30,10], rotation=-90);
      Bonds.eBond B11 annotation (extent=[-70,30; -50,50], rotation=180);
      Utilities.Cj Dbc(
        Is=Is,
        Tau=Taur,
        Cj=Cjc,
        Phi=Phic,
        M=Mc,
        Gb=Gbc,
        Vt=Vt,
        EMin=EMin,
        EMax=EMax) annotation (extent=[-70,30; -90,50]);
      Bonds.eBond B12 annotation (extent=[-30,-50; -10,-30], rotation=180);
      Junctions.J0p3 J0p3_4
        annotation (extent=[-50,-30; -30,-50], rotation=-180);
      Bonds.eBond B13 annotation (extent=[-50,-70; -30,-50], rotation=90);
      Sources.mSf Ib0 annotation (extent=[-50,-90; -30,-70], rotation=-90);
      Bonds.eBond B14 annotation (extent=[-70,-50; -50,-30], rotation=180);
      Utilities.Cj Dbe(
        Is=Is,
        Vt=Vt,
        EMin=EMin,
        EMax=EMax,
        Tau=Tauf,
        Cj=Cje,
        Phi=Phie,
        M=Me,
        Gb=Gbe) annotation (extent=[-70,-50; -90,-30]);
      Utilities.IC0 IC0_1(
        Br=Br,
        Vak=Vak) annotation (extent=[60,40; 80,60], rotation=-90);
      Utilities.IB0 IB0_1(Bf=Bf, Br=Br) annotation (extent=[-80,-90; -60,-70]);
    equation
      connect(J0p3_1.BondCon1, B2.BondCon1) annotation (points=[-6.12303e-016,
            10; -6.00057e-016,10; -6.00057e-016,10.2], style(color=8, rgbcolor=
              {192,192,192}));
      connect(J1p3_1.BondCon2, B2.BondCon2) annotation (points=[6.12303e-016,30;
            6.12303e-016,30], style(color=8, rgbcolor={192,192,192}));
      connect(B1.BondCon1, J1p3_1.BondCon1) annotation (points=[-6.00057e-016,
            50.2; -6.00057e-016,56.1; -6.12303e-016,56.1; -6.12303e-016,50],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p4_1.BondCon3, B1.BondCon2) annotation (points=[0,70;
            6.12303e-016,70], style(color=8, rgbcolor={192,192,192}));
      connect(J0p4_1.BondCon4, C) annotation (points=[0,90; 0,100], style(color=
             8, rgbcolor={192,192,192}));
      connect(J0p3_1.BondCon2, B3.BondCon1) annotation (points=[6.12303e-016,
            -10; -6.00057e-016,-10; -6.00057e-016,-10.2], style(color=8,
            rgbcolor={192,192,192}));
      connect(J1p3_2.BondCon1, B3.BondCon2) annotation (points=[-6.12303e-016,
            -30; 6.12303e-016,-30], style(color=8, rgbcolor={192,192,192}));
      connect(B4.BondCon1, J1p3_2.BondCon2) annotation (points=[-6.00057e-016,
            -50.2; -6.00057e-016,-55.1; 6.12303e-016,-55.1; 6.12303e-016,-50],
          style(color=8, rgbcolor={192,192,192}));
      connect(B4.BondCon2, J0p3_2.BondCon1) annotation (points=[6.12303e-016,
            -70; 6.12303e-016,-70], style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_2.BondCon2, E) annotation (points=[-6.12303e-016,-90; 0,-90;
            0,-100], style(color=8, rgbcolor={192,192,192}));
      connect(B5.BondCon2, J1p3_3.BondCon1) annotation (points=[30,10; 30,10],
          style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_3.BondCon2, B6.BondCon1) annotation (points=[30,-10; 30,
            -10.2], style(color=8, rgbcolor={192,192,192}));
      connect(J0p4_1.BondCon2, B5.BondCon1) annotation (points=[10,80; 30,80;
            30,29.8], style(color=8, rgbcolor={192,192,192}));
      connect(B6.BondCon2, J0p3_2.BondCon3) annotation (points=[30,-30; 30,-80;
            10,-80], style(color=8, rgbcolor={192,192,192}));
      connect(B7.BondCon1, J0p4_1.BondCon1) annotation (points=[-10.2,80; -10,
            80], style(color=8, rgbcolor={192,192,192}));
      connect(C1.BondCon1, B7.BondCon2) annotation (points=[-30,80; -30,80],
          style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_3.BondCon3, B8.eBondCon1) annotation (points=[40,
            -6.12303e-016; 44,-6.12303e-016; 44,1.22461e-015; 40,1.22461e-015],
          style(color=8, rgbcolor={192,192,192}));
      connect(Ic0.BondCon1, B8.fBondCon1) annotation (points=[60,0; 64,0; 64,
            -1.22461e-015; 60,-1.22461e-015], style(color=8, rgbcolor={192,192,
              192}));
      connect(B9.fBondCon1, J1p3_1.BondCon3) annotation (points=[-10,40; -10,40],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_3.BondCon1, B9.eBondCon1) annotation (points=[-30,40; -30,40],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_3.BondCon3, B10.eBondCon1) annotation (points=[-40,30; -40,
            30], style(color=8, rgbcolor={192,192,192}));
      connect(Vbc.BondCon1, B10.fBondCon1) annotation (points=[-40,10; -40,10],
          style(color=8, rgbcolor={192,192,192}));
      connect(B11.fBondCon1, J0p3_3.BondCon2) annotation (points=[-50,40; -50,
            40], style(color=8, rgbcolor={192,192,192}));
      connect(Dbc.BondCon1, B11.eBondCon1) annotation (points=[-70,40; -70,40],
          style(color=8, rgbcolor={192,192,192}));
      connect(B12.fBondCon1, J1p3_2.BondCon3) annotation (points=[-10,-40; -10,
            -40], style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_4.BondCon1, B12.eBondCon1) annotation (points=[-30,-40; -30,
            -40], style(color=8, rgbcolor={192,192,192}));
      connect(B13.eBondCon1, J0p3_4.BondCon3) annotation (points=[-40,-50; -40,
            -50], style(color=8, rgbcolor={192,192,192}));
      connect(Ib0.BondCon1, B13.fBondCon1) annotation (points=[-40,-70; -40,-70],
          style(color=8, rgbcolor={192,192,192}));
      connect(B14.fBondCon1, J0p3_4.BondCon2) annotation (points=[-50,-40; -50,
            -40], style(color=8, rgbcolor={192,192,192}));
      connect(Dbe.BondCon1, B14.eBondCon1) annotation (points=[-70,-40; -70,-40],
          style(color=8, rgbcolor={192,192,192}));
      connect(Vbc.OutPort1, IC0_1.u3) annotation (points=[-40,-11; -40,-20; -20,
            -20; -20,34; 50,34; 50,50; 60,50], style(color=74, rgbcolor={0,0,
              127}));
      connect(Dbc.y, IC0_1.u1) annotation (points=[-90,40; -110,40; -110,112;
            76,112; 76,62], style(color=74, rgbcolor={0,0,127}));
      connect(Dbc.y, IB0_1.u1) annotation (points=[-90,40; -110,40; -110,-74;
            -82,-74], style(color=74, rgbcolor={0,0,127}));
      connect(Dbe.y, IB0_1.u2) annotation (points=[-90,-40; -120,-40; -120,-86;
            -82,-86], style(color=74, rgbcolor={0,0,127}));
      connect(Dbe.y, IC0_1.u2) annotation (points=[-90,-40; -120,-40; -120,122;
            64,122; 64,62], style(color=74, rgbcolor={0,0,127}));
      connect(J0p3_1.BondCon3, B) annotation (points=[-10,-6.12303e-016; -52,
            -6.12303e-016; -52,0; -100,0], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=74,
          rgbfillColor={0,0,185},
          fillPattern=1));
      connect(IB0_1.y, Ib0.s) annotation (points=[-59,-80; -50,-80; -50,-96;
            -40,-96; -40,-89], style(color=74, rgbcolor={0,0,127}));
      connect(IC0_1.y, Ic0.s) annotation (points=[70,39; 70,30; 90,30; 90,0; 79,
            0], style(color=74, rgbcolor={0,0,127}));
    end NPN;

    model NPNth
      "NPN transistor of the Modelica bond graph library with temperature variation"
      parameter Real Bf=50 "Forward beta coefficient";
      parameter Real Br=0.1 "Reverse beta coefficient";
      parameter Modelica.SIunits.Temperature Tnom=300.15
        "Reference temperature";
      parameter Modelica.SIunits.Current Is=1.e-16
        "Transport saturation current";
      parameter Modelica.SIunits.InversePotential Vak=0.02
        "Inverse early voltage";
      parameter Modelica.SIunits.Time Tauf=0.12e-9 "Ideal forward transit time";
      parameter Modelica.SIunits.Time Taur=5e-9 "Ideal reverse transit time";
      parameter Real NF=1 "Forward current emission coefficient";
      parameter Real NR=1 "Reverse current emission coefficient";
      parameter Modelica.SIunits.Capacitance Ccs=1e-12
        "Collector-substrate capacitance";
      parameter Modelica.SIunits.Capacitance Cje=0.4e-12
        "Base-emitter zero bias depletion capacitance";
      parameter Modelica.SIunits.Capacitance Cjc=0.5e-12
        "Base-collector zero bias depletion capacitance";
      parameter Modelica.SIunits.Voltage Phie=0.8
        "Base-emitter diffusion voltage";
      parameter Real Me=0.4 "Base-emitter gradation exponent";
      parameter Modelica.SIunits.Voltage Phic=0.8
        "Base-collector diffusion voltage";
      parameter Real Mc=0.333 "Base-collector gradation exponent";
      parameter Modelica.SIunits.Voltage EG=1.11
        "Energy gap for temperature effect on saturation current";
      parameter Real XTI=3 "Saturation current temperature exponent";
      parameter Real XTB=0 "Forward and reverse beta temperature coefficient";
      parameter Modelica.SIunits.Conductance Gbc=1e-15
        "Base-collector leakage conductance";
      parameter Modelica.SIunits.Conductance Gbe=1e-15
        "Base-emitter leakage conductance";
      parameter Real EMin=-100 "if x < EMin, the exp(x) function is linearized";
      parameter Real EMax=40 "if x > EMax, the exp(x) function is linearized";
      Interfaces.BondCon B "Base"
                           annotation (extent=[-110,-10; -90,10]);
      Interfaces.BondCon C "Collector"
                           annotation (extent=[-10,90; 10,110]);
      Interfaces.BondCon E "Emitter"
                           annotation (extent=[-10,-110; 10,-90]);
      Junctions.J0p3 J0p3_1 annotation (extent=[-10,-10; 10,10], rotation=-90);
      Bonds.Bond B1 annotation (extent=[-10,50; 10,70], rotation=90);
      Bonds.Bond B2 annotation (extent=[-10,10; 10,30], rotation=90);
      Bonds.Bond B3 annotation (extent=[-10,-30; 10,-10], rotation=-90);
      Bonds.Bond B4 annotation (extent=[-10,-70; 10,-50], rotation=-90);
      Junctions.J1p3 J1p3_1 annotation (extent=[-10,30; 10,50], rotation=-90);
      Junctions.J0p4 J0p4_1 annotation (extent=[-10,70; 10,90]);
      Junctions.J1p3 J1p3_2 annotation (extent=[-10,-50; 10,-30], rotation=-90);
      Junctions.J0p3 J0p3_2 annotation (extent=[10,-90; -10,-70], rotation=-90);
      Junctions.J1p3 J1p3_3 annotation (extent=[40,-10; 20,10], rotation=-90);
      Bonds.Bond B5 annotation (extent=[20,10; 40,30], rotation=-90);
      Bonds.Bond B6 annotation (extent=[20,-30; 40,-10], rotation=-90);
      Bonds.Bond B7 annotation (extent=[-30,70; -10,90], rotation=180);
      Passive.C C1(C=Ccs) annotation (extent=[-30,70; -50,90]);
      Bonds.eBond B8 annotation (extent=[40,-10; 60,10], rotation=180);
      Junctions.J0p3 J0p3_3 annotation (extent=[-50,50; -30,30], rotation=-180);
      Bonds.eBond B9 annotation (extent=[-30,30; -10,50], rotation=180);
      Bonds.fBond B10 annotation (extent=[-50,10; -30,30], rotation=-90);
      Sensors.De Vbc annotation (extent=[-50,-10; -30,10], rotation=-90);
      Bonds.eBond B11 annotation (extent=[-70,30; -50,50], rotation=180);
      Bonds.eBond B12 annotation (extent=[-30,-50; -10,-30], rotation=180);
      Junctions.J0p3 J0p3_4
        annotation (extent=[-50,-30; -30,-50], rotation=-180);
      Bonds.eBond B13 annotation (extent=[-50,-70; -30,-50], rotation=90);
      Bonds.eBond B14 annotation (extent=[-70,-50; -50,-30], rotation=180);
      annotation (
        Diagram(
          Text(
            extent=[6,-90; 26,-110],
            style(color=9, rgbcolor={149,164,171}),
            string="E"),
          Text(
            extent=[-108,28; -88,8],
            string="B",
            style(color=9, rgbcolor={149,164,171})),
          Text(
            extent=[6,110; 26,90],
            style(color=9, rgbcolor={149,164,171}),
            string="C"),
          Ellipse(extent=[-112,62; -108,58], style(
              color=9,
              rgbcolor={149,164,171},
              fillColor=74,
              rgbfillColor={0,0,185})),
          Ellipse(extent=[-116,-22; -112,-18], style(
              color=9,
              rgbcolor={149,164,171},
              fillColor=74,
              rgbfillColor={0,0,185})),
          Text(
            extent=[-80,60; -70,52],
            string="ibc",
            style(color=9, rgbcolor={149,164,171})),
          Text(
            extent=[-96,-66; -86,-74],
            string="ibc",
            style(color=9, rgbcolor={149,164,171})),
          Text(
            extent=[78,72; 88,64],
            string="ibc",
            style(color=9, rgbcolor={149,164,171})),
          Text(
            extent=[54,72; 64,64],
            style(color=9, rgbcolor={149,164,171}),
            string="ibe"),
          Text(
            extent=[-96,-86; -86,-94],
            style(color=9, rgbcolor={149,164,171}),
            string="ibe"),
          Text(
            extent=[-80,-20; -70,-28],
            style(color=9, rgbcolor={149,164,171}),
            string="ibe"),
          Text(
            extent=[-36,-12; -26,-20],
            style(color=9, rgbcolor={149,164,171}),
            string="vbc"),
          Text(
            extent=[48,56; 58,48],
            style(color=9, rgbcolor={149,164,171}),
            string="vbc"),
          Text(
            extent=[72,18; 82,10],
            style(color=9, rgbcolor={149,164,171}),
            string="IC0"),
          Text(
            extent=[-30,-72; -20,-80],
            style(color=9, rgbcolor={149,164,171}),
            string="IB0"),
          Text(
            extent=[88,26; 120,0],
            style(color=9, rgbcolor={149,164,171}),
            string="Heat"),
          Ellipse(extent=[48,-12; 52,-8],      style(
              color=9,
              rgbcolor={149,164,171},
              fillColor=74,
              rgbfillColor={0,0,185})),
          Text(
            extent=[50,40; 62,32],
            style(color=9, rgbcolor={149,164,171}),
            string="Tdev"),
          Text(
            extent=[-84,-92; -72,-100],
            style(color=9, rgbcolor={149,164,171}),
            string="Tdev")),
        Documentation(info="<html>
The <b>NPNth</b> element of the bond graph library implements the Ebers-Moll model of a NPN bipolar transistor with temperature variation.  The substrate is not shown on the icon layer, as it is assumed grounded. The transistor is assumed to be vertically diffused, as the substrate is connected to the collector, rather than to the base. <p>

The <b>NPNth</b> bipolar heating transistor is a directed FourPort.  The direction of positive power flow is assumed into the model at the base, <b>B</b>, and at the collector, <b>C</b>, whereas it is assumed out of the model at the emitter, <b>E</b>, and at the heat port, <b>Heat</b>. <p>

The causality of the <b>NPNth</b> model is relatively free.  However, if there is a causality stroke placed at the collector from the outside, the substrate capacitor will assume differential causality. <p>

<hr>

<pre>
Parameters:

 Bf:    Forward beta coefficient (default value = 50)

 Br:    Reverse beta coefficient (default value = 0.1)

 Is:    Transport saturation current (default value = 1e-16 Amp)

 Vak:   Inverse early voltage (default value = 0.02 /Volt)

 Tauf:  Ideal forward transit time (default value = 0.12e-9 sec)

 Taur:  Ideal reverse transit time (default value = 5e-9 sec)

 NF:    Forward current emission coefficient (default value = 1)

 NR:    Reverse current emission coefficient (default value = 1)

 Ccs:   Collector-substrate capacitance (default value = 1e-12 F)

 Cje:   Base-emitter zero bias depletion capacitance (default value = 0.4e-12 F)

 Cjc:   Base-collector zero bias depletion capacitance (default value = 0.5e-12 F)

 Phie:  Base-emitter diffusion voltage (default value = 0.8 Volt)

 Me:    Base-emitter gradation exponent (default value = 0.4)

 Phic:  Base-collector diffusion voltage (default value = 0.8 Volt)

 Mc:    Base-collector gradation exponent (default value = 0.333)

 EG:    Energy gap for temperature effect on saturation current (default value = 1.11 Volt)

 XTI:   Saturation current temperature exponent (default value = 3)

 XTB:   Forward and reverse beta temperature coefficient (default value = 0)

 Gbc:   Base-collector leakage conductance (default value = 1e-15 Mho)

 Gbe:   Base-emitter leakage conductance (default value = 1e-15 Mho)

 EMin:  Minimum exponent (default value = -100)

 EMax:  Maximum exponent (default value = 40)
</pre> <p>

<hr> <p>

References: <p>

Vlach, J. and K. Singal (1983), <I>Computer methods for circuit analysis and design</I>, Van Nostrand Reinhold, New York,
on page 317 ff.
</html>
"),     Icon(
          Text(
            extent=[-84,82; 62,-86],
            style(color=0),
            string="NPN"),
          Line(points=[0,80; 0,-80], style(
              color=9,
              rgbcolor={149,164,171},
              thickness=2,
              fillColor=74,
              rgbfillColor={0,0,185},
              fillPattern=1)),
          Line(points=[-20,-60; 0,-80], style(
              color=9,
              rgbcolor={149,164,171},
              thickness=2,
              fillColor=74,
              rgbfillColor={0,0,185},
              fillPattern=1)),
          Text(
            extent=[6,-90; 26,-110],
            style(color=9, rgbcolor={149,164,171}),
            string="E"),
          Text(
            extent=[-108,28; -88,8],
            string="B",
            style(color=9, rgbcolor={149,164,171})),
          Text(
            extent=[6,110; 26,90],
            style(color=9, rgbcolor={149,164,171}),
            string="C"),
          Line(points=[0,-80; 20,-60], style(
              color=9,
              rgbcolor={149,164,171},
              thickness=2,
              fillColor=74,
              rgbfillColor={0,0,185},
              fillPattern=1)),
          Text(
            extent=[84,28; 116,2],
            style(color=9, rgbcolor={149,164,171}),
            string="Heat"),
          Line(points=[-72,-50; 72,-50], style(color=3, rgbcolor={0,0,255})),
          Line(points=[72,-50; 50,-40], style(color=3, rgbcolor={0,0,255})),
          Line(points=[72,-50; 50,-60], style(color=3, rgbcolor={0,0,255})),
          Text(
            extent=[56,2; 92,-32],
            string="th",
            style(color=0, rgbcolor={0,0,0}))));
      Utilities.IB02 IB0_1(
        Tnom=Tnom,
        Bf=Bf,
        Br=Br,
        XTB=XTB) annotation (extent=[-80,-90; -60,-70]);
      Interfaces.BondCon H "Heat"
                           annotation (extent=[90,-10; 110,10]);
    Modelica.Blocks.Math.Gain Gain1(k=-1)
      annotation (extent=[-14,-90; -34,-110],  rotation=-90);
      Utilities.IC02 IC0_1(
        Tnom=Tnom,
        Br=Br,
        Vak=Vak,
        XTB=XTB) annotation (extent=[60,40; 80,60], rotation=-90);
      Utilities.CjS Dbc(
        Tnom=Tnom,
        Is=Is,
        EG=EG,
        XTI=XTI,
        EMin=EMin,
        EMax=EMax,
        Tau=Taur,
        Cj=Cjc,
        Phi=Phic,
        M=Mc,
        N=NR,
        Gb=Gbc) annotation (extent=[-70,30; -90,50]);
      Utilities.CjS Dbe(
        Tnom=Tnom,
        Is=Is,
        EG=EG,
        XTI=XTI,
        EMin=EMin,
        EMax=EMax,
        Tau=Tauf,
        Cj=Cje,
        Phi=Phie,
        M=Me,
        N=NF,
        Gb=Gbe) annotation (extent=[-70,-50; -90,-30]);
      Bonds.fBond B15 annotation (extent=[56,-60; 76,-40], rotation=90);
      Sensors.De Tdev
                     annotation (extent=[56,-40; 76,-20],  rotation=90);
      Bonds.eBond B16 annotation (extent=[56,-100; 76,-80],  rotation=90);
      Bonds.eBond B17 annotation (extent=[40,-80; 60,-60],   rotation=0);
      Junctions.J0p6 J0p6_1 annotation (extent=[60,-80; 80,-60]);
    Sources.mSf_int Ic0 annotation (extent=[80,10; 60,-10]);
      Bonds.fBond B18 annotation (extent=[70,-30; 90,-10], rotation=90);
    Sources.mSf_int Ib0
      annotation (extent=[-50,-90; -30,-70],   rotation=90);
      Bonds.fBond B19 annotation (extent=[74,-90; 94,-70], rotation=0);
    equation
      connect(J0p3_1.BondCon1,B2. BondCon1) annotation (points=[-6.12303e-016,
            10; -6.12303e-016,10.2; -6.00057e-016,10.2],
                                                       style(color=8, rgbcolor=
              {192,192,192}));
      connect(J1p3_1.BondCon2,B2. BondCon2) annotation (points=[6.12303e-016,30;
            6.12303e-016,30], style(color=8, rgbcolor={192,192,192}));
      connect(B1.BondCon1,J1p3_1. BondCon1) annotation (points=[-6.00057e-016,
            50.2; -6.00057e-016,56.1; 0,50; -6.12303e-016,50],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p4_1.BondCon3,B1. BondCon2) annotation (points=[0,70;
            6.12303e-016,70], style(color=8, rgbcolor={192,192,192}));
      connect(J0p4_1.BondCon4,C)  annotation (points=[0,90; 0,100], style(color=
             8, rgbcolor={192,192,192}));
      connect(J0p3_1.BondCon2,B3. BondCon1) annotation (points=[6.12303e-016,
            -10; 6.12303e-016,-10.2; -6.00057e-016,-10.2],style(color=8,
            rgbcolor={192,192,192}));
      connect(J1p3_2.BondCon1,B3. BondCon2) annotation (points=[-6.12303e-016,
            -30; 6.12303e-016,-30], style(color=8, rgbcolor={192,192,192}));
      connect(B4.BondCon1,J1p3_2. BondCon2) annotation (points=[-6.00057e-016,
            -50.2; -6.00057e-016,-55.1; 0,-50; 6.12303e-016,-50],
          style(color=8, rgbcolor={192,192,192}));
      connect(B4.BondCon2,J0p3_2. BondCon1) annotation (points=[6.12303e-016,
            -70; 6.12303e-016,-70], style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_2.BondCon2,E)  annotation (points=[-6.12303e-016,-90; 0,-90;
            0,-100], style(color=8, rgbcolor={192,192,192}));
      connect(B5.BondCon2,J1p3_3. BondCon1) annotation (points=[30,10; 30,10],
          style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_3.BondCon2,B6. BondCon1) annotation (points=[30,-10; 30,
            -10.2], style(color=8, rgbcolor={192,192,192}));
      connect(J0p4_1.BondCon2,B5. BondCon1) annotation (points=[10,80; 30,80;
            30,29.8], style(color=8, rgbcolor={192,192,192}));
      connect(B6.BondCon2,J0p3_2. BondCon3) annotation (points=[30,-30; 30,-80;
            10,-80], style(color=8, rgbcolor={192,192,192}));
      connect(B7.BondCon1,J0p4_1. BondCon1) annotation (points=[-10.2,80; -10,
            80], style(color=8, rgbcolor={192,192,192}));
      connect(C1.BondCon1,B7. BondCon2) annotation (points=[-30,80; -30,80],
          style(color=8, rgbcolor={192,192,192}));
      connect(B9.fBondCon1,J1p3_1. BondCon3) annotation (points=[-10,40; -10,40],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_3.BondCon1,B9. eBondCon1) annotation (points=[-30,40; -30,40],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_3.BondCon3,B10. eBondCon1) annotation (points=[-40,30; -40,
            30], style(color=8, rgbcolor={192,192,192}));
      connect(Vbc.BondCon1,B10. fBondCon1) annotation (points=[-40,10; -40,10],
          style(color=8, rgbcolor={192,192,192}));
      connect(B11.fBondCon1,J0p3_3. BondCon2) annotation (points=[-50,40; -50,
            40], style(color=8, rgbcolor={192,192,192}));
      connect(B12.fBondCon1,J1p3_2. BondCon3) annotation (points=[-10,-40; -10,
            -40], style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_4.BondCon1,B12. eBondCon1) annotation (points=[-30,-40; -30,
            -40], style(color=8, rgbcolor={192,192,192}));
      connect(B13.eBondCon1,J0p3_4. BondCon3) annotation (points=[-40,-50; -40,
            -50], style(color=8, rgbcolor={192,192,192}));
      connect(B14.fBondCon1,J0p3_4. BondCon2) annotation (points=[-50,-40; -50,
            -40], style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_1.BondCon3,B)  annotation (points=[-10,-6.12303e-016; -52,
            -6.12303e-016; -52,0; -100,0], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=74,
          rgbfillColor={0,0,185},
          fillPattern=1));
      connect(IB0_1.y, Gain1.u) annotation (points=[-59,-80; -56,-80; -56,-118;
            -24,-118; -24,-112], style(color=74, rgbcolor={0,0,127}));
      connect(Vbc.OutPort1, IC0_1.u3) annotation (points=[-40,-11; -40,-20; -20,
            -20; -20,56; 60,56], style(color=74, rgbcolor={0,0,127}));
      connect(Dbc.BondCon1, B11.eBondCon1) annotation (points=[-70,40; -70,40],
          style(color=8, rgbcolor={192,192,192}));
      connect(Dbc.y, IB0_1.u1) annotation (points=[-80,51; -80,60; -110,60;
            -110,-74; -82,-74], style(color=74, rgbcolor={0,0,127}));
      connect(Dbc.y, IC0_1.u1) annotation (points=[-80,51; -80,60; -110,60;
            -110,114; 76,114; 76,62], style(color=74, rgbcolor={0,0,127}));
      connect(Dbe.BondCon1, B14.eBondCon1) annotation (points=[-70,-40; -70,-40],
          style(color=8, rgbcolor={192,192,192}));
      connect(Dbe.y, IB0_1.u2) annotation (points=[-80,-29; -80,-20; -114,-20;
            -114,-86; -82,-86], style(color=74, rgbcolor={0,0,127}));
      connect(Dbe.y, IC0_1.u2) annotation (points=[-80,-29; -80,-20; -114,-20;
            -114,110; 64,110; 64,62], style(color=74, rgbcolor={0,0,127}));
      connect(Tdev.BondCon1, B15.fBondCon1) annotation (points=[66,-40; 66,-40],
          style(color=8, rgbcolor={192,192,192}));
      connect(Tdev.OutPort1, IC0_1.Tdev) annotation (points=[66,-19; 66,-10; 50,
            -10; 50,44; 60,44], style(color=74, rgbcolor={0,0,127}));
      connect(Tdev.OutPort1, IB0_1.Tdev) annotation (points=[66,-19; 66,-10; 50,
            -10; 50,-122; -70,-122; -70,-90], style(color=74, rgbcolor={0,0,127}));
      connect(Dbe.BondCon2, B17.fBondCon1) annotation (points=[-90,-40; -118,
            -40; -118,-126; 34,-126; 34,-70; 40,-70], style(color=8, rgbcolor={
              192,192,192}));
      connect(Dbc.BondCon2, B16.fBondCon1) annotation (points=[-90,40; -122,40;
            -122,-130; 66,-130; 66,-100], style(color=8, rgbcolor={192,192,192}));
      connect(J0p6_1.BondCon1, B17.eBondCon1) annotation (points=[60,-70; 60,-70], style(
            color=8, rgbcolor={192,192,192}));
      connect(J0p6_1.BondCon3, B16.eBondCon1) annotation (points=[66,-80; 66,
            -80], style(color=8, rgbcolor={192,192,192}));
      connect(B15.eBondCon1, J0p6_1.BondCon5) annotation (points=[66,-60; 66,
            -60], style(color=8, rgbcolor={192,192,192}));
      connect(Ic0.BondCon2, B8.fBondCon1) annotation (points=[60,0; 60,
            -1.22461e-015], style(color=8, rgbcolor={192,192,192}));
      connect(B8.eBondCon1, J1p3_3.BondCon3) annotation (points=[40,
            1.22461e-015; 40,-6.12303e-016], style(color=8, rgbcolor={192,192,
              192}));
      connect(B18.fBondCon1, Ic0.BondCon1) annotation (points=[80,-10; 80,0],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p6_1.BondCon6, B18.eBondCon1) annotation (points=[74,-60; 74,
            -40; 80,-40; 80,-30], style(color=8, rgbcolor={192,192,192}));
      connect(J0p6_1.BondCon2, H)  annotation (points=[80,-70; 90,-70; 90,0;
            100,0], style(color=8, rgbcolor={192,192,192}));
      connect(Ib0.BondCon2, B13.fBondCon1) annotation (points=[-40,-70; -40,-70],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p6_1.BondCon4, B19.eBondCon1) annotation (points=[74,-80; 74,
            -80], style(color=8, rgbcolor={192,192,192}));
      connect(B19.fBondCon1, Ib0.BondCon1) annotation (points=[94,-80; 100,-80;
            100,-134; -40,-134; -40,-90], style(color=8, rgbcolor={192,192,192}));
      connect(Gain1.y, Ib0.s) annotation (points=[-24,-89; -24,-80; -31,-80], style(
            color=74, rgbcolor={0,0,127}));
      connect(IC0_1.y, Ic0.s)
        annotation (points=[70,39; 70,9], style(color=74, rgbcolor={0,0,127}));
    end NPNth;

    model PNP "PNP transistor of the Modelica bond graph library"
      parameter Real Bf=50 "Forward beta coefficient";
      parameter Real Br=0.1 "Reverse beta coefficient";
      parameter Modelica.SIunits.Current Is=1.e-16
        "Transport saturation current";
      parameter Modelica.SIunits.InversePotential Vak=0.02
        "Inverse early voltage";
      parameter Modelica.SIunits.Time Tauf=0.12e-9 "Ideal forward transit time";
      parameter Modelica.SIunits.Time Taur=5e-9 "Ideal reverse transit time";
      parameter Modelica.SIunits.Capacitance Ccs=1e-12
        "Collector-substrate capacitance";
      parameter Modelica.SIunits.Capacitance Cje=0.4e-12
        "Base-emitter zero bias depletion capacitance";
      parameter Modelica.SIunits.Capacitance Cjc=0.5e-12
        "Base-collector zero bias depletion capacitance";
      parameter Modelica.SIunits.Voltage Phie=0.8
        "Base-emitter diffusion voltage";
      parameter Real Me=0.4 "Base-emitter gradation exponent";
      parameter Modelica.SIunits.Voltage Phic=0.8
        "Base-collector diffusion voltage";
      parameter Real Mc=0.333 "Base-collector gradation exponent";
      parameter Modelica.SIunits.Conductance Gbc=1e-15
        "Base-collector leakage conductance";
      parameter Modelica.SIunits.Conductance Gbe=1e-15
        "Base-emitter leakage conductance";
      parameter Modelica.SIunits.Voltage Vt=0.02585 "Thermal voltage";
      parameter Real EMin=-100 "if x < EMin, the exp(x) function is linearized";
      parameter Real EMax=40 "if x > EMax, the exp(x) function is linearized";
      annotation (
        Icon(
          Text(
            extent=[-80,82; 88,-86],
            style(color=0),
            string="PNP"),
          Text(
            extent=[-108,28; -88,8],
            string="B",
            style(color=9, rgbcolor={149,164,171})),
          Text(
            extent=[6,110; 26,90],
            style(color=9, rgbcolor={149,164,171}),
            string="C"),
          Text(
            extent=[6,-90; 26,-110],
            style(color=9, rgbcolor={149,164,171}),
            string="E"),
          Line(points=[0,80; 0,-80], style(
              color=9,
              rgbcolor={149,164,171},
              thickness=2,
              fillColor=74,
              rgbfillColor={0,0,185},
              fillPattern=1)),
          Line(points=[0,80; 20,60],    style(
              color=9,
              rgbcolor={149,164,171},
              thickness=2,
              fillColor=74,
              rgbfillColor={0,0,185},
              fillPattern=1)),
          Line(points=[-20,60; 0,80],  style(
              color=9,
              rgbcolor={149,164,171},
              thickness=2,
              fillColor=74,
              rgbfillColor={0,0,185},
              fillPattern=1))),
        Diagram(
          Text(
            extent=[6,-90; 26,-110],
            style(color=9, rgbcolor={149,164,171}),
            string="E"),
          Text(
            extent=[-108,28; -88,8],
            string="B",
            style(color=9, rgbcolor={149,164,171})),
          Text(
            extent=[6,110; 26,90],
            style(color=9, rgbcolor={149,164,171}),
            string="C"),
          Ellipse(extent=[-112,42; -108,38], style(
              color=9,
              rgbcolor={149,164,171},
              fillColor=74,
              rgbfillColor={0,0,185})),
          Ellipse(extent=[-122,-42; -118,-38], style(
              color=9,
              rgbcolor={149,164,171},
              fillColor=74,
              rgbfillColor={0,0,185})),
          Text(
            extent=[-98,48; -88,40],
            style(color=9, rgbcolor={149,164,171}),
            string="icb"),
          Text(
            extent=[-96,-66; -86,-74],
            style(color=9, rgbcolor={149,164,171}),
            string="icb"),
          Text(
            extent=[78,72; 88,64],
            style(color=9, rgbcolor={149,164,171}),
            string="icb"),
          Text(
            extent=[54,72; 64,64],
            style(color=9, rgbcolor={149,164,171}),
            string="ieb"),
          Text(
            extent=[-96,-86; -86,-94],
            style(color=9, rgbcolor={149,164,171}),
            string="ieb"),
          Text(
            extent=[-100,-32; -90,-40],
            style(color=9, rgbcolor={149,164,171}),
            string="ieb"),
          Text(
            extent=[-36,-12; -26,-20],
            style(color=9, rgbcolor={149,164,171}),
            string="vcb"),
          Text(
            extent=[48,58; 58,50],
            style(color=9, rgbcolor={149,164,171}),
            string="vcb"),
          Text(
            extent=[78,8; 88,0],
            style(color=9, rgbcolor={149,164,171}),
            string="IC0"),
          Text(
            extent=[-38,-90; -28,-98],
            style(color=9, rgbcolor={149,164,171}),
            string="IB0")),
        Documentation(info="<html>
The PNP element of the bond graph library implements the Ebers-Moll model of a PNP bipolar transistor.  The substrate is not shown on the icon layer, as it is assumed grounded. The transistor is assumed to be vertically diffused, as the substrate is connected to the collector, rather than to the base. <p>

The PNP bipolar transistor is a directed ThreePort.  The direction of positive power flow is assumed out of the model at the base, <b>B</b>, and at the collector, <b>C</b>, whereas it is assumed into the model at the emitter, <b>E</b>. <p>

The causality of the PNP model is relatively free.  However, if there is a causality stroke placed at the collector from the outside, the substrate capacitor will assume differential causality. <p>

<hr>

<pre>
Parameters:

 Bf:    Forward beta coefficient (default value = 50)

 Br:    Reverse beta coefficient (default value = 0.1)

 Is:    Transport saturation current (default value = 1e-16 Amp)

 Vak:   Inverse early voltage (default value = 0.02 /Volt)

 Tauf:  Ideal forward transit time (default value = 0.12e-9 sec)

 Taur:  Ideal reverse transit time (default value = 5e-9 sec)

 Ccs:   Collector-substrate capacitance (default value = 1e-12 F)

 Cje:   Base-emitter zero bias depletion capacitance (default value = 0.4e-12 F)

 Cjc:   Base-collector zero bias depletion capacitance (default value = 0.5e-12 F)

 Phie:  Base-emitter diffusion voltage (default value = 0.8 Volt)

 Me:    Base-emitter gradation exponent (default value = 0.4)

 Phic:  Base-collector diffusion voltage (default value = 0.8 Volt)

 Mc:    Base-collector gradation exponent (default value = 0.333)

 Gbc:   Base-collector leakage conductance (default value = 1e-15 Mho)

 Gbe:   Base-emitter leakage conductance (default value = 1e-15 Mho)

 Vt:    Thermal voltage (default value = 0.02585 Volt)

 EMin:  Minimum exponent (default value = -100)

 EMax:  Maximum exponent (default value = 40)
</pre> <p>

<hr> <p>

References: <p>

Vlach, J. and K. Singal (1983), <I>Computer methods for circuit analysis and design</I>, Van Nostrand Reinhold, New York,
on page 317 ff.
</html>
"));
      Interfaces.BondCon B "Base"
                           annotation (extent=[-110,-10; -90,10]);
      Interfaces.BondCon C "Collector"
                           annotation (extent=[-10,90; 10,110]);
      Interfaces.BondCon E "Emitter"
                           annotation (extent=[-10,-110; 10,-90]);
      Junctions.J0p3 J0p3_1 annotation (extent=[-10,-10; 10,10], rotation=-90);
      Bonds.Bond B1 annotation (extent=[-10,50; 10,70], rotation=270);
      Bonds.Bond B2 annotation (extent=[-10,10; 10,30], rotation=270);
      Bonds.Bond B3 annotation (extent=[-10,-30; 10,-10], rotation=90);
      Bonds.Bond B4 annotation (extent=[-10,-70; 10,-50], rotation=90);
      Junctions.J1p3 J1p3_1 annotation (extent=[-10,30; 10,50], rotation=-90);
      Junctions.J0p4 J0p4_1 annotation (extent=[-10,70; 10,90]);
      Junctions.J1p3 J1p3_2 annotation (extent=[-10,-50; 10,-30], rotation=-90);
      Junctions.J0p3 J0p3_2 annotation (extent=[10,-90; -10,-70], rotation=-90);
      Junctions.J1p3 J1p3_3 annotation (extent=[40,-10; 20,10], rotation=-90);
      Bonds.Bond B5 annotation (extent=[20,10; 40,30], rotation=90);
      Bonds.Bond B6 annotation (extent=[20,-30; 40,-10], rotation=90);
      Bonds.Bond B7 annotation (extent=[-30,70; -10,90], rotation=180);
      Passive.C C1(C=Ccs) annotation (extent=[-30,70; -50,90]);
      Bonds.eBond B8 annotation (extent=[40,-10; 60,10], rotation=180);
      Sources.mSf Ic0 annotation (extent=[60,-10; 80,10]);
      Junctions.J0p3 J0p3_3 annotation (extent=[-50,50; -30,30], rotation=-180);
      Bonds.eBond B9 annotation (extent=[-30,30; -10,50], rotation=180);
      Bonds.fBond B10 annotation (extent=[-50,10; -30,30], rotation=-90);
      Sensors.De Vcb annotation (extent=[-50,-10; -30,10], rotation=-90);
      Bonds.eBond B11 annotation (extent=[-70,30; -50,50], rotation=180);
      Utilities.Cj Dcb(
        Is=Is,
        Tau=Taur,
        Cj=Cjc,
        Phi=Phic,
        M=Mc,
        Gb=Gbc,
        Vt=Vt,
        EMin=EMin,
        EMax=EMax) annotation (extent=[-70,30; -90,50]);
      Bonds.eBond B12 annotation (extent=[-30,-50; -10,-30], rotation=180);
      Junctions.J0p3 J0p3_4
        annotation (extent=[-50,-30; -30,-50], rotation=-180);
      Bonds.eBond B13 annotation (extent=[-50,-70; -30,-50], rotation=90);
      Sources.mSf Ib0 annotation (extent=[-50,-90; -30,-70], rotation=-90);
      Bonds.eBond B14 annotation (extent=[-70,-50; -50,-30], rotation=180);
      Utilities.Cj Deb(
        Is=Is,
        Vt=Vt,
        EMin=EMin,
        EMax=EMax,
        Tau=Tauf,
        Cj=Cje,
        Phi=Phie,
        M=Me,
        Gb=Gbe) annotation (extent=[-70,-50; -90,-30]);
      Utilities.IC0 IC0_1(
        Br=Br,
        Vak=Vak) annotation (extent=[60,40; 80,60], rotation=-90);
      Utilities.IB0 IB0_1(Bf=Bf, Br=Br) annotation (extent=[-80,-90; -60,-70]);
    equation
      connect(J0p4_1.BondCon4,C)  annotation (points=[0,90; 0,100], style(color=
             8, rgbcolor={192,192,192}));
      connect(J0p3_2.BondCon2,E)  annotation (points=[-6.12303e-016,-90; 0,-90;
            0,-100], style(color=8, rgbcolor={192,192,192}));
      connect(B7.BondCon1,J0p4_1. BondCon1) annotation (points=[-10.2,80; -10,
            80], style(color=8, rgbcolor={192,192,192}));
      connect(C1.BondCon1,B7. BondCon2) annotation (points=[-30,80; -30,80],
          style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_3.BondCon3,B8. eBondCon1) annotation (points=[40,
            -6.12303e-016; 44,-6.12303e-016; 44,1.22461e-015; 40,1.22461e-015],
          style(color=8, rgbcolor={192,192,192}));
      connect(Ic0.BondCon1,B8. fBondCon1) annotation (points=[60,0; 64,0; 64,
            -1.22461e-015; 60,-1.22461e-015], style(color=8, rgbcolor={192,192,
              192}));
      connect(B9.fBondCon1,J1p3_1. BondCon3) annotation (points=[-10,40; -10,40],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_3.BondCon1,B9. eBondCon1) annotation (points=[-30,40; -30,40],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_3.BondCon3,B10. eBondCon1) annotation (points=[-40,30; -40,
            30], style(color=8, rgbcolor={192,192,192}));
      connect(Vcb.BondCon1,B10. fBondCon1) annotation (points=[-40,10; -40,10],
          style(color=8, rgbcolor={192,192,192}));
      connect(B11.fBondCon1,J0p3_3. BondCon2) annotation (points=[-50,40; -50,
            40], style(color=8, rgbcolor={192,192,192}));
      connect(Dcb.BondCon1,B11. eBondCon1) annotation (points=[-70,40; -70,40],
          style(color=8, rgbcolor={192,192,192}));
      connect(B12.fBondCon1,J1p3_2. BondCon3) annotation (points=[-10,-40; -10,
            -40], style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_4.BondCon1,B12. eBondCon1) annotation (points=[-30,-40; -30,
            -40], style(color=8, rgbcolor={192,192,192}));
      connect(B13.eBondCon1,J0p3_4. BondCon3) annotation (points=[-40,-50; -40,
            -50], style(color=8, rgbcolor={192,192,192}));
      connect(Ib0.BondCon1,B13. fBondCon1) annotation (points=[-40,-70; -40,-70],
          style(color=8, rgbcolor={192,192,192}));
      connect(B14.fBondCon1,J0p3_4. BondCon2) annotation (points=[-50,-40; -50,
            -40], style(color=8, rgbcolor={192,192,192}));
      connect(Deb.BondCon1,B14. eBondCon1) annotation (points=[-70,-40; -70,-40],
          style(color=8, rgbcolor={192,192,192}));
      connect(Vcb.OutPort1,IC0_1. u3) annotation (points=[-40,-11; -40,-20; -20,
            -20; -20,34; 50,34; 50,50; 60,50], style(color=74, rgbcolor={0,0,
              127}));
      connect(Dcb.y,IC0_1. u1) annotation (points=[-90,40; -110,40; -110,112;
            76,112; 76,62], style(color=74, rgbcolor={0,0,127}));
      connect(Dcb.y,IB0_1. u1) annotation (points=[-90,40; -110,40; -110,-74;
            -82,-74], style(color=74, rgbcolor={0,0,127}));
      connect(Deb.y,IB0_1. u2) annotation (points=[-90,-40; -120,-40; -120,-86;
            -82,-86], style(color=74, rgbcolor={0,0,127}));
      connect(Deb.y,IC0_1. u2) annotation (points=[-90,-40; -120,-40; -120,122;
            64,122; 64,62], style(color=74, rgbcolor={0,0,127}));
      connect(J0p3_1.BondCon3,B)  annotation (points=[-10,-6.12303e-016; -52,
            -6.12303e-016; -52,0; -100,0], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=74,
          rgbfillColor={0,0,185},
          fillPattern=1));
      connect(J1p3_1.BondCon1, B1.BondCon2) annotation (points=[-6.12303e-016,
            50; -1.83691e-015,50], style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_1.BondCon1, B2.BondCon2) annotation (points=[-6.12303e-016,
            10; -1.83691e-015,10], style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_1.BondCon2, B3.BondCon2) annotation (points=[6.12303e-016,
            -10; 6.12303e-016,-10], style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_2.BondCon2, B4.BondCon2) annotation (points=[6.12303e-016,
            -50; 6.12303e-016,-50], style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_3.BondCon1, B5.BondCon1) annotation (points=[30,10; 30,10.2],
          style(color=8, rgbcolor={192,192,192}));
      connect(B5.BondCon2, J0p4_1.BondCon2) annotation (points=[30,30; 30,80;
            10,80], style(color=8, rgbcolor={192,192,192}));
      connect(B6.BondCon2, J1p3_3.BondCon2) annotation (points=[30,-10; 30,-10],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_2.BondCon3, B6.BondCon1) annotation (points=[10,-80; 30,-80;
            30,-29.8], style(color=8, rgbcolor={192,192,192}));
      connect(B1.BondCon1, J0p4_1.BondCon3) annotation (points=[1.80017e-015,
            69.8; 1.80017e-015,65.9; 0,65.9; 0,70], style(color=8, rgbcolor={
              192,192,192}));
      connect(B2.BondCon1, J1p3_1.BondCon2) annotation (points=[1.80017e-015,
            29.8; 1.80017e-015,27.9; 6.12303e-016,27.9; 6.12303e-016,30], style(
            color=8, rgbcolor={192,192,192}));
      connect(J1p3_2.BondCon1, B3.BondCon1) annotation (points=[-6.12303e-016,
            -30; -6.00057e-016,-30; -6.00057e-016,-29.8], style(color=8,
            rgbcolor={192,192,192}));
      connect(J0p3_2.BondCon1, B4.BondCon1) annotation (points=[6.12303e-016,
            -70; -6.00057e-016,-70; -6.00057e-016,-69.8], style(color=8,
            rgbcolor={192,192,192}));
      connect(IC0_1.y, Ic0.s) annotation (points=[70,39; 70,30; 90,30; 90,0; 79,
            0], style(color=74, rgbcolor={0,0,127}));
      connect(IB0_1.y, Ib0.s) annotation (points=[-59,-80; -50,-80; -50,-96;
            -40,-96; -40,-89], style(color=74, rgbcolor={0,0,127}));
    end PNP;

    model PNPth
      "PNP transistor of the Modelica bond graph library with temperature variation"
      parameter Real Bf=50 "Forward beta coefficient";
      parameter Real Br=0.1 "Reverse beta coefficient";
      parameter Modelica.SIunits.Temperature Tnom=300.15
        "Reference temperature";
      parameter Modelica.SIunits.Current Is=1.e-16
        "Transport saturation current";
      parameter Modelica.SIunits.InversePotential Vak=0.02
        "Inverse early voltage";
      parameter Modelica.SIunits.Time Tauf=0.12e-9 "Ideal forward transit time";
      parameter Modelica.SIunits.Time Taur=5e-9 "Ideal reverse transit time";
      parameter Real NF=1 "Forward current emission coefficient";
      parameter Real NR=1 "Reverse current emission coefficient";
      parameter Modelica.SIunits.Capacitance Ccs=1e-12
        "Collector-substrate capacitance";
      parameter Modelica.SIunits.Capacitance Cje=0.4e-12
        "Base-emitter zero bias depletion capacitance";
      parameter Modelica.SIunits.Capacitance Cjc=0.5e-12
        "Base-collector zero bias depletion capacitance";
      parameter Modelica.SIunits.Voltage Phie=0.8
        "Base-emitter diffusion voltage";
      parameter Real Me=0.4 "Base-emitter gradation exponent";
      parameter Modelica.SIunits.Voltage Phic=0.8
        "Base-collector diffusion voltage";
      parameter Real Mc=0.333 "Base-collector gradation exponent";
      parameter Modelica.SIunits.Voltage EG=1.11
        "Energy gap for temperature effect on saturation current";
      parameter Real XTI=3 "Saturation current temperature exponent";
      parameter Real XTB=0 "Forward and reverse beta temperature coefficient";
      parameter Modelica.SIunits.Conductance Gbc=1e-15
        "Base-collector leakage conductance";
      parameter Modelica.SIunits.Conductance Gbe=1e-15
        "Base-emitter leakage conductance";
      parameter Real EMin=-100 "if x < EMin, the exp(x) function is linearized";
      parameter Real EMax=40 "if x > EMax, the exp(x) function is linearized";
      Interfaces.BondCon B "Base"
                           annotation (extent=[-110,-10; -90,10]);
      Interfaces.BondCon C "Collector"
                           annotation (extent=[-10,90; 10,110]);
      Interfaces.BondCon E "Emitter"
                           annotation (extent=[-10,-110; 10,-90]);
      Junctions.J0p3 J0p3_1 annotation (extent=[-10,-10; 10,10], rotation=-90);
      Bonds.Bond B1 annotation (extent=[-10,50; 10,70], rotation=270);
      Bonds.Bond B2 annotation (extent=[-10,10; 10,30], rotation=270);
      Bonds.Bond B3 annotation (extent=[-10,-30; 10,-10], rotation=90);
      Bonds.Bond B4 annotation (extent=[-10,-70; 10,-50], rotation=90);
      Junctions.J1p3 J1p3_1 annotation (extent=[-10,30; 10,50], rotation=-90);
      Junctions.J0p4 J0p4_1 annotation (extent=[-10,70; 10,90]);
      Junctions.J1p3 J1p3_2 annotation (extent=[-10,-50; 10,-30], rotation=-90);
      Junctions.J0p3 J0p3_2 annotation (extent=[10,-90; -10,-70], rotation=-90);
      Junctions.J1p3 J1p3_3 annotation (extent=[40,-10; 20,10], rotation=-90);
      Bonds.Bond B5 annotation (extent=[20,10; 40,30], rotation=90);
      Bonds.Bond B6 annotation (extent=[20,-30; 40,-10], rotation=90);
      Bonds.Bond B7 annotation (extent=[-30,70; -10,90], rotation=180);
      Passive.C C1(C=Ccs) annotation (extent=[-30,70; -50,90]);
      Bonds.eBond B8 annotation (extent=[40,-10; 60,10], rotation=180);
      Junctions.J0p3 J0p3_3 annotation (extent=[-50,50; -30,30], rotation=-180);
      Bonds.eBond B9 annotation (extent=[-30,30; -10,50], rotation=180);
      Bonds.fBond B10 annotation (extent=[-50,10; -30,30], rotation=-90);
      Sensors.De Vcb annotation (extent=[-50,-10; -30,10], rotation=-90);
      Bonds.eBond B11 annotation (extent=[-70,30; -50,50], rotation=180);
      Bonds.eBond B12 annotation (extent=[-30,-50; -10,-30], rotation=180);
      Junctions.J0p3 J0p3_4
        annotation (extent=[-50,-30; -30,-50], rotation=-180);
      Bonds.eBond B13 annotation (extent=[-50,-70; -30,-50], rotation=90);
      Bonds.eBond B14 annotation (extent=[-70,-50; -50,-30], rotation=180);
      annotation (
        Diagram(
          Text(
            extent=[6,-90; 26,-110],
            style(color=9, rgbcolor={149,164,171}),
            string="E"),
          Text(
            extent=[-108,28; -88,8],
            string="B",
            style(color=9, rgbcolor={149,164,171})),
          Text(
            extent=[6,110; 26,90],
            style(color=9, rgbcolor={149,164,171}),
            string="C"),
          Ellipse(extent=[-112,62; -108,58], style(
              color=9,
              rgbcolor={149,164,171},
              fillColor=74,
              rgbfillColor={0,0,185})),
          Ellipse(extent=[-116,-22; -112,-18], style(
              color=9,
              rgbcolor={149,164,171},
              fillColor=74,
              rgbfillColor={0,0,185})),
          Text(
            extent=[-80,60; -70,52],
            style(color=9, rgbcolor={149,164,171}),
            string="icb"),
          Text(
            extent=[-96,-66; -86,-74],
            style(color=9, rgbcolor={149,164,171}),
            string="icb"),
          Text(
            extent=[78,72; 88,64],
            style(color=9, rgbcolor={149,164,171}),
            string="icb"),
          Text(
            extent=[54,72; 64,64],
            style(color=9, rgbcolor={149,164,171}),
            string="ieb"),
          Text(
            extent=[-96,-86; -86,-94],
            style(color=9, rgbcolor={149,164,171}),
            string="ieb"),
          Text(
            extent=[-80,-20; -70,-28],
            style(color=9, rgbcolor={149,164,171}),
            string="ieb"),
          Text(
            extent=[-36,-12; -26,-20],
            style(color=9, rgbcolor={149,164,171}),
            string="vcb"),
          Text(
            extent=[46,56; 56,48],
            style(color=9, rgbcolor={149,164,171}),
            string="vcb"),
          Text(
            extent=[72,18; 82,10],
            style(color=9, rgbcolor={149,164,171}),
            string="IC0"),
          Text(
            extent=[-28,-72; -18,-80],
            style(color=9, rgbcolor={149,164,171}),
            string="IB0"),
          Ellipse(extent=[48,-12; 52,-8],      style(
              color=9,
              rgbcolor={149,164,171},
              fillColor=74,
              rgbfillColor={0,0,185})),
          Text(
            extent=[88,26; 120,0],
            style(color=9, rgbcolor={149,164,171}),
            string="Heat"),
          Text(
            extent=[52,38; 64,30],
            style(color=9, rgbcolor={149,164,171}),
            string="Tdev"),
          Text(
            extent=[-84,-92; -72,-100],
            style(color=9, rgbcolor={149,164,171}),
            string="Tdev")),
        Icon(
          Text(
            extent=[-82,82; 68,-86],
            style(color=0),
            string="PNP"),
          Line(points=[0,80; 0,-80], style(
              color=9,
              rgbcolor={149,164,171},
              thickness=2,
              fillColor=74,
              rgbfillColor={0,0,185},
              fillPattern=1)),
          Line(points=[0,80; 20,60],    style(
              color=9,
              rgbcolor={149,164,171},
              thickness=2,
              fillColor=74,
              rgbfillColor={0,0,185},
              fillPattern=1)),
          Line(points=[-20,60; 0,80],  style(
              color=9,
              rgbcolor={149,164,171},
              thickness=2,
              fillColor=74,
              rgbfillColor={0,0,185},
              fillPattern=1)),
          Line(points=[-72,-50; 72,-50], style(color=3, rgbcolor={0,0,255})),
          Line(points=[72,-50; 50,-60], style(color=3, rgbcolor={0,0,255})),
          Line(points=[72,-50; 50,-40], style(color=3, rgbcolor={0,0,255})),
          Text(
            extent=[84,28; 116,2],
            style(color=9, rgbcolor={149,164,171}),
            string="Heat"),
          Text(
            extent=[6,110; 26,90],
            style(color=9, rgbcolor={149,164,171}),
            string="C"),
          Text(
            extent=[-108,28; -88,8],
            string="B",
            style(color=9, rgbcolor={149,164,171})),
          Text(
            extent=[6,-90; 26,-110],
            style(color=9, rgbcolor={149,164,171}),
            string="E"),
          Text(
            extent=[58,0; 94,-34],
            string="th",
            style(color=0, rgbcolor={0,0,0}))),
        Documentation(info="<html>
The <b>PNPth</b> element of the bond graph library implements the Ebers-Moll model of a PNP bipolar transistor with temperature variation.  The substrate is not shown on the icon layer, as it is assumed grounded. The transistor is assumed to be vertically diffused, as the substrate is connected to the collector, rather than to the base. <p>

The <b>PNPth</b> bipolar heating transistor is a directed FourPort.  The direction of positive power flow is assumed out of the model at the base, <b>B</b>, at the collector, <b>C</b>, and at the heat port, <b>Heat</b>, whereas it is assumed into the model at the emitter, <b>E</b>. <p>

The causality of the <b>PNPth</b> model is relatively free.  However, if there is a causality stroke placed at the collector from the outside, the substrate capacitor will assume differential causality. <p>

<hr>

<pre>
Parameters:

 Bf:    Forward beta coefficient (default value = 50)

 Br:    Reverse beta coefficient (default value = 0.1)

 Is:    Transport saturation current (default value = 1e-16 Amp)

 Vak:   Inverse early voltage (default value = 0.02 /Volt)

 Tauf:  Ideal forward transit time (default value = 0.12e-9 sec)

 Taur:  Ideal reverse transit time (default value = 5e-9 sec)

 NF:    Forward current emission coefficient (default value = 1)

 NR:    Reverse current emission coefficient (default value = 1)

 Ccs:   Collector-substrate capacitance (default value = 1e-12 F)

 Cje:   Base-emitter zero bias depletion capacitance (default value = 0.4e-12 F)

 Cjc:   Base-collector zero bias depletion capacitance (default value = 0.5e-12 F)

 Phie:  Base-emitter diffusion voltage (default value = 0.8 Volt)

 Me:    Base-emitter gradation exponent (default value = 0.4)

 Phic:  Base-collector diffusion voltage (default value = 0.8 Volt)

 Mc:    Base-collector gradation exponent (default value = 0.333)

 EG:    Energy gap for temperature effect on saturation current (default value = 1.11 Volt)

 XTI:   Saturation current temperature exponent (default value = 3)

 XTB:   Forward and reverse beta temperature coefficient (default value = 0)

 Gbc:   Base-collector leakage conductance (default value = 1e-15 Mho)

 Gbe:   Base-emitter leakage conductance (default value = 1e-15 Mho)

 Vt:    Thermal voltage (default value = 0.02585 Volt)

 EMin:  Minimum exponent (default value = -100)

 EMax:  Maximum exponent (default value = 40)
</pre> <p>

<hr> <p>

References: <p>

Vlach, J. and K. Singal (1983), <I>Computer methods for circuit analysis and design</I>, Van Nostrand Reinhold, New York,
on page 317 ff.
</html>
"));
      Utilities.CjS Dcb(
        Tnom=Tnom,
        Is=Is,
        EG=EG,
        XTI=XTI,
        EMin=EMin,
        EMax=EMax,
        Tau=Taur,
        Cj=Cjc,
        Phi=Phic,
        M=Mc,
        N=NR,
        Gb=Gbc) annotation (extent=[-70,30; -90,50]);
      Utilities.IB02 IB0_1(
        Tnom=Tnom,
        Bf=Bf,
        Br=Br,
        XTB=XTB) annotation (extent=[-80,-90; -60,-70]);
    Modelica.Blocks.Math.Gain Gain1(k=-1)
      annotation (extent=[-14,-90; -34,-110],  rotation=-90);
      Utilities.IC02 IC0_1(
        Tnom=Tnom,
        Br=Br,
        Vak=Vak,
        XTB=XTB) annotation (extent=[60,40; 80,60], rotation=-90);
      Utilities.CjS Deb(
        Tnom=Tnom,
        Is=Is,
        EG=EG,
        XTI=XTI,
        EMin=EMin,
        EMax=EMax,
        Tau=Tauf,
        Cj=Cje,
        Phi=Phie,
        M=Me,
        N=NF,
        Gb=Gbe) annotation (extent=[-70,-50; -90,-30]);
      Interfaces.BondCon H "Heat"
                           annotation (extent=[90,-10; 110,10]);
      Bonds.fBond B15 annotation (extent=[56,-60; 76,-40], rotation=90);
      Bonds.eBond B17 annotation (extent=[40,-80; 60,-60],   rotation=0);
      Bonds.eBond B16 annotation (extent=[56,-100; 76,-80],  rotation=90);
      Sensors.De Tdev
                     annotation (extent=[56,-40; 76,-20],  rotation=90);
      Junctions.J0p6 J0p6_1 annotation (extent=[60,-80; 80,-60]);
    Sources.mSf_int Ic0 annotation (extent=[80,10; 60,-10]);
      Bonds.fBond B18 annotation (extent=[70,-30; 90,-10], rotation=90);
      Bonds.fBond B19 annotation (extent=[74,-90; 94,-70], rotation=0);
    Sources.mSf_int Ib0
      annotation (extent=[-50,-90; -30,-70],   rotation=90);
    equation
      connect(J0p4_1.BondCon4,C)  annotation (points=[0,90; 0,100], style(color=
             8, rgbcolor={192,192,192}));
      connect(J0p3_2.BondCon2,E)  annotation (points=[-6.12303e-016,-90; 0,-90;
            0,-100], style(color=8, rgbcolor={192,192,192}));
      connect(B7.BondCon1,J0p4_1. BondCon1) annotation (points=[-10.2,80; -10,
            80], style(color=8, rgbcolor={192,192,192}));
      connect(C1.BondCon1,B7. BondCon2) annotation (points=[-30,80; -30,80],
          style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_3.BondCon3,B8. eBondCon1) annotation (points=[40,
            -6.12303e-016; 44,-6.12303e-016; 44,1.22461e-015; 40,1.22461e-015],
          style(color=8, rgbcolor={192,192,192}));
      connect(B9.fBondCon1,J1p3_1. BondCon3) annotation (points=[-10,40; -10,40],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_3.BondCon1,B9. eBondCon1) annotation (points=[-30,40; -30,40],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_3.BondCon3,B10. eBondCon1) annotation (points=[-40,30; -40,
            30], style(color=8, rgbcolor={192,192,192}));
      connect(Vcb.BondCon1,B10. fBondCon1) annotation (points=[-40,10; -40,10],
          style(color=8, rgbcolor={192,192,192}));
      connect(B11.fBondCon1,J0p3_3. BondCon2) annotation (points=[-50,40; -50,
            40], style(color=8, rgbcolor={192,192,192}));
      connect(B12.fBondCon1,J1p3_2. BondCon3) annotation (points=[-10,-40; -10,
            -40], style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_4.BondCon1,B12. eBondCon1) annotation (points=[-30,-40; -30,
            -40], style(color=8, rgbcolor={192,192,192}));
      connect(B13.eBondCon1,J0p3_4. BondCon3) annotation (points=[-40,-50; -40,
            -50], style(color=8, rgbcolor={192,192,192}));
      connect(B14.fBondCon1,J0p3_4. BondCon2) annotation (points=[-50,-40; -50,
            -40], style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_1.BondCon3,B)  annotation (points=[-10,-6.12303e-016; -52,
            -6.12303e-016; -52,0; -100,0], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=74,
          rgbfillColor={0,0,185},
          fillPattern=1));
      connect(J1p3_1.BondCon1,B1. BondCon2) annotation (points=[-6.12303e-016,
            50; -1.83691e-015,50], style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_1.BondCon1,B2. BondCon2) annotation (points=[-6.12303e-016,
            10; -1.83691e-015,10], style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_1.BondCon2,B3. BondCon2) annotation (points=[6.12303e-016,
            -10; 6.12303e-016,-10], style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_2.BondCon2,B4. BondCon2) annotation (points=[6.12303e-016,
            -50; 6.12303e-016,-50], style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_3.BondCon1,B5. BondCon1) annotation (points=[30,10; 30,10.2],
          style(color=8, rgbcolor={192,192,192}));
      connect(B5.BondCon2,J0p4_1. BondCon2) annotation (points=[30,30; 30,80;
            10,80], style(color=8, rgbcolor={192,192,192}));
      connect(B6.BondCon2,J1p3_3. BondCon2) annotation (points=[30,-10; 30,-10],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_2.BondCon3,B6. BondCon1) annotation (points=[10,-80; 30,-80;
            30,-29.8], style(color=8, rgbcolor={192,192,192}));
      connect(B1.BondCon1,J0p4_1. BondCon3) annotation (points=[1.80017e-015,
            69.8; 1.80017e-015,65.9; 0,65.9; 0,70], style(color=8, rgbcolor={
              192,192,192}));
      connect(B2.BondCon1,J1p3_1. BondCon2) annotation (points=[1.80017e-015,
            29.8; 1.80017e-015,27.9; 0,30; 6.12303e-016,30],              style(
            color=8, rgbcolor={192,192,192}));
      connect(J1p3_2.BondCon1,B3. BondCon1) annotation (points=[-6.12303e-016,
            -30; -6.12303e-016,-29.8; -6.00057e-016,-29.8],
                                                          style(color=8,
            rgbcolor={192,192,192}));
      connect(J0p3_2.BondCon1,B4. BondCon1) annotation (points=[6.12303e-016,
            -70; 6.12303e-016,-69.8; -6.00057e-016,-69.8],style(color=8,
            rgbcolor={192,192,192}));
      connect(Dcb.BondCon1, B11.eBondCon1) annotation (points=[-70,40; -70,40],
          style(color=8, rgbcolor={192,192,192}));
      connect(IB0_1.y, Gain1.u) annotation (points=[-59,-80; -56,-80; -56,-116;
            -24,-116; -24,-112], style(color=74, rgbcolor={0,0,127}));
      connect(Dcb.y, IB0_1.u1) annotation (points=[-80,51; -80,60; -110,60;
            -110,-74; -82,-74], style(color=74, rgbcolor={0,0,127}));
      connect(Vcb.OutPort1, IC0_1.u3) annotation (points=[-40,-11; -40,-20; -20,
            -20; -20,56; 60,56], style(color=74, rgbcolor={0,0,127}));
      connect(Dcb.y, IC0_1.u1) annotation (points=[-80,51; -80,60; -110,60;
            -110,114; 76,114; 76,62], style(color=74, rgbcolor={0,0,127}));
      connect(Deb.BondCon1, B14.eBondCon1) annotation (points=[-70,-40; -70,-40],
          style(color=8, rgbcolor={192,192,192}));
      connect(Deb.y, IB0_1.u2) annotation (points=[-80,-29; -80,-20; -114,-20;
            -114,-86; -82,-86], style(color=74, rgbcolor={0,0,127}));
      connect(Deb.y, IC0_1.u2) annotation (points=[-80,-29; -80,-20; -114,-20;
            -114,110; 64,110; 64,62], style(color=74, rgbcolor={0,0,127}));
      connect(B15.fBondCon1, Tdev.BondCon1) annotation (points=[66,-40; 66,-40],
          style(color=8, rgbcolor={192,192,192}));
      connect(Tdev.OutPort1, IC0_1.Tdev) annotation (points=[66,-19; 66,-10; 50,
            -10; 50,44; 60,44], style(color=74, rgbcolor={0,0,127}));
      connect(Tdev.OutPort1, IB0_1.Tdev) annotation (points=[66,-19; 66,-10; 50,
            -10; 50,-120; -70,-120; -70,-90], style(color=74, rgbcolor={0,0,127}));
      connect(Deb.BondCon2, B17.fBondCon1) annotation (points=[-90,-40; -118,
            -40; -118,-124; 34,-124; 34,-70; 40,-70], style(color=8, rgbcolor={
              192,192,192}));
      connect(Dcb.BondCon2, B16.fBondCon1) annotation (points=[-90,40; -122,40;
            -122,-128; 66,-128; 66,-100], style(color=8, rgbcolor={192,192,192}));
      connect(J0p6_1.BondCon3, B16.eBondCon1) annotation (points=[66,-80; 66,-80], style(
            color=8, rgbcolor={192,192,192}));
      connect(J0p6_1.BondCon1, B17.eBondCon1) annotation (points=[60,-70; 60,
            -70], style(color=8, rgbcolor={192,192,192}));
      connect(J0p6_1.BondCon5, B15.eBondCon1) annotation (points=[66,-60; 66,
            -60], style(color=8, rgbcolor={192,192,192}));
      connect(Ic0.BondCon2, B8.fBondCon1) annotation (points=[60,0; 63,0; 63,
            -1.22461e-015; 60,-1.22461e-015], style(color=8, rgbcolor={192,192,
              192}));
      connect(J0p6_1.BondCon6, B18.eBondCon1) annotation (points=[74,-60; 74,
            -40; 80,-40; 80,-30], style(color=8, rgbcolor={192,192,192}));
      connect(B18.fBondCon1, Ic0.BondCon1) annotation (points=[80,-10; 80,0],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p6_1.BondCon4, B19.eBondCon1) annotation (points=[74,-80; 74,
            -80], style(color=8, rgbcolor={192,192,192}));
      connect(J0p6_1.BondCon2, H)  annotation (points=[80,-70; 90,-70; 90,0;
            100,0], style(color=8, rgbcolor={192,192,192}));
      connect(Ib0.BondCon2, B13.fBondCon1) annotation (points=[-40,-70; -40,-70],
          style(color=8, rgbcolor={192,192,192}));
      connect(B19.fBondCon1, Ib0.BondCon1) annotation (points=[94,-80; 100,-80;
            100,-132; -40,-132; -40,-90], style(color=8, rgbcolor={192,192,192}));
      connect(IC0_1.y, Ic0.s)
        annotation (points=[70,39; 70,9], style(color=74, rgbcolor={0,0,127}));
      connect(Gain1.y, Ib0.s) annotation (points=[-24,-89; -24,-80; -31,-80], style(
            color=74, rgbcolor={0,0,127}));
    end PNPth;

    package Utilities "Utility models for bond graph semiconductor package"
      annotation (preferedView="info",
        Icon(
          Rectangle(extent=[-100,-100; 80,50], style(
              color=42,
              rgbcolor={127,0,0},
              fillColor=30,
              rgbfillColor={235,235,235},
              fillPattern=1)),
          Polygon(points=[-100,50; -80,70; 100,70; 80,50; -100,50], style(
              color=42,
              rgbcolor={127,0,0},
              fillColor=30,
              rgbfillColor={235,235,235},
              fillPattern=1)),
          Polygon(points=[100,70; 100,-80; 80,-100; 80,50; 100,70], style(
              color=42,
              rgbcolor={127,0,0},
              fillColor=30,
              rgbfillColor={235,235,235},
              fillPattern=1)),
          Text(
            extent=[-85,35; 65,-85],
            string="Library",
            style(color=42, rgbcolor={127,0,0})),
          Text(
            extent=[-120,122; 120,73],
            string="%name",
            style(color=1))), Documentation(info="<html>
This package contains some utility models used inside some of the semiconductor bond graph models.
</html>
"));
      block nlC "Non-linear capacitance computation of Ebers-Moll BJT"
        extends Modelica.Blocks.Interfaces.SISO;
        parameter Modelica.SIunits.Current Is=1.e-16
          "Transport saturation current";
        parameter Modelica.SIunits.Time Tau=5e-9 "Ideal transit time";
        parameter Modelica.SIunits.Voltage Phi=0.8 "Diffusion voltage";
        parameter Modelica.SIunits.Capacitance Cj=0.5e-12
          "Zero bias depletion capacitance";
        parameter Real M=0.333 "Gradation exponent";
        parameter Modelica.SIunits.Voltage Vt=0.02585 "Thermal voltage";
        parameter Real EMin=-100
          "if x < EMin, the exp(x) function is linearized";
        parameter Real EMax=40 "if x > EMax, the exp(x) function is linearized";
        output Modelica.SIunits.Voltage v
          "Voltage across the junction capacitance";
        output Modelica.SIunits.Capacitance C
          "Non-linear junction capacitance value";

      protected
        Modelica.SIunits.Capacitance Cjunc "Junction capacitance";
        Modelica.SIunits.Capacitance Cdepl "Depletion capacitance";
        Real ExMin;
        Real ExMax;

        function pow "Just a helper function for x^y"
          input Real x;
          input Real y;
          output Real z;
        algorithm
          z:=x^y;
        end pow;
        annotation (
          Icon(Text(
              extent=[-76,64; 78,-64],
              string="nl C",
              style(color=0, rgbcolor={0,0,0})), Text(
              extent=[-80,-52; 84,-76],
              style(color=0, rgbcolor={0,0,0}),
              string="Ebers-Moll BJT")),
          Diagram(Text(
              extent=[-76,64; 78,-64],
              string="nl C",
              style(color=0, rgbcolor={0,0,0})), Text(
              extent=[-80,-52; 84,-76],
              style(color=0, rgbcolor={0,0,0}),
              string="Ebers-Moll BJT")),
          Documentation(info="<html>
This block computes the junction capacitance value of a junction capacitance in accordance with the Ebers-Moll junction diode model.  No temperature variation is considered in this model.
</html>
"));
      equation
        ExMin = exp(EMin);
        ExMax = exp(EMax);
        v = u;
        Cjunc = if (v/Vt < EMin) then Tau*Is/Vt*ExMin*(v/Vt - EMin + 1) else
                if (v/Vt > EMax) then Tau*Is/Vt*ExMax*(v/Vt - EMax + 1) else Tau*Is/Vt*exp(v/Vt);
        Cdepl = if (v/Phi > 0) then Cj*(1 + M*v/Phi) else Cj*pow(1 - v/Phi, -M);
        C = Cjunc + Cdepl;
        y = C;
      end nlC;

      block nlC2
        "Non-linear capacitance computation of Ebers-Moll BJT with thermal port"
        extends Modelica.Blocks.Interfaces.SISO;
        constant Real k=1.3806226e-23 "Boltzmann's constant";
        constant Modelica.SIunits.ElectricCharge q=1.6021918e-19
          "Electron charge";
        parameter Modelica.SIunits.Temperature Tnom=300.15
          "Reference temperature";
        parameter Modelica.SIunits.Current Is=1.e-16
          "Transport saturation current";
        parameter Modelica.SIunits.Time Tau=5e-9 "Ideal transit time";
        parameter Modelica.SIunits.Voltage Phi=0.8 "Diffusion voltage";
        parameter Modelica.SIunits.Capacitance Cj=0.5e-12
          "Zero bias depletion capacitance";
        parameter Real M=0.333 "Gradation exponent";
        parameter Modelica.SIunits.Voltage EG=1.11
          "Energy gap for temperature effect on saturation current";
        parameter Real N=1 "Current emission coefficient";
        parameter Real XTI=3 "Saturation current temperature exponent";
        parameter Real EMin=-100
          "if x < EMin, the exp(x) function is linearized";
        parameter Real EMax=40 "if x > EMax, the exp(x) function is linearized";
        output Modelica.SIunits.Voltage v
          "Voltage across the junction capacitance";
        output Modelica.SIunits.Capacitance C
          "Non-linear junction capacitance value";
        output Modelica.SIunits.Voltage Vt "Thermal voltage";
        output Modelica.SIunits.Current ISval
          "Saturation current at device temperature";

      protected
        Modelica.SIunits.Capacitance Cjunc "Junction capacitance";
        Modelica.SIunits.Capacitance Cdepl "Depletion capacitance";
        Real RTemp "Temperature quotient";
        Real ExMin;
        Real ExMax;
        Real et;

        function pow "Just a helper function for x^y"
          input Real x;
          input Real y;
          output Real z;
        algorithm
          z:=x^y;
        end pow;
        annotation (
          Diagram(Text(
              extent=[-78,74; 76,-54],
              string="nl C",
              style(color=0, rgbcolor={0,0,0})), Text(
              extent=[-82,-42; 82,-66],
              style(color=0, rgbcolor={0,0,0}),
              string="Ebers-Moll BJT")),
          Icon(   Text(
              extent=[-78,76; 76,-52],
              string="nl C",
              style(color=0, rgbcolor={0,0,0})), Text(
              extent=[-82,-40; 82,-64],
              style(color=0, rgbcolor={0,0,0}),
              string="Ebers-Moll BJT")),
          Documentation(info="<html>
This block computes the junction capacitance value of a junction capacitance in accordance with the Ebers-Moll junction diode model.  Temperature variation is considered in this model.
</html>
"));
      public
        Modelica.Blocks.Interfaces.RealInput Tdev(redeclare type SignalType =
              Modelica.SIunits.Temperature) "Device temperature"
          annotation (extent=[-10,-110; 10,-90], rotation=90);
      equation
        /* Compute thermal voltage as function of temperature */
        Vt = k*Tdev/q;

        /* Compute temperature dependence of saturation current */
        RTemp = Tdev/Tnom;
        ISval = Is*exp((RTemp - 1)*EG/Vt + XTI*ln(RTemp));

        /* Compute diode characteristic */
        ExMin = exp(EMin);
        ExMax = exp(EMax);
        v = u;
        et = v/(N*Vt);
        Cjunc = Tau*ISval/Vt*(if et < EMin then ExMin*(et - EMin + 1) else
                              if et > EMax then ExMax*(et - EMax + 1) else exp(et));
        Cdepl = if (v/Phi > 0) then Cj*(1 + M*v/Phi) else Cj*pow(1 - v/Phi, -M);
        C = Cjunc + Cdepl;
        y = C;
      end nlC2;

      block IB0 "Non-linear base-emitter current source of the Ebers-Moll BJT"
        extends Modelica.Blocks.Interfaces.SI2SO;
        parameter Real Bf=50 "Forward beta coefficient";
        parameter Real Br=0.1 "Reverse beta coefficient";
        output Modelica.SIunits.Current ibc "Base-collector diode current";
        output Modelica.SIunits.Current ibe "Base-emitter diode current";
        output Modelica.SIunits.Current IB0 "Injected base current";
        annotation (
          Icon(Text(
              extent=[-72,62; 82,-66],
              style(color=0, rgbcolor={0,0,0}),
              string="IB 0"), Text(
              extent=[-76,-54; 88,-78],
              style(color=0, rgbcolor={0,0,0}),
              string="Ebers-Moll BJT"),
            Text(
              extent=[-96,68; -76,52],
              style(color=3, rgbcolor={0,0,255}),
              string="ibc"),
            Text(
              extent=[-96,-52; -76,-68],
              style(color=3, rgbcolor={0,0,255}),
              string="ibe")),
          Diagram(Text(
              extent=[-72,62; 82,-66],
              style(color=0, rgbcolor={0,0,0}),
              string="IB 0"), Text(
              extent=[-76,-54; 88,-78],
              style(color=0, rgbcolor={0,0,0}),
              string="Ebers-Moll BJT"),
            Text(
              extent=[-96,68; -76,52],
              style(color=3, rgbcolor={0,0,255}),
              string="ibc"),
            Text(
              extent=[-96,-52; -76,-68],
              style(color=3, rgbcolor={0,0,255}),
              string="ibe")),
          Documentation(info="<html>
This block computes the injected base current of a BJT in accordance with the Ebers-Moll model. <p>

<pre>
Parameters:

 Bf:   Forward beta coefficient (default value = 50)

 Br:   Reverse beta coefficient (default value = 0.1)
</pre>
</html>
"));
      equation
        ibc = u1;
        ibe = u2;
        IB0 = ibe/Bf + ibc/Br;
        y = IB0;
      end IB0;

      block IB02
        "Non-linear base-emitter current source of the Ebers-Moll BJT with thermal variation"
        extends Modelica.Blocks.Interfaces.SI2SO;
        parameter Modelica.SIunits.Temperature Tnom=300.15
          "Reference temperature";
        parameter Real Bf=50 "Forward beta coefficient";
        parameter Real Br=0.1 "Reverse beta coefficient";
        parameter Real XTB=0 "Forward and reverse beta temperature coefficient";
        output Modelica.SIunits.Current ibc "Base-collector diode current";
        output Modelica.SIunits.Current ibe "Base-emitter diode current";
        output Modelica.SIunits.Current IB0 "Injected base current";

      protected
        Real RTemp "Temperature quotient";
        Real Rxtb "Exponentiated temperature quotient";
        Real BFval "Forward beta coefficient at device temperature";
        Real BRval "Backward beta coefficient at device temperature";
        annotation (Icon(
            Text(
              extent=[-96,68; -76,52],
              style(color=3, rgbcolor={0,0,255}),
              string="ibc"),
            Text(
              extent=[-96,-52; -76,-68],
              style(color=3, rgbcolor={0,0,255}),
              string="ibe"),
               Text(
              extent=[-72,62; 82,-66],
              style(color=0, rgbcolor={0,0,0}),
              string="IB 0"), Text(
              extent=[-76,-54; 88,-78],
              style(color=0, rgbcolor={0,0,0}),
              string="Ebers-Moll BJT")), Diagram(
            Text(
              extent=[-96,68; -76,52],
              style(color=3, rgbcolor={0,0,255}),
              string="ibc"),
            Text(
              extent=[-96,-52; -76,-68],
              style(color=3, rgbcolor={0,0,255}),
              string="ibe"),
               Text(
              extent=[-72,62; 82,-66],
              style(color=0, rgbcolor={0,0,0}),
              string="IB 0"), Text(
              extent=[-76,-54; 88,-78],
              style(color=0, rgbcolor={0,0,0}),
              string="Ebers-Moll BJT")),
          Documentation(info="<html>
This block computes the injected base current of a BJT with temperature variation in accordance with the Ebers-Moll model. <p>

<pre>
Parameters:

 Bf:   Forward beta coefficient (default value = 50)

 Br:   Reverse beta coefficient (default value = 0.1)

 XTB:  Forward and reverse beta temperature coefficient (default value = 0)
</pre>
</html>
"));
      public
        Modelica.Blocks.Interfaces.RealInput Tdev(redeclare type SignalType =
              Modelica.SIunits.Temperature) "Device temperature"
          annotation (extent=[-20,-120; 20,-80], rotation=90);
      equation
        RTemp = Tdev/Tnom;
        Rxtb = RTemp^XTB;
        BFval = Bf*Rxtb;
        BRval = Br*Rxtb;
        ibc = u1;
        ibe = u2;
        IB0 = ibe/BFval + ibc/BRval;
        y = IB0;
      end IB02;

      block IC0
        "Non-linear collector-emitter current source of the Ebers-Moll BJT"
        extends Modelica.Blocks.Interfaces.SI2SO;
        parameter Real Br=0.1 "Reverse beta coefficient";
        parameter Modelica.SIunits.InversePotential Vak=0.02
          "Inverse early voltage";
        output Modelica.SIunits.Current ibc "Base-collector diode current";
        output Modelica.SIunits.Current ibe "Base-emitter diode current";
        output Modelica.SIunits.Voltage vbc "Base-collector voltage";
        output Modelica.SIunits.Current IC0 "Injected collector current";

      protected
        Real qbk "Inverse base charge";
        annotation (
          Icon(Text(
              extent=[-72,62; 82,-66],
              style(color=0, rgbcolor={0,0,0}),
              string="IC 0"), Text(
              extent=[-76,-54; 88,-78],
              style(color=0, rgbcolor={0,0,0}),
              string="Ebers-Moll BJT"),
            Text(
              extent=[-96,-52; -76,-68],
              style(color=3, rgbcolor={0,0,255}),
              string="ibe"),
            Text(
              extent=[-96,68; -76,52],
              style(color=3, rgbcolor={0,0,255}),
              string="ibc"),
            Text(
              extent=[10,-82; 32,-98],
              style(color=3, rgbcolor={0,0,255}),
              string="vbc")),
          Diagram(Text(
              extent=[-72,62; 82,-66],
              style(color=0, rgbcolor={0,0,0}),
              string="IC 0"), Text(
              extent=[-76,-54; 88,-78],
              style(color=0, rgbcolor={0,0,0}),
              string="Ebers-Moll BJT"),
            Text(
              extent=[-96,-52; -76,-68],
              style(color=3, rgbcolor={0,0,255}),
              string="ibe"),
            Text(
              extent=[-96,68; -76,52],
              style(color=3, rgbcolor={0,0,255}),
              string="ibc"),
            Text(
              extent=[10,-82; 32,-98],
              style(color=3, rgbcolor={0,0,255}),
              string="vbc")),
          Documentation(info="<html>
This block computes the injected collector current of a BJT in accordance with the Ebers-Moll model. <p>

<pre>
Parameters:

 Br:   Reverse beta coefficient (default value = 0.1)

 Vak:  Inverse early voltage (default value = 0.02 /Volt)
</pre>
</html>
"));
      public
        Modelica.Blocks.Interfaces.RealInput u3
          "Connector of Real input signal 2"
          annotation (extent=[-20,-120; 20,-80], rotation=90);
      equation
        ibc = u1;
        ibe = u2;
        vbc = u3;
        qbk = 1 - vbc*Vak;
        IC0 = (ibe - ibc)*qbk - ibc/Br;
        y = IC0;
      end IC0;

      block IC02
        "Non-linear collector-emitter current source of the Ebers-Moll BJT with thermal variation"
        extends Modelica.Blocks.Interfaces.SI2SO;
        parameter Modelica.SIunits.Temperature Tnom=300.15
          "Reference temperature";
        parameter Real Br=0.1 "Reverse beta coefficient";
        parameter Modelica.SIunits.InversePotential Vak=0.02
          "Inverse early voltage";
        parameter Real XTB=0 "Forward and reverse beta temperature coefficient";
        output Modelica.SIunits.Current ibc "Base-collector diode current";
        output Modelica.SIunits.Current ibe "Base-emitter diode current";
        output Modelica.SIunits.Voltage vbc "Base-collector voltage";
        output Modelica.SIunits.Current IC0 "Injected collector current";

      protected
        Real RTemp "Temperature quotient";
        Real Rxtb "Exponentiated temperature quotient";
        Real BRval "Backward beta coefficient at device temperature";
        Real qbk "Inverse base charge";
      public
        Modelica.Blocks.Interfaces.RealInput Tdev(redeclare type SignalType =
              Modelica.SIunits.Temperature) "Device temperature"
          annotation (extent=[40,-120; 80,-80], rotation=90);
        annotation (Diagram(
            Text(
              extent=[-98,68; -78,52],
              style(color=3, rgbcolor={0,0,255}),
              string="ibc"),
            Text(
              extent=[-98,-52; -78,-68],
              style(color=3, rgbcolor={0,0,255}),
              string="ibe"),
                  Text(
              extent=[-74,62; 80,-66],
              style(color=0, rgbcolor={0,0,0}),
              string="IC 0"), Text(
              extent=[-78,-54; 86,-78],
              style(color=0, rgbcolor={0,0,0}),
              string="Ebers-Moll BJT"),
            Text(
              extent=[-50,-82; -28,-98],
              style(color=3, rgbcolor={0,0,255}),
              string="vbc")), Icon(
            Text(
              extent=[-98,68; -78,52],
              style(color=3, rgbcolor={0,0,255}),
              string="ibc"),
            Text(
              extent=[-98,-52; -78,-68],
              style(color=3, rgbcolor={0,0,255}),
              string="ibe"),
                  Text(
              extent=[-74,62; 80,-66],
              style(color=0, rgbcolor={0,0,0}),
              string="IC 0"), Text(
              extent=[-78,-54; 86,-78],
              style(color=0, rgbcolor={0,0,0}),
              string="Ebers-Moll BJT"),
            Text(
              extent=[-50,-82; -28,-98],
              style(color=3, rgbcolor={0,0,255}),
              string="vbc")),
          Documentation(info="<html>
This block computes the injected collector current of a BJT with temperature variation in accordance with the Ebers-Moll model. <p>

<pre>
Parameters:

 Br:   Reverse beta coefficient (default value = 0.1)

 Vak:  Inverse early voltage (default value = 0.02 /Volt)

 XTB:  Forward and reverse beta temperature coefficient (default value = 0)
</pre>
</html>
"));
      public
        Modelica.Blocks.Interfaces.RealInput u3
          "Connector of Real input signal 3"
          annotation (extent=[-80,-120; -40,-80],rotation=90);
      equation
        RTemp = Tdev/Tnom;
        Rxtb = RTemp^XTB;
        BRval = Br*Rxtb;
        ibc = u1;
        ibe = u2;
        vbc = u3;
        qbk = 1 - vbc*Vak;
        IC0 = (ibe - ibc)*qbk - ibc/BRval;
        y = IC0;
      end IC02;

      model mC2 "Bondgraphic modulated capacitor element"
        extends Interfaces.ModPassiveOnePort;
        Real C "Bondgraphic Capacitance";
        annotation (
          Icon(Text(
              extent=[-66,78; 54,-82],
              style(color=0),
              string="mC"),    Text(extent=[-72,80; 68,40],   string="%name"),
               Text(
              extent=[50,-2; 80,-54],
              style(color=0),
              string="2")),
          Diagram(
               Text(
              extent=[50,-2; 80,-54],
              style(color=0),
              string="2"),
               Text(
              extent=[-66,78; 54,-82],
              style(color=0),
              string="mC")),
          Documentation(info="<html>
This is an alternative model to the <b>mC</b> modulated capacitance model of the bond graph library.  This model is physically dubious.  Modelers are advised to always use the <b>mC</b> model, never the <b>mC2</b> model.  However, since this is the model that is being used in the Ebers-Moll description of bipolar transistors, it is being reproduced here.  However, it is hidden under utilities to discourage modelers from using it. <p>

<hr> <p>

<pre>
Potential variables:

 e:  Bondgraphic effort variable

 f:  Bondgraphic flow variable, normalized positive for flows into the model

 C:  Capacitance (modulated input signal)
</pre> <p>

<hr> <p>

Equations: <p>

<b>f = C*der(e)</b>
</html>
"));
      equation
        C = s;
        f = C*der(e);
      end mC2;

      model Cj "Junction capacitance model of Ebers-Moll BJT"
        extends Interfaces.PassiveOnePort;
        parameter Modelica.SIunits.Current Is=1.e-16
          "Transport saturation current";
        parameter Modelica.SIunits.Time Tau=5e-9 "Ideal transit time";
        parameter Modelica.SIunits.Capacitance Cj=0.5e-12
          "Zero bias depletion capacitance";
        parameter Modelica.SIunits.Voltage Phi=0.8 "Diffusion voltage";
        parameter Real M=0.333 "Gradation exponent";
        parameter Modelica.SIunits.Conductance Gb=1e-15 "Leakage conductance";
        parameter Modelica.SIunits.Voltage Vt=0.02585 "Thermal voltage";
        parameter Real EMin=-100
          "if x < EMin, the exp(x) function is linearized";
        parameter Real EMax=40 "if x > EMax, the exp(x) function is linearized";
        Modelica.Blocks.Interfaces.RealOutput y
          "Connector of Real output signal"
          annotation (extent=[90,-10; 110,10]);
        annotation (
          Icon(Text(
              extent=[2,-6; 92,-72],
              style(color=0),
              string="Junction"),
               Text(
              extent=[-74,62; 12,-70],
              style(color=0),
              string="C"),     Text(extent=[-70,94; 70,54],   string="%name")),
          Diagram,
          Documentation(info="<html>
The junction capacitance element is a passive OnePort element.  It inherits the effort and flow variables from the passive OnePort.  Junction capacitances are used in semiconductor technology to model the charge accumulated in the vicinity of a pn-junction. <p>

The junction capacitance element has its preferred causality away from the element. <p>

The model also computes the diodic current through the junction and puts it out as an output signal. <p>

<hr>

<pre>
Potential variables:

 e:     Bondgraphic effort variable

 f:     Bondgraphic flow variable, normalized positive for flows into the model

 id:    Diodic current through the junction (output variable)


Parameters:

 Is:    Transport saturation current (default value = 1e-16 Amp)

 Tau:   Ideal transit time (default value = 5e-9 sec)

 Cj:    Zero bias depletion capacitance (default value = 0.5e-12 F)

 Phi:   Diffusion voltage (default value = 0.8 Volt)

 M:     Gradation exponent (default value = 0.333)

 Gb:    Leakage conductance (default value = 1e-15 Mho)

 Vt:    Thermal voltage (default value = 0.02585 Volt)

 EMin:  Minimum exponent (default value = -100)

 EMax:  Maximum exponent (default value = 40)
</pre>
</html>
"));
        Junctions.J0p3 J0p3_1 annotation (extent=[-90,10; -70,-10]);
        Bonds.fBond B1 annotation (extent=[-70,-10; -50,10]);
        Sensors.De vbi annotation (extent=[-50,-10; -30,10]);
        Bonds.eBond B2 annotation (extent=[-90,10; -70,30], rotation=90);
        mC2 Cj1 annotation (extent=[-90,30; -70,50], rotation=90);
        Sources.mSe Vbi annotation (extent=[0,-10; -20,10]);
        Bonds.fBond B3 annotation (extent=[0,-10; 20,10]);
        Junctions.J1p3 J1p3_1 annotation (extent=[20,-10; 40,10]);
        Bonds.eBond B4 annotation (extent=[40,-10; 60,10], rotation=0);
        Sensors.Df id annotation (extent=[60,-10; 80,10]);
        Bonds.fBond B5 annotation (extent=[20,-30; 40,-10], rotation=270);
        Junctions.J0p3 J0p3_2 annotation (extent=[20,-30; 40,-50]);
        Bonds.fBond B6 annotation (extent=[40,-50; 60,-30], rotation=0);
        Bonds.fBond B7 annotation (extent=[0,-50; 20,-30], rotation=180);
        Passive.G G1(G=Gb) annotation (extent=[60,-50; 80,-30]);
        nlC nlC1(
          Is=Is,
          Tau=Tau,
          Phi=Phi,
          Cj=Cj,
          M=M,
          Vt=Vt,
          EMin=EMin,
          EMax=EMax) annotation (extent=[-40,50; -60,70]);
        Dj Dbi(
          Is=Is,
          Vt=Vt,
          EMin=EMin,
          EMax=EMax) annotation (extent=[0,-50; -20,-30]);
      equation
        connect(J0p3_1.BondCon1, BondCon1) annotation (points=[-90,0; -100,0], style(
              color=8, rgbcolor={192,192,192}));
        connect(J0p3_1.BondCon2, B1.eBondCon1) annotation (points=[-70,0; -70,0],
            style(color=8, rgbcolor={192,192,192}));
        connect(B1.fBondCon1, vbi.BondCon1) annotation (points=[-50,0; -50,0],
            style(color=8, rgbcolor={192,192,192}));
        connect(J0p3_1.BondCon3, B2.fBondCon1) annotation (points=[-80,10; -80,
              10], style(color=8, rgbcolor={192,192,192}));
        connect(B2.eBondCon1, Cj1.BondCon1) annotation (points=[-80,30; -80,30],
            style(color=8, rgbcolor={192,192,192}));
        connect(Vbi.BondCon1, B3.eBondCon1) annotation (points=[0,0; 0,0],
            style(color=8, rgbcolor={192,192,192}));
        connect(B3.fBondCon1, J1p3_1.BondCon1) annotation (points=[20,0; 20,0],
            style(color=8, rgbcolor={192,192,192}));
        connect(J1p3_1.BondCon2, B4.fBondCon1) annotation (points=[40,0; 40,0],
            style(color=8, rgbcolor={192,192,192}));
        connect(B4.eBondCon1, id.BondCon1) annotation (points=[60,0; 60,0],
            style(color=8, rgbcolor={192,192,192}));
        connect(id.OutPort1, y) annotation (points=[81,0; 100,0], style(color=
                74, rgbcolor={0,0,127}));
        connect(J0p3_2.BondCon2, B6.eBondCon1) annotation (points=[40,-40; 40,
              -40], style(color=8, rgbcolor={192,192,192}));
        connect(B7.eBondCon1, J0p3_2.BondCon1) annotation (points=[20,-40; 20,
              -40], style(color=8, rgbcolor={192,192,192}));
        connect(B6.fBondCon1, G1.BondCon1) annotation (points=[60,-40; 60,-40],
            style(color=8, rgbcolor={192,192,192}));
        connect(J1p3_1.BondCon3, B5.eBondCon1) annotation (points=[30,-10; 30,
              -10], style(color=8, rgbcolor={192,192,192}));
        connect(B5.fBondCon1, J0p3_2.BondCon3) annotation (points=[30,-30; 30,
              -30], style(color=8, rgbcolor={192,192,192}));
        connect(vbi.OutPort1, nlC1.u) annotation (points=[-29,0; -24,0; -24,60;
              -38,60], style(color=74, rgbcolor={0,0,127}));
        connect(Dbi.BondCon1, B7.fBondCon1) annotation (points=[0,-40; 0,-40],
            style(color=8, rgbcolor={192,192,192}));
        connect(vbi.OutPort1, Vbi.s) annotation (points=[-29,0; -19,0], style(
              color=74, rgbcolor={0,0,127}));
        connect(nlC1.y, Cj1.s) annotation (points=[-61,60; -80,60; -80,49],
            style(color=74, rgbcolor={0,0,127}));
      end Cj;

      model CjS
        "Junction capacitance model of Ebers-Moll BJT with thermal port"
        extends Interfaces.TwoPort;
        parameter Modelica.SIunits.Temperature Tnom=300.15
          "Reference temperature";
        parameter Modelica.SIunits.Current Is=1.e-16
          "Transport saturation current";
        parameter Modelica.SIunits.Time Tau=5e-9 "Ideal transit time";
        parameter Modelica.SIunits.Capacitance Cj=0.5e-12
          "Zero bias depletion capacitance";
        parameter Modelica.SIunits.Voltage Phi=0.8 "Diffusion voltage";
        parameter Real M=0.333 "Gradation exponent";
        parameter Modelica.SIunits.Voltage EG=1.11
          "Energy gap for temperature effect on saturation current";
        parameter Real N=1 "Current emission coefficient";
        parameter Real XTI=3 "Saturation current temperature exponent";
        parameter Modelica.SIunits.Conductance Gb=1e-15 "Leakage conductance";
        parameter Real EMin=-100
          "if x < EMin, the exp(x) function is linearized";
        parameter Real EMax=40 "if x > EMax, the exp(x) function is linearized";
        Modelica.Blocks.Interfaces.RealOutput y
          "Connector of Real output signal"
          annotation (extent=[-10,100; 10,120], rotation=90);
        Junctions.J0p3 J0p3_1 annotation (extent=[-90,10; -70,-10]);
        Bonds.fBond B1 annotation (extent=[-70,-10; -50,10]);
        Sensors.De vbi annotation (extent=[-50,-10; -30,10]);
        Bonds.eBond B2 annotation (extent=[-90,10; -70,30], rotation=90);
        mC2 Cj1 annotation (extent=[-90,30; -70,50], rotation=90);
        Sources.mSe Vbi annotation (extent=[0,-10; -20,10]);
        Bonds.fBond B3 annotation (extent=[0,-10; 20,10]);
        Junctions.J1p3 J1p3_1 annotation (extent=[20,-10; 40,10], rotation=-90);
        Bonds.eBond B4 annotation (extent=[20,10; 40,30],  rotation=90);
        Sensors.Df id annotation (extent=[20,30; 40,50], rotation=90);
        Bonds.fBond B5 annotation (extent=[20,-30; 40,-10], rotation=270);
        Junctions.J0p3 J0p3_2 annotation (extent=[20,-30; 40,-50]);
        Bonds.fBond B6 annotation (extent=[40,-50; 60,-30], rotation=0);
        Bonds.fBond B7 annotation (extent=[0,-50; 20,-30], rotation=180);
        Passive.G G1(G=Gb) annotation (extent=[0,-50; -20,-30]);
        nlC2 nlC1(
          Tnom=Tnom,
          Is=Is,
          Tau=Tau,
          Phi=Phi,
          Cj=Cj,
          M=M,
          EG=EG,
          N=N,
          XTI=XTI,
          EMin=EMin,
          EMax=EMax) annotation (extent=[-40,50; -60,70]);
        DjS DjS1(
          Tnom=Tnom,
          Is=Is,
          EG=EG,
          N=N,
          XTI=XTI,
          EMin=EMin,
          EMax=EMax) annotation (extent=[60,-50; 80,-30]);
        Junctions.J0p3 J0p3_3 annotation (extent=[90,10; 70,-10], rotation=-90);
        Bonds.fBond B8 annotation (extent=[70,10; 90,30], rotation=90);
        Sensors.De Tdev
                       annotation (extent=[70,30; 90,50], rotation=90);
        Bonds.eBond B9 annotation (extent=[70,-30; 90,-10],rotation=90);
      equation
        connect(J0p3_1.BondCon2,B1. eBondCon1) annotation (points=[-70,0; -70,0],
            style(color=8, rgbcolor={192,192,192}));
        connect(B1.fBondCon1,vbi. BondCon1) annotation (points=[-50,0; -50,0],
            style(color=8, rgbcolor={192,192,192}));
        connect(J0p3_1.BondCon3,B2. fBondCon1) annotation (points=[-80,10; -80,
              10], style(color=8, rgbcolor={192,192,192}));
        connect(B2.eBondCon1,Cj1. BondCon1) annotation (points=[-80,30; -80,30],
            style(color=8, rgbcolor={192,192,192}));
        connect(Vbi.BondCon1,B3. eBondCon1) annotation (points=[0,0; 0,0],
            style(color=8, rgbcolor={192,192,192}));
        connect(J0p3_2.BondCon2,B6. eBondCon1) annotation (points=[40,-40; 40,
              -40], style(color=8, rgbcolor={192,192,192}));
        connect(B7.eBondCon1,J0p3_2. BondCon1) annotation (points=[20,-40; 20,
              -40], style(color=8, rgbcolor={192,192,192}));
        connect(B5.fBondCon1,J0p3_2. BondCon3) annotation (points=[30,-30; 30,
              -30], style(color=8, rgbcolor={192,192,192}));
        annotation (Diagram,
          Icon(                Text(extent=[-68,92; 72,52],   string="%name"),
               Text(
              extent=[-88,64; 20,-68],
              style(color=0),
              string="CS"),
               Text(
              extent=[8,4; 98,-62],
              style(color=0),
              string="Junction")),
          Documentation(info="<html>
The junction capacitance source element is a directed TwoPort.  It inherits the effort and flow variables from the directed TwoPort.  Junction capacitances are used in semiconductor technology to model the charge accumulated in the vicinity of a pn-junction. <p>

The junction capacitance element has its preferred causality away from the element. <p>

The model also computes the diodic current through the junction and puts it out as an output signal. <p>

<hr>

<pre>
Potential variables:

 e1:    Bondgraphic effort variable of inflow

 f1:    Bondgraphic flow variable of inflow, normalized positive for flows into the model

 e2:    Bondgraphic effort variable of outflow

 f2:    Bondgraphic flow variable of outflow, normalized positive for flows out of the model

 id:    Diodic current through the junction (output variable)


Parameters:

 Is:    Transport saturation current (default value = 1e-16 Amp)

 Tau:   Ideal transit time (default value = 5e-9 sec)

 Cj:    Zero bias depletion capacitance (default value = 0.5e-12 F)

 Phi:   Diffusion voltage (default value = 0.8 Volt)

 M:     Gradation exponent (default value = 0.333)

 EG:    Energy gap for temperature effect on saturation current (default value = 1.11 Volt)

 N:     Current emission coefficient (default value = 1)

 XTI:   Saturation current temperature exponent (default value = 3)

 Gb:    Leakage conductance (default value = 1e-15 Mho)

 EMin:  Minimum exponent (default value = -100)

 EMax:  Maximum exponent (default value = 40)
</pre>
</html>
"));
        connect(J0p3_1.BondCon1, BondCon1) annotation (points=[-90,0; -100,0],
            style(color=8, rgbcolor={192,192,192}));
        connect(vbi.OutPort1, nlC1.u) annotation (points=[-29,0; -24,0; -24,60;
              -38,60], style(color=74, rgbcolor={0,0,127}));
        connect(G1.BondCon1, B7.fBondCon1) annotation (points=[0,-40; 0,-40],
            style(color=8, rgbcolor={192,192,192}));
        connect(B6.fBondCon1, DjS1.BondCon1) annotation (points=[60,-40; 60,-40],
            style(color=8, rgbcolor={192,192,192}));
        connect(B5.eBondCon1, J1p3_1.BondCon2) annotation (points=[30,-10; 30,
              -10], style(color=8, rgbcolor={192,192,192}));
        connect(B3.fBondCon1, J1p3_1.BondCon3) annotation (points=[20,0; 23,0;
              23,-6.12303e-016; 20,-6.12303e-016], style(color=8, rgbcolor={192,
                192,192}));
        connect(B4.fBondCon1, J1p3_1.BondCon1) annotation (points=[30,10; 30,10],
            style(color=8, rgbcolor={192,192,192}));
        connect(id.BondCon1, B4.eBondCon1) annotation (points=[30,30; 30,30],
            style(color=8, rgbcolor={192,192,192}));
        connect(id.OutPort1, y) annotation (points=[30,51; 30,80; 0,80; 0,110],
            style(color=74, rgbcolor={0,0,127}));
        connect(J0p3_3.BondCon3, BondCon2) annotation (points=[90,6.12303e-016;
              92,6.12303e-016; 92,0; 100,0], style(color=8, rgbcolor={192,192,
                192}));
        connect(B8.eBondCon1, J0p3_3.BondCon2) annotation (points=[80,10; 80,10],
            style(color=8, rgbcolor={192,192,192}));
        connect(Tdev.BondCon1, B8.fBondCon1) annotation (points=[80,30; 80,30],
            style(color=8, rgbcolor={192,192,192}));
        connect(Tdev.OutPort1, nlC1.Tdev) annotation (points=[80,51; 80,60; 0,
              60; 0,40; -50,40; -50,50], style(color=74, rgbcolor={0,0,127}));
        connect(B9.eBondCon1, J0p3_3.BondCon1) annotation (points=[80,-10; 80,-10],
            style(color=8, rgbcolor={192,192,192}));
        connect(DjS1.BondCon2, B9.fBondCon1) annotation (points=[80,-40; 80,-30],
            style(color=8, rgbcolor={192,192,192}));
        connect(vbi.OutPort1, Vbi.s) annotation (points=[-29,0; -19,0], style(
              color=74, rgbcolor={0,0,127}));
        connect(nlC1.y, Cj1.s) annotation (points=[-61,60; -80,60; -80,49],
            style(color=74, rgbcolor={0,0,127}));
      end CjS;

      model Dj "Junction diode model of Ebers-Moll BJT"
        extends Interfaces.PassiveOnePort;
        parameter Modelica.SIunits.Current Is=1.e-16
          "Transport saturation current";
        parameter Modelica.SIunits.Voltage Vt=0.02585
          "Voltage equivalent of temperature (kT/qn)";
        parameter Real EMin=-100
          "if x < EMin, the exp(x) function is linearized";
        parameter Real EMax=40 "if x > EMax, the exp(x) function is linearized";

      protected
        Real ExMin;
        Real ExMax;
        annotation (
          Icon(Text(
              extent=[-76,64; 10,-68],
              style(color=0),
              string="D"),
               Text(
              extent=[-4,-4; 86,-70],
              style(color=0),
              string="Junction"),
                               Text(extent=[-70,96; 70,56],   string="%name")),
          Diagram(
            Line(points=[-80,0; 80,0],   style(color=10)),
            Polygon(points=[70,4; 80,0; 70,-4; 70,4],     style(color=10,
                  fillColor=10)),
            Line(points=[0,80; 0,-80],   style(
                color=10,
                fillColor=10,
                fillPattern=1)),
            Polygon(points=[-4,70; 0,80; 4,70; -4,70],     style(
                color=10,
                fillColor=10,
                fillPattern=1)),
            Text(
              extent=[4,82; 14,72],
              style(
                color=10,
                fillColor=10,
                fillPattern=1),
              string="f"),
            Text(
              extent=[70,-4; 80,-14],
              style(
                color=10,
                fillColor=10,
                fillPattern=1),
              string="e"),
            Line(points=[-80,-4; -28,-4; -6,-2; 0,0],         style(
                color=0,
                thickness=2,
                fillColor=10,
                fillPattern=1)),
            Text(
              extent=[-66,-6; -36,-14],
              style(
                color=10,
                pattern=3,
                fillColor=10,
                fillPattern=1),
              string="blocking"),
            Text(
              extent=[16,50; 48,42],
              style(
                color=10,
                pattern=3,
                fillColor=10,
                fillPattern=1),
              string="conducting"),
            Line(points=[0,0; 6,10; 12,28; 16,66],            style(
                color=0,
                thickness=2,
                fillColor=10,
                fillPattern=1))),
          Documentation(info="<html>
The junction diode element is a passive OnePort element.  It inherits the effort and flow variables from the passive OnePort.  Junction diodes are a special class of non-linear resistors. <p>

The junction diode model is a simple diode with an exponential characteristic. The diode formula essentially is: <p>

<b>f = Is*(exp(e/Vt) - 1)</b> <p>

However, if the exponent <b>e/Vt</b> reaches either an upper limit <b>EMax</b> or a lower limit <b>Emin</b>, the diode characterisic is linearly continued to avoid overflow or underflow, respectively. <p>

The causality of the junction diode element is free. <p>

<hr>

<pre>
Potential variables:

 e:     Bondgraphic effort variable

 f:     Bondgraphic flow variable, normalized positive for flows into the model


Parameters:

 Is:    Transport saturation current (default value = 1e-16 Amp)

 Vt:    Thermal voltage (default value = 0.02585 Volt)

 EMin:  Minimum exponent (default value = -100)

 EMax:  Maximum exponent (default value = 40)
</pre> <p>

<hr> <p>

Equations: <p>

<b>ExMin = exp(EMin)</b> <br>
<b>ExMax = exp(EMax)</b> <br>
<b>f = if (e/Vt &lt; EMin) then Is*(ExMin*(e/Vt - EMin + 1) - 1) else
       if (e/Vt &gt; EMax) then Is*(ExMax*(e/Vt - EMax + 1) - 1) else Is*(exp(e/Vt) - 1)</b>
</html>
"));
      equation
        ExMin = exp(EMin);
        ExMax = exp(EMax);
        f = if (e/Vt < EMin) then Is*(ExMin*(e/Vt - EMin + 1) - 1) else
            if (e/Vt > EMax) then Is*(ExMax*(e/Vt - EMax + 1) - 1) else Is*(exp(e/Vt) - 1);
      end Dj;

      model DjS "Ebers-Moll junction diode model for bipolar transistors"
        extends Interfaces.TwoPort;
        constant Real k=1.3806226e-23 "Boltzmann's constant";
        constant Modelica.SIunits.ElectricCharge q=1.6021918e-19
          "Electron charge";
        parameter Modelica.SIunits.Temperature Tnom=300.15
          "Reference temperature";
        parameter Modelica.SIunits.Current Is=1e-16
          "Saturation current at reference temperature";
        parameter Modelica.SIunits.Voltage EG=1.11
          "Energy gap for temperature effect on saturation current";
        parameter Real N=1 "Current emission coefficient";
        parameter Real XTI=3 "Saturation current temperature exponent";
        parameter Real EMin=-100
          "if x < EMin, the exp(x) function is linearized";
        parameter Real EMax=40 "if x > EMax, the exp(x) function is linearized";
        Modelica.SIunits.Voltage Vt "Thermal voltage";
        Modelica.SIunits.Current ISval
          "Saturation current at device temperature";

      protected
        Real RTemp "Temperature quotient";
        Real ExMin;
        Real ExMax;
        Real et;
        annotation (
          Diagram(
            Line(points=[-80,0; 80,0],   style(color=10)),
            Polygon(points=[70,4; 80,0; 70,-4; 70,4],     style(color=10,
                  fillColor=10)),
            Line(points=[0,80; 0,-80],   style(
                color=10,
                fillColor=10,
                fillPattern=1)),
            Polygon(points=[-4,70; 0,80; 4,70; -4,70],     style(
                color=10,
                fillColor=10,
                fillPattern=1)),
            Text(
              extent=[4,82; 14,72],
              style(
                color=10,
                fillColor=10,
                fillPattern=1),
              string="f"),
            Text(
              extent=[70,-4; 80,-14],
              style(
                color=10,
                fillColor=10,
                fillPattern=1),
              string="e"),
            Line(points=[-80,-4; -28,-4; -6,-2; 0,0],         style(
                color=0,
                thickness=2,
                fillColor=10,
                fillPattern=1)),
            Text(
              extent=[-66,-6; -36,-14],
              style(
                color=10,
                pattern=3,
                fillColor=10,
                fillPattern=1),
              string="blocking"),
            Text(
              extent=[16,50; 48,42],
              style(
                color=10,
                pattern=3,
                fillColor=10,
                fillPattern=1),
              string="conducting"),
            Line(points=[0,0; 6,10; 12,28; 16,66],            style(
                color=0,
                thickness=2,
                fillColor=10,
                fillPattern=1))),
          Icon(Text(
              extent=[-84,74; 20,-82],
              style(color=0),
              string="DS"),
               Text(
              extent=[8,8; 98,-58],
              style(color=0),
              string="Junction"),
                               Text(extent=[-72,102; 68,62],  string="%name")),
          Documentation(info="<html>
The junction diodic source element is a directed TwoPort.  It inherits the effort and flow variables from the directed TwoPort.  Junction diodes are a special class of non-linear resistors. <p>

The junction diode model is a simple diode with an exponential characteristic. The diode formula essentially is: <p>

<b>f = Is*(exp(e/Vt) - 1)</b> <p>

However, if the exponent <b>e/Vt</b> reaches either an upper limit <b>EMax</b> or a lower limit <b>Emin</b>, the diode characterisic is linearly continued to avoid overflow or underflow, respectively. <p>

The causality of the junction diode element is free. <p>

<hr>

<pre>
Potential variables:

 e1:    Bondgraphic effort variable of inflow

 f1:    Bondgraphic flow variable of inflow, normalized positive for flows into the model

 e2:    Bondgraphic effort variable of outflow

 f2:    Bondgraphic flow variable of outflow, normalized positive for flows out of the model


Parameters:

 Tnom:  Reference temperature (default value = 300.15 K)

 Is:    Transport saturation current (default value = 1e-16 Amp)

 EG:    Energy gap for temperature effect on saturation current (default value = 1.11 Volt)

 N:     Current emission coefficient (default value = 1)

 XTI:   Saturation current temperature exponent (default value = 3)

 EMin:  Minimum exponent (default value = -100)

 EMax:  Maximum exponent (default value = 40)
</pre> <p>

<hr> <p>

Equations: <p>

<b>RTemp = e2/Tnom</b> <br>
<b>ISval = IS*exp((RTemp - 1)*EG/Vt + XTI*ln(RTemp))</b> <br>
<b>ExMin = exp(EMin)</b> <br>
<b>ExMax = exp(EMax)</b> <br>
<b>et = e1/(Vt*N)</b> <br>
<b>f1 = ISval*(if et &lt; EMin then ExMin*(et - EMin + 1) - 1 else
               if et &gt; EMax then ExMax*(et - EMax + 1) - 1 else exp(et) - 1)</b> <br>
<b>f2 = 0</b>
</html>
"));
      equation
        /* Compute thermal voltage as function of temperature */
        Vt = k*e2/q;

        /* Compute temperature dependence of saturation current */
        RTemp = e2/Tnom;
        ISval = Is*exp((RTemp - 1)*EG/Vt + XTI*ln(RTemp));

        /* Compute diode characteristic */
        ExMin = exp(EMin);
        ExMax = exp(EMax);
        et = e1/(Vt*N);
        f1 = ISval*(if et < EMin then ExMin*(et - EMin + 1) - 1 else
                    if et > EMax then ExMax*(et - EMax + 1) - 1 else exp(et) - 1);

        /* Compute heat flow */
        f2 = 0;
      end DjS;
    end Utilities;
  end Semiconductors;

  package Sensors "Sensors (detectors) of Modelica bond graph library"
    annotation (
      preferedView="info",
      Icon(
        Rectangle(extent=[-100, -100; 80, 50], style(fillColor=30, fillPattern=
                1)),
        Polygon(points=[-100, 50; -80, 70; 100, 70; 80, 50; -100, 50], style(
              fillColor=30, fillPattern=1)),
        Polygon(points=[100, 70; 100, -80; 80, -100; 80, 50; 100, 70], style(
              fillColor=30, fillPattern=1)),
        Text(
          extent=[-85, 35; 65, -85],
          string="Library",
          style(color=3)),
        Text(
          extent=[-120, 122; 120, 73],
          string="%name",
          style(color=1))),
      Window(
        x=0,
        y=0.6,
        width=0.4,
        height=0.4,
        library=1,
        autolayout=1),
      Documentation(info="<html>
Sensors (detectors) are not among the standard bond graph elements advertized regularly in the bond graph literature.  Yet, they are very useful elements indeed, as they enable the modeler to convert bond graph variables to signal variables.  This is useful for modeling blocks of block diagrams internally by bond graphs.  It is also useful for extracting bond graph variables needed for the modeling of so-called \"activated bonds,\" bond graph elements that are internally modulated by variables extracted elsewhere in the bond graph. <p>

This package contains four types of detectors: <p>

<ol>
<li> effort detectors (<b>De</b>), <br>
<li> flow detectors (<b>Df</b>), <br>
<li> detectors of generalized momentums (<b>Dp</b>), and <br>
<li> detectors of generalized positions (<b>Dq</b>).
</ol> <p>
</html>
"));
    model De "Effort sensor of the Modelica bond graph library"
      extends Interfaces.PassiveOnePort;
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Text(
            extent=[-60, 80; 60, -80],
            string="De",
            style(color=0, fillColor=0))),
        Icon(Text(
            extent=[-60, 80; 60, -80],
            string="De",
            style(color=0, fillColor=0)), Text(extent=[-70, 90; 70, 50], string=
               "%name")),
        Window(
          x=0.45,
          y=0.01,
          width=0.35,
          height=0.63),
        Documentation(info="<html>
The effort sensor (detector) senses the common effort of a 0-junction.  Thus, effort sensors must always be connected to 0-junctions via a bond. <p>

An effort sensor does not affect the behavior of the system at all.  This is accomplished by setting the flow into the sensor equal to zero.  Thus, an effort sensor is in its effects identical with a flow source of zero flow. <p>

The effort sensor is a passive OnePort element.  It inherits the effort and flow variables from the passive OnePort.  The effort sensor has an additional output signal, through which it transmits the effort value. <p>

Since the effort sensor computes the flow (by setting it equal to zero), it has a causality stroke at the sensor element.

<hr>

<pre>
Potential variables:

 e:  Bondgraphic effort variable

 f:  Bondgraphic flow variable, normalized positive for flows into the model
</pre> <p>

<hr> <p>

Equations: <p>

<b>f = 0</b>
</html>
"));
      Modelica.Blocks.Interfaces.RealOutput OutPort1
        "Connector of Real output signal"         annotation (extent=[100, -10;
              120, 10], layer="icon");
    equation
      OutPort1 = e;
      f = 0;
    end De;

    model Df "Flow sensor of the Modelica bond graph library"
      extends Interfaces.PassiveOnePort;
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Text(
            extent=[-60, 80; 60, -80],
            string="Df",
            style(color=0, fillColor=0))),
        Icon(Text(
            extent=[-60, 80; 60, -80],
            string="Df",
            style(color=0, fillColor=0)), Text(extent=[-70, 90; 70, 50], string=
               "%name")),
        Window(
          x=0.45,
          y=0.01,
          width=0.35,
          height=0.63),
        Documentation(info="<html>
The flow sensor (detector) senses the common flow of a 1-junction.  Thus, flow sensors must always be connected to 1-junctions via a bond. <p>

A flow sensor does not affect the behavior of the system at all.  This is accomplished by setting the effort at the sensor equal to zero.  Thus, a flow sensor is in its effects identical with an effort source of zero effort. <p>

The flow sensor is a passive OnePort element.  It inherits the effort and flow variables from the passive OnePort.  The flow sensor has an additional output signal, through which it transmits the flow value. <p>

Since the flow sensor computes the effort (by setting it equal to zero), it has a causality stroke away from the sensor element.

<hr>

<pre>
Potential variables:

 e:  Bondgraphic effort variable

 f:  Bondgraphic flow variable, normalized positive for flows into the model
</pre> <p>

<hr> <p>

Equations: <p>

<b>e = 0</b>
</html>
"));
      Modelica.Blocks.Interfaces.RealOutput OutPort1
        "Connector of Real output signal"         annotation (extent=[100, -10;
              120, 10]);
    equation
      OutPort1 = f;
      e = 0;
    end Df;

    model Dp "Generalized momentum sensor of the Modelica bond graph library"
      extends Interfaces.PassiveOnePort;
      parameter Real p0 "Initial momentum";
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Icon(Text(
            extent=[-58, 80; 62, -80],
            string="Dp",
            style(color=0, fillColor=0)), Text(extent=[-68, 90; 72, 50], string=
               "%name")),
        Diagram,
        Window(
          x=0.45,
          y=0.01,
          width=0.35,
          height=0.63),
        Documentation(info="<html>
The sensor (detector) of generalized momentum senses the common effort of a 0-junction, then integrates it.  Thus, momentum sensors must always be connected to 0-junctions via a bond. <p>

A momentum sensor does not affect the behavior of the system at all.  This is accomplished by setting the flow into the sensor equal to zero.  Thus, a momentum sensor is in its effects similar to a flow source of zero flow. <p>

The momentum sensor is a passive OnePort element.  It inherits the effort and flow variables from the passive OnePort.  The momentum sensor has an additional output signal, through which it transmits the generalized momentum value. <p>

Since the momentum sensor computes the flow (by setting it equal to zero), it has a causality stroke at the sensor element.

<hr>

<pre>
Potential variables:

 e:  Bondgraphic effort variable

 f:  Bondgraphic flow variable, normalized positive for flows into the model

 p:  Generalized momentum
</pre> <p>

<hr> <p>

Equations: <p>

<b>f = 0</b> <br>
<b>der(p) = e</b>
</html>
"));
      Modelica.Blocks.Interfaces.RealOutput OutPort1
        "Connector of Real output signal"         annotation (extent=[100, -10;
              120, 10], layer="icon");
      Modelica.Blocks.Continuous.Integrator Integrator1(y_start=p0)
                                                        annotation (extent=[50,
              -20; 90, 20]);
      De De1 annotation (extent=[-10, -20; 30, 20]);
      Junctions.J0p2 J0p2_1 annotation (extent=[-90, -20; -50, 20]);
      Bonds.fBond B1 annotation (extent=[-50, -20; -10, 20]);
    equation
      connect(Integrator1.y,       OutPort1) annotation (points=[92,0; 110,0], style(
            color=74, rgbcolor={0,0,135}));
      connect(De1.OutPort1,Integrator1.u)       annotation (points=[32, 0; 46,
            0], style(color=74, rgbcolor={0,0,135}));
      connect(BondCon1, J0p2_1.BondCon1) annotation (points=[-100,0; -90,0],
          style(color=8));
      connect(J0p2_1.BondCon2, B1.eBondCon1) annotation (points=[-50, 0; -50, 0],
               style(color=8));
      connect(B1.fBondCon1, De1.BondCon1) annotation (points=[-10, 0; -10, 0],
          style(color=8));
    end Dp;

    model Dq "Generalized position sensor of the Modelica bond graph library"
      extends Interfaces.PassiveOnePort;
      parameter Real q0 "Initial position";
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Icon(Text(
            extent=[-58, 80; 62, -80],
            string="Dq",
            style(color=0, fillColor=0)), Text(extent=[-68, 90; 72, 50], string=
               "%name")),
        Window(
          x=0.45,
          y=0.01,
          width=0.35,
          height=0.63),
        Documentation(info="<html>
The sensor (detector) of generalized position senses the common flow of a 1-junction, then integrates it.  Thus, position sensors must always be connected to 1-junctions via a bond. <p>

A position sensor does not affect the behavior of the system at all.  This is accomplished by setting the effort at the sensor equal to zero.  Thus, a position sensor is in its effects similar to an effort source of zero effort. <p>

The position sensor is a passive OnePort element.  It inherits the effort and flow variables from the passive OnePort.  The position sensor has an additional output signal, through which it transmits the generalized position value. <p>

Since the position sensor computes the effort (by setting it equal to zero), it has a causality stroke away from the sensor element.

<hr>

<pre>
Potential variables:

 e:  Bondgraphic effort variable

 f:  Bondgraphic flow variable, normalized positive for flows into the model

 q:  Generalized position
</pre> <p>

<hr> <p>

Equations: <p>

<b>e = 0</b> <br>
<b>der(q) = f</b>
</html>
"),     Diagram);
      Modelica.Blocks.Interfaces.RealOutput OutPort1
        "Connector of Real output signal"         annotation (extent=[100, -10;
              120, 10], layer="icon");
      Modelica.Blocks.Continuous.Integrator Integrator1(y_start=q0)
                                                        annotation (extent=[50,
              -20; 90, 20]);
      Df Df1 annotation (extent=[-10, -20; 30, 20]);
      Bonds.eBond B1 annotation (extent=[-50,-20; -10,20]);
      Junctions.J1p2 J1p2_1 annotation (extent=[-90,-20; -50,20]);
    equation
      connect(Integrator1.y,       OutPort1) annotation (points=[92,0; 110,0], style(
            color=74, rgbcolor={0,0,135}));
      connect(Df1.OutPort1,Integrator1.u)       annotation (points=[32, 0; 46,
            0], style(color=74, rgbcolor={0,0,135}));
      connect(B1.eBondCon1, Df1.BondCon1) annotation (points=[-10,0; -10,0],
          style(color=8));
      connect(J1p2_1.BondCon2, B1.fBondCon1) annotation (points=[-50,0; -50,0],
               style(color=8));
      connect(BondCon1, J1p2_1.BondCon1) annotation (points=[-100,0; -90,0],
          style(color=8));
    end Dq;

    model Pbond "Power flow sensor"
      extends Bonds.Bond;
      Modelica.Blocks.Interfaces.RealOutput P(redeclare type SignalType =
            Modelica.SIunits.Power) "Power"
        annotation (extent=[-10,80; 10,100], rotation=90);
    equation
      P = BondCon1.e*BondCon1.f;
      annotation (Documentation(info="<html>
The non-directed power flow sensor is built as an acausal bond with an additional signal output that makes available the power flowing through the bond in the direction of the harpoon.
</html>
"), Diagram);
    end Pbond;

    block ePbond "Power flow sensor"
      extends Bonds.eBond;
      Modelica.Blocks.Interfaces.RealOutput P(redeclare type SignalType =
            Modelica.SIunits.Power) "Power"
        annotation (extent=[-10,80; 10,100], rotation=90);
    equation
      P = eBondCon1.e*fBondCon1.f;
      annotation (Documentation(info="<html>
The two directed power flow sensors are built as causal bonds with an additional signal output that makes available the power flowing through the bond in the direction of the harpoon.
</html>
"), Diagram);
    end ePbond;

    block fPbond "Power flow sensor"
      extends Bonds.fBond;
      Modelica.Blocks.Interfaces.RealOutput P(redeclare type SignalType =
            Modelica.SIunits.Power) "Power"
        annotation (extent=[-10,80; 10,100], rotation=90);
    equation
      P = eBondCon1.e*fBondCon1.f;
      annotation (Documentation(info="<html>
The non-directed power flow sensor is built as an acausal bond with an additional signal output that makes available the power flowing through the bond in the direction of the harpoon.
</html>
"), Diagram);
    end fPbond;
  end Sensors;

  package Sources "Sources of Modelica bond graph library"
    annotation (
      preferedView="info",
      Icon(
        Rectangle(extent=[-100, -100; 80, 50], style(fillColor=30, fillPattern=
                1)),
        Polygon(points=[-100, 50; -80, 70; 100, 70; 80, 50; -100, 50], style(
              fillColor=30, fillPattern=1)),
        Polygon(points=[100, 70; 100, -80; 80, -100; 80, 50; 100, 70], style(
              fillColor=30, fillPattern=1)),
        Text(
          extent=[-85, 35; 65, -85],
          string="Library",
          style(color=3)),
        Text(
          extent=[-120, 122; 120, 73],
          string="%name",
          style(color=1))),
      Window(
        x=0.04,
        y=0.26,
        width=0.35,
        height=0.49,
        library=1,
        autolayout=1),
      Documentation(info="<html>
This package contains a collection of effort sources (<b>Se</b>) and flow sources (<b>Sf</b>). <p>

As of now, only the most common types of sources are being provided.  These are: <p>

<ol>
<li> constant sources, <br>
<li> modulated sources, and <br>
<li> sinusoidal sources.
</ol><p>

Other types of sources, such as rectangular sources can easily be constructed using the electrical library as a model.
</html>

"));
    model Se "Constant effort source of the Modelica bond graph library"
      extends Interfaces.ActiveOnePort;
      parameter Real e0=1 "Bondgraphic Effort Source";
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Text(
            extent=[-60, 80; 60, -80],
            string="Se",
            style(color=0))),
        Icon(
          Text(
            extent=[-60, 80; 60, -80],
            string="Se",
            style(color=0)),
          Text(extent=[-70, 90; 70, 50], string="%name"),
          Text(extent=[-70, -50; 70, -90], string="e0=%e0")),
        Window(
          x=0.45,
          y=0.01,
          width=0.35,
          height=0.63),
        Documentation(info="<html>
The constant effort source is an active OnePort element.  It inherits the effort and flow variables from the active OnePort.  The constant effort to be generated is modeled as a parameter. <p>

The effort source has its causality stroke away from the source. <p>

<hr>

<pre>
Potential variables:

 e:   Bondgraphic effort variable

 f:   Bondgraphic flow variable, normalized positive for flows out of the model


Parameters:

 e0:  Constant effort to be generated (default value = 1)
</pre> <p>

<hr> <p>

Equations: <p>

<b>e = e0</b>
</html>
"));
    equation
      e = e0;
    end Se;

    model mSe "Modulated effort source of the Modelica bond graph library"
      extends Interfaces.ModActiveOnePort;
      Real e0 "Bondgraphic Effort Source";
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Text(
            extent=[-60, 80; 60, -80],
            string="mSe",
            style(color=0))),
        Icon(Text(
            extent=[-60, 80; 60, -80],
            string="mSe",
            style(color=0)), Text(extent=[-70, 90; 70, 50], string="%name")),
        Window(
          x=0.45,
          y=0.01,
          width=0.35,
          height=0.63),
        Documentation(info="<html>
The modulated effort source is a modulated active OnePort element.  It inherits the effort and flow variables from the modulated active OnePort.  The modulated effort to be generated is modeled as a real-valued variable. <p>

Modulated sources are commonly used to model actuators of control systems.  They are the most common and convenient way for converting signal variables to bondgraphic variables, thereby allowing individual blocks of a block diagram, e.g. describing a control system, to be modeled internally using bond graphs. <p>

The effort source has its causality stroke away from the source. <p>

<hr>

<pre>
Potential variables:

 e:   Bondgraphic effort variable

 f:   Bondgraphic flow variable, normalized positive for flows out of the model

 e0:  Effort to be generated (modulated input signal)
</pre> <p>

<hr> <p>

Equations: <p>

<b>e = e0</b>
</html>
"));
    equation
      e0 = s;
      e = e0;
    end mSe;

    model sinSe "Sinusoidal effort source of the Modelica bond graph library"
      extends Interfaces.ActiveOnePort;
      parameter Real e0=1 "Amplitude of sine wave";
      parameter Modelica.SIunits.Angle phase=0 "Phase of sine wave";
      parameter Modelica.SIunits.Frequency freqHz=1 "Frequency of sine wave";
      parameter Real offset=0 "Offset of sine wave";
      parameter Modelica.SIunits.Time startTime=0 "Start time of sine wave";
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(
          Line(points=[-80, -90; -80, 84], style(color=8)),
          Polygon(points=[-80, 100; -86, 84; -74, 84; -80, 100], style(color=8,
                  fillColor=8)),
          Line(points=[-99, -40; 85, -40], style(color=8)),
          Polygon(points=[101, -40; 85, -34; 85, -46; 101, -40], style(color=8,
                  fillColor=8)),
          Line(points=[-40, 0; -31.6, 34.2; -26.1, 53.1; -21.3, 66.4; -17.1,
                74.6; -12.9, 79.1; -8.64, 79.8; -4.42, 76.6; -0.201, 69.7; 4.02,
                  59.4; 8.84, 44.1; 14.9, 21.2; 27.5, -30.8; 33, -50.2; 37.8, -
                64.2; 42, -73.1; 46.2, -78.4; 50.5, -80; 54.7, -77.6; 58.9, -
                71.5; 63.1, -61.9; 67.9, -47.2; 74, -24.8; 80, 0], style(color=
                  0, thickness=2)),
          Line(points=[-41, -2; -80, -2], style(color=0, thickness=2)),
          Text(
            extent=[-106, -11; -60, -29],
            string="offset",
            style(color=9)),
          Line(points=[-41, -2; -41, -40], style(color=8, pattern=2)),
          Text(
            extent=[-60, -43; -14, -61],
            string="startTime",
            style(color=9)),
          Text(
            extent=[84, -52; 108, -72],
            string="time",
            style(color=9)),
          Line(points=[-9, 79; 43, 79], style(color=8, pattern=2)),
          Line(points=[-42, -1; 50, 0], style(color=8, pattern=2)),
          Polygon(points=[33, 80; 30, 67; 37, 67; 33, 80], style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[33, 1; 30, 14; 36, 14; 33, 1; 33, 1], style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[33, 79; 33, 0], style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Text(
            extent=[-88, 100; -44, 80],
            string="e",
            style(color=9, fillColor=8)),
          Text(
            extent=[32, 52; 72, 34],
            string="e0",
            style(color=9))),
        Icon(
          Text(
            extent=[-60, 80; 60, -80],
            string="Se",
            style(color=0)),
          Line(points=[-70, 0; -60.2, 29.9; -53.8, 46.5; -48.2, 58.1; -43.3,
                65.2; -38.3, 69.2; -33.4, 69.8; -28.5, 67; -23.6, 61; -18.6, 52;
                  -13, 38.6; -5.98, 18.6; 8.79, -26.9; 15.1, -44; 20.8, -56.2;
                25.7, -64; 30.6, -68.6; 35.5, -70; 40.5, -67.9; 45.4, -62.5;
                50.3, -54.1; 55.9, -41.3; 63, -21.7; 70, 0], style(color=8)),
          Text(extent=[-70, 90; 70, 50], string="%name")),
        Window(
          x=0.45,
          y=0.01,
          width=0.35,
          height=0.63),
        Documentation(info="<html>
The sinusoidal effort source is an active OnePort element.  It inherits the effort and flow variables from the active OnePort.  The sinusoidal effort to be generated is modeled as a set of five parameters. <p>

The effort source has its causality stroke away from the source. <p>

<hr>

<pre>
Potential variables:

 e:          Bondgraphic effort variable

 f:          Bondgraphic flow variable, normalized positive for flows out of the model


Parameters:

 e0:         Amplitude of sine wave to be generated (default value = 1)

 phase:      Phase of sine wave to be generated (default value = 0)

 freqHz:     Frequency of sine wave to be generated (default value = 1)

 offset:     Offset of sine wave to be generated (default value = 0)

 startTime:  Start time of sine wave to be generated (default value = 0)
</pre> <p>

<hr> <p>

Equations: <p>

<b>e = offset + e0*sin(2*pi*freqHz*(time - startTime))</b>
</html>
"));
      Modelica.Blocks.Sources.Sine Sine1(
        amplitude=e0,
        freqHz=freqHz,
        offset=offset,
        phase=phase,
        startTime=startTime)   annotation (extent=[70, 66; 90, 86]);
    equation
      e = Sine1.y;
    end sinSe;

    model stepSe "Step effort source of the Modelica bond graph library"
      extends Interfaces.ActiveOnePort;
      parameter Real e0=1 "Height of step";
      parameter Real offset=0 "Offset of step";
      parameter Modelica.SIunits.Time startTime=0 "Start time of step";
      annotation (
        Icon(
          Text(
            extent=[-62,78; 58,-82],
            string="Se",
            style(color=0)),
          Text(extent=[-72,100; 68,60],  string="%name"),
             Line(points=[-70,-80; 0,-80; 0,60; 69,60],     style(color=8))),
        Diagram(
          Polygon(points=[-80,90; -88,68; -72,68; -80,90],     style(color=8,
                fillColor=8)),
          Line(points=[-80,68; -80,-80],   style(color=8)),
          Line(points=[-80,-18; 0,-18; 0,50; 80,50],     style(color=0, thickness=
                 2)),
          Line(points=[-90,-70; 82,-70],   style(color=8)),
          Polygon(points=[90,-70; 68,-62; 68,-78; 90,-70],     style(color=8,
                fillColor=8)),
          Text(
            extent=[70,-80; 94,-100],
            string="time",
            style(color=9)),
          Text(
            extent=[-21,-72; 25,-90],
            string="startTime",
            style(color=9)),
          Line(points=[0,-17; 0,-71],   style(color=8, pattern=2)),
          Text(
            extent=[-68,-36; -22,-54],
            string="offset",
            style(color=9)),
          Line(points=[-13,50; -13,-17],   style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Polygon(points=[2,50; -19,50; 2,50],    style(color=8, pattern=2)),
          Polygon(points=[-13,-17; -16,-4; -10,-4; -13,-17; -13,-17],      style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[-13,50; -16,37; -9,37; -13,50],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Text(
            extent=[-50,26; -4,8],
            style(color=9),
            string="e0"),
          Polygon(points=[-13,-69; -16,-56; -10,-56; -13,-69; -13,-69],
              style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-13,-18; -13,-70],   style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Polygon(points=[-13,-18; -16,-31; -9,-31; -13,-18],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Text(
            extent=[-91,93; -32,70],
            style(color=9),
            string="e")),
        Documentation(info="<html>
The step effort source is an active OnePort element.  It inherits the effort and flow variables from the active OnePort.  The step effort to be generated is modeled as a set of three parameters. <p>

The effort source has its causality stroke away from the source. <p>

<hr>

<pre>
Potential variables:

 e:          Bondgraphic effort variable

 f:          Bondgraphic flow variable, normalized positive for flows out of the model


Parameters:

 e0:         Height of step to be generated (default value = 1)

 offset:     Offset of step to be generated (default value = 0)

 startTime:  Start time of step to be generated (default value = 0)
</pre> <p>

<hr> <p>

Equations: <p>

<b>e = offset + (if time &lt; startTime then 0 else e0)</b>
</html>
"));
      Modelica.Blocks.Sources.Step Step1(
        offset=offset,
        startTime=startTime,
        height=e0) annotation (extent=[70,68; 90,88]);
    equation
      e = Step1.y;
    end stepSe;

    model rampSe "Ramp effort source of the Modelica bond graph library"
      extends Interfaces.ActiveOnePort;
      parameter Real e0=1 "Height of ramp";
      parameter Modelica.SIunits.Time duration=2 "Duration of ramp";
      parameter Real offset=0 "Offset of ramp";
      parameter Modelica.SIunits.Time startTime=0 "Start time of ramp";
      annotation (
        Icon(
          Text(
            extent=[-60,70; 60,-90],
            string="Se",
            style(color=0)),
          Text(extent=[-70,100; 70,60],  string="%name"),
             Line(points=[-80,-70; -50,-70; 50,50; 80,50],     style(color=8))),
        Diagram(
          Polygon(points=[-80,80; -88,58; -72,58; -80,80],     style(color=8,
                fillColor=8)),
          Line(points=[-80,-30; -20,-30; 50,40],    style(color=0, thickness=2)),
          Line(points=[-90,-80; 82,-80],   style(color=8)),
          Polygon(points=[90,-80; 68,-72; 68,-88; 90,-80],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[-40,-30; -42,-40; -37,-40; -40,-30],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-40,-30; -40,-80],   style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Polygon(points=[-40,-80; -43,-70; -38,-70; -40,-80; -40,-80],
              style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Text(
            extent=[-80,-43; -41,-59],
            string="offset",
            style(color=9)),
          Text(
            extent=[-40,-80; 6,-98],
            string="startTime",
            style(color=9)),
          Text(
            extent=[-83,83; -38,58],
            style(color=9),
            string="e"),
          Text(
            extent=[70,-86; 94,-106],
            string="time",
            style(color=9)),
          Line(points=[-20,-30; -20,-80],   style(color=8, pattern=2)),
          Line(points=[-19,-30; 50,-30],   style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Line(points=[50,40; 101,40],   style(color=0, thickness=2)),
          Line(points=[50,40; 50,-30],   style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Polygon(points=[50,-30; 42,-28; 42,-32; 50,-30],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[-20,-30; -11,-28; -11,-32; -20,-30],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[50,40; 48,30; 53,30; 50,40],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[50,-30; 47,-20; 52,-20; 50,-30; 50,-30],      style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Text(
            extent=[53,15; 82,-3],
            style(color=9),
            string="e0"),
          Text(
            extent=[0,-27; 35,-47],
            string="duration",
            style(color=9))),
        Documentation(info="<html>
The ramp effort source is an active OnePort element.  It inherits the effort and flow variables from the active OnePort.  The ramp effort to be generated is modeled as a set of four parameters. <p>

The effort source has its causality stroke away from the source. <p>

<hr>

<pre>
Potential variables:

 e:          Bondgraphic effort variable

 f:          Bondgraphic flow variable, normalized positive for flows out of the model


Parameters:

 e0:         Height of ramp to be generated (default value = 1)

 duration:   Duration of ramp to be generated (default value = 2)

 offset:     Offset of ramp to be generated (default value = 0)

 startTime:  Start time of ramp to be generated (default value = 0)
</pre> <p>

<hr> <p>

Equations: <p>

<b>e = offset + (if time &lt; startTime then 0 else if time &lt; (startTime + duration)
                 then (time - startTime)*e0/duration else e0);</b>
</html>
"));
      Modelica.Blocks.Sources.Ramp Ramp1(
        height=e0,
        duration=duration,
        offset=offset,
        startTime=startTime) annotation (extent=[68,66; 90,88]);
    equation
      e = Ramp1.y;
    end rampSe;

    model expsinSe
      "Damped sinusoidal effort source of the Modelica bond graph library"
      extends Interfaces.ActiveOnePort;
      parameter Real e0=1 "Amplitude of sine wave";
      parameter Modelica.SIunits.Angle phase=0 "Phase of sine wave";
      parameter Modelica.SIunits.Frequency freqHz=2 "Frequency of sine wave";
      parameter Real damping=1 "Damping coefficient of sine wave";
      parameter Real offset=0 "Offset of sine wave";
      parameter Modelica.SIunits.Time startTime=0 "Start time of sine wave";
      annotation (
        Icon(
          Text(
            extent=[-60,80; 60,-80],
            string="Se",
            style(color=0)),
          Text(extent=[-64,98; 76,58],   string="%name"),
             Line(points=[-70,-4; -65.2,28.3; -62,46.3; -58.7,60.5; -55.5,70.2;
                -52.3,75.3; -49.1,75.6; -45.9,71.3; -42.7,63.1; -38.6,48.2; -33,
                21.8; -25,-17.9; -20.2,-37.7; -16.1,-49.9; -12.1,-57.2; -8.1,
                -59.3; -4.1,-56.5; -0.1,-49.3; 4.77,-36.1; 18.44,9.7; 23.3,22.4;
                28.1,30.8; 32.1,34; 36.9,33.2; 41.8,27.8; 48.2,15.4; 61.1,-14.5;
                67.5,-25.2; 73.1,-29.9; 78.7,-29.9; 85.2,-24.5; 90,-17.8],
                                               style(color=8))),
        Diagram(
          Line(points=[-86,-94; -86,80],   style(color=8)),
          Polygon(points=[-86,96; -92,80; -80,80; -86,96],       style(color=8,
                fillColor=8)),
          Line(points=[-105,-44; 79,-44],  style(color=8)),
          Polygon(points=[95,-44; 79,-38; 79,-50; 95,-44],       style(color=8,
                fillColor=8)),
          Line(points=[-56,-4; -52.1,24.2; -49.5,40; -46.9,52.4; -44.2,60.9;
                -41.6,65.4; -39,65.6; -36.4,61.9; -33.8,54.7; -30.5,41.7; -25.9,
                18.5; -19.4,-16.2; -15.5,-33.5; -12.23,-44.1; -8.96,-50.5;
                -5.698,-52.4; -2.43,-49.9; 0.83,-43.6; 4.8,-32.1; 15.9,8; 19.8,
                19.1; 23.7,26.5; 27,29.3; 30.9,28.5; 34.8,23.8; 40,12.9; 50.5,
                -13.2; 55.7,-22.6; 60.3,-26.7; 64.9,-26.6; 70.1,-22; 74,-16.1],
                                style(color=0, thickness=2)),
          Text(
            extent=[-84,-3; -61,-23],
            string="offset",
            style(color=9)),
          Text(
            extent=[-78,-40; -32,-58],
            string="startTime",
            style(color=9)),
          Text(
            extent=[78,-56; 102,-76],
            string="time",
            style(color=9)),
          Line(points=[-56,-4; 12,-4], style(color=8, pattern=2)),
          Line(points=[-56,-4; -87,-4], style(color=0, thickness=2)),
          Line(points=[-56,73; -56,-4],  style(color=8, pattern=2)),
          Line(points=[12,-5; 12,72],   style(color=8, pattern=2)),
          Line(points=[12,69; -56,69],   style(color=8)),
          Text(
            extent=[-48,84; 3,70],
            string="1/freqHz",
            style(color=9)),
          Polygon(points=[-55,69; -46,71; -46,67; -55,69],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[12,69; 4,71; 4,67; 12,69],       style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-56,-65; -25,-65],   style(color=8)),
          Polygon(points=[-24,-65; -32,-63; -32,-67; -24,-65],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Text(
            extent=[-57,-67; -33,-79],
            string="t",
            style(color=9)),
          Text(
            extent=[-88,-71; 102,-100],
            style(color=9),
            string="e0*exp(-damping*t)*sin(2*pi*freqHz*t+phase)"),
          Line(points=[-56,-4; -56,-44],  style(color=8, pattern=2)),
          Line(points=[-56,-58; -56,-76],   style(color=8, pattern=2)),
          Line(points=[12,-80; -7,-52],   style(color=8, pattern=2)),
          Text(
            extent=[-91,101; -52,82],
            style(color=9),
            string="e")),
        Documentation(info="<html>
The damped sinusoidal effort source is an active OnePort element.  It inherits the effort and flow variables from the active OnePort.  The damped sinusoidal effort to be generated is modeled as a set of six parameters. <p>

The effort source has its causality stroke away from the source. <p>

<hr>

<pre>
Potential variables:

 e:          Bondgraphic effort variable

 f:          Bondgraphic flow variable, normalized positive for flows out of the model


Parameters:

 e0:         Amplitude of damped sine wave to be generated (default value = 1)

 phase:      Phase of damped sine wave to be generated (default value = 0)

 freqHz:     Frequency of damped sine wave to be generated (default value = 2)

 damping:    Damping coefficient of damped sine wave to be generated (default value = 1)

 offset:     Offset of damped sine wave to be generated (default value = 0)

 startTime:  Start time of damped sine wave to be generated (default value = 0)
</pre> <p>

<hr> <p>

Equations: <p>

<b>e = offset + (if time &lt; startTime then 0 else e0*exp(-(time
       - startTime)*damping)*sin(2*pi*freqHz*(time - startTime) + phase))</b>
</html>
"));
      Modelica.Blocks.Sources.ExpSine ExpSine1(
        amplitude=e0,
        freqHz=freqHz,
        phase=phase,
        damping=damping,
        offset=offset,
        startTime=startTime) annotation (extent=[68,66; 90,88]);
    equation
      e = ExpSine1.y;
    end expsinSe;

    model expSe
      "Step response effort source of the Modelica bond graph library"
      extends Interfaces.ActiveOnePort;
      parameter Real e0=1 "Height of step response after infinite time";
      parameter Modelica.SIunits.Time riseTime(min=0) = 0.5 "Rise time";
      parameter Modelica.SIunits.Time riseTimeConst(min=Modelica.Constants.small) = 0.1
        "Rise time constant";
      parameter Modelica.SIunits.Time fallTimeConst(min=Modelica.Constants.small) = riseTimeConst
        "Fall time constant";
      parameter Real offset=0 "Offset of step response";
      parameter Modelica.SIunits.Time startTime=0 "Start time of step response";
      annotation (
        Icon(
          Text(
            extent=[-60,78; 60,-82],
            string="Se",
            style(color=0)),
          Text(extent=[-68,98; 72,58],   string="%name"),
             Line(points=[-70,-69; -67.2,-54.3; -64.3,-41.1; -60.8,-26.6; -57.3,
                -14; -53.7,-3.08; -49.5,8.18; -45.3,17.7; -40.3,27; -34.6,35.5;
                -28.3,43.1; -21.2,49.6; -12.7,55.3; -2.1,60.2; 0,61; 2.12,48.5;
                4.95,33.7; 7.78,20.8; 10.606,9.45; 14.14,-2.7; 17.68,-13; 21.9,
                -23.2; 26.2,-31.6; 31.1,-39.5; 36.8,-46.4; 43.1,-52.3; 50.9,
                -57.5; 60.8,-61.8; 70,-64.4],                            style(
                color=8))),
        Diagram(
          Line(points=[-100,-80; 84,-80],   style(color=8)),
          Polygon(points=[100,-80; 84,-74; 84,-86; 100,-80],     style(color=8,
                fillColor=8)),
          Line(points=[-40,-40; -37.2,-25.3; -34.3,-12.1; -30.8,2.4; -27.3,15;
                -23.7,25.92; -19.5,37.18; -15.3,46.7; -10.3,56; -4.6,64.5; 1.7,
                72.1; 8.8,78.6; 17.3,84.3; 27.9,89.2; 30,90; 32.12,77.5; 34.95,
                62.7; 37.78,49.8; 40.606,38.45; 44.14,26.3; 47.68,16; 51.9,5.8;
                56.2,-2.6; 61.1,-10.5; 66.8,-17.4; 73.1,-23.3; 80.9,-28.5; 90.8,
                -32.8; 100,-35.4],                              style(color=0,
                thickness=2)),
          Polygon(points=[-80,80; -88,58; -72,58; -80,80],     style(color=8,
                fillColor=8)),
          Line(points=[-80,58; -80,-90],   style(color=8)),
          Text(
            extent=[-78,-53; -46,-66],
            string="offset",
            style(color=9)),
          Polygon(points=[-40,-80; -43,-70; -38,-70; -40,-80; -40,-80],
              style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-40,-39; -40,-70],   style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Polygon(points=[-40,-40; -42,-50; -37,-50; -40,-40],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-39,-40; -80,-40],   style(color=0, thickness=2)),
          Text(
            extent=[-59,-81; -13,-99],
            string="startTime",
            style(color=9)),
          Polygon(points=[-41,-40; -32,-38; -32,-42; -41,-40],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-40,-40; 29,-40],   style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Polygon(points=[29,-40; 21,-38; 21,-42; 29,-40],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Text(
            extent=[-26,-22; 19,-38],
            string="riseTime",
            style(color=9)),
          Text(
            extent=[78,-86; 102,-106],
            string="time",
            style(color=9)),
          Line(points=[-100,-80; 84,-80],   style(color=8)),
          Polygon(points=[100,-80; 84,-74; 84,-86; 100,-80],     style(color=8,
                fillColor=8)),
          Line(points=[-40,-40; -37.2,-25.3; -34.3,-12.1; -30.8,2.4; -27.3,15;
                -23.7,25.92; -19.5,37.18; -15.3,46.7; -10.3,56; -4.6,64.5; 1.7,
                72.1; 8.8,78.6; 17.3,84.3; 27.9,89.2; 30,90; 32.12,77.5; 34.95,
                62.7; 37.78,49.8; 40.606,38.45; 44.14,26.3; 47.68,16; 51.9,5.8;
                56.2,-2.6; 61.1,-10.5; 66.8,-17.4; 73.1,-23.3; 80.9,-28.5; 90.8,
                -32.8; 100,-35.4],                              style(color=0,
                thickness=2)),
          Polygon(points=[-80,80; -88,58; -72,58; -80,80],     style(color=8,
                fillColor=8)),
          Line(points=[-80,58; -80,-90],   style(color=8)),
          Text(
            extent=[-82,81; -41,61],
            style(color=9),
            string="e"),
          Text(
            extent=[-78,-53; -46,-66],
            string="offset",
            style(color=9)),
          Polygon(points=[-40,-80; -43,-70; -38,-70; -40,-80; -40,-80],
              style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-40,-39; -40,-70],   style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Polygon(points=[-40,-40; -42,-50; -37,-50; -40,-40],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-39,-40; -80,-40],   style(color=0, thickness=2)),
          Text(
            extent=[-59,-81; -13,-99],
            string="startTime",
            style(color=9)),
          Polygon(points=[-41,-40; -32,-38; -32,-42; -41,-40],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-40,-40; 29,-40],   style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Polygon(points=[29,-40; 21,-38; 21,-42; 29,-40],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Text(
            extent=[-26,-22; 19,-38],
            string="riseTime",
            style(color=9)),
          Text(
            extent=[78,-86; 102,-106],
            string="time",
            style(color=9)),
          Line(points=[30,90; 30,-44],    style(color=8, pattern=2))),
        Documentation(info="<html>
The step response effort source is an active OnePort element.  It inherits the effort and flow variables from the active OnePort.  The step response effort to be generated is modeled as a set of six parameters. <p>

The effort source has its causality stroke away from the source. <p>

<hr>

<pre>
Potential variables:

 e:              Bondgraphic effort variable

 f:              Bondgraphic flow variable, normalized positive for flows out of the model


Parameters:

 e0:             Height of step response after infinite time (default value = 1)

 riseTime:       Rise time of step response to be generated (default value = 0.5)

 riseTimeConst:  Rise time constant of step response to be generated (default value = 0.1)

 fallTimeConst:  Fall time constant of step response to be generated (default value = riseTimeConst)

 offset:         Offset of step response to be generated (default value = 0)

 startTime:      Start time of step response to be generated (default value = 0)
</pre> <p>

<hr> <p>

Equations: <p>

<b>e_riseTime = e0*(1 - exp(-riseTime/riseTimeConst))</b> <br>
<b>e = offset + (if (time &lt; startTime) then 0 else if (time &lt; (startTime
     + riseTime)) then e0*(1 - exp(-(time - startTime)/riseTimeConst)) else
          e_riseTime*exp(-(time - startTime - riseTime)/fallTimeConst));</b>
</html>
"));
      Modelica.Blocks.Sources.Exponentials Exponentials1(
        outMax=e0,
        riseTime=riseTime,
        riseTimeConst=riseTimeConst,
        fallTimeConst=fallTimeConst,
        offset=offset,
        startTime=startTime) annotation (extent=[68,66; 90,88]);
    equation
      e = Exponentials1.y;
    end expSe;

    model pulseSe "Pulse effort source of the Modelica bond graph library"
      extends Interfaces.ActiveOnePort;
      parameter Real e0=1 "Amplitude of pulse";
      parameter Modelica.SIunits.Time width(
        final min=Modelica.Constants.small,
        final max=100) = 50 "Width of pulse in % of period";
      parameter Modelica.SIunits.Time period(final min=Modelica.Constants.small) = 1
        "Time for one period";
      parameter Real offset=0 "Offset of pulse";
      parameter Modelica.SIunits.Time startTime=0 "Start time of pulse";
      annotation (
        Icon(
          Text(
            extent=[-62,70; 58,-90],
            string="Se",
            style(color=0)),
          Text(extent=[-70,100; 70,60],  string="%name"),
             Line(points=[-76,-80; -46,-80; -46,60; -6,60; -6,-80; 34,-80; 35,
                60; 74,60],    style(color=8))),
        Diagram(
          Polygon(points=[-80,86; -88,64; -72,64; -80,86],     style(color=8,
                fillColor=8)),
          Line(points=[-80,64; -80,-84],   style(color=8)),
          Line(points=[-90,-74; 82,-74],   style(color=8)),
          Polygon(points=[90,-74; 68,-66; 68,-82; 90,-74],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[-34,-3; -37,-16; -30,-16; -34,-3],   style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-34,-5; -34,-74],   style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Polygon(points=[-33,-74; -36,-61; -30,-61; -33,-74; -33,-74],
              style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Text(
            extent=[-78,-28; -35,-40],
            string="offset",
            style(color=9)),
          Text(
            extent=[-39,-73; 7,-91],
            string="startTime",
            style(color=9)),
          Text(
            extent=[70,-84; 94,-104],
            string="time",
            style(color=9)),
          Line(points=[-18,-4; -18,-74],  style(color=8, pattern=2)),
          Line(points=[-80,-4; -20,-4; -20,46; 20,46; 20,-4; 40,-4; 40,46; 80,
                46],   style(color=0, thickness=2)),
          Line(points=[-19,84; -19,45],   style(color=8, pattern=2)),
          Line(points=[21,70; 21,46],   style(color=9, pattern=2)),
          Line(points=[41,84; 41,46],   style(color=8, pattern=2)),
          Line(points=[-19,79; 42,79],   style(color=8)),
          Line(points=[-19,65; 21,65],   style(color=8)),
          Text(
            extent=[-9,93; 37,81],
            string="period",
            style(color=9)),
          Text(
            extent=[-18,77; 21,65],
            string="width",
            style(color=9)),
          Line(points=[-52,46; -19,46],   style(color=8, pattern=2)),
          Line(points=[-34,46; -34,-3],  style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Text(
            extent=[-66,28; -25,14],
            style(color=9),
            string="e0"),
          Polygon(points=[-34,45; -37,32; -30,32; -34,45],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[-34,-3; -37,10; -31,10; -34,-3; -34,-3],   style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[80,46; 80,-4; 92,-4],   style(color=0, thickness=2)),
          Polygon(points=[-19,65; -10,67; -10,63; -19,65],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[21,65; 13,67; 13,63; 21,65],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[-19,79; -10,81; -10,77; -19,79],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[41,79; 33,81; 33,77; 41,79],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Text(
            extent=[-70,87; -52,68],
            style(color=9),
            string="e")),
        Documentation(info="<html>
The pulse effort source is an active OnePort element.  It inherits the effort and flow variables from the active OnePort.  The pulse effort to be generated is modeled as a set of five parameters. <p>

The effort source has its causality stroke away from the source. <p>

<hr>

<pre>
Potential variables:

 e:          Bondgraphic effort variable

 f:          Bondgraphic flow variable, normalized positive for flows out of the model


Parameters:

 e0:         Amplitude of pulse to be generated (default value = 1)

 width:      Percentage of period in on state (default value = 50)

 period:     Length of period of pulse to be generated (default value = 1)

 offset:     Offset of pulse to be generated (default value = 0)

 startTime:  Start time of pulse to be generated (default value = 0)
</pre> <p>

<hr> <p>

Equations: <p>

<b>when sample(startTime, period) then</b> <br>
<b>T0 = time</b> <br>
<b>end when</b> <br>
<b>e = offset + (if time &lt; startTime or time &gt;= T0 + T_width then 0 else e0)</b>
</html>
"));
      Modelica.Blocks.Sources.Pulse Pulse1(
        amplitude=e0,
        width=width,
        period=period,
        offset=offset,
        startTime=startTime) annotation (extent=[68,66; 90,88]);
    equation
      e = Pulse1.y;
    end pulseSe;

    model sawSe "Saw tooth effort source of the Modelica bond graph library"
      extends Interfaces.ActiveOnePort;
      parameter Real e0=1 "Amplitude of saw tooth";
      parameter Modelica.SIunits.Time period(final min=Modelica.Constants.small) = 1
        "Time for one period";
      parameter Real offset=0 "Offset of saw tooth";
      parameter Modelica.SIunits.Time startTime=0 "Start time of saw tooth";
      annotation (
        Icon(
          Text(
            extent=[-66,68; 54,-92],
            string="Se",
            style(color=0)),
          Text(extent=[-68,104; 72,64],  string="%name"),
             Line(points=[-78,-79; -58,-79; 2,62; 2,-79; 62,62; 62,-79],
              style(color=8))),
        Diagram(
          Polygon(points=[-80,80; -88,58; -72,58; -80,80],     style(color=8,
                fillColor=8)),
          Line(points=[-80,58; -80,-90],   style(color=8)),
          Line(points=[-90,-80; 82,-80],   style(color=8)),
          Polygon(points=[90,-80; 68,-72; 68,-88; 90,-80],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[-34,-29; -37,-42; -30,-42; -34,-29],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-34,-30; -34,-80],   style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Polygon(points=[-34,-80; -37,-67; -31,-67; -34,-80; -34,-80],
              style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Text(
            extent=[-78,-34; -35,-46],
            string="offset",
            style(color=9)),
          Text(
            extent=[-31,-79; 15,-97],
            string="startTime",
            style(color=9)),
          Text(
            extent=[74,-84; 98,-104],
            string="time",
            style(color=9)),
          Line(points=[-10,-30; -10,-80],   style(color=8, pattern=2)),
          Line(points=[-10,78; -10,-30],   style(color=8, pattern=2)),
          Line(points=[30,78; 30,49],   style(color=8, pattern=2)),
          Line(points=[-10,73; 30,73],   style(color=8)),
          Text(
            extent=[-12,87; 34,75],
            string="period",
            style(color=9)),
          Line(points=[-44,50; 30,50],   style(color=8, pattern=2)),
          Line(points=[-34,37; -34,-17],  style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Text(
            extent=[-66,18; -25,4],
            style(color=9),
            string="e0"),
          Polygon(points=[-34,50; -37,37; -30,37; -34,50],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[-34,-30; -37,-17; -31,-17; -34,-30; -34,-30],    style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[-10,73; -1,75; -1,71; -10,73],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[30,73; 22,75; 22,71; 30,73],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-80,-30; -10,-30; 30,50; 30,-30; 72,50; 72,-30],
              style(color=0, thickness=2)),
          Text(
            extent=[-69,82; -50,62],
            style(color=9),
            string="e")),
        Documentation(info="<html>
The saw tooth effort source is an active OnePort element.  It inherits the effort and flow variables from the active OnePort.  The saw tooth effort to be generated is modeled as a set of four parameters. <p>

The effort source has its causality stroke away from the source. <p>

<hr>

<pre>
Potential variables:

 e:          Bondgraphic effort variable

 f:          Bondgraphic flow variable, normalized positive for flows out of the model


Parameters:

 e0:         Amplitude of saw tooth to be generated (default value = 1)

 period:     Length of period of saw tooth to be generated (default value = 1)

 offset:     Offset of saw tooth to be generated (default value = 0)

 startTime:  Start time of saw tooth to be generated (default value = 0)
</pre> <p>

<hr> <p>

Equations: <p>

<b>when sample(startTime, period) then</b> <br>
<b>T0 = time</b> <br>
<b>end when</b> <br>
<b>e = offset + (if time &lt; startTime then 0 else (e0/period)*(time - T0))</b>
</html>
"));
      Modelica.Blocks.Sources.SawTooth SawTooth1(
        amplitude=e0,
        period=period,
        offset=offset,
        startTime=startTime) annotation (extent=[68,66; 90,88]);
    equation
      e = SawTooth1.y;
    end sawSe;

    model trapSe "Trapezoidal effort source of the Modelica bond graph library"
      extends Interfaces.ActiveOnePort;
      parameter Real e0=1 "Amplitude of trapezoidal function";
      parameter Modelica.SIunits.Time rising(final min=0) = 0
        "Rising duration of trapezoid";
      parameter Modelica.SIunits.Time width(final min=0) = 0.5
        "Width duration of trapezoid";
      parameter Modelica.SIunits.Time falling(final min=0) = 0
        "Falling duration of trapezoid";
      parameter Modelica.SIunits.Time period(final min=Modelica.Constants.small) = 1
        "Time for one period";
      parameter Integer nperiod=-1
        "Number of periods (< 0 means infinite number of periods)";
      parameter Real offset=0 "Offset of trapezoidal function";
      parameter Modelica.SIunits.Time startTime=0
        "Start time of trapezoidal function";
      annotation (
        Icon(
          Text(
            extent=[-56,78; 64,-82],
            string="Se",
            style(color=0)),
          Text(extent=[-64,104; 76,64],  string="%name"),
             Line(points=[-79,-78; -58,-78; -28,62; 3,62; 32,-78; 53,-78; 82,62],
                         style(color=8))),
        Diagram(
          Polygon(points=[-80,80; -88,58; -72,58; -80,80],     style(color=8,
                fillColor=8)),
          Line(points=[-80,58; -80,-90],   style(color=8)),
          Line(points=[-90,-80; 82,-80],   style(color=8)),
          Polygon(points=[90,-80; 68,-72; 68,-88; 90,-80],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[-45,-40; -47,-51; -43,-51; -45,-40],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-45,-41; -45,-80],   style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Polygon(points=[-45,-80; -47,-70; -43,-70; -45,-80; -45,-80],
              style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Text(
            extent=[-86,-53; -43,-65],
            string="offset",
            style(color=9)),
          Text(
            extent=[-47,-79; -1,-97],
            string="startTime",
            style(color=9)),
          Text(
            extent=[76,-84; 100,-104],
            string="time",
            style(color=9)),
          Line(points=[-30,71; -30,-80],   style(color=8, pattern=2)),
          Line(points=[-10,49; -10,30],   style(color=8, pattern=2)),
          Line(points=[20,49; 20,29],   style(color=9, pattern=2)),
          Line(points=[40,49; 40,-40],   style(color=8, pattern=2)),
          Line(points=[-29,46; 40,46],   style(color=8)),
          Text(
            extent=[-8,60; 21,50],
            string="width",
            style(color=9)),
          Line(points=[-42,30; -10,30],   style(color=8, pattern=2)),
          Line(points=[-39,30; -39,-29],   style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Text(
            extent=[-67,6; -30,-8],
            style(color=9),
            string="e0"),
          Polygon(points=[-29,46; -22,48; -22,44; -29,46],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[-10,46; -17,48; -17,44; -10,46],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-80,-40; -30,-40; -9,36; 21,36; 40,-40; 60,-40; 80,36;
                100,36],      style(color=0, thickness=2)),
          Polygon(points=[-39,30; -41,19; -37,19; -39,30],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[-39,-39; -41,-29; -37,-29; -39,-39; -39,-39],
              style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[60,71; 60,-40],   style(color=8, pattern=2)),
          Polygon(points=[39,46; 32,48; 32,44; 39,46],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[20,46; 27,48; 27,44; 20,46],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[20,46; 13,48; 13,44; 20,46],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[-12,46; -5,48; -5,44; -12,46],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Text(
            extent=[-34,60; -5,50],
            string="rising",
            style(color=9)),
          Text(
            extent=[16,60; 45,50],
            string="falling",
            style(color=9)),
          Text(
            extent=[-75,85; -46,62],
            style(color=9),
            string="e"),
          Line(points=[-20,66; 61,66],   style(color=8)),
          Polygon(points=[-29,66; -20,68; -20,64; -29,66],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[61,66; 53,68; 53,64; 61,66],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Text(
            extent=[-2,76; 25,67],
            string="period",
            style(color=9))),
        Documentation(info="<html>
The trapezoidal effort source is an active OnePort element.  It inherits the effort and flow variables from the active OnePort.  The trapezoidal effort to be generated is modeled as a set of eight parameters. <p>

The effort source has its causality stroke away from the source. <p>

<hr>

<pre>
Potential variables:

 e:          Bondgraphic effort variable

 f:          Bondgraphic flow variable, normalized positive for flows out of the model


Parameters:

 e0:         Amplitude of trapezoid to be generated (default value = 1)

 rising:     Length of rising time of trapezoid to be generated (default value = 0)

 width :     Length of on state of trapezoid to be generated (default value = 0.5)

 falling:    Length of falling time of trapezoid to be generated (default value = 0)

 period:     Length of period of trapezoid to be generated (default value = 1)

 nperiod:    Number of periods of trapezoid to be generated
             an infinite number of periods is being generated if nperiod &lt; 0 (default value = -1)

 offset:     Offset of trapezoid to be generated (default value = 0)

 startTime:  Start time of trapezoid to be generated (default value = 0)
</pre> <p>

<hr> <p>

Equations: <p>

<b>when pre(counter2) &lt;&gt; 0 and sample(startTime, period) then</b> <br>
<b>T0 = time</b> <br>
<b>counter2 = pre(counter)</b> <br>
<b>counter = pre(counter) - (if pre(counter) &gt; 0 then 1 else 0)</b> <br>
<b>end when</b> <br>
<b>e = offset + (if (time &lt; startTime or counter2 == 0 or time &gt;= T0 +
    T_falling) then 0 else if (time &lt; T0 + T_rising) then (time - T0)*e0/T_rising
    else if (time &lt; T0 + T_width) then e0 else
    (T0 + T_falling - time)*e0/(T_falling - T_width))</b>
</html>
"));
      Modelica.Blocks.Sources.Trapezoid Trapezoid1(
        amplitude=e0,
        rising=rising,
        width=width,
        falling=falling,
        period=period,
        nperiod=nperiod,
        offset=offset,
        startTime=startTime) annotation (extent=[68,66; 90,88]);
    equation
      e = Trapezoid1.y;
    end trapSe;

    model tableSe "Tabular effort source of the Modelica bond graph library"
      extends Interfaces.ActiveOnePort;
      parameter Real table[:, :]=[0, 0; 1, 1; 2, 4]
        "Table matrix (time = first column, voltage = second column)";
      parameter Real offset=0 "Offset of time function";
      parameter Modelica.SIunits.Time startTime=0 "Start time of time function";
      annotation (
        Icon(
          Text(
            extent=[-58,78; 62,-82],
            string="Se",
            style(color=0)),
          Text(extent=[-66,104; 74,64],  string="%name"),
             Line(points=[-64,-62; -64,58; 36,58; 36,-62; -64,-62; -64,-32; 36,
                -32; 36,-2; -64,-2; -64,28; 36,28; 36,58; -14,58; -14,-63],
                     style(color=8))),
        Diagram(
          Polygon(points=[-80,80; -88,58; -72,58; -80,80],     style(color=8,
                fillColor=8)),
          Line(points=[-80,58; -80,-90],   style(color=8)),
          Line(points=[-90,-80; 82,-80],   style(color=8)),
          Polygon(points=[90,-80; 68,-72; 68,-88; 90,-80],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Rectangle(extent=[-20,80; 30,-40],   style(
              color=7,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-20,-40; -20,80; 80,80; 80,-40; -20,-40; -20,-10; 80,-10;
                80,20; -20,20; -20,50; 80,50; 80,80; 30,80; 30,-41],
              style(color=0)),
          Text(
            extent=[-77,-52; -38,-68],
            string="offset",
            style(color=9)),
          Polygon(points=[-31,-40; -33,-50; -28,-50; -31,-40],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[-30,-80; -33,-70; -28,-70; -30,-80; -30,-80],
              style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-31,-41; -31,-80],   style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Line(points=[-20,-42; -20,-80],   style(color=8, pattern=2)),
          Text(
            extent=[-38,-80; 8,-98],
            string="startTime",
            style(color=9)),
          Line(points=[-20,-40; -80,-40],   style(color=8, pattern=2)),
          Text(
            extent=[72,-87; 97,-99],
            string="time",
            style(color=9)),
          Text(
            extent=[-15,73; 24,58],
            string="time",
            style(color=0)),
          Text(
            extent=[33,73; 76,57],
            string="v",
            style(color=0)),
          Text(
            extent=[-73,84; -50,62],
            style(color=9),
            string="e")),
        Documentation(info="<html>
The tabular effort source is an active OnePort element.  It inherits the effort and flow variables from the active OnePort.  The tabular effort to be generated is modeled as a two-dimensional parameter table plus a set of two scalar parameters. <p>

The effort source has its causality stroke away from the source. <p>

<hr>

<pre>
Potential variables:

 e:          Bondgraphic effort variable

 f:          Bondgraphic flow variable, normalized positive for flows out of the model


Parameters:

 table:      Two-dimensional table with supporting values (default value = [0, 0; 1, 1; 2, 4])

 offset:     Offset of time function to be generated (default value = 0)

 startTime:  Start time of time function to be generated (default value = 0)
</pre> <p>

<hr> <p>

The model generates an effort source by <b>linear interpolation</b> in
a table. The time points and effort values are stored in a matrix
<b>table[i,j]</b>, where the first column table[:,1] contains the
time points, and the second column contains the effort to be interpolated.
The table interpolation has the following proporties: <p>

<ul>
<li>The time points need to be <b>monotonically increasing</b>.
<li><b>Discontinuities</b> are allowed, by providing the same
    time point twice in the table.
<li>Values <b>outside</b> of the table range, are computed by
    <b>extrapolation</b> through the last or first two points of the
    table.
<li>If the table has only <b>one row</b>, no interpolation is performed and
    the voltage value is just returned independantly of the
    actual time instant, i.e., this is a constant voltage source.
<li>Via parameters <b>startTime</b> and <b>offset</b> the curve defined
    by the table can be shifted both in time and in the voltage.
<li>The table is implemented in a numerically sound way by
    generating <b>time events</b> at interval boundaries,
    in order to not integrate over a discontinuous or not differentiable
    points.
</ul> <p>

<hr> <p>

<b>Example:</b> <p>

<pre>
   table = [0  0
            1  0
            1  1
            2  4
            3  9
            4 16]
If, e.g., time = 1.0, the effort e =  0.0 (before event), 1.0 (after event)
    e.g., time = 1.5, the effort e =  2.5,
    e.g., time = 2.0, the effort e =  4.0,
    e.g., time = 5.0, the effort e = 23.0 (i.e. extrapolation).
</pre>
</html>
"));
      Modelica.Blocks.Sources.TimeTable TimeTable1(
        table=table,
        offset=offset,
        startTime=startTime) annotation (extent=[68,68; 90,90]);
    equation
      e = TimeTable1.y;
    end tableSe;

    model mSe_int "Internal modulated effort source of the bond graph library"
      extends Interfaces.ModTwoPort;
      Real e0 "Bondgraphic Effort Source";
      annotation (
        Diagram(Text(
            extent=[-60,80; 60,-80],
            string="mSe",
            style(color=0))),
        Icon(   Text(
            extent=[-60,80; 60,-80],
            string="mSe",
            style(color=0)), Text(extent=[-70,90; 70,50],   string="%name")),
        Documentation(info="<html>
The modulated internal effort source is a modulated directed TwoPort element.  It inherits the effort and flow variables from the modulated directed TwoPort.  The modulated effort to be generated is modeled as a real-valued variable. <p>

The modulated internal effort source is a modulated effort source on its secondary side.  The primary side provides the power needed by the effort source.  This element can e.g. be used to model a DC power supply together with the power cable that draws the required energy from the grid. <p>

The effort source has its secondary causality stroke away from the source.  Its primary causality stroke is free. <p>

<hr>

<pre>
Potential variables:

 e1:  Bondgraphic effort variable of inflow

 f1:  Bondgraphic flow variable of inflow, normalized positive for flows into the model

 e2:  Bondgraphic effort variable of outflow

 f2:  Bondgraphic flow variable of outflow, normalized positive for flows out of the model

 e0:  Effort to be generated (modulated input signal)
</pre> <p>

<hr> <p>

Equations: <p>

<b>e2 = e0</b> <br>
<b>e1*f1 = e2*f2</b>
</html>
"));
    equation
      e0 = s;
      e2 = e0;
      e1*f1 = e2*f2;
    end mSe_int;

    model Sf "Constant flow source of the Modelica bond graph library"
      extends Interfaces.ActiveOnePort;
      parameter Real f0=1 "Bondgraphic Flow Source";
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Text(
            extent=[-60, 80; 60, -80],
            string="Sf",
            style(color=0))),
        Icon(
          Text(
            extent=[-60, 80; 60, -80],
            string="Sf",
            style(color=0)),
          Text(extent=[-70, 90; 70, 50], string="%name"),
          Text(extent=[-70, -50; 70, -90], string="f0=%f0")),
        Window(
          x=0.45,
          y=0.01,
          width=0.35,
          height=0.63),
        Documentation(info="<html>
The constant flow source is an active OnePort element.  It inherits the effort and flow variables from the active OnePort.  The constant flow to be generated is modeled as a parameter. <p>

The flow source has its causality stroke at the source. <p>

<hr>

<pre>
Potential variables:

 e:   Bondgraphic effort variable

 f:   Bondgraphic flow variable, normalized positive for flows out of the model


Parameters:

 f0:  Constant flow to be generated (default value = 1)
</pre> <p>

<hr> <p>

Equations: <p>

<b>f = f0</b>
</html>
"));
    equation
      f = f0;
    end Sf;

    model mSf "Modulated flow source of the Modelica bond graph library"
      extends Interfaces.ModActiveOnePort;
      Real f0 "Bondgraphic Effort Source";
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Text(
            extent=[-60, 80; 60, -80],
            string="mSf",
            style(color=0))),
        Icon(Text(
            extent=[-60, 80; 60, -80],
            string="mSf",
            style(color=0)), Text(extent=[-70, 90; 70, 50], string="%name")),
        Window(
          x=0.45,
          y=0.01,
          width=0.35,
          height=0.63),
        Documentation(info="<html>
The modulated flow source is a modulated active OnePort element.  It inherits the effort and flow variables from the modulated active OnePort.  The modulated flow to be generated is modeled as a real-valued variable. <p>

Modulated sources are commonly used to model actuators of control systems.  They are the most common and convenient way for converting signal variables to bondgraphic variables, thereby allowing individual blocks of a block diagram, e.g. describing a control system, to be modeled internally using bond graphs. <p>

The flow source has its causality stroke at the source. <p>

<hr>

<pre>
Potential variables:

 e:   Bondgraphic effort variable

 f:   Bondgraphic flow variable, normalized positive for flows out of the model

 f0:  Flow to be generated (modulated input signal)
</pre> <p>

<hr> <p>

Equations: <p>

<b>f = f0</b>
</html>
"));
    equation
      f0 = s;
      f = f0;
    end mSf;

    model sinSf "Sinusoidal flow source of the Modelica bond graph library"
      extends Interfaces.ActiveOnePort;
      parameter Real f0=1 "Amplitude of sine wave";
      parameter Modelica.SIunits.Angle phase=0 "Phase of sine wave";
      parameter Modelica.SIunits.Frequency freqHz=1 "Frequency of sine wave";
      parameter Real offset=0 "Offset of sine wave";
      parameter Modelica.SIunits.Time startTime=0 "Start time of sine wave";
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(
          Line(points=[-80, -90; -80, 84], style(color=8)),
          Polygon(points=[-80, 100; -86, 84; -74, 84; -80, 100], style(color=8,
                  fillColor=8)),
          Line(points=[-99, -40; 85, -40], style(color=8)),
          Polygon(points=[101, -40; 85, -34; 85, -46; 101, -40], style(color=8,
                  fillColor=8)),
          Line(points=[-40, 0; -31.6, 34.2; -26.1, 53.1; -21.3, 66.4; -17.1,
                74.6; -12.9, 79.1; -8.64, 79.8; -4.42, 76.6; -0.201, 69.7; 4.02,
                  59.4; 8.84, 44.1; 14.9, 21.2; 27.5, -30.8; 33, -50.2; 37.8, -
                64.2; 42, -73.1; 46.2, -78.4; 50.5, -80; 54.7, -77.6; 58.9, -
                71.5; 63.1, -61.9; 67.9, -47.2; 74, -24.8; 80, 0], style(color=
                  0, thickness=2)),
          Line(points=[-41, -2; -80, -2], style(color=0, thickness=2)),
          Text(
            extent=[-106, -11; -60, -29],
            string="offset",
            style(color=9)),
          Line(points=[-41, -2; -41, -40], style(color=8, pattern=2)),
          Text(
            extent=[-60, -43; -14, -61],
            string="startTime",
            style(color=9)),
          Text(
            extent=[84, -52; 108, -72],
            string="time",
            style(color=9)),
          Line(points=[-9, 79; 43, 79], style(color=8, pattern=2)),
          Line(points=[-42, -1; 50, 0], style(color=8, pattern=2)),
          Polygon(points=[33, 80; 30, 67; 37, 67; 33, 80], style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[33, 1; 30, 14; 36, 14; 33, 1; 33, 1], style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[33, 79; 33, 0], style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Text(
            extent=[32, 52; 72, 34],
            string="f0",
            style(color=9)),
          Text(
            extent=[-86, 96; -44, 78],
            string="f",
            style(color=9))),
        Icon(
          Text(
            extent=[-60, 80; 60, -80],
            string="Sf",
            style(color=0)),
          Line(points=[-70, 0; -60.2, 29.9; -53.8, 46.5; -48.2, 58.1; -43.3,
                65.2; -38.3, 69.2; -33.4, 69.8; -28.5, 67; -23.6, 61; -18.6, 52;
                  -13, 38.6; -5.98, 18.6; 8.79, -26.9; 15.1, -44; 20.8, -56.2;
                25.7, -64; 30.6, -68.6; 35.5, -70; 40.5, -67.9; 45.4, -62.5;
                50.3, -54.1; 55.9, -41.3; 63, -21.7; 70, 0], style(color=8)),
          Text(extent=[-70, 90; 70, 50], string="%name")),
        Window(
          x=0.45,
          y=0.01,
          width=0.35,
          height=0.63),
        Documentation(info="<html>
The sinusoidal flow source is an active OnePort element.  It inherits the effort and flow variables from the active OnePort.  The sinusoidal flow to be generated is modeled as a set of five parameters. <p>

The flow source has its causality stroke at the source. <p>

<hr>

<pre>
Potential variables:

 e:          Bondgraphic effort variable

 f:          Bondgraphic flow variable, normalized positive for flows out of the model


Parameters:

 f0:         Amplitude of sine wave to be generated (default value = 1)

 phase:      Phase of sine wave to be generated (default value = 0)

 freqHz:     Frequency of sine wave to be generated (default value = 1)

 offset:     Offset of sine wave to be generated (default value = 0)

 startTime:  Start time of sine wave to be generated (default value = 0)
</pre> <p>

<hr> <p>

Equations: <p>

<b>f = offset + f0*sin(2*pi*freqHz*(time - startTime))</b>
</html>
"));
      Modelica.Blocks.Sources.Sine Sine1(
        amplitude=f0,
        freqHz=freqHz,
        offset=offset,
        phase=phase,
        startTime=startTime)   annotation (extent=[70, 66; 90, 86]);
    equation
      f = Sine1.y;
    end sinSf;

    model stepSf "Step flow source of the Modelica bond graph library"
      extends Interfaces.ActiveOnePort;
      parameter Real f0=1 "Height of step";
      parameter Real offset=0 "Offset of step";
      parameter Modelica.SIunits.Time startTime=0 "Start time of step";
      annotation (
        Icon(
          Text(
            extent=[-70,74; 36,-78],
            string="Sf",
            style(color=0)),
          Text(extent=[-76,102; 64,62],  string="%name"),
             Line(points=[-70,-80; 0,-80; 0,60; 69,60],     style(color=8))),
        Diagram(
          Polygon(points=[-80,90; -88,68; -72,68; -80,90],     style(color=8,
                fillColor=8)),
          Line(points=[-80,68; -80,-80],   style(color=8)),
          Line(points=[-80,-18; 0,-18; 0,50; 80,50],     style(color=0, thickness=
                 2)),
          Line(points=[-90,-70; 82,-70],   style(color=8)),
          Polygon(points=[90,-70; 68,-62; 68,-78; 90,-70],     style(color=8,
                fillColor=8)),
          Text(
            extent=[70,-80; 94,-100],
            string="time",
            style(color=9)),
          Text(
            extent=[-21,-72; 25,-90],
            string="startTime",
            style(color=9)),
          Line(points=[0,-17; 0,-71],   style(color=8, pattern=2)),
          Text(
            extent=[-68,-36; -22,-54],
            string="offset",
            style(color=9)),
          Line(points=[-13,50; -13,-17],   style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Polygon(points=[2,50; -19,50; 2,50],    style(color=8, pattern=2)),
          Polygon(points=[-13,-17; -16,-4; -10,-4; -13,-17; -13,-17],      style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[-13,50; -16,37; -9,37; -13,50],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Text(
            extent=[-50,26; -4,8],
            style(color=9),
            string="f0"),
          Polygon(points=[-13,-69; -16,-56; -10,-56; -13,-69; -13,-69],
              style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-13,-18; -13,-70],   style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Polygon(points=[-13,-18; -16,-31; -9,-31; -13,-18],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Text(
            extent=[-91,93; -32,70],
            style(color=9),
            string="f")),
        Documentation(info="<html>
The step flow source is an active OnePort element.  It inherits the effort and flow variables from the active OnePort.  The step flow to be generated is modeled as a set of three parameters. <p>

The flow source has its causality stroke at the source. <p>

<hr>

<pre>
Potential variables:

 e:          Bondgraphic effort variable

 f:          Bondgraphic flow variable, normalized positive for flows out of the model


Parameters:

 f0:         Height of step to be generated (default value = 1)

 offset:     Offset of step to be generated (default value = 0)

 startTime:  Start time of step to be generated (default value = 0)
</pre> <p>

<hr> <p>

Equations: <p>

<b>f = offset + (if time &lt; startTime then 0 else f0)</b>
</html>
"));
      Modelica.Blocks.Sources.Step Step1(
        offset=offset,
        startTime=startTime,
        height=f0) annotation (extent=[70,68; 90,88]);
    equation
      f = Step1.y;
    end stepSf;

    model rampSf "Ramp flow source of the Modelica bond graph library"
      extends Interfaces.ActiveOnePort;
      parameter Real f0=1 "Height of ramp";
      parameter Modelica.SIunits.Time duration=2 "Duration of ramp";
      parameter Real offset=0 "Offset of ramp";
      parameter Modelica.SIunits.Time startTime=0 "Start time of ramp";
      annotation (
        Icon(
          Text(
            extent=[-60,64; 36,-68],
            style(color=0),
            string="Sf"),
          Text(extent=[-70,106; 70,66],  string="%name"),
             Line(points=[-80,-60; -50,-60; 50,60; 80,60],     style(color=8))),
        Diagram(
          Polygon(points=[-80,80; -88,58; -72,58; -80,80],     style(color=8,
                fillColor=8)),
          Line(points=[-80,-30; -20,-30; 50,40],    style(color=0, thickness=2)),
          Line(points=[-90,-80; 82,-80],   style(color=8)),
          Polygon(points=[90,-80; 68,-72; 68,-88; 90,-80],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[-40,-30; -42,-40; -37,-40; -40,-30],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-40,-30; -40,-80],   style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Polygon(points=[-40,-80; -43,-70; -38,-70; -40,-80; -40,-80],
              style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Text(
            extent=[-80,-43; -41,-59],
            string="offset",
            style(color=9)),
          Text(
            extent=[-40,-80; 6,-98],
            string="startTime",
            style(color=9)),
          Text(
            extent=[-83,83; -38,58],
            style(color=9),
            string="f"),
          Text(
            extent=[70,-86; 94,-106],
            string="time",
            style(color=9)),
          Line(points=[-20,-30; -20,-80],   style(color=8, pattern=2)),
          Line(points=[-19,-30; 50,-30],   style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Line(points=[50,40; 101,40],   style(color=0, thickness=2)),
          Line(points=[50,40; 50,-30],   style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Polygon(points=[50,-30; 42,-28; 42,-32; 50,-30],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[-20,-30; -11,-28; -11,-32; -20,-30],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[50,40; 48,30; 53,30; 50,40],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[50,-30; 47,-20; 52,-20; 50,-30; 50,-30],      style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Text(
            extent=[53,15; 82,-3],
            style(color=9),
            string="f0"),
          Text(
            extent=[0,-27; 35,-47],
            string="duration",
            style(color=9))),
        Documentation(info="<html>
The ramp flow source is an active OnePort element.  It inherits the effort and flow variables from the active OnePort.  The ramp flow to be generated is modeled as a set of four parameters. <p>

The flow source has its causality stroke at the source. <p>

<hr>

<pre>
Potential variables:

 e:          Bondgraphic effort variable

 f:          Bondgraphic flow variable, normalized positive for flows out of the model


Parameters:

 f0:         Height of ramp to be generated (default value = 1)

 duration:   Duration of ramp to be generated (default value = 2)

 offset:     Offset of ramp to be generated (default value = 0)

 startTime:  Start time of ramp to be generated (default value = 0)
</pre> <p>

<hr> <p>

Equations: <p>

<b>f = offset + (if time &lt; startTime then 0 else if time &lt; (startTime + duration)
                 then (time - startTime)*f0/duration else f0);</b>
</html>
"));
      Modelica.Blocks.Sources.Ramp Ramp1(
        duration=duration,
        offset=offset,
        startTime=startTime,
        height=f0)           annotation (extent=[68,66; 90,88]);
    equation
      f = Ramp1.y;
    end rampSf;

    model expsinSf
      "Damped sinusoidal flow source of the Modelica bond graph library"
      extends Interfaces.ActiveOnePort;
      parameter Real f0=1 "Amplitude of sine wave";
      parameter Modelica.SIunits.Angle phase=0 "Phase of sine wave";
      parameter Modelica.SIunits.Frequency freqHz=2 "Frequency of sine wave";
      parameter Real damping=1 "Damping coefficient of sine wave";
      parameter Real offset=0 "Offset of sine wave";
      parameter Modelica.SIunits.Time startTime=0 "Start time of sine wave";
      annotation (
        Icon(
          Text(
            extent=[-70,62; 24,-64],
            style(color=0),
            string="Sf"),
          Text(extent=[-68,102; 72,62],  string="%name"),
             Line(points=[-80,-4; -75.2,28.3; -72,46.3; -68.7,60.5; -65.5,70.2;
                -62.3,75.3; -59.1,75.6; -55.9,71.3; -52.7,63.1; -48.6,48.2; -43,
                21.8; -35,-17.9; -30.2,-37.7; -26.1,-49.9; -22.1,-57.2; -18.1,
                -59.3; -14.1,-56.5; -10.1,-49.3; -5.23,-36.1; 8.44,9.7; 13.3,
                22.4; 18.1,30.8; 22.1,34; 26.9,33.2; 31.8,27.8; 38.2,15.4; 51.1,
                -14.5; 57.5,-25.2; 63.1,-29.9; 68.7,-29.9; 75.2,-24.5; 80,-17.8],
                                               style(color=8))),
        Diagram(
          Line(points=[-88,-92; -88,82],   style(color=8)),
          Polygon(points=[-88,98; -94,82; -82,82; -88,98],       style(color=8,
                fillColor=8)),
          Line(points=[-107,-42; 77,-42],  style(color=8)),
          Polygon(points=[93,-42; 77,-36; 77,-48; 93,-42],       style(color=8,
                fillColor=8)),
          Line(points=[-58,-2; -54.1,26.2; -51.5,42; -48.9,54.4; -46.2,62.9;
                -43.6,67.4; -41,67.6; -38.4,63.9; -35.8,56.7; -32.5,43.7; -27.9,
                20.5; -21.4,-14.2; -17.5,-31.5; -14.23,-42.1; -10.96,-48.5;
                -7.698,-50.4; -4.43,-47.9; -1.17,-41.6; 2.8,-30.1; 13.9,10;
                17.8,21.1; 21.7,28.5; 25,31.3; 28.9,30.5; 32.8,25.8; 38,14.9;
                48.5,-11.2; 53.7,-20.6; 58.3,-24.7; 62.9,-24.6; 68.1,-20; 72,
                -14.1],         style(color=0, thickness=2)),
          Text(
            extent=[-86,-1; -63,-21],
            string="offset",
            style(color=9)),
          Text(
            extent=[-80,-38; -34,-56],
            string="startTime",
            style(color=9)),
          Text(
            extent=[76,-54; 100,-74],
            string="time",
            style(color=9)),
          Line(points=[-58,-2; 10,-2], style(color=8, pattern=2)),
          Line(points=[-58,-2; -89,-2], style(color=0, thickness=2)),
          Line(points=[-58,75; -58,-2],  style(color=8, pattern=2)),
          Line(points=[10,-3; 10,74],   style(color=8, pattern=2)),
          Line(points=[10,71; -58,71],   style(color=8)),
          Text(
            extent=[-50,86; 1,72],
            string="1/freqHz",
            style(color=9)),
          Polygon(points=[-57,71; -48,73; -48,69; -57,71],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[10,71; 2,73; 2,69; 10,71],       style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-58,-63; -27,-63],   style(color=8)),
          Polygon(points=[-26,-63; -34,-61; -34,-65; -26,-63],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Text(
            extent=[-59,-65; -35,-77],
            string="t",
            style(color=9)),
          Text(
            extent=[-90,-69; 100,-98],
            style(color=9),
            string="f0*exp(-damping*t)*sin(2*pi*freqHz*t+phase)"),
          Line(points=[-58,-2; -58,-42],  style(color=8, pattern=2)),
          Line(points=[-58,-56; -58,-74],   style(color=8, pattern=2)),
          Line(points=[10,-78; -9,-50],   style(color=8, pattern=2)),
          Text(
            extent=[-93,101; -54,82],
            style(color=9),
            string="f")),
        Documentation(info="<html>
The damped sinusoidal flow source is an active OnePort element.  It inherits the effort and flow variables from the active OnePort.  The damped sinusoidal flow to be generated is modeled as a set of six parameters. <p>

The flow source has its causality stroke at the source. <p>

<hr>

<pre>
Potential variables:

 e:          Bondgraphic effort variable

 f:          Bondgraphic flow variable, normalized positive for flows out of the model


Parameters:

 f0:         Amplitude of damped sine wave to be generated (default value = 1)

 phase:      Phase of damped sine wave to be generated (default value = 0)

 freqHz:     Frequency of damped sine wave to be generated (default value = 2)

 damping:    Damping coefficient of damped sine wave to be generated (default value = 1)

 offset:     Offset of damped sine wave to be generated (default value = 0)

 startTime:  Start time of damped sine wave to be generated (default value = 0)
</pre> <p>

<hr> <p>

Equations: <p>

<b>f = offset + (if time &lt; startTime then 0 else f0*exp(-(time
       - startTime)*damping)*sin(2*pi*freqHz*(time - startTime) + phase))</b>
</html>
"));
      Modelica.Blocks.Sources.ExpSine ExpSine1(
        freqHz=freqHz,
        phase=phase,
        damping=damping,
        offset=offset,
        startTime=startTime,
        amplitude=f0) annotation (extent=[66,68; 88,90]);
    equation
      f = ExpSine1.y;
    end expsinSf;

    model expSf "Step response flow source of the Modelica bond graph library"
      extends Interfaces.ActiveOnePort;
      parameter Real f0=1 "Height of step response after infinite time";
      parameter Modelica.SIunits.Time riseTime(min=0) = 0.5 "Rise time";
      parameter Modelica.SIunits.Time riseTimeConst(min=Modelica.Constants.small) = 0.1
        "Rise time constant";
      parameter Modelica.SIunits.Time fallTimeConst(min=Modelica.Constants.small) = riseTimeConst
        "Fall time constant";
      parameter Real offset=0 "Offset of step response";
      parameter Modelica.SIunits.Time startTime=0 "Start time of step response";
      annotation (
        Icon(
          Text(
            extent=[-60,78; 34,-80],
            style(color=0),
            string="Sf"),
          Text(extent=[-68,98; 72,58],   string="%name"),
             Line(points=[-70,-69; -67.2,-54.3; -64.3,-41.1; -60.8,-26.6; -57.3,
                -14; -53.7,-3.08; -49.5,8.18; -45.3,17.7; -40.3,27; -34.6,35.5;
                -28.3,43.1; -21.2,49.6; -12.7,55.3; -2.1,60.2; 0,61; 2.12,48.5;
                4.95,33.7; 7.78,20.8; 10.606,9.45; 14.14,-2.7; 17.68,-13; 21.9,
                -23.2; 26.2,-31.6; 31.1,-39.5; 36.8,-46.4; 43.1,-52.3; 50.9,
                -57.5; 60.8,-61.8; 70,-64.4],                            style(
                color=8))),
        Diagram(
          Line(points=[-100,-80; 84,-80],   style(color=8)),
          Polygon(points=[100,-80; 84,-74; 84,-86; 100,-80],     style(color=8,
                fillColor=8)),
          Line(points=[-40,-40; -37.2,-25.3; -34.3,-12.1; -30.8,2.4; -27.3,15;
                -23.7,25.92; -19.5,37.18; -15.3,46.7; -10.3,56; -4.6,64.5; 1.7,
                72.1; 8.8,78.6; 17.3,84.3; 27.9,89.2; 30,90; 32.12,77.5; 34.95,
                62.7; 37.78,49.8; 40.606,38.45; 44.14,26.3; 47.68,16; 51.9,5.8;
                56.2,-2.6; 61.1,-10.5; 66.8,-17.4; 73.1,-23.3; 80.9,-28.5; 90.8,
                -32.8; 100,-35.4],                              style(color=0,
                thickness=2)),
          Polygon(points=[-80,80; -88,58; -72,58; -80,80],     style(color=8,
                fillColor=8)),
          Line(points=[-80,58; -80,-90],   style(color=8)),
          Text(
            extent=[-78,-53; -46,-66],
            string="offset",
            style(color=9)),
          Polygon(points=[-40,-80; -43,-70; -38,-70; -40,-80; -40,-80],
              style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-40,-39; -40,-70],   style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Polygon(points=[-40,-40; -42,-50; -37,-50; -40,-40],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-39,-40; -80,-40],   style(color=0, thickness=2)),
          Text(
            extent=[-59,-81; -13,-99],
            string="startTime",
            style(color=9)),
          Polygon(points=[-41,-40; -32,-38; -32,-42; -41,-40],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-40,-40; 29,-40],   style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Polygon(points=[29,-40; 21,-38; 21,-42; 29,-40],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Text(
            extent=[-26,-22; 19,-38],
            string="riseTime",
            style(color=9)),
          Text(
            extent=[78,-86; 102,-106],
            string="time",
            style(color=9)),
          Line(points=[-100,-80; 84,-80],   style(color=8)),
          Polygon(points=[100,-80; 84,-74; 84,-86; 100,-80],     style(color=8,
                fillColor=8)),
          Line(points=[-40,-40; -37.2,-25.3; -34.3,-12.1; -30.8,2.4; -27.3,15;
                -23.7,25.92; -19.5,37.18; -15.3,46.7; -10.3,56; -4.6,64.5; 1.7,
                72.1; 8.8,78.6; 17.3,84.3; 27.9,89.2; 30,90; 32.12,77.5; 34.95,
                62.7; 37.78,49.8; 40.606,38.45; 44.14,26.3; 47.68,16; 51.9,5.8;
                56.2,-2.6; 61.1,-10.5; 66.8,-17.4; 73.1,-23.3; 80.9,-28.5; 90.8,
                -32.8; 100,-35.4],                              style(color=0,
                thickness=2)),
          Polygon(points=[-80,80; -88,58; -72,58; -80,80],     style(color=8,
                fillColor=8)),
          Line(points=[-80,58; -80,-90],   style(color=8)),
          Text(
            extent=[-82,81; -41,61],
            style(color=9),
            string="f"),
          Text(
            extent=[-78,-53; -46,-66],
            string="offset",
            style(color=9)),
          Polygon(points=[-40,-80; -43,-70; -38,-70; -40,-80; -40,-80],
              style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-40,-39; -40,-70],   style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Polygon(points=[-40,-40; -42,-50; -37,-50; -40,-40],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-39,-40; -80,-40],   style(color=0, thickness=2)),
          Text(
            extent=[-59,-81; -13,-99],
            string="startTime",
            style(color=9)),
          Polygon(points=[-41,-40; -32,-38; -32,-42; -41,-40],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-40,-40; 29,-40],   style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Polygon(points=[29,-40; 21,-38; 21,-42; 29,-40],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Text(
            extent=[-26,-22; 19,-38],
            string="riseTime",
            style(color=9)),
          Text(
            extent=[78,-86; 102,-106],
            string="time",
            style(color=9)),
          Line(points=[30,90; 30,-44],    style(color=8, pattern=2))),
        Documentation(info="<html>
The step response flow source is an active OnePort element.  It inherits the effort and flow variables from the active OnePort.  The step response flow to be generated is modeled as a set of six parameters. <p>

The flow source has its causality stroke at the source. <p>

<hr>

<pre>
Potential variables:

 e:              Bondgraphic effort variable

 f:              Bondgraphic flow variable, normalized positive for flows out of the model


Parameters:

 f0:             Height of step response after infinite time (default value = 1)

 riseTime:       Rise time of step response to be generated (default value = 0.5)

 riseTimeConst:  Rise time constant of step response to be generated (default value = 0.1)

 fallTimeConst:  Fall time constant of step response to be generated (default value = riseTimeConst)

 offset:         Offset of step response to be generated (default value = 0)

 startTime:      Start time of step response to be generated (default value = 0)
</pre> <p>

<hr> <p>

Equations: <p>

<b>f_riseTime = f0*(1 - exp(-riseTime/riseTimeConst))</b> <br>
<b>f = offset + (if (time &lt; startTime) then 0 else if (time &lt; (startTime
     + riseTime)) then f0*(1 - exp(-(time - startTime)/riseTimeConst)) else
          f_riseTime*exp(-(time - startTime - riseTime)/fallTimeConst));</b>
</html>
"));
      Modelica.Blocks.Sources.Exponentials Exponentials1(
        riseTime=riseTime,
        riseTimeConst=riseTimeConst,
        fallTimeConst=fallTimeConst,
        offset=offset,
        startTime=startTime,
        outMax=f0) annotation (extent=[68,66; 90,88]);
    equation
      f = Exponentials1.y;
    end expSf;

    model pulseSf "Pulse flow source of the Modelica bond graph library"
      extends Interfaces.ActiveOnePort;
      parameter Real f0=1 "Amplitude of saw tooth";
      parameter Modelica.SIunits.Time width(
        final min=Modelica.Constants.small,
        final max=100) = 50 "Width of saw tooth in % of period";
      parameter Modelica.SIunits.Time period(final min=Modelica.Constants.small) = 1
        "Time for one period";
      parameter Real offset=0 "Offset of saw tooth";
      parameter Modelica.SIunits.Time startTime=0 "Start time of saw tooth";
      annotation (
        Icon(
          Text(
            extent=[-62,58; 50,-60],
            style(color=0),
            string="Sf"),
          Text(extent=[-70,100; 70,60],  string="%name"),
             Line(points=[-76,-80; -46,-80; -46,60; -6,60; -6,-80; 34,-80; 35,
                60; 74,60],    style(color=8))),
        Diagram(
          Polygon(points=[-80,90; -88,68; -72,68; -80,90],     style(color=8,
                fillColor=8)),
          Line(points=[-80,68; -80,-80],   style(color=8)),
          Line(points=[-90,-70; 82,-70],   style(color=8)),
          Polygon(points=[90,-70; 68,-62; 68,-78; 90,-70],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[-34,1; -37,-12; -30,-12; -34,1],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-34,-1; -34,-70],   style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Polygon(points=[-33,-70; -36,-57; -30,-57; -33,-70; -33,-70],
              style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Text(
            extent=[-78,-24; -35,-36],
            string="offset",
            style(color=9)),
          Text(
            extent=[-39,-69; 7,-87],
            string="startTime",
            style(color=9)),
          Text(
            extent=[70,-80; 94,-100],
            string="time",
            style(color=9)),
          Line(points=[-18,0; -18,-70],   style(color=8, pattern=2)),
          Line(points=[-80,0; -20,0; -20,50; 20,50; 20,0; 40,0; 40,50; 80,50],
                       style(color=0, thickness=2)),
          Line(points=[-19,88; -19,49],   style(color=8, pattern=2)),
          Line(points=[21,74; 21,50],   style(color=9, pattern=2)),
          Line(points=[41,88; 41,50],   style(color=8, pattern=2)),
          Line(points=[-19,83; 42,83],   style(color=8)),
          Line(points=[-19,69; 21,69],   style(color=8)),
          Text(
            extent=[-9,97; 37,85],
            string="period",
            style(color=9)),
          Text(
            extent=[-18,81; 21,69],
            string="width",
            style(color=9)),
          Line(points=[-52,50; -19,50],   style(color=8, pattern=2)),
          Line(points=[-34,50; -34,1],   style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Text(
            extent=[-66,32; -25,18],
            style(color=9),
            string="f0"),
          Polygon(points=[-34,49; -37,36; -30,36; -34,49],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[-34,1; -37,14; -31,14; -34,1; -34,1],      style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[80,50; 80,0; 92,0],     style(color=0, thickness=2)),
          Polygon(points=[-19,69; -10,71; -10,67; -19,69],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[21,69; 13,71; 13,67; 21,69],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[-19,83; -10,85; -10,81; -19,83],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[41,83; 33,85; 33,81; 41,83],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Text(
            extent=[-74,91; -56,72],
            style(color=9),
            string="f")),
        Documentation(info="<html>
The pulse flow source is an active OnePort element.  It inherits the effort and flow variables from the active OnePort.  The pulse flow to be generated is modeled as a set of five parameters. <p>

The flow source has its causality stroke at the source. <p>

<hr>

<pre>
Potential variables:

 e:          Bondgraphic effort variable

 f:          Bondgraphic flow variable, normalized positive for flows out of the model


Parameters:

 f0:         Amplitude of pulse to be generated (default value = 1)

 width:      Percentage of period in on state (default value = 50)

 period:     Length of period of pulse to be generated (default value = 1)

 offset:     Offset of pulse to be generated (default value = 0)

 startTime:  Start time of pulse to be generated (default value = 0)
</pre> <p>

<hr> <p>

Equations: <p>

<b>when sample(startTime, period) then</b> <br>
<b>T0 = time</b> <br>
<b>end when</b> <br>
<b>f = offset + (if time &lt; startTime or time &gt;= T0 + T_width then 0 else f0)</b>
</html>
"));
      Modelica.Blocks.Sources.Pulse Pulse1(
        width=width,
        period=period,
        offset=offset,
        startTime=startTime,
        amplitude=f0) annotation (extent=[68,70; 90,92]);
    equation
      f = Pulse1.y;
    end pulseSf;

    model sawSf "Saw tooth flow source of the Modelica bond graph library"
      extends Interfaces.ActiveOnePort;
      parameter Real f0=1 "Amplitude of saw tooth";
      parameter Modelica.SIunits.Time period(final min=Modelica.Constants.small) = 1
        "Time for one period";
      parameter Real offset=0 "Offset of saw tooth";
      parameter Modelica.SIunits.Time startTime=0 "Start time of saw tooth";
      annotation (
        Icon(
          Text(
            extent=[-58,50; 38,-70],
            style(color=0),
            string="Sf"),
          Text(extent=[-72,104; 68,64],  string="%name"),
             Line(points=[-82,-79; -62,-79; -2,62; -2,-79; 58,62; 58,-79],
              style(color=8))),
        Diagram(
          Polygon(points=[-80,80; -88,58; -72,58; -80,80],     style(color=8,
                fillColor=8)),
          Line(points=[-80,58; -80,-90],   style(color=8)),
          Line(points=[-90,-80; 82,-80],   style(color=8)),
          Polygon(points=[90,-80; 68,-72; 68,-88; 90,-80],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[-34,-29; -37,-42; -30,-42; -34,-29],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-34,-30; -34,-80],   style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Polygon(points=[-34,-80; -37,-67; -31,-67; -34,-80; -34,-80],
              style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Text(
            extent=[-78,-34; -35,-46],
            string="offset",
            style(color=9)),
          Text(
            extent=[-31,-79; 15,-97],
            string="startTime",
            style(color=9)),
          Text(
            extent=[74,-84; 98,-104],
            string="time",
            style(color=9)),
          Line(points=[-10,-30; -10,-80],   style(color=8, pattern=2)),
          Line(points=[-10,78; -10,-30],   style(color=8, pattern=2)),
          Line(points=[30,78; 30,49],   style(color=8, pattern=2)),
          Line(points=[-10,73; 30,73],   style(color=8)),
          Text(
            extent=[-12,87; 34,75],
            string="period",
            style(color=9)),
          Line(points=[-44,50; 30,50],   style(color=8, pattern=2)),
          Line(points=[-34,37; -34,-17],  style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Text(
            extent=[-66,18; -25,4],
            style(color=9),
            string="f0"),
          Polygon(points=[-34,50; -37,37; -30,37; -34,50],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[-34,-30; -37,-17; -31,-17; -34,-30; -34,-30],    style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[-10,73; -1,75; -1,71; -10,73],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[30,73; 22,75; 22,71; 30,73],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-80,-30; -10,-30; 30,50; 30,-30; 72,50; 72,-30],
              style(color=0, thickness=2)),
          Text(
            extent=[-75,82; -56,62],
            style(color=9),
            string="f")),
        Documentation(info="<html>
The saw tooth flow source is an active OnePort element.  It inherits the effort and flow variables from the active OnePort.  The saw tooth flow to be generated is modeled as a set of four parameters. <p>

The flow source has its causality stroke at the source. <p>

<hr>

<pre>
Potential variables:

 e:          Bondgraphic effort variable

 f:          Bondgraphic flow variable, normalized positive for flows out of the model


Parameters:

 f0:         Amplitude of saw tooth to be generated (default value = 1)

 period:     Length of period of saw tooth to be generated (default value = 1)

 offset:     Offset of saw tooth to be generated (default value = 0)

 startTime:  Start time of saw tooth to be generated (default value = 0)
</pre> <p>

<hr> <p>

Equations: <p>

<b>when sample(startTime, period) then</b> <br>
<b>T0 = time</b> <br>
<b>end when</b> <br>
<b>f = offset + (if time &lt; startTime then 0 else (f0/period)*(time - T0))</b>
</html>
"));
      Modelica.Blocks.Sources.SawTooth SawTooth1(
        period=period,
        offset=offset,
        startTime=startTime,
        amplitude=f0) annotation (extent=[68,66; 90,88]);
    equation
      f = SawTooth1.y;
    end sawSf;

    model trapSf "Trapezoidal flow source of the Modelica bond graph library"
      extends Interfaces.ActiveOnePort;
      parameter Real f0=1 "Amplitude of trapezoidal function";
      parameter Modelica.SIunits.Time rising(final min=0) = 0
        "Rising duration of trapezoid";
      parameter Modelica.SIunits.Time width(final min=0) = 0.5
        "Width duration of trapezoid";
      parameter Modelica.SIunits.Time falling(final min=0) = 0
        "Falling duration of trapezoid";
      parameter Modelica.SIunits.Time period(final min=Modelica.Constants.small) = 1
        "Time for one period";
      parameter Integer nperiod=-1
        "Number of periods (< 0 means infinite number of periods)";
      parameter Real offset=0 "Offset of trapezoidal function";
      parameter Modelica.SIunits.Time startTime=0
        "Start time of trapezoidal function";
      annotation (
        Icon(
          Text(
            extent=[-60,46; 36,-68],
            style(color=0),
            string="Sf"),
          Text(extent=[-66,104; 74,64],  string="%name"),
             Line(points=[-81,-78; -60,-78; -30,62; 1,62; 30,-78; 51,-78; 80,62],
                         style(color=8))),
        Diagram(
          Polygon(points=[-80,80; -88,58; -72,58; -80,80],     style(color=8,
                fillColor=8)),
          Line(points=[-80,58; -80,-90],   style(color=8)),
          Line(points=[-90,-80; 82,-80],   style(color=8)),
          Polygon(points=[90,-80; 68,-72; 68,-88; 90,-80],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[-45,-40; -47,-51; -43,-51; -45,-40],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-45,-41; -45,-80],   style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Polygon(points=[-45,-80; -47,-70; -43,-70; -45,-80; -45,-80],
              style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Text(
            extent=[-86,-53; -43,-65],
            string="offset",
            style(color=9)),
          Text(
            extent=[-47,-79; -1,-97],
            string="startTime",
            style(color=9)),
          Text(
            extent=[76,-84; 100,-104],
            string="time",
            style(color=9)),
          Line(points=[-30,71; -30,-80],   style(color=8, pattern=2)),
          Line(points=[-10,49; -10,30],   style(color=8, pattern=2)),
          Line(points=[20,49; 20,29],   style(color=9, pattern=2)),
          Line(points=[40,49; 40,-40],   style(color=8, pattern=2)),
          Line(points=[-29,46; 40,46],   style(color=8)),
          Text(
            extent=[-8,60; 21,50],
            string="width",
            style(color=9)),
          Line(points=[-42,30; -10,30],   style(color=8, pattern=2)),
          Line(points=[-39,30; -39,-29],   style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Text(
            extent=[-67,6; -30,-8],
            style(color=9),
            string="f0"),
          Polygon(points=[-29,46; -22,48; -22,44; -29,46],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[-10,46; -17,48; -17,44; -10,46],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-80,-40; -30,-40; -9,36; 21,36; 40,-40; 60,-40; 80,36;
                100,36],      style(color=0, thickness=2)),
          Polygon(points=[-39,30; -41,19; -37,19; -39,30],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[-39,-39; -41,-29; -37,-29; -39,-39; -39,-39],
              style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[60,71; 60,-40],   style(color=8, pattern=2)),
          Polygon(points=[39,46; 32,48; 32,44; 39,46],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[20,46; 27,48; 27,44; 20,46],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[20,46; 13,48; 13,44; 20,46],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[-12,46; -5,48; -5,44; -12,46],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Text(
            extent=[-34,60; -5,50],
            string="rising",
            style(color=9)),
          Text(
            extent=[16,60; 45,50],
            string="falling",
            style(color=9)),
          Text(
            extent=[-77,83; -48,60],
            style(color=9),
            string="f"),
          Line(points=[-20,66; 61,66],   style(color=8)),
          Polygon(points=[-29,66; -20,68; -20,64; -29,66],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[61,66; 53,68; 53,64; 61,66],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Text(
            extent=[-2,76; 25,67],
            string="period",
            style(color=9))),
        Documentation(info="<html>
The trapezoidal flow source is an active OnePort element.  It inherits the effort and flow variables from the active OnePort.  The trapezoidal flow to be generated is modeled as a set of eight parameters. <p>

The flow source has its causality stroke at the source. <p>

<hr>

<pre>
Potential variables:

 e:          Bondgraphic effort variable

 f:          Bondgraphic flow variable, normalized positive for flows out of the model


Parameters:

 f0:         Amplitude of trapezoid to be generated (default value = 1)

 rising:     Length of rising time of trapezoid to be generated (default value = 0)

 width :     Length of on state of trapezoid to be generated (default value = 0.5)

 falling:    Length of falling time of trapezoid to be generated (default value = 0)

 period:     Length of period of trapezoid to be generated (default value = 1)

 nperiod:    Number of periods of trapezoid to be generated
             an infinite number of periods is being generated if nperiod &lt; 0 (default value = -1)

 offset:     Offset of trapezoid to be generated (default value = 0)

 startTime:  Start time of trapezoid to be generated (default value = 0)
</pre> <p>

<hr> <p>

Equations: <p>

<b>when pre(counter2) &lt;&gt; 0 and sample(startTime, period) then</b> <br>
<b>T0 = time</b> <br>
<b>counter2 = pre(counter)</b> <br>
<b>counter = pre(counter) - (if pre(counter) &gt; 0 then 1 else 0)</b> <br>
<b>end when</b> <br>
<b>f = offset + (if (time &lt; startTime or counter2 == 0 or time &gt;= T0 +
    T_falling) then 0 else if (time &lt; T0 + T_rising) then (time - T0)*f0/T_rising
    else if (time &lt; T0 + T_width) then f0 else
    (T0 + T_falling - time)*f0/(T_falling - T_width))</b>
</html>
"));
      Modelica.Blocks.Sources.Trapezoid Trapezoid1(
        rising=rising,
        width=width,
        falling=falling,
        period=period,
        nperiod=nperiod,
        offset=offset,
        startTime=startTime,
        amplitude=f0) annotation (extent=[68,66; 90,88]);
    equation
      f = Trapezoid1.y;
    end trapSf;

    model tableSf "Tabular flow source of the Modelica bond graph library"
      extends Interfaces.ActiveOnePort;
      parameter Real table[:, :]=[0, 0; 1, 1; 2, 4]
        "Table matrix (time = first column, voltage = second column)";
      parameter Real offset=0 "Offset of time function";
      parameter Modelica.SIunits.Time startTime=0 "Start time of time function";
      annotation (
        Icon(
          Text(
            extent=[-54,66; 46,-68],
            style(color=0),
            string="Sf"),
          Text(extent=[-66,102; 74,62],  string="%name"),
             Line(points=[-64,-64; -64,56; 36,56; 36,-64; -64,-64; -64,-34; 36,
                -34; 36,-4; -64,-4; -64,26; 36,26; 36,56; -14,56; -14,-65],
                     style(color=8))),
        Diagram(
          Polygon(points=[-80,80; -88,58; -72,58; -80,80],     style(color=8,
                fillColor=8)),
          Line(points=[-80,58; -80,-90],   style(color=8)),
          Line(points=[-90,-80; 82,-80],   style(color=8)),
          Polygon(points=[90,-80; 68,-72; 68,-88; 90,-80],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Rectangle(extent=[-20,80; 30,-40],   style(
              color=7,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-20,-40; -20,80; 80,80; 80,-40; -20,-40; -20,-10; 80,-10;
                80,20; -20,20; -20,50; 80,50; 80,80; 30,80; 30,-41],
              style(color=0)),
          Text(
            extent=[-77,-52; -38,-68],
            string="offset",
            style(color=9)),
          Polygon(points=[-31,-40; -33,-50; -28,-50; -31,-40],     style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Polygon(points=[-30,-80; -33,-70; -28,-70; -30,-80; -30,-80],
              style(
              color=8,
              fillColor=8,
              fillPattern=1)),
          Line(points=[-31,-41; -31,-80],   style(
              color=8,
              pattern=1,
              thickness=1,
              arrow=0)),
          Line(points=[-20,-42; -20,-80],   style(color=8, pattern=2)),
          Text(
            extent=[-38,-80; 8,-98],
            string="startTime",
            style(color=9)),
          Line(points=[-20,-40; -80,-40],   style(color=8, pattern=2)),
          Text(
            extent=[72,-87; 97,-99],
            string="time",
            style(color=9)),
          Text(
            extent=[-15,73; 24,58],
            string="time",
            style(color=0)),
          Text(
            extent=[33,73; 76,57],
            string="v",
            style(color=0)),
          Text(
            extent=[-77,80; -54,58],
            style(color=9),
            string="f")),
        Documentation(info="<html>
The tabular flow source is an active OnePort element.  It inherits the effort and flow variables from the active OnePort.  The tabular flow to be generated is modeled as a two-dimensional parameter table plus a set of two scalar parameters. <p>

The flow source has its causality stroke at the source. <p>

<hr>

<pre>
Potential variables:

 e:          Bondgraphic effort variable

 f:          Bondgraphic flow variable, normalized positive for flows out of the model


Parameters:

 table:      Two-dimensional table with supporting values (default value = [0, 0; 1, 1; 2, 4])

 offset:     Offset of time function to be generated (default value = 0)

 startTime:  Start time of time function to be generated (default value = 0)
</pre> <p>

<hr> <p>

The model generates a flow source by <b>linear interpolation</b> in
a table. The time points and flow values are stored in a matrix
<b>table[i,j]</b>, where the first column table[:,1] contains the
time points, and the second column contains the flow to be interpolated.
The table interpolation has the following proporties: <p>

<ul>
<li>The time points need to be <b>monotonically increasing</b>.
<li><b>Discontinuities</b> are allowed, by providing the same
    time point twice in the table.
<li>Values <b>outside</b> of the table range, are computed by
    <b>extrapolation</b> through the last or first two points of the
    table.
<li>If the table has only <b>one row</b>, no interpolation is performed and
    the voltage value is just returned independantly of the
    actual time instant, i.e., this is a constant voltage source.
<li>Via parameters <b>startTime</b> and <b>offset</b> the curve defined
    by the table can be shifted both in time and in the voltage.
<li>The table is implemented in a numerically sound way by
    generating <b>time events</b> at interval boundaries,
    in order to not integrate over a discontinuous or not differentiable
    points.
</ul> <p>

<hr> <p>

<b>Example:</b> <p>

<pre>
   table = [0  0
            1  0
            1  1
            2  4
            3  9
            4 16]
If, e.g., time = 1.0, the flow f =  0.0 (before event), 1.0 (after event)
    e.g., time = 1.5, the flow f =  2.5,
    e.g., time = 2.0, the flow f =  4.0,
    e.g., time = 5.0, the flow f = 23.0 (i.e. extrapolation).
</pre>
</html>
"));
      Modelica.Blocks.Sources.TimeTable TimeTable1(
        table=table,
        offset=offset,
        startTime=startTime) annotation (extent=[68,68; 90,90]);
    equation
      f = TimeTable1.y;
    end tableSf;

    model mSf2
      extends Interfaces.ModTwoPort;
      annotation (
        Coordsys(
          extent=[-100, -100; 100, 100],
          grid=[2, 2],
          component=[20, 20]),
        Diagram(Text(
            extent=[-60, 80; 60, -80],
            string="mSf",
            style(color=0))),
        Icon(Text(
            extent=[-60, 80; 60, -80],
            string="mSf",
            style(color=0)), Text(extent=[-70, 90; 70, 50], string="%name")),
        Window(
          x=0.45,
          y=0.01,
          width=0.35,
          height=0.63),
        Documentation(info="<html>
The modulated internal flow source is a modulated directed TwoPort element.  It inherits the effort and flow variables from the modulated directed TwoPort. <p>

The internal source takes power out of the system to the left, and feeds it into the system to the right.  The modulation signal is used to set the <b>flow</b> of the inflow.  The <b>flow</b> of the outflow is computed such that power in equals power out. <p>

The internal flow source is primarily used in the computation of convective flows, when entropy flow has to accompany mass flow. <p>

The modulated internal flow source has two causality strokes. <p>

<hr>

<pre>
Potential variables:

 e1:  Bondgraphic effort variable of inflow

 f1:  Bondgraphic flow variable of inflow, normalized positive for flows into the model

 e2:  Bondgraphic effort variable of outflow

 f2:  Bondgraphic flow variable of outflow, normalized positive for flows out of the model

 s:   Modulation input signal
</pre> <p>

<hr> <p>

Equations: <p>

<b>f1 = s</b> <br>
<b>f2 = f1*(e1/e2)</b>
</html>
"));
    equation
      f1 = s;
      e1*f1 = e2*f2;
    end mSf2;

    model mSf_int "Internal modulated flow source of the bond graph library"
      extends Interfaces.ModTwoPort;
      Real f0 "Bondgraphic Flow Source";
      annotation (
        Icon(   Text(
            extent=[-62,80; 58,-80],
            style(color=0),
            string="mSf"),   Text(extent=[-70,90; 70,50],   string="%name")),
        Diagram(Text(
            extent=[-62,80; 58,-80],
            style(color=0),
            string="mSf")),
        Documentation(info="<html>
The modulated internal flow source is a modulated directed TwoPort element.  It inherits the effort and flow variables from the modulated directed TwoPort.  The modulated flow to be generated is modeled as a real-valued variable. <p>

The modulated internal flow source is a modulated flow source on its secondary side.  The primary side provides the power needed by the flow source.  This element can e.g. be used to model a DC current supply together with the power cable that draws the required energy from the grid. <p>

The flow source has its secondary causality stroke away from the source.  Its primary causality stroke is free. <p>

<hr>

<pre>
Potential variables:

 e1:  Bondgraphic effort variable of inflow

 f1:  Bondgraphic flow variable of inflow, normalized positive for flows into the model

 e2:  Bondgraphic effort variable of outflow

 f2:  Bondgraphic flow variable of outflow, normalized positive for flows out of the model

 f0:  Flow to be generated (modulated input signal)
</pre> <p>

<hr> <p>

Equations: <p>

<b>f2 = f0</b> <br>
<b>e1*f1 = e2*f2</b>
</html>
"));
    equation
      f0 = s;
      f2 = f0;
      e1*f1 = e2*f2;
    end mSf_int;

    model mSP "Source of power"
      extends Interfaces.ModActiveOnePort;
      Real P0 "Bondgraphic Power Source";
      annotation (
        Diagram(Text(
            extent=[-62, 80; 58, -80],
            style(color=0),
            string="mSP")),
        Icon(Text(
            extent=[-62, 80; 58, -80],
            style(color=0),
            string="mSP"), Text(extent=[-70, 92; 70, 52], string="%name")),
        Documentation(info="<html>
The modulated power source is a modulated active OnePort element.  It inherits the effort and flow variables from the modulated active OnePort.  The modulated power to be generated is modeled as a real-valued variable. <p>

Modulated sources are commonly used to model actuators of control systems.  They are the most common and convenient way for converting signal variables to bondgraphic variables, thereby allowing individual blocks of a block diagram, e.g. describing a control system, to be modeled internally using bond graphs. <p>

The power source has free causality. <p>

<hr>

<pre>
Potential variables:

 e:   Bondgraphic effort variable

 f:   Bondgraphic flow variable, normalized positive for flows out of the model

 P0:  Power to be generated (modulated input signal)
</pre> <p>

<hr> <p>

Equations: <p>

<b>e*f = P0</b>
</html>
"));
    equation
      P0 = s;
      P0 = e*f;
    end mSP;
  end Sources;

    package Spice "Spice-style semiconductor models"
      annotation (preferedView="info",
        Icon(
          Rectangle(extent=[-100,-100; 80,50],   style(fillColor=30,
                fillPattern=
                  1)),
          Polygon(points=[-100,50; -80,70; 100,70; 80,50; -100,50],      style(
                fillColor=30, fillPattern=1)),
          Polygon(points=[100,70; 100,-80; 80,-100; 80,50; 100,70],      style(
                fillColor=30, fillPattern=1)),
          Text(
            extent=[-85,35; 65,-85],
            string="Library",
            style(color=3)),
          Text(
            extent=[-120,122; 120,73],
            string="%name",
            style(color=1))), Documentation(info="<html>
This package contains Spice-style models of electronic circuit elements built in bond graph technology.  The library implements facets of a dialect of <b>HSpice</b>, called <b>BBSpice</b>, further developed from HSpice at <i>Burr Brown, Inc.</i> in Tucson Arizona, a company that has meanwhile been acquired by <i>Texas Instruments, Inc.</i> <p>

<hr> <p>

<b>References:</b> <p>

<ol>
<li> Cellier, F.E. (1991), <i>Continuous System Modeling</i>, Springer-Verlag, New York, Chapter 6.
<li> Massobrio, G. and P. Antognetti (1993), <i>Semiconductor Device Modeling with Spice</i>,
     2<sup>nd</sup> edition, McGraw Hill, New York.
</ol>
</html>
"));
      package Interfaces "Interfaces for Spice-style electronic circuit models"
      annotation (preferedView="info",
          Icon(
            Rectangle(extent=[-100,-100; 80,50],   style(
                color=58,
                fillColor=30,
                fillPattern=1)),
            Polygon(points=[-100,50; -80,70; 100,70; 80,50; -100,50],      style(
                color=58,
                fillColor=30,
                fillPattern=1)),
            Polygon(points=[100,70; 100,-80; 80,-100; 80,50; 100,70],      style(
                color=58,
                fillColor=30,
                fillPattern=1)),
            Text(
              extent=[-85,35; 65,-85],
              string="Library",
              style(color=58)),
            Text(
              extent=[-120,122; 120,73],
              string="%name",
              style(color=1))), Documentation(info="<html>
This package contains some common interfaces for Spice-style electronic circuit models.
</html>
"));
        partial model SpiceTwoPort
        "Special twoPort element used by Spice models"
          extends BondLib.Interfaces.TwoPort;
          parameter Modelica.SIunits.Temperature Tnom=300.15
          "Reference temperature";
          Modelica.SIunits.Temperature DTemp
          "Difference between circuit temperature and reference temperature";
          Real DTempSq "Square of DTemp";
        annotation (Icon(
              Text(
                extent=[24,-4; 92,-34],
                string="Spice",
                style(color=0, rgbcolor={0,0,0})),
              Text(extent=[-40,86; 40,56],   string="%name")), Diagram(
              Text(
                extent=[24,-4; 92,-34],
                string="Spice",
                style(color=0, rgbcolor={0,0,0}))));
        equation
          DTemp = e2 - Tnom;
          DTempSq = DTemp*DTemp;
        end SpiceTwoPort;
      end Interfaces;

      model RS "Spice-style resistor model"
        extends Interfaces.SpiceTwoPort;
        parameter Modelica.SIunits.Resistance R=1
        "Resistance at reference temperature";
        parameter Real TR1(unit="1/K") = 0 "Linear temperature coefficient";
        parameter Real TR2(unit="1/K2") = 0 "Quadratic temperature coefficient";
        parameter Real Area=1 "Relative area occupied by resistor";
        Modelica.SIunits.Resistance Rval
        "Resistance value at circuit temperature";
        annotation (
          Icon(
            Text(
              extent=[-82,76; 28,-80],
              string="RS",
              style(color=0)),
            Text(extent=[-72,-58; 66,-86],   string="R=%R")),
          Diagram(
            Text(
              extent=[-82,76; 28,-80],
              string="RS",
              style(color=0))),
          Documentation(info="<html>
The Spice-style resistor has a reference resistance value computed at the reference temperature, <b>Tnom</b>, and for a relative area of <b>Area=1</b>.  It changes its value as a function of the circuit temperature and the relative area occupied by the resistor. <p>

<hr> <p>

<pre>
Parameters:

 R:     Reference resistance (default value = 1 Ohm)

 Tnom:  Reference temperature (default value = 300.15 <sup>o</sup>K)

 TR1:   Linear temperature variation coefficient (default value = 0 1/<sup>o</sup>K)

 TR2:   Quadratic temperature variation coefficient (default value = 0 1/<sup>o</sup>K<sup>2</sup>)

 Area:  Relative area occupied by resistor (default value = 1)
</pre> <p>

<hr> <p>

Equations: <p>

<b>DTemp = e2 - Tnom</b> <br>
<b>Rval = R*(1 + TR1*DTemp + TR2*DTemp*DTemp)/Area</b> <br>
<b>e1 = Rval*f1</b> <br>
<b>f2 = (e1/e2)*f1</b>
</html>
"));
      equation
        if R > 0 then
          Rval = R*(1 + TR1*DTemp + TR2*DTempSq)/Area;
         else
          Rval = 0;
        end if;
        e1 = Rval*f1;
        e1*f1 = e2*f2;
      end RS;

      model CS "Spice-style capacitor model"
        extends Interfaces.SpiceTwoPort;
        parameter Modelica.SIunits.Capacitance C=1
        "Capacitance at reference temperature";
        parameter Real TC1(unit="1/K") = 0 "Linear temperature coefficient";
        parameter Real TC2(unit="1/K2") = 0 "Quadratic temperature coefficient";
        parameter Real Area=1 "Relative area occupied by capacitor";
        Modelica.SIunits.Capacitance Cval
        "Capacitance value at circuit temperature";
        annotation (
          Icon(
            Text(
              extent=[-84,76; 26,-80],
              style(color=0),
              string="CS"),
            Text(extent=[-62,-58; 48,-86], string="C=%C")),
          Diagram(
            Text(
              extent=[-84,76; 26,-80],
              style(color=0),
              string="CS")),
          Documentation(info="<html>
The Spice-style capacitor has a reference capacitance value computed at the reference temperature, <b>Tnom</b>, and for a relative area of <b>Area=1</b>.  It changes its value as a function of the circuit temperature and the relative area occupied by the capacitor. <p>

<hr> <p>

<pre>
Parameters:

 C:     Reference capacitance (default value = 1 F)

 Tnom:  Reference temperature (default value = 300.15 <sup>o</sup>K)

 TC1:   Linear temperature variation coefficient (default value = 0 1/<sup>o</sup>K)

 TC2:   Quadratic temperature variation coefficient (default value = 0 1/<sup>o</sup>K<sup>2</sup>)

 Area:  Relative area occupied by capacitor (default value = 1)
</pre> <p>

<hr> <p>

Equations: <p>

<b>DTemp = e2 - Tnom</b> <br>
<b>Cval = C*(1 + TC1*DTemp + TC2*DTemp*DTemp)*Area</b> <br>
<b>q1 = Cval*e1</b> <br>
<b>der(q1) = f1</b> <br>
<b>f2 = 0</b>
</html>
"));
      equation
        if C > 0 then
          Cval = C*(1 + TC1*DTemp + TC2*DTempSq)*Area;
         else
          Cval = 0;
        end if;
        f1 = Cval*der(e1);
        f2 = 0;
      end CS;

      model LS "Spice-style inductor model"
        extends Interfaces.SpiceTwoPort;
        parameter Modelica.SIunits.Inductance L=1
        "Inductance at reference temperature";
        parameter Real TL1(unit="1/K") = 0 "Linear temperature coefficient";
        parameter Real TL2(unit="1/K2") = 0 "Quadratic temperature coefficient";
        Modelica.SIunits.Inductance Lval
        "Inductance value at circuit temperature";
        annotation (
          Icon(
            Text(
              extent=[-82,76; 28,-80],
              style(color=0),
              string="LS"),
            Text(extent=[-74,-58; 70,-88], string="L=%L")),
          Diagram(
            Text(
              extent=[-82,76; 28,-80],
              style(color=0),
              string="LS")),
          Documentation(info="<html>
The Spice-style inductor has a reference inductance value computed at the reference temperature, <b>Tnom</b>.  It changes its value as a function of the circuit temperature. <p>

<hr> <p>

<pre>
Parameters:

 L:     Reference inductance (default value = 1 H)

 Tnom:  Reference temperature (default value = 300.15 <sup>o</sup>K)

 TL1:   Linear temperature variation coefficient (default value = 0 1/<sup>o</sup>K)

 TL2:   Quadratic temperature variation coefficient (default value = 0 1/<sup>o</sup>K<sup>2</sup>)
</pre> <p>

<hr> <p>

Equations: <p>

<b>DTemp = e2 - Tnom</b> <br>
<b>Lval = L*(1 + TL1*DTemp + TL2*DTemp*DTemp)</b> <br>
<b>p1 = Lval*f1</b> <br>
<b>der(p1) = e1</b> <br>
<b>f2 = 0</b>
</html>
"));
      equation
        if L > 0 then
          Lval = L*(1 + TL1*DTemp + TL2*DTempSq);
         else
          Lval = 0;
        end if;
        e1 = Lval*der(f1);
        f2 = 0;
      end LS;

        model DS
      "Spice-style diode model used inside the bipolar transistor models"
          extends BondLib.Interfaces.TwoPort;
          constant Real inf=Modelica.Constants.inf;
          constant Modelica.SIunits.Entropy k=Modelica.Constants.k
        "Boltzmann's constant";
          constant Modelica.SIunits.ElectricCharge q=1.6021892e-19
        "Electron charge";

          /* General parameters */
          parameter Modelica.SIunits.Temperature Tnom=300.15
        "Reference temperature";
          parameter Integer Level(min=1,max=3) = 1
        "DIODE modeling level (check documentation window for details)";
          parameter Real Area(min=Modelica.Constants.small) = 1
        "Relative device area"
            annotation(Dialog(enable=Level == 1));

          /* Process parameters */
          parameter Modelica.SIunits.Length TOX(min=Modelica.Constants.small) = 1e-7
        "Thin oxide thickness"
            annotation(Dialog(group="Process parameters"));

          /* Basic Electrical parameters */
          parameter Modelica.SIunits.Voltage PB(min=0) = 0.8
        "Built-in junction potential at reference temperature"
            annotation(Dialog(group="Electrical parameters", enable=(Level==1 or Level==3)));
          parameter Modelica.SIunits.Voltage BV(min=0) = 0
        "Reverse breakdown voltage at reference temperature"
            annotation(Dialog(group="Electrical parameters", enable=(Level==1 or Level==3)));
          parameter Modelica.SIunits.Current IBV(min=0) = 0
        "Reverse breakdown current"
            annotation(Dialog(group="Electrical parameters", enable=(Level==1 or Level==3)));
          parameter Modelica.SIunits.Voltage PHP(min=0) = 0.8
        "Built-in perimeter potential at reference temperature"
            annotation(Dialog(group="Electrical parameters", enable=(Level==1 or Level==3)));
          parameter Real N=1 "Current emission coefficient"
            annotation(Dialog(group="Electrical parameters", enable=(Level==1 or Level==3)));
          parameter Modelica.SIunits.Current IS(min=Modelica.Constants.small) = 1e-14
        "Saturation current at reference temperature"
            annotation(Dialog(group="Electrical parameters", enable=Level == 1));
          parameter Modelica.SIunits.Current ISW(min=0) = 0
        "Saturation current of perimeter at reference temperature"
            annotation(Dialog(group="Electrical parameters", enable=Level == 1));
          parameter Real PJ(min=0) = 1 "Relative perimeter width"
            annotation(Dialog(group="Electrical parameters", enable=Level == 1));

          /* High current beta degradation effect parameters */
          parameter Modelica.SIunits.Current IKF(min=Modelica.Constants.small) = inf
        "Forward knee current"
            annotation(Dialog(group="High current beta degradation effect", enable=(Level==1 or Level==3)));
          parameter Modelica.SIunits.Current IKR(min=Modelica.Constants.small) = inf
        "Reverse knee current"
            annotation(Dialog(group="High current beta degradation effect", enable=(Level==1 or Level==3)));

          /* Parasitic resistor parameters */
          parameter Modelica.SIunits.Resistance RS(min=0) = 0
        "Resistance at reference temperature"
            annotation(Dialog(group="Parasitic resistor parameters"));

          /* Junction capacitance parameters */
          parameter Modelica.SIunits.Capacitance CJ(min=0) = 0.5e-12
        "Zero-bias bulk capacitance at reference temperature"
            annotation(Dialog(group="Junction capacitance parameters", enable=Level == 1));
          parameter Real MJ(min=0.1,max=0.9) = 0.33
        "Bulk junction grading coefficient"
            annotation(Dialog(group="Junction capacitance parameters", enable=(Level==1 or Level==3)));
          parameter Modelica.SIunits.Capacitance CJSW(min=0) = 0.5e-12
        "Zero-bias perimeter capacitance at reference temperature"
            annotation(Dialog(group="Junction capacitance parameters", enable=Level == 1));
          parameter Real MJSW(min=0.1,max=0.9) = 0.33
        "Perimeter capacitance grading coefficient"
            annotation(Dialog(group="Junction capacitance parameters", enable=(Level==1 or Level==3)));
          parameter Real FC(min=0,max=0.95) = 0.5
        "Forward-bias depletion capacitance coefficient"
            annotation(Dialog(group="Junction capacitance parameters", enable=(Level==1 or Level==3)));

          /* Transit time parameters */
          parameter Modelica.SIunits.Time TT(min=0) = 0
        "Transit time at reference temperature"
            annotation(Dialog(group="Transit time parameters", enable=(Level==1 or Level==3)));

          /* Temperature compensation parameters */
          parameter Real TRS(unit="1/K") = 0
        "Linear temperature coefficient of resistance"
            annotation(Dialog(group="Temperature compensation parameters"));
          parameter Real TCV(unit="V/K")=0
        "Linear temperature coefficient of breakdown voltage"
            annotation(Dialog(group="Temperature compensation parameters", enable=(Level==1 or Level==3)));
          parameter Real TTT1(unit="1/K") = 0
        "Linear temperature coefficient of transit time"
            annotation(Dialog(group="Temperature compensation parameters", enable=(Level==1 or Level==3)));
          parameter Real TTT2(unit="1/K2") = 0
        "Quadratic temperature coefficient of transit time"
            annotation(Dialog(group="Temperature compensation parameters", enable=(Level==1 or Level==3)));
          parameter Real XTI(min=0) = 3
        "Saturation current temperature exponent"
            annotation(Dialog(group="Temperature compensation parameters", enable=(Level==1 or Level==3)));
          parameter Modelica.SIunits.Voltage EG(min=0.1) = 1.11
        "Energy gap at 0K (EG=1.11: Si, EG=0.73: Schottky barrier diode, EG=0.71: Ge)"
            annotation(Dialog(group="Temperature compensation parameters", enable=(Level==1 or Level==3)));

          /* Level=2 General parameters */
          parameter Integer M(min=1) = 1 "Diode multiplier"
            annotation(Dialog(tab="Level=2", enable=Level>1));

          /* Level=2 Geometric parameters */
          parameter Modelica.SIunits.Length L(min=0) = 0 "Diode length"
            annotation(Dialog(tab="Level=2", group="Geometric parameters", enable=Level>1));
          parameter Modelica.SIunits.Length W(min=0) = 0 "Diode width"
            annotation(Dialog(tab="Level=2", group="Geometric parameters", enable=Level>1));

          /* Level=2 Electrical parameters */
          parameter Real JF(min=0,unit="A/V2") = 0
        "Forward Fowler-Nordheim current coefficient"
            annotation(Dialog(tab="Level=2", group="Electrical parameters", enable=Level == 2));
          parameter Real JR(min=0,unit="A/V2") = 0
        "Reverse Fowler-Nordheim current coefficient"
            annotation(Dialog(tab="Level=2", group="Electrical parameters", enable=Level == 2));
          parameter Modelica.SIunits.ElectricFieldStrength EF(min=0) = 0
        "Forward Fowler-Nordheim critical electrical field"
            annotation(Dialog(tab="Level=2", group="Electrical parameters", enable=Level == 2));
          parameter Modelica.SIunits.ElectricFieldStrength ER(min=0) = 0
        "Reverse Fowler-Nordheim critical electrical field"
            annotation(Dialog(tab="Level=2", group="Electrical parameters", enable=Level == 2));

          /* Level=3 Geometric parameters */
          parameter Modelica.SIunits.Length LP(min=0) = 0 "Polysilicon length"
            annotation(Dialog(tab="Level=3", group="Geometric parameters", enable=Level == 3));
          parameter Modelica.SIunits.Length WP(min=0) = 0 "Polysilicon width"
            annotation(Dialog(tab="Level=3", group="Geometric parameters", enable=Level == 3));
          parameter Modelica.SIunits.Length XP(min=0) = 0
        "Widening of polysilicon due to masking and etching"
            annotation(Dialog(tab="Level=3", group="Geometric parameters", enable=Level == 3));
          parameter Modelica.SIunits.Length XOI(min=0) = 0
        "Polysilicon thickness"
            annotation(Dialog(tab="Level=3", group="Geometric parameters", enable=Level == 3));
          parameter Modelica.SIunits.Length LM(min=0) = 0 "Metal length"
            annotation(Dialog(tab="Level=3", group="Geometric parameters", enable=Level == 3));
          parameter Modelica.SIunits.Length WM(min=0) = 0 "Metal width"
            annotation(Dialog(tab="Level=3", group="Geometric parameters", enable=Level == 3));
          parameter Modelica.SIunits.Length XM(min=0) = 0
        "Widening of metal due to masking and etching"
            annotation(Dialog(tab="Level=3", group="Geometric parameters", enable=Level == 3));
          parameter Modelica.SIunits.Length XOM(min=0) = 0 "Metal thickness"
            annotation(Dialog(tab="Level=3", group="Geometric parameters", enable=Level == 3));

          /* Level=3 Electrical parameters */
          parameter Modelica.SIunits.CurrentDensity JS(min=Modelica.Constants.small) = 1e-4
        "Saturation current density at reference temperature"
            annotation(Dialog(tab="Level=3", group="Electrical parameters", enable=Level == 3));
          parameter Real JSW(unit="A/m", min=0) = 0
        "Saturation current density of perimeter at reference temperature"
            annotation(Dialog(tab="Level=3", group="Electrical parameters", enable=Level == 3));

          /* Level=3 Junction capacitance parameters */
          parameter Real CJ2(min=Modelica.Constants.small,unit="F/m2") = 1e-4
        "Zero-bias bulk capacitance per square meter at reference temperature"
            annotation(Dialog(tab="Level=3", group="Junction capacitance parameters", enable=Level == 3));
          parameter Real CJSW2(min=0,unit="F/m") = 1e-9
        "Zero-bias perimeter capacitance per meter at reference temperature"
            annotation(Dialog(tab="Level=3", group="Junction capacitance parameters", enable=Level == 3));

          /* Numerical parameters */
          parameter Real EMin=-100
        "if x < EMin, the exp(x) function is linearized"
            annotation(Dialog(tab="Advanced", group="Numerical parameters", enable=(Level==1 or Level==3)));
          parameter Real EMax=40
        "if x > EMax, the exp(x) function is linearized"
            annotation(Dialog(tab="Advanced", group="Numerical parameters", enable=(Level==1 or Level==3)));
          parameter Modelica.SIunits.Conductance GminDC(min=Modelica.Constants.small) = 1e-16
        "Leakage conductance"
            annotation(Dialog(tab="Advanced", group="Numerical parameters", enable=(Level==1 or Level==3)));

    protected
          parameter Modelica.SIunits.Area AREAeff = W*L*M;
          parameter Modelica.SIunits.Length PJeff = 2*(W + L)*M;
          parameter Modelica.SIunits.Voltage Vt = k*Tnom/q;
          parameter Modelica.SIunits.Current ISeff = if Level==1 then Area*IS + PJ*ISW else
                                                     if Level==3 then AREAeff*JS + PJeff*JSW else 0;
          parameter Modelica.SIunits.Current Ibreak = -ISeff*(exp(-BV/(N*Vt)) - 1);
          parameter Modelica.SIunits.Current IBVval = if IBV < Ibreak then Ibreak else IBV;
          parameter Modelica.SIunits.Voltage BVval = if BV > 0 then BVval - (N*Vt)*ln(IBVval/Ibreak) else 0;
        annotation (
          Icon(
              Text(
                extent=[-82,76; 28,-80],
                style(color=0),
              string="DS"), Text(
            extent=[8,-10; 108,-38],
            string="Spice",
            style(
              color=0,
              rgbcolor={0,0,0},
              pattern=0))),
          Diagram,
          Documentation(info="<html>
The Spice-style diode offers three modeling levels [1].  The <b>Level=1</b> model characterizes a junction diode by its electrical properties.  This model is similar to the models of the junction diodes used within the BJT and MOSFET models, but it is more accurate.  In particular, it includes a description of avalanche breakdown behavior.  The <b>Level=3</b> model is similar to the <b>Level=1</b> model, but here, the diode is characterized by its physical parameters, such as length and width of the junction.  Also, this model is yet more accurate as it allows the modeler to specify a polysilicon and a metal layer, in which case the additional capacitances associated with these layers will be computed as well.  The <b>Level=2</b> model is an entirely different device.  It represents essentially a non-linear resistor with a quadratic characteristic on both sides.  Thus, the Fowler-Nordheim diode is a symmetric device. <p>

<hr> <p>

<pre>
<b>Parameters:</b>

 <font color=red><b>Level:   DIODE modeling level (default value = 1)
            Level=1: Non-geometric junction diode
            Level=2: Fowler-Nordheim diode
            Level=3: Geometric junction diode</b></font>

 Area:    Relative area occupied by the diode (default value = 1) <font color=red>Level 1</font>


<b>Process Parameters:</b>

 TOX:     Thin oxide thickness (default value = 1e-7 m) <font color=red>Levels 1-3</font>


<b>Basic Electrical Parameters:</b>

 PB:      Built-in junction potential at reference temperature (default value = 0.8 Volt) <font color=red>Levels 1,3</font>

 BV:      Reverse breakdown voltage at reference temperature (default value = 0 Volt) <font color=red>Levels 1,3</font>
            <font color=blue><b>BV must be specified as a positive number.  A value of 0 indicates that the breakdown voltage is infinite.</b></font>

 IBV:     Reverse breakdown current (default value = 0 Amp) <font color=red>Levels 1,3</font>
            <font color=blue><b>IBV must be specified as a positive number.  A value of 0 indicates that the breakdown voltage is infinite.
            Either BV or IBV (the diodic current at breakdown) can be specified.  If both are specified and
            the values are inconsistent with each other, they will be suitably adjusted to match.</b></font>

 PHP:     Built-in perimeter potential at reference temperature (default value = 0.8 Volt) <font color=red>Levels 1,3</font>

 N:       Current emission coefficient (default value = 1) <font color=red>Levels 1,3</font>

 IS:      Saturation current at reference temperature (default value = 1e-14 Amp) <font color=red>Level 1</font>

 ISW:     Saturation current of perimeter at reference temperature (default value = 0 Amp) <font color=red>Level 1</font>

 PJ:      Relative perimeter width (default value = 1) <font color=red>Level 1</font>


<b>High Current Beta Degradation Effect Parameters:</b>

 IKF:     Forward knee current (default value = &infin; Amp) <font color=red>Levels 1,3</font>

 IKR:     Reverse knee current (default value = &infin; Amp) <font color=red>Levels 1,3</font>


<b>Parasitic Resistor Parameters:</b>

 RS:      Resistance at reference temperature (default value = 0 Ohm) <font color=red>Levels 1-3</font>


<b>Junction Capacitance Parameters:</b>

 CJ:      Zero-bias bulk capacitance at reference temperature (default value = 0.5e-12 F) <font color=red>Level 1</font>

 MJ:      Bulk junction grading coefficient (default value = 0.33) <font color=red>Levels 1,3</font>

 CJSW:    Zero-bias perimeter capacitance at reference temperature (default value = 0.5e-12 F) <font color=red>Level 1</font>

 MJSW:    Perimeter capacitance grading coefficient (default value = 0.33) <font color=red>Levels 1,3</font>

 FC:      Forward-bias depletion capacitance coefficient (default value = 0.5) <font color=red>Levels 1,3</font>


<b>Transit Time Parameters:</b>

 TT:      Transit time at reference temperature (default value = 0 sec) <font color=red>Levels 1,3</font>


<b>Temperature Compensation Parameters:</b>

 TRS:     Linear temperature coefficient of resistance (default value = 0 1/K) <font color=red>Levels 1-3</font>

 TCV:     Linear temperature coefficient of breakdown voltage (default value = 0 V/K) <font color=red>Levels 1,3</font>

 TTT1:    Linear temperature coefficient of transit time (default value = 0 1/K) <font color=red>Levels 1,3</font>

 TTT2:    Quadratic temperature coefficient of transit time (default value = 0 1/K2) <font color=red>Levels 1,3</font>

 Tnom:    Reference temperature (default value = 300.15 K) <font color=red>Levels 1-3</font>

 XTI:     Temperature exponent of saturation current (default value = 3) <font color=red>Levels 1,3</font>
            <font color=blue><b>XTI = 3 for <i>pn</i>-junction
            XTI = 2 for Schottky barrier diode</b></font>

 EG:      Energy gap for temperature effect on saturation current at 0 K (default value = 1.11 Volt) <font color=red>Levels 1,3</font>
            <font color=blue><b>EG = 1.11 for Si
            EG = 0.73 for Schottky barrier diode
            EG = 0.71 for Ge</b></font>


<b>Level=2 General Parameters:</b>

 M:       Diode multiplier (default value = 1) <font color=red>Levels 2,3</font>


<b>Level=2 Geometric Parameters:</b>

 L:       Diode length (default value = 0 m) <font color=red>Levels 2,3</font>

 W:       Diode width (default value = 0 m) <font color=red>Levels 2,3</font>


<b>Level=2 Electrical Parameters:</b>

 JF:       Forward Fowler-Nordheim current coefficient (default value = 0 A/V<sup>2</sup>) <font color=red>Level 2</font>

 JR:       Reverse Fowler-Nordheim current coefficient (default value = 0 A/V<sup>2</sup>) <font color=red>Level 2</font>

 EF:       Forward Fowler-Nordheim critical electrical field (default value = 0 V/m) <font color=red>Level 2</font>

 ER:       Reverse Fowler-Nordheim critical electrical field (default value = 0 V/m) <font color=red>Level 2</font>


<b>Level=3 Geometric Parameters:</b>

 LP:       Polysilicon length (default value = 0 m) <font color=red>Level 3</font>

 WP:       Polysilicon width (default value = 0 m) <font color=red>Level 3</font>

 XP:       Widening of polysilicon due to masking and etching (default value = 0 m) <font color=red>Level 3</font>

 XOI:      Polysilicon thickness (default value = 0 m) <font color=red>Level 3</font>

 LM:       Metal length (default value = 0 m) <font color=red>Level 3</font>

 WM:       Metal width (default value = 0 m) <font color=red>Level 3</font>

 XM:       Widening of metal due to masking and etching (default value = 0 m) <font color=red>Level 3</font>

 XOM:      Metal thickness (default value = 0 m) <font color=red>Level 3</font>


<b>Level=3 Electrical Parameters:</b>

 JS:      Saturation current density at reference temperature (default value = 1e-4 Amp/m<sup>2</sup>) <font color=red>Level 3</font>

 JSW:     Saturation current density of perimeter at reference temperature (default value = 0 Amp/m<sup>2</sup>) <font color=red>Level 3</font>


<b>Level=2 Junction Capacitance Parameters:</b>

 CJ2:     Zero-bias bulk capacitance per meter square at reference temperature (default value = 1e-4 F/m<sup>2</sup>) <font color=red>Level 3</font>

 CJSW2:   Zero-bias perimeter capacitance per meter at reference temperature (default value = 1e-9 F/m) <font color=red>Level 3</font>


<b>Numerical Parameters:</b>

 EMin:    if x &lt; EMin, the exp(x) function is linearized (default value = -100) <font color=red>Levels 1,3</font>

 EMax:    if x &gt; EMax, the exp(x) function is linearized (default value = 40) <font color=red>Levels 1,3</font>

 GminDC:  Leakage conductance (default value = 1e-16 mho) <font color=red>Levels 1,3</font>
</pre> <p>

<hr> <p>

<b>References:</b> <p>

<ol>
<li> Massobrio, G. and P. Antognetti (1993), <i>Semiconductor Device Modeling with Spice</i>,
     2<sup>nd</sup> edition, McGraw Hill, New York.
</ol>
</html>
"));
    public
          Junctions.J0p2 J0p2_1 annotation (extent=[-90,-10; -70,10]);
          Junctions.J0p5 J0p5_1 annotation (extent=[70,-10; 90,10]);
          Bonds.Bond B1 annotation (extent=[-70,-10; -50,10]);
          Junctions.J1p3 J1p3_1 annotation (extent=[-50,-10; -30,10]);
          Bonds.Bond B2 annotation (extent=[-30,-10; -10,10]);
          Junctions.J0p4 J0p4_1 annotation (extent=[-10,-10; 10,10]);
          Bonds.fBond B4 annotation (extent=[50,-10; 70,10], rotation=180);
          Bonds.eBond B3 annotation (extent=[10,-10; 30,10], rotation=180);
          Bonds.Bond B5 annotation (extent=[-50,-30; -30,-10], rotation=-90);
          Bonds.eBond B6 annotation (extent=[-50,-90; -30,-70], rotation=270);
          Bonds.Bond B7 annotation (extent=[-10,10; 10,30], rotation=90);
          Utilities.CjD CjD1(
        Level=Level,
        Tnom=Tnom,
        IS=IS,
        N=N,
        TT=TT,
        CJ=CJ,
        PB=PB,
        MJ=MJ,
        EG=EG,
        XTI=XTI,
        FC=FC,
        BV=BVval,
        IBV=IBVval,
        TCV=TCV,
        TTT1=TTT1,
        TTT2=TTT2,
        ISW=ISW,
        CJSW=CJSW,
        PHP=PHP,
        MJSW=MJSW,
        TOX=TOX,
        GminDC=GminDC,
        Area=Area,
        PJ=PJ,
        EMin=EMin,
        EMax=EMax,
        L=L,
        W=W,
        M=M,
        JS=JS,
        CJ2=CJ2,
        JSW=JSW,
        CJSW2=CJSW2,
        LP=LP,
        WP=WP,
        XP=XP,
        XOI=XOI,
        LM=LM,
        WM=WM,
        XM=XM,
        XOM=XOM,
        JF=JF,
        JR=JR,
        EF=EF,
        ER=ER) annotation (extent=[10,30; -10,70], rotation=90);
          Utilities.ID0 ID0(
        Level=Level,
        N=N,
        IKF=IKF,
        IKR=IKR) annotation (extent=[20,60; 40,40]);
          Bonds.eBond B8 annotation (extent=[-10,70; 10,90], rotation=90);
          Modelica.Blocks.Math.Gain Gain1(k=-1) annotation (extent=[50,44; 62,56]);
          Bonds.fBond B9 annotation (extent=[70,10; 90,30], rotation=90);
          Sensors.De Tdev annotation (extent=[70,30; 90,50], rotation=90);
          Bonds.fBond B10 annotation (extent=[-10,-30; 10,-10], rotation=270);
          Sensors.De vd annotation (extent=[-10,-50; 10,-30], rotation=270);
          BondLib.Spice.RS RS1(
        Tnom=Tnom,
        R=RS,
        TR1=TRS,
        TR2=0,
        Area=Area) annotation (extent=[-50,-70; -30,-30], rotation=-90);
          BondLib.Sources.mSf_int mSf_int1 annotation (extent=[50,10; 30,-10]);
        equation
          connect(J0p2_1.BondCon1, BondCon1) annotation (points=[-90,0; -100,0],
          style(color=8, rgbcolor={192,192,192}));
          connect(J0p5_1.BondCon2, BondCon2) annotation (points=[90,0; 100,0],
          style(color=8, rgbcolor={192,192,192}));
          connect(B1.BondCon1, J0p2_1.BondCon2) annotation (points=[-69.8,0; -70,0],
          style(color=8, rgbcolor={192,192,192}));
          connect(B1.BondCon2, J1p3_1.BondCon1) annotation (points=[-50,0; -50,0],
          style(color=8, rgbcolor={192,192,192}));
          connect(J1p3_1.BondCon2, B2.BondCon1) annotation (points=[-30,0; -29.8,0],
          style(color=8, rgbcolor={192,192,192}));
          connect(B2.BondCon2, J0p4_1.BondCon1) annotation (points=[-10,0; -10,0],
          style(color=8, rgbcolor={192,192,192}));
          connect(J0p4_1.BondCon2, B3.eBondCon1) annotation (points=[10,0; 15,0;
            15,1.22461e-015; 10,1.22461e-015],
                                            style(color=8, rgbcolor={192,192,
              192}));
          connect(B4.eBondCon1, J0p5_1.BondCon1) annotation (points=[70,
            -1.22461e-015; 70,0], style(color=8, rgbcolor={192,192,192}));
          connect(B5.BondCon1, J1p3_1.BondCon3) annotation (points=[-40,-10.2; -40,
            -10], style(color=8, rgbcolor={192,192,192}));
          connect(B7.BondCon1, J0p4_1.BondCon4) annotation (points=[
            -6.00057e-016,10.2; -6.00057e-016,17.1; 0,17.1; 0,10],
                                                     style(color=8, rgbcolor={
              192,192,192}));
          connect(CjD1.BondCon1, B7.BondCon2) annotation (points=[6.12303e-016,
            30; 6.12303e-016,30],
                              style(color=8, rgbcolor={192,192,192}));
          connect(CjD1.id, ID0.u) annotation (points=[10,50; 18,50], style(color=74,
            rgbcolor={0,0,127}));
          connect(B8.fBondCon1, CjD1.BondCon2) annotation (points=[
            -6.12303e-016,70; -6.12303e-016,70],
                               style(color=8, rgbcolor={192,192,192}));
          connect(Gain1.u, ID0.y) annotation (points=[48.8,50; 41,50], style(color=
              74, rgbcolor={0,0,127}));
          connect(B9.eBondCon1, J0p5_1.BondCon5) annotation (points=[80,10; 80,10],
          style(color=8, rgbcolor={192,192,192}));
          connect(Tdev.BondCon1, B9.fBondCon1) annotation (points=[80,30; 80,30],
          style(color=8, rgbcolor={192,192,192}));
          connect(Tdev.OutPort1, ID0.Tdev) annotation (points=[80,51; 80,70; 30,70;
            30,60], style(color=74, rgbcolor={0,0,127}));
          connect(B10.eBondCon1, J0p4_1.BondCon3) annotation (points=[
            1.83691e-015,-10; 0,-10],
                         style(color=8, rgbcolor={192,192,192}));
          connect(vd.BondCon1, B10.fBondCon1) annotation (points=[1.83691e-015,
            -30; -1.83691e-015,-30],
                                style(color=8, rgbcolor={192,192,192}));
          connect(vd.OutPort1, ID0.v) annotation (points=[-2.0206e-015,-51;
            -2.0206e-015,-60; 20,-60; 20,20; 30,20; 30,40], style(color=74,
            rgbcolor={0,0,127}));
          connect(B6.eBondCon1, J0p5_1.BondCon3) annotation (points=[-40,-90; -40,
            -96; 76,-96; 76,-10], style(color=8, rgbcolor={192,192,192}));
          connect(B8.eBondCon1, J0p5_1.BondCon4) annotation (points=[
            6.12303e-016,90; 0,90; 0,94; 110,94; 110,-20; 84,-20; 84,-10],
                                                              style(color=8,
            rgbcolor={192,192,192}));
          connect(RS1.BondCon1, B5.BondCon2) annotation (points=[-40,-30; -40,-30],
          style(color=8, rgbcolor={192,192,192}));
          connect(RS1.BondCon2, B6.fBondCon1) annotation (points=[-40,-70; -40,-70],
          style(color=8, rgbcolor={192,192,192}));
          connect(B3.fBondCon1, mSf_int1.BondCon2) annotation (points=[30,
            -1.22461e-015; 30,0], style(color=8, rgbcolor={192,192,192}));
          connect(B4.fBondCon1, mSf_int1.BondCon1) annotation (points=[50,
            1.22461e-015; 50,0], style(color=8, rgbcolor={192,192,192}));
          connect(Gain1.y, mSf_int1.s) annotation (points=[62.6,50; 66,50; 66,20;
            40,20; 40,9], style(color=74, rgbcolor={0,0,127}));
        end DS;

      model NPNlat "Spice-style laterally diffused NPN bipolar transistor"
        constant Real inf=Modelica.Constants.inf;

        /* General parameters */
        parameter Integer Level(min=1,max=2) = 1 "Transistor modeling level" annotation(Evaluate=true);
        parameter Real Area=1 "Relative area occupied by device";

        /* DC model parameters */
        parameter Real BF=100
        "Maximum forward current gain at reference temperature"
          annotation(Dialog(group="DC model parameters"));
        parameter Real BR=1
        "Maximum reverse current gain at reference temperature"
          annotation(Dialog(group="DC model parameters"));
        parameter Modelica.SIunits.Current IS=1e-16
        "Saturation current at reference temperature"   annotation(Dialog(group="DC model parameters"));
        parameter Real NF=1 "Forward current emission coefficient"
          annotation(Dialog(group="DC model parameters"));
        parameter Real NR=1 "Reverse current emission coefficient"
          annotation(Dialog(group="DC model parameters"));
        parameter Modelica.SIunits.Current ISS=IS
        "Saturation current used for current injection"   annotation(Dialog(group="DC model parameters", enable=Level == 2));

        /* Low current beta degradation effect parameters */
        parameter Real C4(min=0) = 0
        "Base-collector leakage current coefficient"
          annotation(Dialog(group="Low current beta degradation effect parameters", enable=Level == 2));
        parameter Modelica.SIunits.Current ISC(min=0) = 0
        "Base-collector leakage saturation current at reference temperature (ISC = C4*IS)"
          annotation(Dialog(group="Low current beta degradation effect parameters", enable=Level == 2));
        parameter Real C2(min=0) = 0
        "Base-emitterr leakage current coefficient"
          annotation(Dialog(group="Low current beta degradation effect parameters", enable=Level == 2));
        parameter Modelica.SIunits.Current ISE(min=0) = 0
        "Base-emitter leakage saturation current at reference temperature (ISE = C2*IS)"
          annotation(Dialog(group="Low current beta degradation effect parameters", enable=Level == 2));
        parameter Real NC=2
        "Low-current base-collector leakage emission coefficient"
          annotation(Dialog(group="Low current beta degradation effect parameters", enable=Level == 2));
        parameter Real NE=1.5
        "Low-current base-emitter leakage emission coefficient"
          annotation(Dialog(group="Low current beta degradation effect parameters", enable=Level == 2));

        /* High current beta degradation effect parameters */
        parameter Modelica.SIunits.Current IKF=inf
        "Corner for forward beta high-current roll-off"
          annotation(Dialog(group="High current beta degradation effect", enable=Level == 2));
        parameter Modelica.SIunits.Current IKR=inf
        "Corner for reverse beta high-current roll-off"
          annotation(Dialog(group="High current beta degradation effect", enable=Level == 2));

        /* Base width modulation parameters */
        parameter Modelica.SIunits.Voltage VAF=inf "Forward early voltage"
          annotation(Dialog(group="Base width modulation parameters"));
        parameter Modelica.SIunits.Voltage VAR=inf "Reverse early voltage"
          annotation(Dialog(group="Base width modulation parameters", enable=Level == 2));

        /* Parasitic Resistor Parameters */
        parameter Modelica.SIunits.Current IRB=inf
        "Current where base resistance falls halfway to minimum value"
          annotation(Dialog(group="Parasitic resistor parameters", enable=Level == 2));
        parameter Modelica.SIunits.Resistance RB(min=0) = 0
        "Zero-bias base resistance"   annotation(Dialog(group="Parasitic resistor parameters", enable=Level == 2));
        parameter Modelica.SIunits.Resistance RBM(min=0) = RB
        "Minimum base resistance at high currents"   annotation(Dialog(group="Parasitic resistor parameters", enable=Level == 2));
        parameter Modelica.SIunits.Resistance RC(min=0) = 0
        "Collector resistance"
          annotation(Dialog(group="Parasitic resistor parameters", enable=Level == 2));
        parameter Modelica.SIunits.Resistance RE(min=0) = 0
        "Emitter resistance"
          annotation(Dialog(group="Parasitic resistor parameters", enable=Level == 2));

        /* Junction capacitor parameters */
        parameter Modelica.SIunits.Capacitance CJC(min=Modelica.Constants.small) = 1e-12
        "Zero-bias base-collector depletion capacitance at reference temperature (CJC > 0)"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Real MJC(min=0.3,max=0.5) = 0.33
        "Base-collector junction grading coefficient"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Modelica.SIunits.Voltage VJC=0.75
        "Base-collector built-in potential at reference temperature"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Modelica.SIunits.Capacitance CJE(min=Modelica.Constants.small) = 1e-12
        "Zero-bias base-emitter depletion capacitance at reference temperature (CJE > 0)"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Real MJE(min=0.3,max=0.5) = 0.33
        "Base-emitter junction grading coefficient"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Modelica.SIunits.Voltage VJE=0.75
        "Base-emitter built-in potential at reference temperature"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Modelica.SIunits.Capacitance CJS(min=Modelica.Constants.small) = 1e-12
        "Zero-bias substrate depletion capacitance at reference temperature"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Real MJS(min=0.3,max=0.5) = 0.33
        "Substrate junction grading coefficient"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Modelica.SIunits.Voltage VJS=0.75
        "Substrate built-in potential at reference temperature"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Real XCJC(min=0,max=1) = 1
        "Fraction of base-collector depletion capacitance connected to internal base node"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Real FC(min=0,max=1) = 0.5
        "Depletion capacitance factor for linearization"
          annotation(Dialog(group="Junction capacitor parameters", enable=Level == 2));

        /* Transit time parameters */
        parameter Modelica.SIunits.Time TF(min=0) = 0
        "Ideal forward transit time"
          annotation(Dialog(group="Transit time parameters"));
        parameter Modelica.SIunits.Time TR(min=0) = 0
        "Ideal reverse transit time"
          annotation(Dialog(group="Transit time parameters"));

        /* Temperature compensation and area parameters */
        parameter Modelica.SIunits.Temperature Tnom=300.15
        "Reference temperature"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Real XTI=3 "Saturation current temperature exponent"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Real XTB=0 "Forward and reverse beta temperature coefficient"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Modelica.SIunits.Voltage EG=1.11
        "Energy gap for temperature effect on saturation current"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Real TRB1(unit="1/K") = 0
        "Linear temperature coefficient of zero-bias base resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));
        parameter Real TRB2(unit="1/K2") = 0
        "Quadratic temperature coefficient of zero-bias base resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));
        parameter Real TRM1(unit="1/K") = 0
        "Linear temperature coefficient of minimum base resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));
        parameter Real TRM2(unit="1/K2") = 0
        "Quadratic temperature coefficient of minimum base resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));
        parameter Real TRC1(unit="1/K") = 0
        "Linear temperature coefficient of collector resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));
        parameter Real TRC2(unit="1/K2") = 0
        "Quadratic temperature coefficient of collector resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));
        parameter Real TRE1(unit="1/K") = 0
        "Linear temperature coefficient of emitter resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));
        parameter Real TRE2(unit="1/K2") = 0
        "Quadratic temperature coefficient of emitter resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));

        /* Initialization parameters */
        parameter Modelica.SIunits.Time rampTime=0
        "Substrate potential ramping time (if unconnected)"
          annotation(Dialog(tab="Advanced", group="Initialization parameters"));

        /* Numerical parameters */
        parameter Real EMin=-100
        "if x < EMin, the exp(x) function is linearized"
          annotation(Dialog(tab="Advanced", group="Numerical parameters"));
        parameter Real EMax=40 "if x > EMax, the exp(x) function is linearized"
          annotation(Dialog(tab="Advanced", group="Numerical parameters"));
        parameter Modelica.SIunits.Conductance GminDC(min=0) = 1e-19
        "Leakage conductance"
          annotation(Dialog(tab="Advanced", group="Numerical parameters"));

    protected
        parameter Modelica.SIunits.Current ISCeff = if ISC > 0 then ISC else C4*IS;
        parameter Modelica.SIunits.Current ISEeff = if ISE > 0 then ISE else C2*IS;
      annotation (Coordsys(extent=[-120,-120; 120,120]), Diagram(
          Text(
            extent=[-126,22; -114,8],
            string="B",
            style(color=10, rgbcolor={95,95,95})),
          Text(
            extent=[-20,128; -8,114],
            style(color=10, rgbcolor={95,95,95}),
            string="C"),
          Text(
            extent=[-20,-114; -8,-128],
            style(color=10, rgbcolor={95,95,95}),
            string="E"),
          Text(
            extent=[108,-6; 134,-22],
            style(color=10, rgbcolor={95,95,95}),
            string="Heat"),
          Text(
            extent=[78,50; 104,34],
            style(color=10, rgbcolor={95,95,95}),
            string="Heat"),
          Text(
            extent=[42,-34; 52,-46],
            style(color=10, rgbcolor={95,95,95}),
            string="qb"),
          Text(
            extent=[-28,66; -18,54],
            style(color=10, rgbcolor={95,95,95}),
            string="qb"),
          Text(
            extent=[0,20; 10,8],
            style(color=10, rgbcolor={95,95,95}),
            string="ix"),
          Text(
            extent=[-60,70; -50,58],
            style(color=10, rgbcolor={95,95,95}),
            string="ix"),
          Text(
            extent=[-126,-38; -114,-52],
            style(color=10, rgbcolor={95,95,95}),
            string="S")),
        Icon(
          Text(
            extent=[-126,24; -114,10],
            string="B",
            style(color=10, rgbcolor={95,95,95})),
          Text(
            extent=[10,126; 22,112],
            style(color=10, rgbcolor={95,95,95}),
            string="C"),
          Text(
            extent=[8,-114; 20,-128],
            style(color=10, rgbcolor={95,95,95}),
            string="E"),
          Text(
            extent=[106,26; 132,6],
            style(color=10, rgbcolor={95,95,95}),
            string="Heat"),
            Text(
              extent=[-96,84; 68,-92],
              style(color=0),
              string="NPN"),
            Text(
              extent=[62,4; 102,-38],
              style(color=0),
            string="lat"),
            Line(points=[0,100; 0,-100],
                                       style(
                color=9,
                rgbcolor={149,164,171},
                thickness=2,
                fillColor=74,
                rgbfillColor={0,0,185},
                fillPattern=1)),
            Line(points=[0,-100; 20,-80],style(
                color=9,
                rgbcolor={149,164,171},
                thickness=2,
                fillColor=74,
                rgbfillColor={0,0,185},
                fillPattern=1)),
            Line(points=[-20,-80; 0,-100],style(
                color=9,
                rgbcolor={149,164,171},
                thickness=2,
                fillColor=74,
                rgbfillColor={0,0,185},
                fillPattern=1)),
          Line(points=[-100,-60; 100,-60], style(color=3, rgbcolor={0,0,255})),
          Line(points=[80,-80; 100,-60], style(color=3, rgbcolor={0,0,255})),
          Line(points=[80,-40; 100,-60], style(color=3, rgbcolor={0,0,255})),
          Text(
            extent=[-80,100; 80,60],
            style(color=3, rgbcolor={0,0,255}),
            string="%name"),
          Text(
            extent=[-88,-72; -44,-108],
            string="%Area x",
            style(color=3, rgbcolor={0,0,255}))),
        Documentation(info="<html>
The <b>NPNlat</b> element of the Spice bond graph library implements a full-fledges Spice-stye Gummel-Poon model of the laterally diffused NPN bipolar transistor [1-3]. The model contains a <b>NPNint</b> model implementing the inner parts of the NPN transistor up to the internal nodes. <p>

The NPN bipolar transistor is a directed FourPort.  The direction of positive power flow is assumed into the model at the base, <b>B</b>, and at the collector, <b>C</b>, whereas it is assumed out of the model at the emitter, <b>E</b>, and at the <b>Heat</b> port. <p>

The causality of the NPN model is free. <p>

<hr>

<pre>
<b>Parameters:</b>

 <font color=red><b>Level:   Transistor modeling level (default value = 2)
            Level = 1: Ebers-Moll model
            Level = 2: Gummel-Poon model</b></font>

 Area:    Relative area occupied by device (default value = 1) <font color=red>Levels 1,2</font>


<b>DC Model Parameters:</b>

 BF:      Maximum forward current gain at reference temperature (default value = 100) <font color=red>Levels 1,2</font>

 BR:      Maximum reverse current gain at reference temperature (default value = 1) <font color=red>Levels 1,2</font>

 IS:      Saturation current at reference temperature (default value = 1e-16 Amp) <font color=red>Levels 1,2</font>

 NF:      Forward current emission coefficient (default value = 1) <font color=red>Levels 1,2</font>

 NR:      Reverse current emission coefficient (default value = 1) <font color=red>Levels 1,2</font>

 ISS:     Saturation current for injection (default value = IS Amp) <font color=red>Levels 2</font>


<b>Low Current Beta Degradation Effect Parameters:</b>

 C4:      Base-collector leakage current coefficient (default value = 0) <font color=red>Levels 2</font>

 ISC:     Base-collector leakage saturation current at reference temperature (default value = 0 Amp) <font color=red>Levels 2</font>
          <font color=blue><b>ISCeff = if ISC &gt; 0 then ISC else C4*IS</b></font>

 C2:      Base-emitter leakage current coefficient (default value = 0) <font color=red>Levels 2</font>

 ISE:     Base-emitter leakage saturation current at reference temperature (default value = 0 Amp) <font color=red>Levels 2</font>
          <font color=blue><b>ISEeff = if ISE &gt; 0 then ISE else C2*IS</b></font>

 NC:      Low-current base-collector leakage emission coefficient (default value = 2) <font color=red>Levels 2</font>

 NE:      Low-current base-emitter leakage emission coefficient (default value = 1.5) <font color=red>Levels 2</font>


<b>High Current Beta Degradation Effect Parameters:</b>

 IKF:     Corner for forward beta high-current roll-off (default value = &infin; Amp) <font color=red>Levels 2</font>

 IKR:     Corner for reverse beta high-current roll-off (default value = &infin; Amp) <font color=red>Levels 2</font>


<b>Base Width Modulation Parameters:</b>

 VAF:     Forward early voltage (default value = &infin; Volt) <font color=red>Levels 1,2</font>

 VAR:     Reverse early voltage (default value = &infin; Volt) <font color=red>Levels 2</font>


<b>Parasitic Resistor Parameters:</b>

 IRB:     Current where base resistance falls halfway to minimum value (default value = &infin; Amp) <font color=red>Levels 2</font>

 RB:      Zero-bias base resistance (default value = 0 Ohm) <font color=red>Levels 2</font>

 RBM:     Minimum base resistance at high currents (default value = RB Ohm) <font color=red>Levels 2</font>

 RC:      Collector resistance (default value = 0 Ohm) <font color=red>Levels 2</font>

 RE:      Emitter resistance (default value = 0 Ohm) <font color=red>Levels 2</font>


<b>Junction Capacitor Parameters:</b>

 CJC:     Zero-bias base-collector depletion capacitance at reference temperature (CJC > 0) (default value = 1e-12 F) <font color=red>Levels 1,2</font>

 MJC:     Base-collector junction grading coefficient (default value = 0.33) <font color=red>Levels 1,2</font>

 VJC:     Base-collector built-in potential at reference temperature (default value = 0.75 Volt) <font color=red>Levels 1,2</font>

 CJE:     Zero-bias base-emitter depletion capacitance at reference temperature (CJE > 0) (default value = 1e-12 F) <font color=red>Levels 1,2</font>

 MJE:     Base-emitter junction grading coefficient (default value = 0.33) <font color=red>Levels 1,2</font>

 VJE:     Base-emitter built-in potential at reference temperature (default value = 0.75 Volt) <font color=red>Levels 1,2</font>

 CJS:     Zero-bias substrate depletion capacitance at reference temperature (CJS > 0) (default value = 1e-12 F) <font color=red>Levels 1,2</font>

 MJS:     Substrate junction grading coefficient (default value = 0.33) <font color=red>Levels 1,2</font>

 VJS:     Substrate built-in potential at reference temperature (default value = 0.75 Volt) <font color=red>Levels 1,2</font>

 XCJC:    Fraction of base-collector depletion capacitance connected to internal base node (default value = 1) <font color=red>Levels 1,2</font>

 FC:      Depletion capacitance factor for linearization (default value = 0.5) <font color=red>Levels 2</font>


<b>Transit Time Parameters:</b>

 TF:      Ideal forward transit time (default value = 0 sec) <font color=red>Levels 1,2</font>

 TR:      Ideal reverse transit time (default value = 0 sec) <font color=red>Levels 1,2</font>


<b>Temperature Compensation and Area Parameters:</b>

 Tnom:    Reference temperature (default value = 300.15 K) <font color=red>Levels 1,2</font>

 XTI:     Saturation current temperature exponent (default value = 3) <font color=red>Levels 1,2</font>

 XTB:     Forward and reverse beta temperature coefficient (default value = 0) <font color=red>Levels 1,2</font>

 EG:      Energy gap for temperature effect on saturation current (default value = 1.11 Volt) <font color=red>Levels 1,2</font>

 TRB1:    Linear temperature coefficient of zero-bias base resistance (default value = 0 1/K) <font color=red>Levels 2</font>

 TRB2:    Quadratic temperature coefficient of zero-bias base resistance (default value = 0 1/(K<sup>2</sup>)) <font color=red>Levels 2</font>

 TRM1:    Linear temperature coefficient of mimimum base resistance (default value = 0 1/K) <font color=red>Levels 2</font>

 TRM2:    Quadratic temperature coefficient of minimum base resistance (default value = 0 1/(K<sup>2</sup>)) <font color=red>Levels 2</font>

 TRC1:    Linear temperature coefficient of collector resistance (default value = 0 1/K) <font color=red>Levels 2</font>

 TRC2:    Quadratic temperature coefficient of collector resistance (default value = 0 1/(K<sup>2</sup>)) <font color=red>Levels 2</font>

 TRE1:    Linear temperature coefficient of emitter resistance (default value = 0 1/K) <font color=red>Levels 2</font>

 TRE2:    Quadratic temperature coefficient of emitter resistance (default value = 0 1/(K<sup>2</sup>)) <font color=red>Levels 2</font>


<b>Numerical Parameters:</b>

 EMin:    Minimum exponent for linearization of junction current (default value = -100) <font color=red>Levels 1,2</font>

 EMax:    Maximum exponent for linearization of junction current (default value = 40) <font color=red>Levels 1,2</font>

 GminDC:  Leakage conductance (default value = 1e-19 Mho) <font color=red>Levels 1,2</font>
</pre> <p>

<hr> <p>

<b>References:</b> <p>

<ol>
<li>Cellier, F.E. (1991), <i>Continuous System Modeling</i>, Springer-Verlag, New York.
<li>Hild, D.R. and F.E. Cellier (1994), \"Object-oriented electronic circuit modeling using Dymola,\"
    <i>Proc. OOS'94, SCS Object Oriented Simulation Conference</i>, Tempe, AZ, pp.68-75.
<li>Hild, D.R. (1993), <i>Circuit Modeling in Dymola</i>, MS Thesis, Dept. of Electr. &amp; Comp. Engr.,
    University of Arizona, Tucson.
<li> Massobrio, G. and P. Antognetti (1993), <i>Semiconductor Device Modeling with Spice</i>,
     2<sup>nd</sup> edition, McGraw Hill, New York.
<li>Schweisguth, M.C. and F.E. Cellier (1999), \"<a href=\"http://www.inf.ethz.ch/~fcellier/Publikationen/BJT.pdf\">A
    bond graph model of the bipolar junction transistor</a>,\" <i>Proc. SCS Intl. Conf. on Bond Graph Modeling</i>,
    San Francisco, CA, pp.344-349.
<li>Schweisguth, M.C. (1997), <a href=\"http://www.inf.ethz.ch/~fcellier/Publikationen/schweisguth_ms.pdf\"><i>Semiconductor
    Modeling with Bondgraphs</i></a>, MS Thesis, Dept. of Electr. &amp; Comp. Engr., University of Arizona, Tucson.
</ol>
</html>
"));
    public
      BondLib.Interfaces.BondCon B "Base"
        annotation (extent=[-130,-10; -110,10]);
      BondLib.Interfaces.BondCon E "Emitter"
        annotation (extent=[-10,-130; 10,-110]);
      BondLib.Interfaces.BondCon H "Heat" annotation (extent=[110,-10; 130,10]);
      BondLib.Interfaces.BondCon C "Collector"
                                          annotation (extent=[-10,110; 10,130]);
      BondLib.Spice.Utilities.NPNint Qni(
        BF=BF,
        BR=BR,
        IS=IS,
        NF=NF,
        NR=NR,
        GminDC=GminDC,
        NC=NC,
        NE=NE,
        VAF=VAF,
        VAR=VAR,
        IKF=IKF,
        IKR=IKR,
        CJC=CJC,
        MJC=MJC,
        VJC=VJC,
        CJE=CJE,
        MJE=MJE,
        VJE=VJE,
        XCJC=XCJC,
        TF=TF,
        TR=TR,
        Tnom=Tnom,
        XTI=XTI,
        XTB=XTB,
        EG=EG,
        Area=Area,
        EMin=EMin,
        EMax=EMax,
        FC=FC,
        Level=Level,
        ISC=ISCeff,
        ISE=ISEeff,
        ISS=ISS) annotation (extent=[0,-40; 40,0]);
      BondLib.Bonds.Bond B1 annotation (extent=[-20,-30; 0,-10], rotation=0);
      BondLib.Junctions.J0p3 J0p3_1
        annotation (extent=[-40,-30; -20,-10], rotation=90);
      BondLib.Bonds.Bond B2 annotation (extent=[-50,-10; -30,10], rotation=0);
      BondLib.Junctions.J1p3 J1p3_1 annotation (extent=[-70,10; -50,-10]);
      BondLib.Bonds.Bond B3 annotation (extent=[-90,-10; -70,10], rotation=0);
      BondLib.Bonds.Bond B4 annotation (extent=[-70,10; -50,30], rotation=90);
      BondLib.Spice.Utilities.RBS Rb(
        Tnom=Tnom,
        RB=RB,
        RBM=RBM,
        IRB=IRB,
        TRB1=TRB1,
        TRB2=TRB2,
        TRM1=TRM1,
        TRM2=TRM2,
        Area=Area,
        Level=Level)
                   annotation (extent=[-60,30; -20,50], rotation=0);
      BondLib.Bonds.eBond B5 annotation (extent=[-20,30; 0,50]);
      BondLib.Bonds.Bond B6 annotation (extent=[-50,-50; -30,-30], rotation=180);
      BondLib.Junctions.J1p3 J1p3_2 annotation (extent=[-70,-50; -50,-30]);
      BondLib.Bonds.Bond B7 annotation (extent=[-90,-50; -70,-30], rotation=180);
      BondLib.Junctions.J0p2 J0p2_2
        annotation (extent=[-110,-60; -90,-40], rotation=90);
      BondLib.Bonds.Bond B9 annotation (extent=[-70,-70; -50,-50], rotation=270);
      BondLib.Bonds.eBond B10 annotation (extent=[-20,-90; 0,-70]);
      BondLib.Bonds.Bond B11 annotation (extent=[10,40; 30,60], rotation=270);
      BondLib.Junctions.J1p3 J1p3_3
        annotation (extent=[30,80; 10,60], rotation=-90);
      BondLib.Bonds.Bond B12 annotation (extent=[10,80; 30,100], rotation=270);
      BondLib.Junctions.J0p2 J0p2_3
        annotation (extent=[10,100; 30,120], rotation=90);
      BondLib.Bonds.Bond B13 annotation (extent=[30,60; 50,80], rotation=0);
      BondLib.Bonds.eBond B14 annotation (extent=[90,60; 110,80], rotation=0);
      BondLib.Bonds.Bond B15 annotation (extent=[10,-60; 30,-40], rotation=270);
      BondLib.Junctions.J1p3 J1p3_4
        annotation (extent=[30,-60; 10,-80], rotation=-90);
      BondLib.Bonds.Bond B16 annotation (extent=[10,-100; 30,-80], rotation=270);
      BondLib.Junctions.J0p2 J0p2_4
        annotation (extent=[10,-120; 30,-100], rotation=90);
      BondLib.Bonds.Bond B17 annotation (extent=[30,-80; 50,-60], rotation=0);
      BondLib.Bonds.eBond B18 annotation (extent=[90,-80; 110,-60], rotation=0);
      BondLib.Bonds.eBond B19 annotation (extent=[40,-30; 60,-10]);
      BondLib.Junctions.J0p6 J0p6_1 annotation (extent=[60,20; 80,40]);
      BondLib.Bonds.Bond B20 annotation (extent=[10,0; 30,20], rotation=270);
      BondLib.Junctions.J0p3 J0p3_2
        annotation (extent=[10,20; 30,40], rotation=90);
      BondLib.Junctions.J0p3 J0p3_3
        annotation (extent=[-110,-10; -90,10], rotation=180);
      BondLib.Bonds.Bond B21 annotation (extent=[-110,10; -90,30], rotation=90);
      BondLib.Junctions.J1p3 J1p3_5
        annotation (extent=[-110,30; -90,50], rotation=90);
      BondLib.Bonds.Bond B22 annotation (extent=[-110,50; -90,70], rotation=90);
      BondLib.Bonds.Bond B23 annotation (extent=[-90,50; -70,70], rotation=90);
      BondLib.Bonds.eBond B24 annotation (extent=[-40,70; -20,90]);
      BondLib.Bonds.eBond B25 annotation (extent=[80,20; 100,40]);
      BondLib.Junctions.J0p3 J0p3_4
        annotation (extent=[100,20; 120,40], rotation=180);
      Utilities.CbxS Cbx   annotation (extent=[-80,70; -40,90]);
      Utilities.RS2 Rc(
        Tnom=Tnom,
        R=RC,
        TR1=TRC1,
        TR2=TRC2,
        Area=Area,
        Level=Level) annotation (extent=[50,58; 90,82]);
      Utilities.RS2 Re(
        Tnom=Tnom,
        Area=Area,
        Level=Level,
        R=RE,
        TR1=TRE1,
        TR2=TRE2) annotation (extent=[50,-82; 90,-58]);
      BondLib.Interfaces.BondCon S "Substrate"
        annotation (extent=[-130,-70; -110,-50]);
      Utilities.CjS Dbs(
        Tnom=Tnom,
        XTI=XTI,
        XTB=XTB,
        EG=EG,
        Area=Area,
        Level=Level,
        BE=false,
        EMin=EMin,
        EMax=EMax,
        MJ=MJS,
        VJ=VJS,
        XCJ=1,
        Tau=0,
        CJ=CJS,
        FC=0,
        IS=0,
        GminDC=0,
        N=1)       annotation (extent=[-60,-94; -20,-66]);
      Modelica.Blocks.Sources.Constant C0(k=0)
        annotation (extent=[-66,-116; -54,-104]);
      Modelica.Blocks.Sources.Constant qb(k=1)
        annotation (extent=[-46,-116; -34,-104]);
      equation
      connect(B2.BondCon2, J0p3_1.BondCon2) annotation (points=[-30,0; -30,-10],
          style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_1.BondCon2, B2.BondCon1) annotation (points=[-50,0; -49.8,0],
          style(color=8, rgbcolor={192,192,192}));
      connect(B3.BondCon2, J1p3_1.BondCon1) annotation (points=[-70,0; -70,0],
          style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_1.BondCon3, B4.BondCon1) annotation (points=[-60,10; -60,
            10.2], style(color=8, rgbcolor={192,192,192}));
      connect(Rb.BondCon1, B4.BondCon2)  annotation (points=[-60,40; -60,30],
          style(color=8, rgbcolor={192,192,192}));
      connect(Rb.BondCon2, B5.fBondCon1)  annotation (points=[-20,40; -20,40],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_1.BondCon1, B6.BondCon1) annotation (points=[-30,-30; -30,
            -40; -30.2,-40], style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_2.BondCon2, B6.BondCon2) annotation (points=[-50,-40; -50,
            -40], style(color=8, rgbcolor={192,192,192}));
      connect(B7.BondCon1, J1p3_2.BondCon1) annotation (points=[-70.2,-40; -70,
            -40], style(color=8, rgbcolor={192,192,192}));
      connect(J0p2_2.BondCon2, B7.BondCon2) annotation (points=[-100,-40; -90,
            -40],          style(color=8, rgbcolor={192,192,192}));
      connect(B9.BondCon1, J1p3_2.BondCon3) annotation (points=[-60,-50.2; -60,
            -50], style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_3.BondCon1, B11.BondCon1) annotation (points=[20,60; 20,59.9;
            20,59.8; 20,59.8], style(color=8, rgbcolor={192,192,192}));
      connect(B12.BondCon2, J1p3_3.BondCon2) annotation (points=[20,80; 20,80],
          style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_3.BondCon3, B13.BondCon1) annotation (points=[30,70; 30.2,70],
          style(color=8, rgbcolor={192,192,192}));
      connect(B15.BondCon2, J1p3_4.BondCon2) annotation (points=[20,-60; 20,-60],
          style(color=8, rgbcolor={192,192,192}));
      connect(B16.BondCon1, J1p3_4.BondCon1) annotation (points=[20,-80.2; 20,
            -80.1; 20,-80.1; 20,-80], style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_4.BondCon3, B17.BondCon1) annotation (points=[30,-70; 30.2,
            -70], style(color=8, rgbcolor={192,192,192}));
      connect(J0p2_3.BondCon1, B12.BondCon1) annotation (points=[20,100; 20,
            99.9; 20,99.8; 20,99.8], style(color=8, rgbcolor={192,192,192}));
      connect(B16.BondCon2, J0p2_4.BondCon2) annotation (points=[20,-100; 20,
            -100], style(color=8, rgbcolor={192,192,192}));
      connect(J0p2_4.BondCon1, E)        annotation (points=[20,-120; 0,-120],
          style(color=8, rgbcolor={192,192,192}));
      connect(B11.BondCon2, J0p3_2.BondCon2) annotation (points=[20,40; 20,40],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_2.BondCon1, B20.BondCon1) annotation (points=[20,20; 20,19.9;
            20,19.8; 20,19.8], style(color=8, rgbcolor={192,192,192}));
      connect(J0p2_3.BondCon2, C)        annotation (points=[20,120; 0,120],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_3.BondCon2, B)        annotation (points=[-110,1.22461e-015;
            -116,1.22461e-015; -114,0; -120,0], style(color=8, rgbcolor={192,
              192,192}));
      connect(J0p3_3.BondCon1, B3.BondCon1) annotation (points=[-90,
            -1.22461e-015; -90,0; -89.8,0], style(color=8, rgbcolor={192,192,
              192}));
      connect(J0p3_3.BondCon3, B21.BondCon1) annotation (points=[-100,10; -100,
            10.2], style(color=8, rgbcolor={192,192,192}));
      connect(B21.BondCon2, J1p3_5.BondCon1) annotation (points=[-100,30; -100,
            30], style(color=8, rgbcolor={192,192,192}));
      connect(B22.BondCon1, J1p3_5.BondCon2) annotation (points=[-100,50.2;
            -100,50], style(color=8, rgbcolor={192,192,192}));
      connect(B23.BondCon1, J1p3_5.BondCon3) annotation (points=[-80,50.2; -80,
            40; -90,40], style(color=8, rgbcolor={192,192,192}));
      connect(B22.BondCon2, J0p3_2.BondCon3) annotation (points=[-100,70; -100,
            94; 40,94; 40,30; 30,30], style(color=8, rgbcolor={192,192,192}));
      connect(J0p6_1.BondCon2, B25.fBondCon1) annotation (points=[80,30; 80,30],
          style(color=8, rgbcolor={192,192,192}));
      connect(B25.eBondCon1, J0p3_4.BondCon2) annotation (points=[100,30; 100,
            30], style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_4.BondCon1, H)        annotation (points=[120,30; 120,0],
          style(color=8, rgbcolor={192,192,192}));
      connect(B14.eBondCon1, J0p3_4.BondCon3) annotation (points=[110,70; 110,
            40], style(color=8, rgbcolor={192,192,192}));
      connect(B19.eBondCon1, J0p6_1.BondCon1) annotation (points=[60,-20; 60,30],
          style(color=8, rgbcolor={192,192,192}));
      connect(B5.eBondCon1, J0p6_1.BondCon5) annotation (points=[0,40; 0,48; 66,
            48; 66,40], style(color=8, rgbcolor={192,192,192}));
      connect(B24.eBondCon1, J0p6_1.BondCon6) annotation (points=[-20,80; 0,80;
            0,52; 74,52; 74,40], style(color=8, rgbcolor={192,192,192}));
      connect(B18.eBondCon1, J0p6_1.BondCon3) annotation (points=[110,-70; 110,
            -40; 66,-40; 66,20], style(color=8, rgbcolor={192,192,192}));
      connect(B10.eBondCon1, J0p6_1.BondCon4) annotation (points=[0,-80; 8,-80;
            8,-90; 114,-90; 114,-36; 74,-36; 74,20], style(color=8, rgbcolor={
              192,192,192}));
      connect(Qni.qb, Rb.qb)      annotation (points=[41.4286,-33.7143; 52,
            -33.7143; 52,-48; -10,-48; -10,56; -40,56; -40,50], style(color=74,
            rgbcolor={0,0,127}));
      connect(B1.BondCon1, J0p3_1.BondCon3) annotation (points=[-19.8,-20; -20,
            -20], style(color=8, rgbcolor={192,192,192}));
      connect(Cbx.BondCon2, B24.fBondCon1)   annotation (points=[-40,80; -40,80],
          style(color=8, rgbcolor={192,192,192}));
      connect(Cbx.BondCon1, B23.BondCon2)   annotation (points=[-80,80; -80,70],
          style(color=8, rgbcolor={192,192,192}));
      connect(Qni.ix, Cbx.ix)       annotation (points=[10,0; 10,20; -70,20;
            -70,60; -60,60; -60,70], style(color=74, rgbcolor={0,0,127}));
      connect(Rc.BondCon1, B13.BondCon2) annotation (points=[50,70; 50,70],
          style(color=8, rgbcolor={192,192,192}));
      connect(Rc.BondCon2, B14.fBondCon1) annotation (points=[90,70; 90,70],
          style(color=8, rgbcolor={192,192,192}));
      connect(Re.BondCon1, B17.BondCon2) annotation (points=[50,-70; 50,-70],
          style(color=8, rgbcolor={192,192,192}));
      connect(Re.BondCon2, B18.fBondCon1) annotation (points=[90,-70; 90,-70],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p2_2.BondCon1, S)        annotation (points=[-100,-60; -120,-60],
          style(color=8, rgbcolor={192,192,192}));
      connect(Dbs.BondCon2, B10.fBondCon1) annotation (points=[-20,-80; -20,-80],
          style(color=8, rgbcolor={192,192,192}));
      connect(B9.BondCon2, Dbs.BondCon1) annotation (points=[-60,-70; -60,-80],
          style(color=8, rgbcolor={192,192,192}));
      connect(C0.y, Dbs.C0) annotation (points=[-53.4,-110; -50,-110; -50,-94],
          style(color=74, rgbcolor={0,0,127}));
      connect(qb.y, Dbs.qb) annotation (points=[-33.4,-110; -30,-110; -30,-94],
          style(color=74, rgbcolor={0,0,127}));
      connect(B1.BondCon2, Qni.Bint) annotation (points=[0,-20; 0,-20], style(
            color=8, rgbcolor={192,192,192}));
      connect(B15.BondCon1, Qni.Eint) annotation (points=[20,-40.2; 20,-40.1;
            20,-40.1; 20,-40], style(color=8, rgbcolor={192,192,192}));
      connect(B19.fBondCon1, Qni.H) annotation (points=[40,-20; 40,-20], style(
            color=8, rgbcolor={192,192,192}));
      connect(B20.BondCon2, Qni.Cint) annotation (points=[20,0; 20,0], style(
            color=8, rgbcolor={192,192,192}));
      end NPNlat;

      model NPNvert "Spice-style vertically diffused NPN bipolar transistor"
        constant Real inf=Modelica.Constants.inf;

        /* General parameters */
        parameter Integer Level(min=1,max=2) = 1 "Transistor modeling level" annotation(Evaluate=true);
        parameter Real Area=1 "Relative area occupied by device";

        /* DC model parameters */
        parameter Real BF=100
        "Maximum forward current gain at reference temperature"
          annotation(Dialog(group="DC model parameters"));
        parameter Real BR=1
        "Maximum reverse current gain at reference temperature"
          annotation(Dialog(group="DC model parameters"));
        parameter Modelica.SIunits.Current IS=1e-16
        "Saturation current at reference temperature"   annotation(Dialog(group="DC model parameters"));
        parameter Real NF=1 "Forward current emission coefficient"
          annotation(Dialog(group="DC model parameters"));
        parameter Real NR=1 "Reverse current emission coefficient"
          annotation(Dialog(group="DC model parameters"));
        parameter Modelica.SIunits.Current ISS=IS
        "Saturation current used for current injection"   annotation(Dialog(group="DC model parameters", enable=Level == 2));

        /* Low current beta degradation effect parameters */
        parameter Real C4(min=0) = 0
        "Base-collector leakage current coefficient"
          annotation(Dialog(group="Low current beta degradation effect parameters", enable=Level == 2));
        parameter Modelica.SIunits.Current ISC(min=0) = 0
        "Base-collector leakage saturation current at reference temperature (ISC = C4*IS)"
          annotation(Dialog(group="Low current beta degradation effect parameters", enable=Level == 2));
        parameter Real C2(min=0) = 0
        "Base-emitterr leakage current coefficient"
          annotation(Dialog(group="Low current beta degradation effect parameters", enable=Level == 2));
        parameter Modelica.SIunits.Current ISE(min=0) = 0
        "Base-emitter leakage saturation current at reference temperature (ISE = C2*IS)"
          annotation(Dialog(group="Low current beta degradation effect parameters", enable=Level == 2));
        parameter Real NC=2
        "Low-current base-collector leakage emission coefficient"
          annotation(Dialog(group="Low current beta degradation effect parameters", enable=Level == 2));
        parameter Real NE=1.5
        "Low-current base-emitter leakage emission coefficient"
          annotation(Dialog(group="Low current beta degradation effect parameters", enable=Level == 2));

        /* High current beta degradation effect parameters */
        parameter Modelica.SIunits.Current IKF=inf
        "Corner for forward beta high-current roll-off"
          annotation(Dialog(group="High current beta degradation effect", enable=Level == 2));
        parameter Modelica.SIunits.Current IKR=inf
        "Corner for reverse beta high-current roll-off"
          annotation(Dialog(group="High current beta degradation effect", enable=Level == 2));

        /* Base width modulation parameters */
        parameter Modelica.SIunits.Voltage VAF=inf "Forward early voltage"
          annotation(Dialog(group="Base width modulation parameters"));
        parameter Modelica.SIunits.Voltage VAR=inf "Reverse early voltage"
          annotation(Dialog(group="Base width modulation parameters", enable=Level == 2));

        /* Parasitic Resistor Parameters */
        parameter Modelica.SIunits.Current IRB=inf
        "Current where base resistance falls halfway to minimum value"
          annotation(Dialog(group="Parasitic resistor parameters", enable=Level == 2));
        parameter Modelica.SIunits.Resistance RB(min=0) = 0
        "Zero-bias base resistance"   annotation(Dialog(group="Parasitic resistor parameters", enable=Level == 2));
        parameter Modelica.SIunits.Resistance RBM(min=0) = RB
        "Minimum base resistance at high currents"   annotation(Dialog(group="Parasitic resistor parameters", enable=Level == 2));
        parameter Modelica.SIunits.Resistance RC(min=0) = 0
        "Collector resistance"
          annotation(Dialog(group="Parasitic resistor parameters", enable=Level == 2));
        parameter Modelica.SIunits.Resistance RE(min=0) = 0
        "Emitter resistance"
          annotation(Dialog(group="Parasitic resistor parameters", enable=Level == 2));

        /* Junction capacitor parameters */
        parameter Modelica.SIunits.Capacitance CJC(min=Modelica.Constants.small) = 1e-12
        "Zero-bias base-collector depletion capacitance at reference temperature (CJC > 0)"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Real MJC(min=0.3,max=0.5) = 0.33
        "Base-collector junction grading coefficient"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Modelica.SIunits.Voltage VJC=0.75
        "Base-collector built-in potential at reference temperature"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Modelica.SIunits.Capacitance CJE(min=Modelica.Constants.small) = 1e-12
        "Zero-bias base-emitter depletion capacitance at reference temperature (CJE > 0)"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Real MJE(min=0.3,max=0.5) = 0.33
        "Base-emitter junction grading coefficient"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Modelica.SIunits.Voltage VJE=0.75
        "Base-emitter built-in potential at reference temperature"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Modelica.SIunits.Capacitance CJS(min=Modelica.Constants.small) = 1e-12
        "Zero-bias substrate depletion capacitance at reference temperature"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Real MJS(min=0.3,max=0.5) = 0.33
        "Substrate junction grading coefficient"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Modelica.SIunits.Voltage VJS=0.75
        "Substrate built-in potential at reference temperature"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Real XCJC(min=0,max=1) = 1
        "Fraction of base-collector depletion capacitance connected to internal base node"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Real FC(min=0,max=1) = 0.5
        "Depletion capacitance factor for linearization"
          annotation(Dialog(group="Junction capacitor parameters", enable=Level == 2));

        /* Transit time parameters */
        parameter Modelica.SIunits.Time TF(min=0) = 0
        "Ideal forward transit time"
          annotation(Dialog(group="Transit time parameters"));
        parameter Modelica.SIunits.Time TR(min=0) = 0
        "Ideal reverse transit time"
          annotation(Dialog(group="Transit time parameters"));

        /* Temperature compensation and area parameters */
        parameter Modelica.SIunits.Temperature Tnom=300.15
        "Reference temperature"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Real XTI=3 "Saturation current temperature exponent"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Real XTB=0 "Forward and reverse beta temperature coefficient"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Modelica.SIunits.Voltage EG=1.11
        "Energy gap for temperature effect on saturation current"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Real TRB1(unit="1/K") = 0
        "Linear temperature coefficient of zero-bias base resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));
        parameter Real TRB2(unit="1/K2") = 0
        "Quadratic temperature coefficient of zero-bias base resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));
        parameter Real TRM1(unit="1/K") = 0
        "Linear temperature coefficient of minimum base resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));
        parameter Real TRM2(unit="1/K2") = 0
        "Quadratic temperature coefficient of minimum base resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));
        parameter Real TRC1(unit="1/K") = 0
        "Linear temperature coefficient of collector resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));
        parameter Real TRC2(unit="1/K2") = 0
        "Quadratic temperature coefficient of collector resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));
        parameter Real TRE1(unit="1/K") = 0
        "Linear temperature coefficient of emitter resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));
        parameter Real TRE2(unit="1/K2") = 0
        "Quadratic temperature coefficient of emitter resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));

        /* Initialization parameters */
        parameter Modelica.SIunits.Time rampTime=0
        "Substrate potential ramping time (if unconnected)"
          annotation(Dialog(tab="Advanced", group="Initialization parameters"));

        /* Numerical parameters */
        parameter Real EMin=-100
        "if x < EMin, the exp(x) function is linearized"
          annotation(Dialog(tab="Advanced", group="Numerical parameters"));
        parameter Real EMax=40 "if x > EMax, the exp(x) function is linearized"
          annotation(Dialog(tab="Advanced", group="Numerical parameters"));
        parameter Modelica.SIunits.Conductance GminDC(min=0) = 1e-19
        "Leakage conductance"
          annotation(Dialog(tab="Advanced", group="Numerical parameters"));

    protected
        parameter Modelica.SIunits.Current ISCeff = if ISC > 0 then ISC else C4*IS;
        parameter Modelica.SIunits.Current ISEeff = if ISE > 0 then ISE else C2*IS;
    public
      BondLib.Interfaces.BondCon B "Base"
        annotation (extent=[-130,-10; -110,10]);
      BondLib.Interfaces.BondCon E "Emitter"
        annotation (extent=[-10,-130; 10,-110]);
      BondLib.Interfaces.BondCon H "Heat" annotation (extent=[110,-10; 130,10]);
      BondLib.Spice.Utilities.NPNint Qni(
        BF=BF,
        BR=BR,
        IS=IS,
        NF=NF,
        NR=NR,
        GminDC=GminDC,
        NC=NC,
        NE=NE,
        VAF=VAF,
        VAR=VAR,
        IKF=IKF,
        IKR=IKR,
        CJC=CJC,
        MJC=MJC,
        VJC=VJC,
        CJE=CJE,
        MJE=MJE,
        VJE=VJE,
        XCJC=XCJC,
        TF=TF,
        TR=TR,
        Tnom=Tnom,
        XTI=XTI,
        XTB=XTB,
        EG=EG,
        Area=Area,
        EMin=EMin,
        EMax=EMax,
        FC=FC,
        Level=Level,
        ISC=ISCeff,
        ISE=ISEeff,
        ISS=ISS) annotation (extent=[0,-40; 40,0]);
      BondLib.Bonds.Bond B2 annotation (extent=[-50,-10; -30,10], rotation=0);
      BondLib.Junctions.J1p3 J1p3_1 annotation (extent=[-70,-10; -50,10]);
      BondLib.Bonds.Bond B3 annotation (extent=[-90,-10; -70,10], rotation=0);
      BondLib.Bonds.Bond B4 annotation (extent=[-70,-30; -50,-10], rotation=270);
      BondLib.Spice.Utilities.RBS Rb(
        Tnom=Tnom,
        RB=RB,
        RBM=RBM,
        IRB=IRB,
        TRB1=TRB1,
        TRB2=TRB2,
        TRM1=TRM1,
        TRM2=TRM2,
        Area=Area,
        Level=Level)
                   annotation (extent=[-60,-50; -20,-30], rotation=0);
      BondLib.Bonds.eBond B5 annotation (extent=[-20,-50; 0,-30]);
      BondLib.Junctions.J1p3 J1p3_2 annotation (extent=[50,40; 70,20]);
      BondLib.Bonds.Bond B7 annotation (extent=[30,20; 50,40], rotation=180);
      BondLib.Bonds.Bond B9 annotation (extent=[50,40; 70,60], rotation=90);
      BondLib.Bonds.eBond B10 annotation (extent=[110,80; 130,100], rotation=90);
      BondLib.Bonds.Bond B11 annotation (extent=[10,40; 30,60], rotation=270);
      BondLib.Junctions.J1p3 J1p3_3
        annotation (extent=[10,80; 30,60], rotation=-90);
      BondLib.Bonds.Bond B12 annotation (extent=[10,80; 30,100], rotation=270);
      BondLib.Junctions.J0p2 J0p2_3
        annotation (extent=[10,100; 30,120], rotation=90);
      BondLib.Bonds.Bond B13 annotation (extent=[-10,60; 10,80], rotation=180);
      BondLib.Bonds.eBond B14 annotation (extent=[-70,60; -50,80], rotation=180);
      BondLib.Bonds.Bond B15 annotation (extent=[10,-60; 30,-40], rotation=270);
      BondLib.Junctions.J1p3 J1p3_4
        annotation (extent=[10,-60; 30,-80], rotation=-90);
      BondLib.Bonds.Bond B16 annotation (extent=[10,-100; 30,-80], rotation=270);
      BondLib.Junctions.J0p2 J0p2_4
        annotation (extent=[10,-120; 30,-100], rotation=90);
      BondLib.Bonds.Bond B17 annotation (extent=[-10,-80; 10,-60], rotation=180);
      BondLib.Bonds.eBond B18
        annotation (extent=[-70,-80; -50,-60], rotation=180);
      BondLib.Bonds.eBond B19 annotation (extent=[40,-30; 60,-10]);
      BondLib.Junctions.J0p6 J0p6_1 annotation (extent=[60,-100; 80,-80]);
      BondLib.Bonds.Bond B20 annotation (extent=[10,0; 30,20], rotation=270);
      BondLib.Junctions.J0p3 J0p3_3
        annotation (extent=[-110,-10; -90,10], rotation=180);
      BondLib.Bonds.Bond B21 annotation (extent=[-110,10; -90,30], rotation=90);
      BondLib.Junctions.J1p3 J1p3_5
        annotation (extent=[-110,30; -90,50], rotation=90);
      BondLib.Bonds.Bond B22 annotation (extent=[-110,50; -90,70], rotation=90);
      BondLib.Bonds.Bond B23 annotation (extent=[-90,30; -70,50], rotation=0);
      BondLib.Bonds.eBond B24 annotation (extent=[-30,30; -10,50]);
      BondLib.Bonds.eBond B25 annotation (extent=[80,-100; 100,-80]);
      BondLib.Junctions.J0p3 J0p3_4
        annotation (extent=[100,-100; 120,-80], rotation=180);
      BondLib.Interfaces.BondCon C "Collector"
                                          annotation (extent=[-10,110; 10,130]);
      annotation (
        Diagram(
          Text(
            extent=[-126,22; -114,8],
            string="B",
            style(color=10, rgbcolor={95,95,95})),
          Text(
            extent=[-20,128; -8,114],
            style(color=10, rgbcolor={95,95,95}),
            string="C"),
          Text(
            extent=[-20,-114; -8,-128],
            style(color=10, rgbcolor={95,95,95}),
            string="E"),
          Text(
            extent=[92,-6; 118,-22],
            style(color=10, rgbcolor={95,95,95}),
            string="Heat"),
          Text(
            extent=[82,-96; 108,-112],
            style(color=10, rgbcolor={95,95,95}),
            string="Heat"),
          Text(
            extent=[38,-34; 48,-46],
            style(color=10, rgbcolor={95,95,95}),
            string="qb"),
          Text(
            extent=[-38,-14; -28,-26],
            style(color=10, rgbcolor={95,95,95}),
            string="qb"),
          Text(
            extent=[0,10; 10,-2],
            style(color=10, rgbcolor={95,95,95}),
            string="ix"),
          Text(
            extent=[-50,30; -40,18],
            style(color=10, rgbcolor={95,95,95}),
            string="ix"),
          Text(
            extent=[-126,-36; -114,-50],
            style(color=10, rgbcolor={95,95,95}),
            string="S")),
        Coordsys(extent=[-120,-120; 120,120]),
        Documentation(info="<html>
The <b>NPNvert</b> element of the Spice bond graph library implements a full-fledges Spice-stye Gummel-Poon model of the vertically diffused NPN bipolar transistor [1-3]. The model contains a <b>NPNint</b> model implementing the inner parts of the NPN transistor up to the internal nodes. <p>

The NPN bipolar transistor is a directed FourPort.  The direction of positive power flow is assumed into the model at the base, <b>B</b>, and at the collector, <b>C</b>, whereas it is assumed out of the model at the emitter, <b>E</b>, and at the <b>Heat</b> port. <p>

The causality of the NPN model is free. <p>

<hr>

<pre>
<b>Parameters:</b>

 <font color=red><b>Level:   Transistor modeling level (default value = 2)
            Level = 1: Ebers-Moll model
            Level = 2: Gummel-Poon model</b></font>

 Area:    Relative area occupied by device (default value = 1) <font color=red>Levels 1,2</font>


<b>DC Model Parameters:</b>

 BF:      Maximum forward current gain at reference temperature (default value = 100) <font color=red>Levels 1,2</font>

 BR:      Maximum reverse current gain at reference temperature (default value = 1) <font color=red>Levels 1,2</font>

 IS:      Saturation current at reference temperature (default value = 1e-16 Amp) <font color=red>Levels 1,2</font>

 NF:      Forward current emission coefficient (default value = 1) <font color=red>Levels 1,2</font>

 NR:      Reverse current emission coefficient (default value = 1) <font color=red>Levels 1,2</font>

 ISS:     Saturation current for injection (default value = IS Amp) <font color=red>Levels 2</font>


<b>Low Current Beta Degradation Effect Parameters:</b>

 C4:      Base-collector leakage current coefficient (default value = 0) <font color=red>Levels 2</font>

 ISC:     Base-collector leakage saturation current at reference temperature (default value = 0 Amp) <font color=red>Levels 2</font>
          <font color=blue><b>ISCeff = if ISC &gt; 0 then ISC else C4*IS</b></font>

 C2:      Base-emitter leakage current coefficient (default value = 0) <font color=red>Levels 2</font>

 ISE:     Base-emitter leakage saturation current at reference temperature (default value = 0 Amp) <font color=red>Levels 2</font>
          <font color=blue><b>ISEeff = if ISE &gt; 0 then ISE else C2*IS</b></font>

 NC:      Low-current base-collector leakage emission coefficient (default value = 2) <font color=red>Levels 2</font>

 NE:      Low-current base-emitter leakage emission coefficient (default value = 1.5) <font color=red>Levels 2</font>


<b>High Current Beta Degradation Effect Parameters:</b>

 IKF:     Corner for forward beta high-current roll-off (default value = &infin; Amp) <font color=red>Levels 2</font>

 IKR:     Corner for reverse beta high-current roll-off (default value = &infin; Amp) <font color=red>Levels 2</font>


<b>Base Width Modulation Parameters:</b>

 VAF:     Forward early voltage (default value = &infin; Volt) <font color=red>Levels 1,2</font>

 VAR:     Reverse early voltage (default value = &infin; Volt) <font color=red>Levels 2</font>


<b>Parasitic Resistor Parameters:</b>

 IRB:     Current where base resistance falls halfway to minimum value (default value = &infin; Amp) <font color=red>Levels 2</font>

 RB:      Zero-bias base resistance (default value = 0 Ohm) <font color=red>Levels 2</font>

 RBM:     Minimum base resistance at high currents (default value = RB Ohm) <font color=red>Levels 2</font>

 RC:      Collector resistance (default value = 0 Ohm) <font color=red>Levels 2</font>

 RE:      Emitter resistance (default value = 0 Ohm) <font color=red>Levels 2</font>


<b>Junction Capacitor Parameters:</b>

 CJC:     Zero-bias base-collector depletion capacitance at reference temperature (CJC > 0) (default value = 1e-12 F) <font color=red>Levels 1,2</font>

 MJC:     Base-collector junction grading coefficient (default value = 0.33) <font color=red>Levels 1,2</font>

 VJC:     Base-collector built-in potential at reference temperature (default value = 0.75 Volt) <font color=red>Levels 1,2</font>

 CJE:     Zero-bias base-emitter depletion capacitance at reference temperature (CJE > 0) (default value = 1e-12 F) <font color=red>Levels 1,2</font>

 MJE:     Base-emitter junction grading coefficient (default value = 0.33) <font color=red>Levels 1,2</font>

 VJE:     Base-emitter built-in potential at reference temperature (default value = 0.75 Volt) <font color=red>Levels 1,2</font>

 CJS:     Zero-bias substrate depletion capacitance at reference temperature (CJS > 0) (default value = 1e-12 F) <font color=red>Levels 1,2</font>

 MJS:     Substrate junction grading coefficient (default value = 0.33) <font color=red>Levels 1,2</font>

 VJS:     Substrate built-in potential at reference temperature (default value = 0.75 Volt) <font color=red>Levels 1,2</font>

 XCJC:    Fraction of base-collector depletion capacitance connected to internal base node (default value = 1) <font color=red>Levels 1,2</font>

 FC:      Depletion capacitance factor for linearization (default value = 0.5) <font color=red>Levels 2</font>


<b>Transit Time Parameters:</b>

 TF:      Ideal forward transit time (default value = 0 sec) <font color=red>Levels 1,2</font>

 TR:      Ideal reverse transit time (default value = 0 sec) <font color=red>Levels 1,2</font>


<b>Temperature Compensation and Area Parameters:</b>

 Tnom:    Reference temperature (default value = 300.15 K) <font color=red>Levels 1,2</font>

 XTI:     Saturation current temperature exponent (default value = 3) <font color=red>Levels 1,2</font>

 XTB:     Forward and reverse beta temperature coefficient (default value = 0) <font color=red>Levels 1,2</font>

 EG:      Energy gap for temperature effect on saturation current (default value = 1.11 Volt) <font color=red>Levels 1,2</font>

 TRB1:    Linear temperature coefficient of zero-bias base resistance (default value = 0 1/K) <font color=red>Levels 2</font>

 TRB2:    Quadratic temperature coefficient of zero-bias base resistance (default value = 0 1/(K<sup>2</sup>)) <font color=red>Levels 2</font>

 TRM1:    Linear temperature coefficient of mimimum base resistance (default value = 0 1/K) <font color=red>Levels 2</font>

 TRM2:    Quadratic temperature coefficient of minimum base resistance (default value = 0 1/(K<sup>2</sup>)) <font color=red>Levels 2</font>

 TRC1:    Linear temperature coefficient of collector resistance (default value = 0 1/K) <font color=red>Levels 2</font>

 TRC2:    Quadratic temperature coefficient of collector resistance (default value = 0 1/(K<sup>2</sup>)) <font color=red>Levels 2</font>

 TRE1:    Linear temperature coefficient of emitter resistance (default value = 0 1/K) <font color=red>Levels 2</font>

 TRE2:    Quadratic temperature coefficient of emitter resistance (default value = 0 1/(K<sup>2</sup>)) <font color=red>Levels 2</font>


<b>Numerical Parameters:</b>

 EMin:    Minimum exponent for linearization of junction current (default value = -100) <font color=red>Levels 1,2</font>

 EMax:    Maximum exponent for linearization of junction current (default value = 40) <font color=red>Levels 1,2</font>

 GminDC:  Leakage conductance (default value = 1e-19 Mho) <font color=red>Levels 1,2</font>
</pre> <p>

<hr> <p>

<b>References:</b> <p>

<ol>
<li>Cellier, F.E. (1991), <i>Continuous System Modeling</i>, Springer-Verlag, New York.
<li>Hild, D.R. and F.E. Cellier (1994), \"Object-oriented electronic circuit modeling using Dymola,\"
    <i>Proc. OOS'94, SCS Object Oriented Simulation Conference</i>, Tempe, AZ, pp.68-75.
<li>Hild, D.R. (1993), <i>Circuit Modeling in Dymola</i>, MS Thesis, Dept. of Electr. &amp; Comp. Engr.,
    University of Arizona, Tucson.
<li> Massobrio, G. and P. Antognetti (1993), <i>Semiconductor Device Modeling with Spice</i>,
     2<sup>nd</sup> edition, McGraw Hill, New York.
<li>Schweisguth, M.C. and F.E. Cellier (1999), \"<a href=\"http://www.inf.ethz.ch/~fcellier/Publikationen/BJT.pdf\">A
    bond graph model of the bipolar junction transistor</a>,\" <i>Proc. SCS Intl. Conf. on Bond Graph Modeling</i>,
    San Francisco, CA, pp.344-349.
<li>Schweisguth, M.C. (1997), <a href=\"http://www.inf.ethz.ch/~fcellier/Publikationen/schweisguth_ms.pdf\"><i>Semiconductor
    Modeling with Bondgraphs</i></a>, MS Thesis, Dept. of Electr. &amp; Comp. Engr., University of Arizona, Tucson.
</ol>
</html>
"),     Icon(
            Text(
              extent=[-102,84; 62,-92],
              style(color=0),
              string="NPN"),
            Line(points=[0,100; 0,-100],
                                       style(
                color=9,
                rgbcolor={149,164,171},
                thickness=2,
                fillColor=74,
                rgbfillColor={0,0,185},
                fillPattern=1)),
            Line(points=[-20,-80; 0,-100],style(
                color=9,
                rgbcolor={149,164,171},
                thickness=2,
                fillColor=74,
                rgbfillColor={0,0,185},
                fillPattern=1)),
            Line(points=[0,-100; 20,-80],style(
                color=9,
                rgbcolor={149,164,171},
                thickness=2,
                fillColor=74,
                rgbfillColor={0,0,185},
                fillPattern=1)),
          Text(
            extent=[-88,-72; -44,-108],
            string="%Area x",
            style(color=3, rgbcolor={0,0,255})),
          Line(points=[80,-80; 100,-60], style(color=3, rgbcolor={0,0,255})),
          Line(points=[80,-40; 100,-60], style(color=3, rgbcolor={0,0,255})),
          Line(points=[-100,-60; 100,-60], style(color=3, rgbcolor={0,0,255})),
            Text(
              extent=[58,4; 110,-42],
              style(color=0),
            string="vert"),
          Text(
            extent=[-80,100; 80,60],
            style(color=3, rgbcolor={0,0,255}),
            string="%name"),
          Text(
            extent=[-126,24; -114,10],
            string="B",
            style(color=10, rgbcolor={95,95,95})),
          Text(
            extent=[10,126; 22,112],
            style(color=10, rgbcolor={95,95,95}),
            string="C"),
          Text(
            extent=[106,26; 132,6],
            style(color=10, rgbcolor={95,95,95}),
            string="Heat"),
          Text(
            extent=[8,-114; 20,-128],
            style(color=10, rgbcolor={95,95,95}),
            string="E")));
      BondLib.Junctions.J0p4 J0p4_1 annotation (extent=[10,20; 30,40]);
      Utilities.CbxS Cbx   annotation (extent=[-70,30; -30,50]);
      Utilities.RS2 Rc(
        Tnom=Tnom,
        R=RC,
        TR1=TRC1,
        TR2=TRC2,
        Area=Area,
        Level=Level) annotation (extent=[-10,58; -50,82]);
      Utilities.RS2 Re(
        Tnom=Tnom,
        Area=Area,
        Level=Level,
        R=RE,
        TR1=TRE1,
        TR2=TRE2) annotation (extent=[-10,-82; -50,-58]);
      Junctions.J0p2 J0p2_2
        annotation (extent=[90,20; 110,40],     rotation=180);
      BondLib.Bonds.Bond B8 annotation (extent=[70,20; 90,40], rotation=180);
      BondLib.Interfaces.BondCon S "Substrate"
        annotation (extent=[-130,-70; -110,-50]);
      Utilities.CjS Dsc(
        Tnom=Tnom,
        XTI=XTI,
        XTB=XTB,
        EG=EG,
        Area=Area,
        Level=Level,
        BE=false,
        EMin=EMin,
        EMax=EMax,
        MJ=MJS,
        VJ=VJS,
        XCJ=1,
        Tau=0,
        FC=0,
        IS=0,
        GminDC=0,
        CJ=CJS,
        N=1)       annotation (extent=[70,66; 110,94]);
      Modelica.Blocks.Sources.Constant C0(k=0)
        annotation (extent=[96,44; 84,56]);
      Modelica.Blocks.Sources.Constant qb(k=1)
        annotation (extent=[116,44; 104,56]);
      equation
      connect(J1p3_1.BondCon2, B2.BondCon1) annotation (points=[-50,0; -49.8,0],
          style(color=8, rgbcolor={192,192,192}));
      connect(B3.BondCon2, J1p3_1.BondCon1) annotation (points=[-70,0; -70,0],
          style(color=8, rgbcolor={192,192,192}));
      connect(Rb.BondCon2, B5.fBondCon1)  annotation (points=[-20,-40; -20,-40],
          style(color=8, rgbcolor={192,192,192}));
      connect(B7.BondCon1, J1p3_2.BondCon1) annotation (points=[49.8,30; 50,30],
          style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_3.BondCon1, B11.BondCon1) annotation (points=[20,60; 20,59.9;
            20,59.8; 20,59.8], style(color=8, rgbcolor={192,192,192}));
      connect(B12.BondCon2, J1p3_3.BondCon2) annotation (points=[20,80; 20,80],
          style(color=8, rgbcolor={192,192,192}));
      connect(B15.BondCon2, J1p3_4.BondCon2) annotation (points=[20,-60; 20,-60],
          style(color=8, rgbcolor={192,192,192}));
      connect(B16.BondCon1, J1p3_4.BondCon1) annotation (points=[20,-80.2; 20,
            -80.1; 20,-80.1; 20,-80], style(color=8, rgbcolor={192,192,192}));
      connect(J0p2_3.BondCon1, B12.BondCon1) annotation (points=[20,100; 20,
            99.9; 20,99.8; 20,99.8], style(color=8, rgbcolor={192,192,192}));
      connect(B16.BondCon2, J0p2_4.BondCon2) annotation (points=[20,-100; 20,
            -100], style(color=8, rgbcolor={192,192,192}));
      connect(J0p2_4.BondCon1, E)        annotation (points=[20,-120; 0,-120],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p2_3.BondCon2, C)        annotation (points=[20,120; 0,120],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_3.BondCon2, B)        annotation (points=[-110,1.22461e-015;
            -115,1.22461e-015; -115,0; -120,0], style(color=8, rgbcolor={192,
              192,192}));
      connect(J0p3_3.BondCon1, B3.BondCon1) annotation (points=[-90,
            -1.22461e-015; -89.9,-1.22461e-015; -89.9,0; -89.8,0], style(color=
              8, rgbcolor={192,192,192}));
      connect(J0p3_3.BondCon3, B21.BondCon1) annotation (points=[-100,10; -100,
            10.2], style(color=8, rgbcolor={192,192,192}));
      connect(B21.BondCon2, J1p3_5.BondCon1) annotation (points=[-100,30; -100,
            30], style(color=8, rgbcolor={192,192,192}));
      connect(B22.BondCon1, J1p3_5.BondCon2) annotation (points=[-100,50.2;
            -100,50], style(color=8, rgbcolor={192,192,192}));
      connect(J0p6_1.BondCon2, B25.fBondCon1) annotation (points=[80,-90; 80,
            -90], style(color=8, rgbcolor={192,192,192}));
      connect(B25.eBondCon1, J0p3_4.BondCon2) annotation (points=[100,-90; 100,
            -90], style(color=8, rgbcolor={192,192,192}));
      connect(B4.BondCon1, J1p3_1.BondCon3) annotation (points=[-60,-10.2; -60,
            -10], style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_5.BondCon3, B23.BondCon1) annotation (points=[-90,40; -89.8,
            40], style(color=8, rgbcolor={192,192,192}));
      connect(J0p4_1.BondCon4, B11.BondCon2) annotation (points=[20,40; 20,40],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p4_1.BondCon3, B20.BondCon1) annotation (points=[20,20; 25,20;
            25,19.8; 20,19.8], style(color=8, rgbcolor={192,192,192}));
      connect(B4.BondCon2, Rb.BondCon1)  annotation (points=[-60,-30; -60,-40],
          style(color=8, rgbcolor={192,192,192}));
      connect(Qni.qb, Rb.qb)      annotation (points=[41.4286,-33.7143; 50,
            -33.7143; 50,-48; -10,-48; -10,-24; -40,-24; -40,-30], style(color=
              74, rgbcolor={0,0,127}));
      connect(B13.BondCon1, J1p3_3.BondCon3) annotation (points=[9.8,70; 10,70],
          style(color=8, rgbcolor={192,192,192}));
      connect(B22.BondCon2, J0p4_1.BondCon1) annotation (points=[-100,70; -100,
            88; 0,88; 0,30; 10,30], style(color=8, rgbcolor={192,192,192}));
      connect(J0p4_1.BondCon2, B7.BondCon2) annotation (points=[30,30; 30,30],
          style(color=8, rgbcolor={192,192,192}));
      connect(B9.BondCon1, J1p3_2.BondCon3) annotation (points=[60,40.2; 60,40],
          style(color=8, rgbcolor={192,192,192}));
      connect(B17.BondCon1, J1p3_4.BondCon3) annotation (points=[9.8,-70; 10,
            -70], style(color=8, rgbcolor={192,192,192}));
      connect(B10.eBondCon1, J0p3_4.BondCon1) annotation (points=[120,100; 120,
            108; 130,108; 130,-90; 120,-90], style(color=8, rgbcolor={192,192,
              192}));
      connect(B5.eBondCon1, J0p6_1.BondCon1) annotation (points=[0,-40; 8,-40;
            8,-52; 60,-52; 60,-90], style(color=8, rgbcolor={192,192,192}));
      connect(B19.eBondCon1, J0p6_1.BondCon5) annotation (points=[60,-20; 66,
            -20; 66,-80], style(color=8, rgbcolor={192,192,192}));
      connect(B24.eBondCon1, J0p6_1.BondCon6) annotation (points=[-10,40; -6,40;
            -6,12; 74,12; 74,-80], style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_4.BondCon3, H)        annotation (points=[110,-80; 120,-80;
            120,0], style(color=8, rgbcolor={192,192,192}));
      connect(B18.eBondCon1, J0p6_1.BondCon3) annotation (points=[-70,-70; -76,
            -70; -76,-88; 48,-88; 48,-100; 66,-100], style(color=8, rgbcolor={
              192,192,192}));
      connect(B14.eBondCon1, J0p6_1.BondCon4) annotation (points=[-70,70; -80,
            70; -80,-92; 44,-92; 44,-104; 74,-104; 74,-100], style(color=8,
            rgbcolor={192,192,192}));
      connect(Cbx.BondCon1, B23.BondCon2)   annotation (points=[-70,40; -70,40],
          style(color=8, rgbcolor={192,192,192}));
      connect(Cbx.BondCon2, B24.fBondCon1)   annotation (points=[-30,40; -30,40],
          style(color=8, rgbcolor={192,192,192}));
      connect(Qni.ix, Cbx.ix)       annotation (points=[10,0; 10,10; -50,10;
            -50,30], style(color=74, rgbcolor={0,0,127}));
      connect(Rc.BondCon1, B13.BondCon2) annotation (points=[-10,70; -10,70],
          style(color=8, rgbcolor={192,192,192}));
      connect(Rc.BondCon2, B14.fBondCon1) annotation (points=[-50,70; -50,70],
          style(color=8, rgbcolor={192,192,192}));
      connect(Re.BondCon1, B17.BondCon2) annotation (points=[-10,-70; -10,-70],
          style(color=8, rgbcolor={192,192,192}));
      connect(Re.BondCon2, B18.fBondCon1) annotation (points=[-50,-70; -50,-70],
          style(color=8, rgbcolor={192,192,192}));
      connect(B8.BondCon2, J1p3_2.BondCon2) annotation (points=[70,30; 70,30],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p2_2.BondCon2, B8.BondCon1) annotation (points=[90,30; 89.8,30],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p2_2.BondCon1, S)        annotation (points=[110,30; 134,30;
            134,-130; -120,-130; -120,-60], style(color=8, rgbcolor={192,192,
              192}));
      connect(C0.y, Dsc.C0) annotation (points=[83.4,50; 80,50; 80,66], style(
            color=74, rgbcolor={0,0,127}));
      connect(qb.y, Dsc.qb) annotation (points=[103.4,50; 100,50; 100,66],
          style(color=74, rgbcolor={0,0,127}));
      connect(Dsc.BondCon2, B10.fBondCon1) annotation (points=[110,80; 120,80],
          style(color=8, rgbcolor={192,192,192}));
      connect(Dsc.BondCon1, B9.BondCon2) annotation (points=[70,80; 60,80; 60,
            60], style(color=8, rgbcolor={192,192,192}));
      connect(Qni.Bint, B2.BondCon2) annotation (points=[0,-20; -20,-20; -20,0;
            -30,0], style(color=8, rgbcolor={192,192,192}));
      connect(B15.BondCon1, Qni.Eint) annotation (points=[20,-40.2; 20,-40.1;
            20,-40.1; 20,-40], style(color=8, rgbcolor={192,192,192}));
      connect(B19.fBondCon1, Qni.H) annotation (points=[40,-20; 40,-20], style(
            color=8, rgbcolor={192,192,192}));
      connect(B20.BondCon2, Qni.Cint) annotation (points=[20,0; 20,0], style(
            color=8, rgbcolor={192,192,192}));
      end NPNvert;

      model PNPlat "Spice-style laterally diffused PNP bipolar transistor"
        constant Real inf=Modelica.Constants.inf;

        /* General parameters */
        parameter Integer Level(min=1,max=2) = 1 "Transistor modeling level" annotation(Evaluate=true);
        parameter Real Area=1 "Relative area occupied by device";

        /* DC model parameters */
        parameter Real BF=100
        "Maximum forward current gain at reference temperature"
          annotation(Dialog(group="DC model parameters"));
        parameter Real BR=1
        "Maximum reverse current gain at reference temperature"
          annotation(Dialog(group="DC model parameters"));
        parameter Modelica.SIunits.Current IS=1e-16
        "Saturation current at reference temperature"   annotation(Dialog(group="DC model parameters"));
        parameter Real NF=1 "Forward current emission coefficient"
          annotation(Dialog(group="DC model parameters"));
        parameter Real NR=1 "Reverse current emission coefficient"
          annotation(Dialog(group="DC model parameters"));
        parameter Modelica.SIunits.Current ISS=IS
        "Saturation current used for current injection"   annotation(Dialog(group="DC model parameters", enable=Level == 2));

        /* Low current beta degradation effect parameters */
        parameter Real C4(min=0) = 0
        "Base-collector leakage current coefficient"
          annotation(Dialog(group="Low current beta degradation effect parameters", enable=Level == 2));
        parameter Modelica.SIunits.Current ISC(min=0) = 0
        "Base-collector leakage saturation current at reference temperature (ISC = C4*IS)"
          annotation(Dialog(group="Low current beta degradation effect parameters", enable=Level == 2));
        parameter Real C2(min=0) = 0
        "Base-emitterr leakage current coefficient"
          annotation(Dialog(group="Low current beta degradation effect parameters", enable=Level == 2));
        parameter Modelica.SIunits.Current ISE(min=0) = 0
        "Base-emitter leakage saturation current at reference temperature (ISE = C2*IS)"
          annotation(Dialog(group="Low current beta degradation effect parameters", enable=Level == 2));
        parameter Real NC=2
        "Low-current base-collector leakage emission coefficient"
          annotation(Dialog(group="Low current beta degradation effect parameters", enable=Level == 2));
        parameter Real NE=1.5
        "Low-current base-emitter leakage emission coefficient"
          annotation(Dialog(group="Low current beta degradation effect parameters", enable=Level == 2));

        /* High current beta degradation effect parameters */
        parameter Modelica.SIunits.Current IKF=inf
        "Corner for forward beta high-current roll-off"
          annotation(Dialog(group="High current beta degradation effect", enable=Level == 2));
        parameter Modelica.SIunits.Current IKR=inf
        "Corner for reverse beta high-current roll-off"
          annotation(Dialog(group="High current beta degradation effect", enable=Level == 2));

        /* Base width modulation parameters */
        parameter Modelica.SIunits.Voltage VAF=inf "Forward early voltage"
          annotation(Dialog(group="Base width modulation parameters"));
        parameter Modelica.SIunits.Voltage VAR=inf "Reverse early voltage"
          annotation(Dialog(group="Base width modulation parameters", enable=Level == 2));

        /* Parasitic Resistor Parameters */
        parameter Modelica.SIunits.Current IRB=inf
        "Current where base resistance falls halfway to minimum value"
          annotation(Dialog(group="Parasitic resistor parameters", enable=Level == 2));
        parameter Modelica.SIunits.Resistance RB(min=0) = 0
        "Zero-bias base resistance"   annotation(Dialog(group="Parasitic resistor parameters", enable=Level == 2));
        parameter Modelica.SIunits.Resistance RBM(min=0) = RB
        "Minimum base resistance at high currents"   annotation(Dialog(group="Parasitic resistor parameters", enable=Level == 2));
        parameter Modelica.SIunits.Resistance RC(min=0) = 0
        "Collector resistance"
          annotation(Dialog(group="Parasitic resistor parameters", enable=Level == 2));
        parameter Modelica.SIunits.Resistance RE(min=0) = 0
        "Emitter resistance"
          annotation(Dialog(group="Parasitic resistor parameters", enable=Level == 2));

        /* Junction capacitor parameters */
        parameter Modelica.SIunits.Capacitance CJC(min=Modelica.Constants.small) = 1e-12
        "Zero-bias base-collector depletion capacitance at reference temperature (CJC > 0)"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Real MJC(min=0.3,max=0.5) = 0.33
        "Base-collector junction grading coefficient"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Modelica.SIunits.Voltage VJC=0.75
        "Base-collector built-in potential at reference temperature"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Modelica.SIunits.Capacitance CJE(min=Modelica.Constants.small) = 1e-12
        "Zero-bias base-emitter depletion capacitance at reference temperature (CJE > 0)"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Real MJE(min=0.3,max=0.5) = 0.33
        "Base-emitter junction grading coefficient"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Modelica.SIunits.Voltage VJE=0.75
        "Base-emitter built-in potential at reference temperature"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Modelica.SIunits.Capacitance CJS(min=Modelica.Constants.small) = 1e-12
        "Zero-bias substrate depletion capacitance at reference temperature"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Real MJS(min=0.3,max=0.5) = 0.33
        "Substrate junction grading coefficient"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Modelica.SIunits.Voltage VJS=0.75
        "Substrate built-in potential at reference temperature"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Real XCJC(min=0,max=1) = 1
        "Fraction of base-collector depletion capacitance connected to internal base node"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Real FC(min=0,max=1) = 0.5
        "Depletion capacitance factor for linearization"
          annotation(Dialog(group="Junction capacitor parameters", enable=Level == 2));

        /* Transit time parameters */
        parameter Modelica.SIunits.Time TF(min=0) = 0
        "Ideal forward transit time"
          annotation(Dialog(group="Transit time parameters"));
        parameter Modelica.SIunits.Time TR(min=0) = 0
        "Ideal reverse transit time"
          annotation(Dialog(group="Transit time parameters"));

        /* Temperature compensation and area parameters */
        parameter Modelica.SIunits.Temperature Tnom=300.15
        "Reference temperature"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Real XTI=3 "Saturation current temperature exponent"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Real XTB=0 "Forward and reverse beta temperature coefficient"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Modelica.SIunits.Voltage EG=1.11
        "Energy gap for temperature effect on saturation current"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Real TRB1(unit="1/K") = 0
        "Linear temperature coefficient of zero-bias base resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));
        parameter Real TRB2(unit="1/K2") = 0
        "Quadratic temperature coefficient of zero-bias base resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));
        parameter Real TRM1(unit="1/K") = 0
        "Linear temperature coefficient of minimum base resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));
        parameter Real TRM2(unit="1/K2") = 0
        "Quadratic temperature coefficient of minimum base resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));
        parameter Real TRC1(unit="1/K") = 0
        "Linear temperature coefficient of collector resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));
        parameter Real TRC2(unit="1/K2") = 0
        "Quadratic temperature coefficient of collector resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));
        parameter Real TRE1(unit="1/K") = 0
        "Linear temperature coefficient of emitter resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));
        parameter Real TRE2(unit="1/K2") = 0
        "Quadratic temperature coefficient of emitter resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));

        /* Initialization parameters */
        parameter Modelica.SIunits.Time rampTime=0
        "Substrate potential ramping time (if unconnected)"
          annotation(Dialog(tab="Advanced", group="Initialization parameters"));

        /* Numerical parameters */
        parameter Real EMin=-100
        "if x < EMin, the exp(x) function is linearized"
          annotation(Dialog(tab="Advanced", group="Numerical parameters"));
        parameter Real EMax=40 "if x > EMax, the exp(x) function is linearized"
          annotation(Dialog(tab="Advanced", group="Numerical parameters"));
        parameter Modelica.SIunits.Conductance GminDC(min=0) = 1e-19
        "Leakage conductance"
          annotation(Dialog(tab="Advanced", group="Numerical parameters"));

    protected
        parameter Modelica.SIunits.Current ISCeff = if ISC > 0 then ISC else C4*IS;
        parameter Modelica.SIunits.Current ISEeff = if ISE > 0 then ISE else C2*IS;
      annotation (
        Coordsys(extent=[-120,-120; 120,120]),
        Diagram(
          Text(
            extent=[-126,22; -114,8],
            string="B",
            style(color=10, rgbcolor={95,95,95})),
          Text(
            extent=[-20,128; -8,114],
            style(color=10, rgbcolor={95,95,95}),
            string="C"),
          Text(
            extent=[-20,-114; -8,-128],
            style(color=10, rgbcolor={95,95,95}),
            string="E"),
          Text(
            extent=[108,-6; 134,-22],
            style(color=10, rgbcolor={95,95,95}),
            string="Heat"),
          Text(
            extent=[78,50; 104,34],
            style(color=10, rgbcolor={95,95,95}),
            string="Heat"),
          Text(
            extent=[42,-34; 52,-46],
            style(color=10, rgbcolor={95,95,95}),
            string="qb"),
          Text(
            extent=[-28,66; -18,54],
            style(color=10, rgbcolor={95,95,95}),
            string="qb"),
          Text(
            extent=[-60,70; -50,58],
            style(color=10, rgbcolor={95,95,95}),
            string="ix"),
          Text(
            extent=[0,16; 10,4],
            style(color=10, rgbcolor={95,95,95}),
            string="ix"),
          Text(
            extent=[-126,-38; -114,-52],
            style(color=10, rgbcolor={95,95,95}),
            string="S")),
        Documentation(info="<html>
The <b>PNPlat</b> element of the Spice bond graph library implements a full-fledges Spice-stye Gummel-Poon model of the laterally diffused PNP bipolar transistor [1-3]. The model contains a <b>PNPint</b> model implementing the inner parts of the PNP transistor up to the internal nodes. <p>

The PNP bipolar transistor is a directed FourPort.  The direction of positive power flow is assumed into the model at the emitter, <b>E</b>, whereas it is assumed out of the model at the base, <b>B</b>, at the collector, <b>C</b>, and at the <b>Heat</b> port. <p>

The causality of the PNP model is free. <p>

<hr>

<pre>
<b>Parameters:</b>

 <font color=red><b>Level:   Transistor modeling level (default value = 2)
            Level = 1: Ebers-Moll model
            Level = 2: Gummel-Poon model</b></font>

 Area:    Relative area occupied by device (default value = 1) <font color=red>Levels 1,2</font>


<b>DC Model Parameters:</b>

 BF:      Maximum forward current gain at reference temperature (default value = 100) <font color=red>Levels 1,2</font>

 BR:      Maximum reverse current gain at reference temperature (default value = 1) <font color=red>Levels 1,2</font>

 IS:      Saturation current at reference temperature (default value = 1e-16 Amp) <font color=red>Levels 1,2</font>

 NF:      Forward current emission coefficient (default value = 1) <font color=red>Levels 1,2</font>

 NR:      Reverse current emission coefficient (default value = 1) <font color=red>Levels 1,2</font>

 ISS:     Saturation current for injection (default value = IS Amp) <font color=red>Levels 2</font>


<b>Low Current Beta Degradation Effect Parameters:</b>

 C4:      Base-collector leakage current coefficient (default value = 0) <font color=red>Levels 2</font>

 ISC:     Base-collector leakage saturation current at reference temperature (default value = 0 Amp) <font color=red>Levels 2</font>
          <font color=blue><b>ISCeff = if ISC &gt; 0 then ISC else C4*IS</b></font>

 C2:      Base-emitter leakage current coefficient (default value = 0) <font color=red>Levels 2</font>

 ISE:     Base-emitter leakage saturation current at reference temperature (default value = 0 Amp) <font color=red>Levels 2</font>
          <font color=blue><b>ISEeff = if ISE &gt; 0 then ISE else C2*IS</b></font>

 NC:      Low-current base-collector leakage emission coefficient (default value = 2) <font color=red>Levels 2</font>

 NE:      Low-current base-emitter leakage emission coefficient (default value = 1.5) <font color=red>Levels 2</font>


<b>High Current Beta Degradation Effect Parameters:</b>

 IKF:     Corner for forward beta high-current roll-off (default value = &infin; Amp) <font color=red>Levels 2</font>

 IKR:     Corner for reverse beta high-current roll-off (default value = &infin; Amp) <font color=red>Levels 2</font>


<b>Base Width Modulation Parameters:</b>

 VAF:     Forward early voltage (default value = &infin; Volt) <font color=red>Levels 1,2</font>

 VAR:     Reverse early voltage (default value = &infin; Volt) <font color=red>Levels 2</font>


<b>Parasitic Resistor Parameters:</b>

 IRB:     Current where base resistance falls halfway to minimum value (default value = &infin; Amp) <font color=red>Levels 2</font>

 RB:      Zero-bias base resistance (default value = 0 Ohm) <font color=red>Levels 2</font>

 RBM:     Minimum base resistance at high currents (default value = RB Ohm) <font color=red>Levels 2</font>

 RC:      Collector resistance (default value = 0 Ohm) <font color=red>Levels 2</font>

 RE:      Emitter resistance (default value = 0 Ohm) <font color=red>Levels 2</font>


<b>Junction Capacitor Parameters:</b>

 CJC:     Zero-bias base-collector depletion capacitance at reference temperature (CJC > 0) (default value = 1e-12 F) <font color=red>Levels 1,2</font>

 MJC:     Base-collector junction grading coefficient (default value = 0.33) <font color=red>Levels 1,2</font>

 VJC:     Base-collector built-in potential at reference temperature (default value = 0.75 Volt) <font color=red>Levels 1,2</font>

 CJE:     Zero-bias base-emitter depletion capacitance at reference temperature (CJE > 0) (default value = 1e-12 F) <font color=red>Levels 1,2</font>

 MJE:     Base-emitter junction grading coefficient (default value = 0.33) <font color=red>Levels 1,2</font>

 VJE:     Base-emitter built-in potential at reference temperature (default value = 0.75 Volt) <font color=red>Levels 1,2</font>

 CJS:     Zero-bias substrate depletion capacitance at reference temperature (CJS > 0) (default value = 1e-12 F) <font color=red>Levels 1,2</font>

 MJS:     Substrate junction grading coefficient (default value = 0.33) <font color=red>Levels 1,2</font>

 VJS:     Substrate built-in potential at reference temperature (default value = 0.75 Volt) <font color=red>Levels 1,2</font>

 XCJC:    Fraction of base-collector depletion capacitance connected to internal base node (default value = 1) <font color=red>Levels 1,2</font>

 FC:      Depletion capacitance factor for linearization (default value = 0.5) <font color=red>Levels 2</font>


<b>Transit Time Parameters:</b>

 TF:      Ideal forward transit time (default value = 0 sec) <font color=red>Levels 1,2</font>

 TR:      Ideal reverse transit time (default value = 0 sec) <font color=red>Levels 1,2</font>


<b>Temperature Compensation and Area Parameters:</b>

 Tnom:    Reference temperature (default value = 300.15 K) <font color=red>Levels 1,2</font>

 XTI:     Saturation current temperature exponent (default value = 3) <font color=red>Levels 1,2</font>

 XTB:     Forward and reverse beta temperature coefficient (default value = 0) <font color=red>Levels 1,2</font>

 EG:      Energy gap for temperature effect on saturation current (default value = 1.11 Volt) <font color=red>Levels 1,2</font>

 TRB1:    Linear temperature coefficient of zero-bias base resistance (default value = 0 1/K) <font color=red>Levels 2</font>

 TRB2:    Quadratic temperature coefficient of zero-bias base resistance (default value = 0 1/(K<sup>2</sup>)) <font color=red>Levels 2</font>

 TRM1:    Linear temperature coefficient of mimimum base resistance (default value = 0 1/K) <font color=red>Levels 2</font>

 TRM2:    Quadratic temperature coefficient of minimum base resistance (default value = 0 1/(K<sup>2</sup>)) <font color=red>Levels 2</font>

 TRC1:    Linear temperature coefficient of collector resistance (default value = 0 1/K) <font color=red>Levels 2</font>

 TRC2:    Quadratic temperature coefficient of collector resistance (default value = 0 1/(K<sup>2</sup>)) <font color=red>Levels 2</font>

 TRE1:    Linear temperature coefficient of emitter resistance (default value = 0 1/K) <font color=red>Levels 2</font>

 TRE2:    Quadratic temperature coefficient of emitter resistance (default value = 0 1/(K<sup>2</sup>)) <font color=red>Levels 2</font>


<b>Numerical Parameters:</b>

 EMin:    Minimum exponent for linearization of junction current (default value = -100) <font color=red>Levels 1,2</font>

 EMax:    Maximum exponent for linearization of junction current (default value = 40) <font color=red>Levels 1,2</font>

 GminDC:  Leakage conductance (default value = 1e-19 Mho) <font color=red>Levels 1,2</font>
</pre> <p>

<hr> <p>

<b>References:</b> <p>

<ol>
<li>Cellier, F.E. (1991), <i>Continuous System Modeling</i>, Springer-Verlag, New York.
<li>Hild, D.R. and F.E. Cellier (1994), \"Object-oriented electronic circuit modeling using Dymola,\"
    <i>Proc. OOS'94, SCS Object Oriented Simulation Conference</i>, Tempe, AZ, pp.68-75.
<li>Hild, D.R. (1993), <i>Circuit Modeling in Dymola</i>, MS Thesis, Dept. of Electr. &amp; Comp. Engr.,
    University of Arizona, Tucson.
<li> Massobrio, G. and P. Antognetti (1993), <i>Semiconductor Device Modeling with Spice</i>,
     2<sup>nd</sup> edition, McGraw Hill, New York.
<li>Schweisguth, M.C. and F.E. Cellier (1999), \"<a href=\"http://www.inf.ethz.ch/~fcellier/Publikationen/BJT.pdf\">A
    bond graph model of the bipolar junction transistor</a>,\" <i>Proc. SCS Intl. Conf. on Bond Graph Modeling</i>,
    San Francisco, CA, pp.344-349.
<li>Schweisguth, M.C. (1997), <a href=\"http://www.inf.ethz.ch/~fcellier/Publikationen/schweisguth_ms.pdf\"><i>Semiconductor
    Modeling with Bondgraphs</i></a>, MS Thesis, Dept. of Electr. &amp; Comp. Engr., University of Arizona, Tucson.
</ol>
</html>
"),     Icon(
            Text(
              extent=[-96,84; 68,-92],
              style(color=0),
            string="PNP"),
            Text(
              extent=[64,4; 104,-38],
              style(color=0),
            string="lat"),
          Text(
            extent=[-80,100; 80,60],
            style(color=3, rgbcolor={0,0,255}),
            string="%name"),
          Text(
            extent=[-126,24; -114,10],
            string="B",
            style(color=10, rgbcolor={95,95,95})),
          Text(
            extent=[10,126; 22,112],
            style(color=10, rgbcolor={95,95,95}),
            string="C"),
          Text(
            extent=[106,26; 132,6],
            style(color=10, rgbcolor={95,95,95}),
            string="Heat"),
          Text(
            extent=[-88,-72; -44,-108],
            string="%Area x",
            style(color=3, rgbcolor={0,0,255})),
            Line(points=[0,100; 0,-100],
                                       style(
                color=9,
                rgbcolor={149,164,171},
                thickness=2,
                fillColor=74,
                rgbfillColor={0,0,185},
                fillPattern=1)),
            Line(points=[0,100; 20,80],   style(
                color=9,
                rgbcolor={149,164,171},
                thickness=2,
                fillColor=74,
                rgbfillColor={0,0,185},
                fillPattern=1)),
            Line(points=[-20,80; 0,100], style(
                color=9,
                rgbcolor={149,164,171},
                thickness=2,
                fillColor=74,
                rgbfillColor={0,0,185},
                fillPattern=1)),
          Line(points=[-100,-60; 100,-60], style(color=3, rgbcolor={0,0,255})),
          Line(points=[80,-80; 100,-60], style(color=3, rgbcolor={0,0,255})),
          Line(points=[80,-40; 100,-60], style(color=3, rgbcolor={0,0,255})),
          Text(
            extent=[8,-114; 20,-128],
            style(color=10, rgbcolor={95,95,95}),
            string="E")));
    public
      BondLib.Interfaces.BondCon B "Base"
        annotation (extent=[-130,-10; -110,10]);
      BondLib.Interfaces.BondCon E "Emitter"
        annotation (extent=[-10,-130; 10,-110]);
      BondLib.Interfaces.BondCon H "Heat" annotation (extent=[110,-10; 130,10]);
      BondLib.Interfaces.BondCon C "Collector"
                                          annotation (extent=[-10,110; 10,130]);
      BondLib.Bonds.Bond B1 annotation (extent=[-20,-30; 0,-10], rotation=180);
      BondLib.Junctions.J0p3 J0p3_1
        annotation (extent=[-40,-30; -20,-10], rotation=90);
      BondLib.Bonds.Bond B2 annotation (extent=[-50,-10; -30,10], rotation=180);
      BondLib.Junctions.J1p3 J1p3_1 annotation (extent=[-70,10; -50,-10]);
      BondLib.Bonds.Bond B3 annotation (extent=[-90,-10; -70,10], rotation=180);
      BondLib.Bonds.Bond B4 annotation (extent=[-70,10; -50,30], rotation=90);
      BondLib.Spice.Utilities.RBS Rb(
        Tnom=Tnom,
        RB=RB,
        RBM=RBM,
        IRB=IRB,
        TRB1=TRB1,
        TRB2=TRB2,
        TRM1=TRM1,
        TRM2=TRM2,
        Area=Area,
        Level=Level)
                   annotation (extent=[-60,30; -20,50], rotation=0);
      BondLib.Bonds.eBond B5 annotation (extent=[-20,30; 0,50]);
      BondLib.Bonds.Bond B6 annotation (extent=[-50,-50; -30,-30], rotation=0);
      BondLib.Junctions.J1p3 J1p3_2 annotation (extent=[-70,-50; -50,-30]);
      BondLib.Bonds.Bond B9 annotation (extent=[-70,-70; -50,-50], rotation=270);
      BondLib.Bonds.eBond B10 annotation (extent=[-20,-90; 0,-70]);
      BondLib.Bonds.Bond B11 annotation (extent=[10,40; 30,60], rotation=90);
      BondLib.Junctions.J1p3 J1p3_3
        annotation (extent=[30,80; 10,60], rotation=-90);
      BondLib.Bonds.Bond B12 annotation (extent=[10,80; 30,100], rotation=90);
      BondLib.Junctions.J0p2 J0p2_3
        annotation (extent=[10,100; 30,120], rotation=90);
      BondLib.Bonds.Bond B13 annotation (extent=[30,60; 50,80], rotation=0);
      BondLib.Bonds.eBond B14 annotation (extent=[90,60; 110,80], rotation=0);
      BondLib.Bonds.Bond B15 annotation (extent=[10,-60; 30,-40], rotation=90);
      BondLib.Junctions.J1p3 J1p3_4
        annotation (extent=[30,-60; 10,-80], rotation=-90);
      BondLib.Bonds.Bond B16 annotation (extent=[10,-100; 30,-80], rotation=90);
      BondLib.Junctions.J0p2 J0p2_4
        annotation (extent=[10,-120; 30,-100], rotation=90);
      BondLib.Bonds.Bond B17 annotation (extent=[30,-80; 50,-60], rotation=0);
      BondLib.Bonds.eBond B18 annotation (extent=[90,-80; 110,-60], rotation=0);
      BondLib.Bonds.eBond B19 annotation (extent=[40,-30; 60,-10]);
      BondLib.Junctions.J0p6 J0p6_1 annotation (extent=[60,20; 80,40]);
      BondLib.Bonds.Bond B20 annotation (extent=[10,0; 30,20], rotation=90);
      BondLib.Junctions.J0p3 J0p3_2
        annotation (extent=[10,20; 30,40], rotation=90);
      BondLib.Junctions.J0p3 J0p3_3
        annotation (extent=[-110,-10; -90,10], rotation=180);
      BondLib.Bonds.Bond B21 annotation (extent=[-110,10; -90,30], rotation=90);
      BondLib.Junctions.J1p3 J1p3_5
        annotation (extent=[-110,30; -90,50], rotation=90);
      BondLib.Bonds.Bond B22 annotation (extent=[-110,50; -90,70], rotation=90);
      BondLib.Bonds.Bond B23 annotation (extent=[-90,50; -70,70], rotation=90);
      BondLib.Bonds.eBond B24 annotation (extent=[-40,70; -20,90]);
      BondLib.Bonds.eBond B25 annotation (extent=[80,20; 100,40]);
      BondLib.Junctions.J0p3 J0p3_4
        annotation (extent=[100,20; 120,40], rotation=180);
      BondLib.Spice.Utilities.PNPint Qpi(
        BF=BF,
        BR=BR,
        IS=IS,
        NF=NF,
        NR=NR,
        GminDC=GminDC,
        NC=NC,
        NE=NE,
        VAF=VAF,
        VAR=VAR,
        IKF=IKF,
        IKR=IKR,
        CJC=CJC,
        MJC=MJC,
        VJC=VJC,
        CJE=CJE,
        MJE=MJE,
        VJE=VJE,
        XCJC=XCJC,
        TF=TF,
        TR=TR,
        Tnom=Tnom,
        XTI=XTI,
        XTB=XTB,
        EG=EG,
        Area=Area,
        EMin=EMin,
        EMax=EMax,
        FC=FC,
        Level=Level,
        ISC=ISCeff,
        ISE=ISEeff,
        ISS=ISS) annotation (extent=[0,-40; 40,0]);
      Utilities.CbxS Cbx   annotation (extent=[-80,70; -40,90]);
      Utilities.RS2 Rc(
        Tnom=Tnom,
        R=RC,
        TR1=TRC1,
        TR2=TRC2,
        Area=Area,
        Level=Level) annotation (extent=[50,58; 90,82]);
      Utilities.RS2 Re(
        Tnom=Tnom,
        Area=Area,
        Level=Level,
        R=RE,
        TR1=TRE1,
        TR2=TRE2) annotation (extent=[50,-82; 90,-58]);
      BondLib.Bonds.Bond B7 annotation (extent=[-90,-50; -70,-30], rotation=0);
      Junctions.J0p2 J0p2_2
        annotation (extent=[-110,-60; -90,-40], rotation=270);
      BondLib.Interfaces.BondCon S "Substrate"
        annotation (extent=[-130,-70; -110,-50]);
      Utilities.CjS Dsb(
        Tnom=Tnom,
        XTI=XTI,
        XTB=XTB,
        EG=EG,
        Area=Area,
        Level=Level,
        BE=false,
        EMin=EMin,
        EMax=EMax,
        MJ=MJS,
        VJ=VJS,
        XCJ=1,
        Tau=0,
        IS=0,
        GminDC=0,
        CJ=CJS,
        FC=0,
        N=1)       annotation (extent=[-60,-94; -20,-66]);
      Modelica.Blocks.Sources.Constant C0(k=0)
        annotation (extent=[-66,-116; -54,-104]);
      Modelica.Blocks.Sources.Constant qb(k=1)
        annotation (extent=[-46,-116; -34,-104]);
      equation
      connect(J1p3_1.BondCon3, B4.BondCon1) annotation (points=[-60,10; -60,
            10.2], style(color=8, rgbcolor={192,192,192}));
      connect(Rb.BondCon1, B4.BondCon2)  annotation (points=[-60,40; -60,30],
          style(color=8, rgbcolor={192,192,192}));
      connect(Rb.BondCon2, B5.fBondCon1)  annotation (points=[-20,40; -20,40],
          style(color=8, rgbcolor={192,192,192}));
      connect(B9.BondCon1, J1p3_2.BondCon3) annotation (points=[-60,-50.2; -60,
            -50], style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_3.BondCon3, B13.BondCon1) annotation (points=[30,70; 30.2,70],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p2_4.BondCon1, E)        annotation (points=[20,-120; 0,-120],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p2_3.BondCon2, C)        annotation (points=[20,120; 0,120],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_3.BondCon3, B21.BondCon1) annotation (points=[-100,10; -100,
            10.2], style(color=8, rgbcolor={192,192,192}));
      connect(B21.BondCon2, J1p3_5.BondCon1) annotation (points=[-100,30; -100,
            30], style(color=8, rgbcolor={192,192,192}));
      connect(B22.BondCon1, J1p3_5.BondCon2) annotation (points=[-100,50.2;
            -100,50], style(color=8, rgbcolor={192,192,192}));
      connect(B23.BondCon1, J1p3_5.BondCon3) annotation (points=[-80,50.2; -80,
            40; -90,40], style(color=8, rgbcolor={192,192,192}));
      connect(B22.BondCon2, J0p3_2.BondCon3) annotation (points=[-100,70; -100,
            94; 40,94; 40,30; 30,30], style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_4.BondCon1, H)        annotation (points=[120,30; 120,0],
          style(color=8, rgbcolor={192,192,192}));
      connect(B14.eBondCon1, J0p3_4.BondCon3) annotation (points=[110,70; 110,
            40], style(color=8, rgbcolor={192,192,192}));
      connect(B19.eBondCon1, J0p6_1.BondCon1) annotation (points=[60,-20; 60,30],
          style(color=8, rgbcolor={192,192,192}));
      connect(B5.eBondCon1, J0p6_1.BondCon5) annotation (points=[0,40; 0,48; 66,
            48; 66,40], style(color=8, rgbcolor={192,192,192}));
      connect(B24.eBondCon1, J0p6_1.BondCon6) annotation (points=[-20,80; 0,80;
            0,52; 74,52; 74,40], style(color=8, rgbcolor={192,192,192}));
      connect(B18.eBondCon1, J0p6_1.BondCon3) annotation (points=[110,-70; 110,
            -40; 66,-40; 66,20], style(color=8, rgbcolor={192,192,192}));
      connect(B10.eBondCon1, J0p6_1.BondCon4) annotation (points=[0,-80; 8,-80;
            8,-90; 114,-90; 114,-36; 74,-36; 74,20], style(color=8, rgbcolor={
              192,192,192}));
      connect(B12.BondCon2, J0p2_3.BondCon1) annotation (points=[20,100; 20,100],
          style(color=8, rgbcolor={192,192,192}));
      connect(B12.BondCon1, J1p3_3.BondCon2) annotation (points=[20,80.2; 20,80],
          style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_3.BondCon1, B11.BondCon2) annotation (points=[20,60; 20,60],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_2.BondCon2, B11.BondCon1) annotation (points=[20,40; 20,40.2],
          style(color=8, rgbcolor={192,192,192}));
      connect(B20.BondCon2, J0p3_2.BondCon1) annotation (points=[20,20; 20,20],
          style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_4.BondCon2, B15.BondCon1) annotation (points=[20,-60; 20,
            -59.8], style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_4.BondCon1, B16.BondCon2) annotation (points=[20,-80; 20,-80],
          style(color=8, rgbcolor={192,192,192}));
      connect(B16.BondCon1, J0p2_4.BondCon2) annotation (points=[20,-99.8; 20,
            -100], style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_1.BondCon3, B1.BondCon2) annotation (points=[-20,-20; -20,
            -20], style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_1.BondCon2, B2.BondCon1) annotation (points=[-30,-10; -30,
            -1.20011e-015; -30.2,-1.20011e-015],     style(color=8, rgbcolor={
              192,192,192}));
      connect(B3.BondCon2, J0p3_3.BondCon1) annotation (points=[-90,
            1.22461e-015; -90,-1.22461e-015], style(color=8, rgbcolor={192,192,
              192}));
      connect(B6.BondCon1, J1p3_2.BondCon2) annotation (points=[-49.8,-40; -50,
            -40], style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_1.BondCon1, B6.BondCon2) annotation (points=[-30,-30; -30,
            -40], style(color=8, rgbcolor={192,192,192}));
      connect(Cbx.BondCon2, B24.fBondCon1)   annotation (points=[-40,80; -40,80],
          style(color=8, rgbcolor={192,192,192}));
      connect(Cbx.BondCon1, B23.BondCon2)   annotation (points=[-80,80; -80,70],
          style(color=8, rgbcolor={192,192,192}));
      connect(Qpi.qb, Rb.qb)      annotation (points=[41.4286,-33.7143; 52,
            -33.7143; 52,-48; -10,-48; -10,56; -40,56; -40,50], style(color=74,
            rgbcolor={0,0,127}));
      connect(Qpi.ix, Cbx.ix)       annotation (points=[10,0; 10,20; -70,20;
            -70,60; -60,60; -60,70], style(color=74, rgbcolor={0,0,127}));
      connect(Rc.BondCon1, B13.BondCon2) annotation (points=[50,70; 50,70],
          style(color=8, rgbcolor={192,192,192}));
      connect(Rc.BondCon2, B14.fBondCon1) annotation (points=[90,70; 90,70],
          style(color=8, rgbcolor={192,192,192}));
      connect(Re.BondCon2, B18.fBondCon1) annotation (points=[90,-70; 90,-70],
          style(color=8, rgbcolor={192,192,192}));
      connect(B7.BondCon2, J1p3_2.BondCon1) annotation (points=[-70,-40; -70,
            -40], style(color=8, rgbcolor={192,192,192}));
      connect(B7.BondCon1, J0p2_2.BondCon1) annotation (points=[-89.8,-40; -100,
            -40], style(color=8, rgbcolor={192,192,192}));
      connect(J0p2_2.BondCon2, S)        annotation (points=[-100,-60; -120,-60],
          style(color=8, rgbcolor={192,192,192}));
      connect(C0.y, Dsb.C0) annotation (points=[-53.4,-110; -50,-110; -50,-94],
          style(color=74, rgbcolor={0,0,127}));
      connect(qb.y, Dsb.qb) annotation (points=[-33.4,-110; -30,-110; -30,-94],
          style(color=74, rgbcolor={0,0,127}));
      connect(Dsb.BondCon2, B10.fBondCon1) annotation (points=[-20,-80; -20,-80],
          style(color=8, rgbcolor={192,192,192}));
      connect(B9.BondCon2, Dsb.BondCon1) annotation (points=[-60,-70; -60,-80],
          style(color=8, rgbcolor={192,192,192}));
      connect(B2.BondCon2, J1p3_1.BondCon2) annotation (points=[-50,
            1.22461e-015; -50,0], style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_1.BondCon1, B3.BondCon1) annotation (points=[-70,0; -70,
            -1.20011e-015; -70.2,-1.20011e-015], style(color=8, rgbcolor={192,
              192,192}));
      connect(B17.BondCon2, Re.BondCon1) annotation (points=[50,-70; 50,-70],
          style(color=8, rgbcolor={192,192,192}));
      connect(B17.BondCon1, J1p3_4.BondCon3) annotation (points=[30.2,-70; 30,
            -70], style(color=8, rgbcolor={192,192,192}));
      connect(J0p6_1.BondCon2, B25.fBondCon1) annotation (points=[80,30; 80,30],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_4.BondCon2, B25.eBondCon1) annotation (points=[100,30; 100,
            30], style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_3.BondCon2, B)        annotation (points=[-110,1.22461e-015;
            -116,1.22461e-015; -116,0; -120,0], style(color=8, rgbcolor={192,
              192,192}));
      connect(Qpi.Bint, B1.BondCon1) annotation (points=[0,-20; -0.2,-20],
          style(color=8, rgbcolor={192,192,192}));
      connect(Qpi.Eint, B15.BondCon2) annotation (points=[20,-40; 20,-40],
          style(color=8, rgbcolor={192,192,192}));
      connect(B19.fBondCon1, Qpi.H) annotation (points=[40,-20; 40,-20], style(
            color=8, rgbcolor={192,192,192}));
      connect(B20.BondCon1, Qpi.Cint) annotation (points=[20,0.2; 20,0], style(
            color=8, rgbcolor={192,192,192}));
      end PNPlat;

      model PNPvert "Spice-style vertically diffused PNP bipolar transistor"
        constant Real inf=Modelica.Constants.inf;

        /* General parameters */
        parameter Integer Level(min=1,max=2) = 1 "Transistor modeling level" annotation(Evaluate=true);
        parameter Real Area=1 "Relative area occupied by device";

        /* DC model parameters */
        parameter Real BF=100
        "Maximum forward current gain at reference temperature"
          annotation(Dialog(group="DC model parameters"));
        parameter Real BR=1
        "Maximum reverse current gain at reference temperature"
          annotation(Dialog(group="DC model parameters"));
        parameter Modelica.SIunits.Current IS=1e-16
        "Saturation current at reference temperature"   annotation(Dialog(group="DC model parameters"));
        parameter Real NF=1 "Forward current emission coefficient"
          annotation(Dialog(group="DC model parameters"));
        parameter Real NR=1 "Reverse current emission coefficient"
          annotation(Dialog(group="DC model parameters"));
        parameter Modelica.SIunits.Current ISS=IS
        "Saturation current used for current injection"   annotation(Dialog(group="DC model parameters", enable=Level == 2));

        /* Low current beta degradation effect parameters */
        parameter Real C4(min=0) = 0
        "Base-collector leakage current coefficient"
          annotation(Dialog(group="Low current beta degradation effect parameters", enable=Level == 2));
        parameter Modelica.SIunits.Current ISC(min=0) = 0
        "Base-collector leakage saturation current at reference temperature (ISC = C4*IS)"
          annotation(Dialog(group="Low current beta degradation effect parameters", enable=Level == 2));
        parameter Real C2(min=0) = 0
        "Base-emitterr leakage current coefficient"
          annotation(Dialog(group="Low current beta degradation effect parameters", enable=Level == 2));
        parameter Modelica.SIunits.Current ISE(min=0) = 0
        "Base-emitter leakage saturation current at reference temperature (ISE = C2*IS)"
          annotation(Dialog(group="Low current beta degradation effect parameters", enable=Level == 2));
        parameter Real NC=2
        "Low-current base-collector leakage emission coefficient"
          annotation(Dialog(group="Low current beta degradation effect parameters", enable=Level == 2));
        parameter Real NE=1.5
        "Low-current base-emitter leakage emission coefficient"
          annotation(Dialog(group="Low current beta degradation effect parameters", enable=Level == 2));

        /* High current beta degradation effect parameters */
        parameter Modelica.SIunits.Current IKF=inf
        "Corner for forward beta high-current roll-off"
          annotation(Dialog(group="High current beta degradation effect", enable=Level == 2));
        parameter Modelica.SIunits.Current IKR=inf
        "Corner for reverse beta high-current roll-off"
          annotation(Dialog(group="High current beta degradation effect", enable=Level == 2));

        /* Base width modulation parameters */
        parameter Modelica.SIunits.Voltage VAF=inf "Forward early voltage"
          annotation(Dialog(group="Base width modulation parameters"));
        parameter Modelica.SIunits.Voltage VAR=inf "Reverse early voltage"
          annotation(Dialog(group="Base width modulation parameters", enable=Level == 2));

        /* Parasitic Resistor Parameters */
        parameter Modelica.SIunits.Current IRB=inf
        "Current where base resistance falls halfway to minimum value"
          annotation(Dialog(group="Parasitic resistor parameters", enable=Level == 2));
        parameter Modelica.SIunits.Resistance RB(min=0) = 0
        "Zero-bias base resistance"   annotation(Dialog(group="Parasitic resistor parameters", enable=Level == 2));
        parameter Modelica.SIunits.Resistance RBM(min=0) = RB
        "Minimum base resistance at high currents"   annotation(Dialog(group="Parasitic resistor parameters", enable=Level == 2));
        parameter Modelica.SIunits.Resistance RC(min=0) = 0
        "Collector resistance"
          annotation(Dialog(group="Parasitic resistor parameters", enable=Level == 2));
        parameter Modelica.SIunits.Resistance RE(min=0) = 0
        "Emitter resistance"
          annotation(Dialog(group="Parasitic resistor parameters", enable=Level == 2));

        /* Junction capacitor parameters */
        parameter Modelica.SIunits.Capacitance CJC(min=Modelica.Constants.small) = 1e-12
        "Zero-bias base-collector depletion capacitance at reference temperature (CJC > 0)"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Real MJC(min=0.3,max=0.5) = 0.33
        "Base-collector junction grading coefficient"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Modelica.SIunits.Voltage VJC=0.75
        "Base-collector built-in potential at reference temperature"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Modelica.SIunits.Capacitance CJE(min=Modelica.Constants.small) = 1e-12
        "Zero-bias base-emitter depletion capacitance at reference temperature (CJE > 0)"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Real MJE(min=0.3,max=0.5) = 0.33
        "Base-emitter junction grading coefficient"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Modelica.SIunits.Voltage VJE=0.75
        "Base-emitter built-in potential at reference temperature"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Modelica.SIunits.Capacitance CJS(min=Modelica.Constants.small) = 1e-12
        "Zero-bias substrate depletion capacitance at reference temperature"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Real MJS(min=0.3,max=0.5) = 0.33
        "Substrate junction grading coefficient"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Modelica.SIunits.Voltage VJS=0.75
        "Substrate built-in potential at reference temperature"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Real XCJC(min=0,max=1) = 1
        "Fraction of base-collector depletion capacitance connected to internal base node"
          annotation(Dialog(group="Junction capacitor parameters"));
        parameter Real FC(min=0,max=1) = 0.5
        "Depletion capacitance factor for linearization"
          annotation(Dialog(group="Junction capacitor parameters", enable=Level == 2));

        /* Transit time parameters */
        parameter Modelica.SIunits.Time TF(min=0) = 0
        "Ideal forward transit time"
          annotation(Dialog(group="Transit time parameters"));
        parameter Modelica.SIunits.Time TR(min=0) = 0
        "Ideal reverse transit time"
          annotation(Dialog(group="Transit time parameters"));

        /* Temperature compensation and area parameters */
        parameter Modelica.SIunits.Temperature Tnom=300.15
        "Reference temperature"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Real XTI=3 "Saturation current temperature exponent"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Real XTB=0 "Forward and reverse beta temperature coefficient"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Modelica.SIunits.Voltage EG=1.11
        "Energy gap for temperature effect on saturation current"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Real TRB1(unit="1/K") = 0
        "Linear temperature coefficient of zero-bias base resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));
        parameter Real TRB2(unit="1/K2") = 0
        "Quadratic temperature coefficient of zero-bias base resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));
        parameter Real TRM1(unit="1/K") = 0
        "Linear temperature coefficient of minimum base resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));
        parameter Real TRM2(unit="1/K2") = 0
        "Quadratic temperature coefficient of minimum base resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));
        parameter Real TRC1(unit="1/K") = 0
        "Linear temperature coefficient of collector resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));
        parameter Real TRC2(unit="1/K2") = 0
        "Quadratic temperature coefficient of collector resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));
        parameter Real TRE1(unit="1/K") = 0
        "Linear temperature coefficient of emitter resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));
        parameter Real TRE2(unit="1/K2") = 0
        "Quadratic temperature coefficient of emitter resistance"
          annotation(Dialog(group="Temperature compensation parameters", enable=Level == 2));

        /* Initialization parameters */
        parameter Modelica.SIunits.Time rampTime=0
        "Substrate potential ramping time (if unconnected)"
          annotation(Dialog(tab="Advanced", group="Initialization parameters"));

        /* Numerical parameters */
        parameter Real EMin=-100
        "if x < EMin, the exp(x) function is linearized"
          annotation(Dialog(tab="Advanced", group="Numerical parameters"));
        parameter Real EMax=40 "if x > EMax, the exp(x) function is linearized"
          annotation(Dialog(tab="Advanced", group="Numerical parameters"));
        parameter Modelica.SIunits.Conductance GminDC(min=0) = 1e-19
        "Leakage conductance"
          annotation(Dialog(tab="Advanced", group="Numerical parameters"));

    protected
        parameter Modelica.SIunits.Current ISCeff = if ISC > 0 then ISC else C4*IS;
        parameter Modelica.SIunits.Current ISEeff = if ISE > 0 then ISE else C2*IS;
    public
      BondLib.Interfaces.BondCon B "Base"
        annotation (extent=[-130,-10; -110,10]);
      BondLib.Interfaces.BondCon E "Emitter"
        annotation (extent=[-10,-130; 10,-110]);
      BondLib.Interfaces.BondCon H "Heat" annotation (extent=[110,-10; 130,10]);
      BondLib.Interfaces.BondCon C "Collector"
                                          annotation (extent=[-10,110; 10,130]);
      BondLib.Bonds.Bond B2 annotation (extent=[-50,-10; -30,10], rotation=180);
      BondLib.Junctions.J1p3 J1p3_1 annotation (extent=[-70,-10; -50,10]);
      BondLib.Bonds.Bond B3 annotation (extent=[-90,-10; -70,10], rotation=180);
      BondLib.Bonds.Bond B4 annotation (extent=[-70,-30; -50,-10], rotation=270);
      BondLib.Spice.Utilities.RBS Rb(
        Tnom=Tnom,
        RB=RB,
        RBM=RBM,
        IRB=IRB,
        TRB1=TRB1,
        TRB2=TRB2,
        TRM1=TRM1,
        TRM2=TRM2,
        Area=Area,
        Level=Level)
                   annotation (extent=[-60,-50; -20,-30], rotation=0);
      BondLib.Bonds.eBond B5 annotation (extent=[-20,-50; 0,-30]);
      BondLib.Bonds.Bond B6 annotation (extent=[70,20; 90,40], rotation=0);
      BondLib.Junctions.J1p3 J1p3_2 annotation (extent=[50,40; 70,20]);
      BondLib.Bonds.Bond B7 annotation (extent=[30,20; 50,40], rotation=0);
      BondLib.Bonds.Bond B9 annotation (extent=[50,40; 70,60], rotation=90);
      BondLib.Bonds.eBond B10 annotation (extent=[110,80; 130,100], rotation=90);
      BondLib.Bonds.Bond B11 annotation (extent=[10,40; 30,60], rotation=90);
      BondLib.Junctions.J1p3 J1p3_3
        annotation (extent=[10,80; 30,60], rotation=-90);
      BondLib.Bonds.Bond B12 annotation (extent=[10,80; 30,100], rotation=90);
      BondLib.Junctions.J0p2 J0p2_3
        annotation (extent=[10,100; 30,120], rotation=90);
      BondLib.Bonds.Bond B13 annotation (extent=[-10,60; 10,80], rotation=180);
      BondLib.Bonds.eBond B14 annotation (extent=[-70,60; -50,80], rotation=180);
      BondLib.Bonds.Bond B15 annotation (extent=[10,-60; 30,-40], rotation=90);
      BondLib.Junctions.J1p3 J1p3_4
        annotation (extent=[10,-60; 30,-80], rotation=-90);
      BondLib.Bonds.Bond B16 annotation (extent=[10,-100; 30,-80], rotation=90);
      BondLib.Junctions.J0p2 J0p2_4
        annotation (extent=[10,-120; 30,-100], rotation=90);
      BondLib.Bonds.Bond B17 annotation (extent=[-10,-80; 10,-60], rotation=180);
      BondLib.Bonds.eBond B18
        annotation (extent=[-70,-80; -50,-60], rotation=180);
      BondLib.Bonds.eBond B19 annotation (extent=[40,-30; 60,-10]);
      BondLib.Junctions.J0p6 J0p6_1 annotation (extent=[60,-100; 80,-80]);
      BondLib.Bonds.Bond B20 annotation (extent=[10,0; 30,20], rotation=90);
      BondLib.Junctions.J0p3 J0p3_3
        annotation (extent=[-110,-10; -90,10], rotation=180);
      BondLib.Bonds.Bond B21 annotation (extent=[-110,10; -90,30], rotation=90);
      BondLib.Junctions.J1p3 J1p3_5
        annotation (extent=[-110,30; -90,50], rotation=90);
      BondLib.Bonds.Bond B22 annotation (extent=[-110,50; -90,70], rotation=90);
      BondLib.Bonds.Bond B23 annotation (extent=[-90,30; -70,50], rotation=0);
      BondLib.Bonds.eBond B24 annotation (extent=[-30,30; -10,50]);
      BondLib.Bonds.eBond B25 annotation (extent=[80,-100; 100,-80]);
      BondLib.Junctions.J0p3 J0p3_4
        annotation (extent=[100,-100; 120,-80], rotation=180);
      BondLib.Junctions.J0p4 J0p4_1 annotation (extent=[10,20; 30,40]);
      annotation (
        Diagram(
          Text(
            extent=[-126,22; -114,8],
            string="B",
            style(color=10, rgbcolor={95,95,95})),
          Text(
            extent=[-20,128; -8,114],
            style(color=10, rgbcolor={95,95,95}),
            string="C"),
          Text(
            extent=[-20,-114; -8,-128],
            style(color=10, rgbcolor={95,95,95}),
            string="E"),
          Text(
            extent=[106,22; 132,6],
            style(color=10, rgbcolor={95,95,95}),
            string="Heat"),
          Text(
            extent=[82,-96; 108,-112],
            style(color=10, rgbcolor={95,95,95}),
            string="Heat"),
          Text(
            extent=[38,-34; 48,-46],
            style(color=10, rgbcolor={95,95,95}),
            string="qb"),
          Text(
            extent=[-38,-14; -28,-26],
            style(color=10, rgbcolor={95,95,95}),
            string="qb"),
          Text(
            extent=[0,10; 10,-2],
            style(color=10, rgbcolor={95,95,95}),
            string="ix"),
          Text(
            extent=[-50,30; -40,18],
            style(color=10, rgbcolor={95,95,95}),
            string="ix"),
          Text(
            extent=[-126,-36; -114,-50],
            style(color=10, rgbcolor={95,95,95}),
            string="S")),
        Coordsys(extent=[-120,-120; 120,120]),
        Icon(
          Text(
            extent=[-80,100; 80,60],
            style(color=3, rgbcolor={0,0,255}),
            string="%name"),
            Line(points=[0,100; 0,-100],
                                       style(
                color=9,
                rgbcolor={149,164,171},
                thickness=2,
                fillColor=74,
                rgbfillColor={0,0,185},
                fillPattern=1)),
            Text(
              extent=[-94,84; 70,-92],
              style(color=0),
            string="PNP"),
            Text(
              extent=[58,4; 110,-42],
              style(color=0),
            string="vert"),
          Line(points=[-100,-60; 100,-60], style(color=3, rgbcolor={0,0,255})),
          Line(points=[80,-80; 100,-60], style(color=3, rgbcolor={0,0,255})),
          Line(points=[80,-40; 100,-60], style(color=3, rgbcolor={0,0,255})),
            Line(points=[-20,80; 0,100], style(
                color=9,
                rgbcolor={149,164,171},
                thickness=2,
                fillColor=74,
                rgbfillColor={0,0,185},
                fillPattern=1)),
            Line(points=[0,100; 20,80],   style(
                color=9,
                rgbcolor={149,164,171},
                thickness=2,
                fillColor=74,
                rgbfillColor={0,0,185},
                fillPattern=1)),
          Text(
            extent=[-88,-72; -44,-108],
            string="%Area x",
            style(color=3, rgbcolor={0,0,255})),
          Text(
            extent=[8,-114; 20,-128],
            style(color=10, rgbcolor={95,95,95}),
            string="E"),
          Text(
            extent=[106,26; 132,6],
            style(color=10, rgbcolor={95,95,95}),
            string="Heat"),
          Text(
            extent=[10,126; 22,112],
            style(color=10, rgbcolor={95,95,95}),
            string="C"),
          Text(
            extent=[-126,24; -114,10],
            string="B",
            style(color=10, rgbcolor={95,95,95}))),
        Documentation(info="<html>
The <b>PNPvert</b> element of the Spice bond graph library implements a full-fledges Spice-stye Gummel-Poon model of the vertically diffused PNP bipolar transistor [1-3]. The model contains a <b>PNPint</b> model implementing the inner parts of the PNP transistor up to the internal nodes. <p>

The PNP bipolar transistor is a directed FourPort.  The direction of positive power flow is assumed into the model at the emitter, <b>E</b>, whereas it is assumed out of the model at the base, <b>B</b>, at the collector, <b>C</b>, and at the <b>Heat</b> port. <p>

The causality of the PNP model is free. <p>

<hr>

<pre>
<b>Parameters:</b>

 <font color=red><b>Level:   Transistor modeling level (default value = 2)
            Level = 1: Ebers-Moll model
            Level = 2: Gummel-Poon model</b></font>

 Area:    Relative area occupied by device (default value = 1) <font color=red>Levels 1,2</font>


<b>DC Model Parameters:</b>

 BF:      Maximum forward current gain at reference temperature (default value = 100) <font color=red>Levels 1,2</font>

 BR:      Maximum reverse current gain at reference temperature (default value = 1) <font color=red>Levels 1,2</font>

 IS:      Saturation current at reference temperature (default value = 1e-16 Amp) <font color=red>Levels 1,2</font>

 NF:      Forward current emission coefficient (default value = 1) <font color=red>Levels 1,2</font>

 NR:      Reverse current emission coefficient (default value = 1) <font color=red>Levels 1,2</font>

 ISS:     Saturation current for injection (default value = IS Amp) <font color=red>Levels 2</font>


<b>Low Current Beta Degradation Effect Parameters:</b>

 C4:      Base-collector leakage current coefficient (default value = 0) <font color=red>Levels 2</font>

 ISC:     Base-collector leakage saturation current at reference temperature (default value = 0 Amp) <font color=red>Levels 2</font>
          <font color=blue><b>ISCeff = if ISC &gt; 0 then ISC else C4*IS</b></font>

 C2:      Base-emitter leakage current coefficient (default value = 0) <font color=red>Levels 2</font>

 ISE:     Base-emitter leakage saturation current at reference temperature (default value = 0 Amp) <font color=red>Levels 2</font>
          <font color=blue><b>ISEeff = if ISE &gt; 0 then ISE else C2*IS</b></font>

 NC:      Low-current base-collector leakage emission coefficient (default value = 2) <font color=red>Levels 2</font>

 NE:      Low-current base-emitter leakage emission coefficient (default value = 1.5) <font color=red>Levels 2</font>


<b>High Current Beta Degradation Effect Parameters:</b>

 IKF:     Corner for forward beta high-current roll-off (default value = &infin; Amp) <font color=red>Levels 2</font>

 IKR:     Corner for reverse beta high-current roll-off (default value = &infin; Amp) <font color=red>Levels 2</font>


<b>Base Width Modulation Parameters:</b>

 VAF:     Forward early voltage (default value = &infin; Volt) <font color=red>Levels 1,2</font>

 VAR:     Reverse early voltage (default value = &infin; Volt) <font color=red>Levels 2</font>


<b>Parasitic Resistor Parameters:</b>

 IRB:     Current where base resistance falls halfway to minimum value (default value = &infin; Amp) <font color=red>Levels 2</font>

 RB:      Zero-bias base resistance (default value = 0 Ohm) <font color=red>Levels 2</font>

 RBM:     Minimum base resistance at high currents (default value = RB Ohm) <font color=red>Levels 2</font>

 RC:      Collector resistance (default value = 0 Ohm) <font color=red>Levels 2</font>

 RE:      Emitter resistance (default value = 0 Ohm) <font color=red>Levels 2</font>


<b>Junction Capacitor Parameters:</b>

 CJC:     Zero-bias base-collector depletion capacitance at reference temperature (CJC > 0) (default value = 1e-12 F) <font color=red>Levels 1,2</font>

 MJC:     Base-collector junction grading coefficient (default value = 0.33) <font color=red>Levels 1,2</font>

 VJC:     Base-collector built-in potential at reference temperature (default value = 0.75 Volt) <font color=red>Levels 1,2</font>

 CJE:     Zero-bias base-emitter depletion capacitance at reference temperature (CJE > 0) (default value = 1e-12 F) <font color=red>Levels 1,2</font>

 MJE:     Base-emitter junction grading coefficient (default value = 0.33) <font color=red>Levels 1,2</font>

 VJE:     Base-emitter built-in potential at reference temperature (default value = 0.75 Volt) <font color=red>Levels 1,2</font>

 CJS:     Zero-bias substrate depletion capacitance at reference temperature (CJS > 0) (default value = 1e-12 F) <font color=red>Levels 1,2</font>

 MJS:     Substrate junction grading coefficient (default value = 0.33) <font color=red>Levels 1,2</font>

 VJS:     Substrate built-in potential at reference temperature (default value = 0.75 Volt) <font color=red>Levels 1,2</font>

 XCJC:    Fraction of base-collector depletion capacitance connected to internal base node (default value = 1) <font color=red>Levels 1,2</font>

 FC:      Depletion capacitance factor for linearization (default value = 0.5) <font color=red>Levels 2</font>


<b>Transit Time Parameters:</b>

 TF:      Ideal forward transit time (default value = 0 sec) <font color=red>Levels 1,2</font>

 TR:      Ideal reverse transit time (default value = 0 sec) <font color=red>Levels 1,2</font>


<b>Temperature Compensation and Area Parameters:</b>

 Tnom:    Reference temperature (default value = 300.15 K) <font color=red>Levels 1,2</font>

 XTI:     Saturation current temperature exponent (default value = 3) <font color=red>Levels 1,2</font>

 XTB:     Forward and reverse beta temperature coefficient (default value = 0) <font color=red>Levels 1,2</font>

 EG:      Energy gap for temperature effect on saturation current (default value = 1.11 Volt) <font color=red>Levels 1,2</font>

 TRB1:    Linear temperature coefficient of zero-bias base resistance (default value = 0 1/K) <font color=red>Levels 2</font>

 TRB2:    Quadratic temperature coefficient of zero-bias base resistance (default value = 0 1/(K<sup>2</sup>)) <font color=red>Levels 2</font>

 TRM1:    Linear temperature coefficient of mimimum base resistance (default value = 0 1/K) <font color=red>Levels 2</font>

 TRM2:    Quadratic temperature coefficient of minimum base resistance (default value = 0 1/(K<sup>2</sup>)) <font color=red>Levels 2</font>

 TRC1:    Linear temperature coefficient of collector resistance (default value = 0 1/K) <font color=red>Levels 2</font>

 TRC2:    Quadratic temperature coefficient of collector resistance (default value = 0 1/(K<sup>2</sup>)) <font color=red>Levels 2</font>

 TRE1:    Linear temperature coefficient of emitter resistance (default value = 0 1/K) <font color=red>Levels 2</font>

 TRE2:    Quadratic temperature coefficient of emitter resistance (default value = 0 1/(K<sup>2</sup>)) <font color=red>Levels 2</font>


<b>Numerical Parameters:</b>

 EMin:    Minimum exponent for linearization of junction current (default value = -100) <font color=red>Levels 1,2</font>

 EMax:    Maximum exponent for linearization of junction current (default value = 40) <font color=red>Levels 1,2</font>

 GminDC:  Leakage conductance (default value = 1e-19 Mho) <font color=red>Levels 1,2</font>
</pre> <p>

<hr> <p>

<b>References:</b> <p>

<ol>
<li>Cellier, F.E. (1991), <i>Continuous System Modeling</i>, Springer-Verlag, New York.
<li>Hild, D.R. and F.E. Cellier (1994), \"Object-oriented electronic circuit modeling using Dymola,\"
    <i>Proc. OOS'94, SCS Object Oriented Simulation Conference</i>, Tempe, AZ, pp.68-75.
<li>Hild, D.R. (1993), <i>Circuit Modeling in Dymola</i>, MS Thesis, Dept. of Electr. &amp; Comp. Engr.,
    University of Arizona, Tucson.
<li> Massobrio, G. and P. Antognetti (1993), <i>Semiconductor Device Modeling with Spice</i>,
     2<sup>nd</sup> edition, McGraw Hill, New York.
<li>Schweisguth, M.C. and F.E. Cellier (1999), \"<a href=\"http://www.inf.ethz.ch/~fcellier/Publikationen/BJT.pdf\">A
    bond graph model of the bipolar junction transistor</a>,\" <i>Proc. SCS Intl. Conf. on Bond Graph Modeling</i>,
    San Francisco, CA, pp.344-349.
<li>Schweisguth, M.C. (1997), <a href=\"http://www.inf.ethz.ch/~fcellier/Publikationen/schweisguth_ms.pdf\"><i>Semiconductor
    Modeling with Bondgraphs</i></a>, MS Thesis, Dept. of Electr. &amp; Comp. Engr., University of Arizona, Tucson.
</ol>
</html>
"));
      BondLib.Spice.Utilities.PNPint Qpi(
        BF=BF,
        BR=BR,
        IS=IS,
        NF=NF,
        NR=NR,
        GminDC=GminDC,
        NC=NC,
        NE=NE,
        VAF=VAF,
        VAR=VAR,
        IKF=IKF,
        IKR=IKR,
        CJC=CJC,
        MJC=MJC,
        VJC=VJC,
        CJE=CJE,
        MJE=MJE,
        VJE=VJE,
        XCJC=XCJC,
        TF=TF,
        TR=TR,
        Tnom=Tnom,
        XTI=XTI,
        XTB=XTB,
        EG=EG,
        Area=Area,
        EMin=EMin,
        EMax=EMax,
        FC=FC,
        Level=Level,
        ISC=ISCeff,
        ISE=ISEeff,
        ISS=ISS) annotation (extent=[0,-40; 40,0]);
      BondLib.Junctions.J0p2 J0p2_1
        annotation (extent=[90,20; 110,40], rotation=180);
      Utilities.CbxS Cbx   annotation (extent=[-70,30; -30,50]);
      Utilities.RS2 Rc(
        Tnom=Tnom,
        R=RC,
        TR1=TRC1,
        TR2=TRC2,
        Area=Area,
        Level=Level) annotation (extent=[-10,58; -50,82]);
      Utilities.RS2 Re(
        Tnom=Tnom,
        Area=Area,
        Level=Level,
        R=RE,
        TR1=TRE1,
        TR2=TRE2) annotation (extent=[-10,-82; -50,-58]);
      BondLib.Interfaces.BondCon S "Substrate"
        annotation (extent=[-130,-70; -110,-50]);
      Utilities.CjS Dcs(
        Tnom=Tnom,
        XTI=XTI,
        XTB=XTB,
        EG=EG,
        Area=Area,
        Level=Level,
        BE=false,
        EMin=EMin,
        EMax=EMax,
        MJ=MJS,
        VJ=VJS,
        XCJ=1,
        Tau=0,
        IS=0,
        GminDC=0,
        CJ=CJS,
        FC=0,
        N=1)       annotation (extent=[70,66; 110,94]);
      Modelica.Blocks.Sources.Constant C0(k=0)
        annotation (extent=[96,44; 84,56]);
      Modelica.Blocks.Sources.Constant qb(k=1)
        annotation (extent=[116,44; 104,56]);
      equation
      connect(Rb.BondCon2, B5.fBondCon1)  annotation (points=[-20,-40; -20,-40],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p2_4.BondCon1, E)        annotation (points=[20,-120; 0,-120],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p2_3.BondCon2, C)        annotation (points=[20,120; 0,120],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_3.BondCon2, B)        annotation (points=[-110,1.22461e-015;
            -115,1.22461e-015; -115,0; -120,0], style(color=8, rgbcolor={192,
              192,192}));
      connect(J0p3_3.BondCon3, B21.BondCon1) annotation (points=[-100,10; -100,
            10.2], style(color=8, rgbcolor={192,192,192}));
      connect(B21.BondCon2, J1p3_5.BondCon1) annotation (points=[-100,30; -100,
            30], style(color=8, rgbcolor={192,192,192}));
      connect(B22.BondCon1, J1p3_5.BondCon2) annotation (points=[-100,50.2;
            -100,50], style(color=8, rgbcolor={192,192,192}));
      connect(J0p6_1.BondCon2, B25.fBondCon1) annotation (points=[80,-90; 80,
            -90], style(color=8, rgbcolor={192,192,192}));
      connect(B25.eBondCon1, J0p3_4.BondCon2) annotation (points=[100,-90; 100,
            -90], style(color=8, rgbcolor={192,192,192}));
      connect(B4.BondCon1, J1p3_1.BondCon3) annotation (points=[-60,-10.2; -60,
            -10], style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_5.BondCon3, B23.BondCon1) annotation (points=[-90,40; -89.8,
            40], style(color=8, rgbcolor={192,192,192}));
      connect(B4.BondCon2, Rb.BondCon1)  annotation (points=[-60,-30; -60,-40],
          style(color=8, rgbcolor={192,192,192}));
      connect(B13.BondCon1, J1p3_3.BondCon3) annotation (points=[9.8,70; 10,70],
          style(color=8, rgbcolor={192,192,192}));
      connect(B22.BondCon2, J0p4_1.BondCon1) annotation (points=[-100,70; -100,
            88; 0,88; 0,30; 10,30], style(color=8, rgbcolor={192,192,192}));
      connect(B9.BondCon1, J1p3_2.BondCon3) annotation (points=[60,40.2; 60,40],
          style(color=8, rgbcolor={192,192,192}));
      connect(B17.BondCon1, J1p3_4.BondCon3) annotation (points=[9.8,-70; 10,
            -70], style(color=8, rgbcolor={192,192,192}));
      connect(B10.eBondCon1, J0p3_4.BondCon1) annotation (points=[120,100; 120,
            108; 132,108; 132,-90; 120,-90], style(color=8, rgbcolor={192,192,
              192}));
      connect(B5.eBondCon1, J0p6_1.BondCon1) annotation (points=[0,-40; 8,-40;
            8,-52; 60,-52; 60,-90], style(color=8, rgbcolor={192,192,192}));
      connect(B19.eBondCon1, J0p6_1.BondCon5) annotation (points=[60,-20; 66,
            -20; 66,-80], style(color=8, rgbcolor={192,192,192}));
      connect(B24.eBondCon1, J0p6_1.BondCon6) annotation (points=[-10,40; -6,40;
            -6,12; 74,12; 74,-80], style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_4.BondCon3, H)        annotation (points=[110,-80; 120,-80;
            120,0], style(color=8, rgbcolor={192,192,192}));
      connect(B18.eBondCon1, J0p6_1.BondCon3) annotation (points=[-70,-70; -76,
            -70; -76,-88; 48,-88; 48,-100; 66,-100], style(color=8, rgbcolor={
              192,192,192}));
      connect(B14.eBondCon1, J0p6_1.BondCon4) annotation (points=[-70,70; -80,
            70; -80,-92; 44,-92; 44,-104; 74,-104; 74,-100], style(color=8,
            rgbcolor={192,192,192}));
      connect(Qpi.qb, Rb.qb)      annotation (points=[41.4286,-33.7143; 50,
            -33.7143; 50,-48; 12,-48; 12,-24; -40,-24; -40,-30], style(color=74,
            rgbcolor={0,0,127}));
      connect(B2.BondCon2, J1p3_1.BondCon2) annotation (points=[-50,
            1.22461e-015; -41,1.22461e-015; -41,0; -50,0], style(color=8,
            rgbcolor={192,192,192}));
      connect(B3.BondCon1, J1p3_1.BondCon1) annotation (points=[-70.2,
            -1.20011e-015; -70,-1.20011e-015; -70,0], style(color=8, rgbcolor={
              192,192,192}));
      connect(B3.BondCon2, J0p3_3.BondCon1) annotation (points=[-90,
            1.22461e-015; -90,-1.22461e-015], style(color=8, rgbcolor={192,192,
              192}));
      connect(J0p2_3.BondCon1, B12.BondCon2) annotation (points=[20,100; 20,100],
          style(color=8, rgbcolor={192,192,192}));
      connect(B12.BondCon1, J1p3_3.BondCon2) annotation (points=[20,80.2; 20,80],
          style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_3.BondCon1, B11.BondCon2) annotation (points=[20,60; 20,60],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p4_1.BondCon4, B11.BondCon1) annotation (points=[20,40; 20,40.2],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p4_1.BondCon3, B20.BondCon2) annotation (points=[20,20; 20,20],
          style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_4.BondCon2, B15.BondCon1) annotation (points=[20,-60; 20,
            -59.8], style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_4.BondCon1, B16.BondCon2) annotation (points=[20,-80; 20,-80],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p2_4.BondCon2, B16.BondCon1) annotation (points=[20,-100; 20,
            -99.8], style(color=8, rgbcolor={192,192,192}));
      connect(J0p4_1.BondCon2, B7.BondCon1) annotation (points=[30,30; 30.2,30],
          style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_2.BondCon1, B7.BondCon2) annotation (points=[50,30; 50,30],
          style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_2.BondCon2, B6.BondCon1) annotation (points=[70,30; 70.2,30],
          style(color=8, rgbcolor={192,192,192}));
      connect(Cbx.BondCon1, B23.BondCon2)   annotation (points=[-70,40; -70,40],
          style(color=8, rgbcolor={192,192,192}));
      connect(Cbx.BondCon2, B24.fBondCon1)   annotation (points=[-30,40; -30,40],
          style(color=8, rgbcolor={192,192,192}));
      connect(Qpi.ix, Cbx.ix)       annotation (points=[10,0; 10,10; -50,10;
            -50,30], style(color=74, rgbcolor={0,0,127}));
      connect(Rc.BondCon2, B14.fBondCon1) annotation (points=[-50,70; -50,70],
          style(color=8, rgbcolor={192,192,192}));
      connect(Rc.BondCon1, B13.BondCon2) annotation (points=[-10,70; -10,70],
          style(color=8, rgbcolor={192,192,192}));
      connect(Re.BondCon2, B18.fBondCon1) annotation (points=[-50,-70; -50,-70],
          style(color=8, rgbcolor={192,192,192}));
      connect(Re.BondCon1, B17.BondCon2) annotation (points=[-10,-70; -10,-70],
          style(color=8, rgbcolor={192,192,192}));
      connect(B6.BondCon2, J0p2_1.BondCon2) annotation (points=[90,30; 90,30],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p2_1.BondCon1, S)        annotation (points=[110,30; 136,30;
            136,-130; -120,-130; -120,-60], style(color=8, rgbcolor={192,192,
              192}));
      connect(C0.y, Dcs.C0) annotation (points=[83.4,50; 80,50; 80,66], style(
            color=74, rgbcolor={0,0,127}));
      connect(qb.y, Dcs.qb) annotation (points=[103.4,50; 100,50; 100,66],
          style(color=74, rgbcolor={0,0,127}));
      connect(Dcs.BondCon2, B10.fBondCon1) annotation (points=[110,80; 120,80],
          style(color=8, rgbcolor={192,192,192}));
      connect(Dcs.BondCon1, B9.BondCon2) annotation (points=[70,80; 60,80; 60,
            60], style(color=8, rgbcolor={192,192,192}));
      connect(Qpi.Bint, B2.BondCon1) annotation (points=[0,-20; -20,-20; -20,
            -1.20011e-015; -30.2,-1.20011e-015], style(color=8, rgbcolor={192,
              192,192}));
      connect(Qpi.Eint, B15.BondCon2) annotation (points=[20,-40; 20,-40],
          style(color=8, rgbcolor={192,192,192}));
      connect(B19.fBondCon1, Qpi.H) annotation (points=[40,-20; 40,-20], style(
            color=8, rgbcolor={192,192,192}));
      connect(B20.BondCon1, Qpi.Cint) annotation (points=[20,0.2; 20,0], style(
            color=8, rgbcolor={192,192,192}));
      end PNPvert;

      model NMOS "Spice-style NMOS transistor in bond graph technology"
        constant Real pi=Modelica.Constants.pi;
        constant Modelica.SIunits.Entropy k=Modelica.Constants.k
        "Boltzmann's constant";
        constant Modelica.SIunits.Permittivity e0=Modelica.Constants.epsilon_0
        "Permittivity of free space";
        constant Modelica.SIunits.ElectricCharge q=1.6021892e-19
        "Electron charge";
        constant Modelica.SIunits.ElectronNumberDensity ni=1.45e16
        "Intrinsic carrier concentration at 300 K";
        constant Real kox=3.9 "Dielectric constant of SiO2";
        constant Real ks=11.8 "Dielectric constant of Si";
        constant Real GapC1(unit="V/K")=7.02E-4
        "Sze's first bandgap correction factor Silicon";
        constant Modelica.SIunits.Temperature GapC2=1108.0
        "Sze's second bandgap correction factor Silicon";
        constant Modelica.SIunits.Voltage Fn0=3.25
        "Work function constant for computation of flat band voltage";
        constant Modelica.SIunits.Length TOXdef = 1e-7
        "Default oxide thickness";

        /* General parameters */
        parameter Integer Level(min=0,max=4) = 1 "Level of MOSFET model";

        /* Basic Geometric parameters */
        parameter Modelica.SIunits.Length L(min=0) = 1e-4 "Channel length"
          annotation(Dialog(group="Geometric parameters"));
        parameter Modelica.SIunits.Length W(min=0) = 1e-4 "Channel width"
          annotation(Dialog(group="Geometric parameters"));

        /* Basic Process parameters */
        parameter Real TPG(min=1,max=2) = 1 "Type of gate material"
          annotation(Dialog(group="Process parameters"));
        parameter Modelica.SIunits.Length TOX=TOXdef "Thin oxide thickness"
          annotation(Dialog(group="Process parameters"));
        parameter Real COX(unit="F/m2") = 0 "Specific capacitance of SiO2"
          annotation(Dialog(group="Process parameters"));
        parameter Modelica.SIunits.ElectronNumberDensity NSUB= 0
        "Substrate doping"
          annotation(Dialog(group="Process parameters"));

        /* Basic Electrical parameters */
        parameter Real U0(unit="m2/(V.s)") = 0
        "Surface mobility at reference temperature"
          annotation(Dialog(group="Electrical parameters"));
        parameter Modelica.SIunits.Voltage VT0=0
        "Zero-bias threshold voltage (VT0>0: enhancement type, VT0<0: depletion type)"
          annotation(Dialog(group="Electrical parameters", enable=(Level==0 or Level==1)));
        parameter Modelica.SIunits.InversePotential LAMBDA(min=0) = 0
        "Channel length modulation"
          annotation(Dialog(group="Electrical parameters"));
        parameter Modelica.SIunits.Length LD(min=0) = 0 "Lateral diffusion"
          annotation(Dialog(group="Electrical parameters"));
        parameter Modelica.SIunits.Length WD(min=0) = 0 "Width diffusion"
          annotation(Dialog(group="Electrical parameters"));
        parameter Modelica.SIunits.Transconductance KP=0
        "Transconductance parameter at reference temperature"
          annotation(Dialog(group="Electrical parameters"));
        parameter Real GAMMA(unit="V^0.5") = 0 "Body-effect parameter"
          annotation(Dialog(group="Electrical parameters"));
        parameter Modelica.SIunits.Voltage PHI=0
        "Surface inversion potential at reference temperature"
          annotation(Dialog(group="Electrical parameters"));
        parameter Real NSS(unit="1/m2") = 0 "Surface state density"
          annotation(Dialog(group="Electrical parameters"));

        /* Basic temperature compensation parameters */
        parameter Modelica.SIunits.Temperature Tnom=300.15
        "Reference temperature"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Modelica.SIunits.Voltage EG=1.11
        "Energy gap for temperature effect on saturation current at 0 K"
          annotation(Dialog(group="Temperature compensation parameters"));

        /* Level=1 electrical parameters */
        parameter Modelica.SIunits.Area AD=0 "Drain diffusion area"
          annotation(Dialog(tab="Level=1", group="Electrical parameters", enable=Level>0));
        parameter Modelica.SIunits.Length PD=0 "Drain perimeter width"
          annotation(Dialog(tab="Level=1", group="Electrical parameters", enable=Level>0));
        parameter Modelica.SIunits.Area AS=0 "Source diffusion area"
          annotation(Dialog(tab="Level=1", group="Electrical parameters", enable=Level>0));
        parameter Modelica.SIunits.Length PS=0 "Source perimeter width"
          annotation(Dialog(tab="Level=1", group="Electrical parameters", enable=Level>0));
        parameter Modelica.SIunits.Current ISD(min=0) = 0
        "Drain junction saturation current at reference temperature"
          annotation(Dialog(tab="Level=1", group="Electrical parameters", enable=Level>0));
        parameter Modelica.SIunits.Current ISS(min=0) = 0
        "Source junction saturation current at reference temperature"
          annotation(Dialog(tab="Level=1", group="Electrical parameters", enable=Level>0));
        parameter Modelica.SIunits.CurrentDensity JS(min=0) = 0
        "Saturation current density at reference temperature"
          annotation(Dialog(tab="Level=1", group="Electrical parameters", enable=Level>0));
        parameter Modelica.SIunits.Voltage PB(min=0.2) = 0.8
        "Built-in potential at reference temperature"
          annotation(Dialog(tab="Level=1", group="Electrical parameters", enable=Level>0));

        /* Level=1 parasitic resistance parameters */
        parameter Modelica.SIunits.Resistance RD(min=0) = 0.1
        "Drain Ohmic resistance"
          annotation(Dialog(tab="Level=1", group="Parasitic resistance parameters", enable=Level>0));
        parameter Modelica.SIunits.Resistance RS(min=0) = 0.1
        "Source Ohmic resistance"
          annotation(Dialog(tab="Level=1", group="Parasitic resistance parameters", enable=Level>0));

        /* Level=1 junction capacitance parameters */
        parameter Real CJ(unit="F/m2",min=0) = 0
        "Zero-bias bulk capacitance per square meter at reference temperature"
          annotation(Dialog(tab="Level=1", group="Junction capacitance parameters", enable=Level>0));
        parameter Real MJ(min=0.3,max=0.5) = 0.33
        "Bulk junction grading coefficient"
          annotation(Dialog(tab="Level=1", group="Junction capacitance parameters", enable=Level>0));
        parameter Real CJSW(unit="F/m",min=0) = 1e-9
        "Zero-bias perimeter capacitance per meter at reference temperature"
          annotation(Dialog(tab="Level=1", group="Junction capacitance parameters", enable=Level>0));
        parameter Real MJSW(min=0.3,max=0.5) = 0.33
        "Perimeter capacitance grading coefficient"
          annotation(Dialog(tab="Level=1", group="Junction capacitance parameters", enable=Level>0));
        parameter Real FC(min=0,max=0.95)= 0.5
        "Forward-bias depletion capacitance coefficient"
          annotation(Dialog(tab="Level=1", group="Junction capacitance parameters", enable=Level>0));
        parameter Real CGB0(unit="F/m",min=0) = 0
        "Gate-bulk overlap capacitance per meter"
          annotation(Dialog(tab="Level=1", group="Junction capacitance parameters", enable=Level>0));
        parameter Real CGD0(unit="F/m",min=0) = 0
        "Gate-drain overlap capacitance per meter"
          annotation(Dialog(tab="Level=1", group="Junction capacitance parameters", enable=Level>0));
        parameter Real CGS0(unit="F/m",min=0) = 0
        "Gate-source overlap capacitance per meter"
          annotation(Dialog(tab="Level=1", group="Junction capacitance parameters", enable=Level>0));

        /* Level=1 temperature compensation parameters */
        parameter Real TRD1(unit="1/K") = 0
        "Linear temperature coefficient of drain resistance"
          annotation(Dialog(tab="Level=1", group="Temperature compensation parameters", enable=Level>0));
        parameter Real TRD2(unit="1/K2") = 0
        "Quadratic temperature coefficient of drain resistance"
          annotation(Dialog(tab="Level=1", group="Temperature compensation parameters", enable=Level>0));
        parameter Real TRS1(unit="1/K") = 0
        "Linear temperature coefficient of source resistance"
          annotation(Dialog(tab="Level=1", group="Temperature compensation parameters", enable=Level>0));
        parameter Real TRS2(unit="1/K2") = 0
        "Quadratic temperature coefficient of source resistance"
          annotation(Dialog(tab="Level=1", group="Temperature compensation parameters", enable=Level>0));

        /* Level=2 process parameters */
        parameter Modelica.SIunits.Length XJ=0 "Metallurgical junction depth"
          annotation(Dialog(tab="Level=2", group="Process parameters", enable=Level>1));

        /* Level=2 electrical parameters */
        parameter Modelica.SIunits.ElectricFieldStrength UCRIT=1e6
        "Critical electric field for mobility"
          annotation(Dialog(tab="Level=2", group="Electrical parameters", enable=(Level==2 or Level==4)));
        parameter Real UEXP=0 "Exponential coefficient for mobility"
          annotation(Dialog(tab="Level=2", group="Electrical parameters", enable=(Level==2 or Level==4)));
        parameter Real UTRA(min=0,max=0.5) = 0 "Transverse field coefficient"
          annotation(Dialog(tab="Level=2", group="Electrical parameters", enable=(Level==2 or Level==4)));
        parameter Modelica.SIunits.ElectricFieldStrength ECRIT=0
        "Critical electric field for pinch-off"
          annotation(Dialog(tab="Level=2", group="Electrical parameters", enable=(Level==2 or Level==4)));
        parameter Real DELTA=0 "Width effect on threshold voltage"
          annotation(Dialog(tab="Level=2", group="Electrical parameters", enable=Level>1));
        parameter Modelica.SIunits.Velocity VMAX=0
        "Maximum drift velocity of carriers (VMAX=0: Frohman-Grove model, VMAX>0: Baum-Beneking model)"
          annotation(Dialog(tab="Level=2", group="Electrical parameters", enable=Level>1));
        parameter Real NFS(unit="1/m2") = 0 "Surface fast state density"
          annotation(Dialog(tab="Level=2", group="Electrical parameters", enable=Level>1));
        parameter Real NEFF(min=Modelica.Constants.small) = 1
        "Total channel charge coefficient"
          annotation(Dialog(tab="Level=2", group="Electrical parameters", enable=Level>1));
        parameter Real XQC=0
        "Coefficient of channel charge share (XQC<=0.5: Ward-Dutton model, XQC>0.5: Meyer model)"
          annotation(Dialog(tab="Level=2", group="Electrical parameters", enable=Level>1));

        /* Level=3 electrical parameters */
        parameter Real KAPPA(min=Modelica.Constants.small) = 0.2
        "Saturation field factor"
          annotation(Dialog(tab="Level=3", group="Electrical parameters", enable=Level == 3));
        parameter Real ETA=0 "Static feedback on threshold voltage"
          annotation(Dialog(tab="Level=3", group="Electrical parameters", enable=Level == 3));
        parameter Modelica.SIunits.InversePotential THETA=0
        "Mobility modulation"
          annotation(Dialog(tab="Level=3", group="Electrical parameters", enable=Level == 3));

        /* Numerical parameters */
        parameter Real EMin=-100
        "if x < EMin, the exp(x) function is linearized"
          annotation(Dialog(tab="Advanced", group="Numerical parameters", enable=Level>0));
        parameter Real EMax=40 "if x > EMax, the exp(x) function is linearized"
          annotation(Dialog(tab="Advanced", group="Numerical parameters", enable=Level>0));
        parameter Modelica.SIunits.Conductance GminDC=1e-12
        "Leakage conductance"
          annotation(Dialog(tab="Advanced", group="Numerical parameters"));

        /* Implement Spice-style MODCHK */
    protected
        parameter Integer Type = 1 "Type=1 for NMOS; Type=-1 for PMOS";
        parameter Modelica.SIunits.Permittivity es = ks*e0
        "Permittivity of silicon";
        parameter Modelica.SIunits.Permittivity eox = kox*e0
        "Permittivity of SiO2";
        parameter Modelica.SIunits.Length LDeff = LD;
        parameter Modelica.SIunits.Length Leff = L - 2*LDeff;
        parameter Modelica.SIunits.Length Weff = W - 2*WD;
        parameter Modelica.SIunits.Length TOXeff = if TOX == TOXdef and COX > 0 then eox/COX else TOX;
        parameter Real COXeff(unit="F/m2") = if COX > 0 then COX else eox/TOX;
        parameter Real CGB0eff(unit="F/m") = CGB0;
        parameter Real CGD0eff(unit="F/m") = CGD0;
        parameter Real CGS0eff(unit="F/m") = CGS0;
        parameter Real U0def(unit="m2/(V.s)") = 0.06;
        parameter Real U0tmp(unit="m2/(V.s)") = if KP == 0 and U0 == 0 then U0def else U0;
        parameter Modelica.SIunits.Transconductance KPeff = if KP > 0 then KP else U0tmp*COXeff;
        parameter Real U0eff(unit="m2/(V.s)") = if KP > 0 and U0 == 0 then KP/COXeff else U0tmp;
        parameter Real GAMMAtmp(unit="V^0.5") = if GAMMA == 0 and NSUB == 0 then 0.35 else GAMMA;
        parameter Modelica.SIunits.ElectronNumberDensity NSUBeff = if NSUB > 0 then NSUB else (GAMMAtmp*COXeff)^2/(2*es*q);
        parameter Real GAMMAeff(unit="V^0.5") = if GAMMA == 0 and NSUB > 0 then sqrt(2*es*q*NSUB)/COXeff else GAMMAtmp;
        parameter Modelica.SIunits.InversePotential LAMBDAeff = if LAMBDA > 0 then LAMBDA else
                                                                if Level <> 2 then sqrt(2*es/(q*NSUBeff)) else LAMBDA;
        parameter Modelica.SIunits.Voltage EGref = EG - GapC1*Tnom*Tnom/(Tnom + GapC2)
        "Energy gap for temperature effect on saturation current at reference temperature";
        parameter Modelica.SIunits.ElectronNumberDensity XNI = ni*(Tnom/300)^1.5 * exp(EGref*q/(2*k)*(1/300 - 1/Tnom))
        "Intrinsic carrier concentration at reference temperature";
        parameter Modelica.SIunits.Voltage VTref = k*Tnom/q
        "Thermal voltage at reference temperature";
        parameter Modelica.SIunits.Voltage PHIeff = if PHI > 0 then PHI else if NSUB > 0 then max({2*VTref*ln(NSUB/XNI),0.1}) else 0.6;
        parameter Modelica.SIunits.Voltage FermiG = Type*TPG*EGref/2;
        parameter Modelica.SIunits.Voltage Fn1 = if TPG == 0 then 3.2 else Fn0 + EGref/2 - FermiG;
        parameter Modelica.SIunits.Voltage FermiS = Type*PHIeff/2;
        parameter Modelica.SIunits.Voltage Fn2 = Fn1 - Fn0 - EGref/2 - FermiS;
        parameter Modelica.SIunits.Voltage VFB = Fn2 - NSS*q/COXeff;
        parameter Modelica.SIunits.Voltage Vcorr = Type*(GAMMAeff*sqrt(PHIeff) + PHIeff);
        parameter Modelica.SIunits.Voltage VT0eff = if Type*VT0 > 0 then VT0 else VFB + Vcorr;
        parameter Modelica.SIunits.Voltage VFBeff = if Type*VT0 > 0 then VT0 - Vcorr else VFB;
        parameter Real XD(unit="m/V^0.5") = sqrt(2*es/(q*NSUBeff))
        "Depletion factor";
        parameter Real CJeff(unit="F/m2") = if CJ > 0 then CJ else sqrt(es*q*NSUBeff/(2*PB));
        parameter Modelica.SIunits.Voltage EGeff = if Level == 3 and EG == 0 then 0.2 else EGref;
        parameter Modelica.SIunits.Voltage VBP = UCRIT*es/COXeff
        "Gate to channel critical voltage";
        parameter Modelica.SIunits.Current ISDeff = if ISD > 0 then ISD else if AD*JS > 0 then AD*JS else 1e-15;
        parameter Modelica.SIunits.Current ISSeff = if ISS > 0 then ISS else if AS*JS > 0 then AS*JS else 1e-15;
    public
      BondLib.Interfaces.BondCon G "Gate"
        annotation (extent=[-130,-10; -110,10]);
      BondLib.Interfaces.BondCon D "Drain"
        annotation (extent=[-10,110; 10,130]);
      BondLib.Interfaces.BondCon B "Bulk"
        annotation (extent=[110,-10; 130,10]);
      BondLib.Interfaces.BondCon S "Source"
        annotation (extent=[-10,-130; 10,-110]);
      BondLib.Interfaces.BondCon H "Heat"
        annotation (extent=[-70,-130; -50,-110]);
      annotation (
        Diagram(
          Text(
            extent=[-90,-114; -68,-128],
            style(color=10, rgbcolor={135,135,135}),
            string="Heat"),
          Text(
            extent=[2,-114; 24,-128],
            style(color=10, rgbcolor={135,135,135}),
            string="S"),
          Text(
            extent=[110,20; 132,6],
            style(color=10, rgbcolor={135,135,135}),
            string="B"),
          Text(
            extent=[4,126; 26,112],
            style(color=10, rgbcolor={135,135,135}),
            string="D"),
          Text(
            extent=[-130,20; -108,6],
            style(color=10, rgbcolor={135,135,135}),
            string="G"),
            Text(
              extent=[-138,60; -124,54],
              string="vds",
              style(color=10, rgbcolor={135,135,135})),
            Text(
              extent=[-138,-50; -124,-56],
              string="vds",
              style(color=10, rgbcolor={135,135,135})),
            Text(
              extent=[-54,136; -40,130],
              string="vds",
              style(color=10, rgbcolor={135,135,135})),
            Text(
              extent=[-98,136; -84,130],
              style(color=10, rgbcolor={135,135,135}),
            string="vbs"),
            Text(
              extent=[-138,94; -124,88],
              style(color=10, rgbcolor={135,135,135}),
            string="vbs"),
            Text(
              extent=[-140,-84; -126,-90],
              style(color=10, rgbcolor={135,135,135}),
            string="vbs")),
        Coordsys(extent=[-120,-120; 120,120]),
        Documentation(info="<html>
The Spice-style n-channel MOSFET model contains the internal MOSFET plus the three external gate capacitances and the drain and source resistances. <p>

<hr> <p>

<pre>
<b>Parameters:</b>

 <font color=red><b>Level:   MOSFET modeling level (default value = 3)
            Level=0: Static injection model
            Level=1: Shichman-Hodges model
            Level=2: Grove-Frohman model
            Level=3: Empirical model
            Level=4: Simplified Grove-Frohman model</b></font>


<b>Basic Geometric Parameters:</b>

 L:       Channel length (default value = 1e-4 m) <font color=red>Levels 0-4</font>

 W:       Channel width (default value = 1e-4 m) <font color=red>Levels 0-4</font>


<b>Basic Process Parameters:</b>

 TPG:     Type of gate material (default value = 1) <font color=red>Levels 0-4</font>
            <font color=blue><b>TPG = 0:  Aluminum gate</b></font>
            <font color=blue><b>TPG = 1:  Polysilicon gate</b></font>

 TOX:     Thin oxide thickness (default value = 1e-7 m) <font color=red>Levels 0-4</font>

 COX:     Specific capacitance of SiO2 (default value = 0 F/m<sup>2</sup>) <font color=red>Levels 0-4</font>

 NSUB:    Substrate doping (default value = 0 1/m<sup>3</sup>) <font color=red>Levels 0-4</font>


<b>Basic Electrical Parameters:</b>

 U0:      Surface mobility at reference temperature (default value = 0 m<sup>2</sup>/(V*s)) <font color=red>Levels 0-4</font>

 VT0:     Zero-bias threshold voltage (default value = 0 Volt) <font color=red>Levels 0,1</font>
            <font color=blue><b>VT0 &gt; 0 for enhancement MOSFET
            VT0 &lt; 0 for depletion MOSFET</b></font>

 LAMBDA:  Channel length modulation  (default value = 0 1/Volt) <font color=red>Levels 0-4</font>

 LD:      Lateral diffusion (default value = 0 m) <font color=red>Levels 0-4</font>

 WD:      Width diffusion (default value = 0 m) <font color=red>Levels 0-4</font>

 KP:      Transconductance parameter at reference temperature (default value = 0 Amp/Volt<sup>2</sup>) <font color=red>Levels 0-4</font>

 GAMMA:   Body-effect parameter (default value = 0 Volt<sup>0.5</sup>) <font color=red>Levels 0-4</font>

 PHI:     Surface inversion potential at reference temperature (default value = 0 Volt) <font color=red>Levels 0-4</font>

 NSS:     Surface state density (default value = 0 1/m<sup>2</sup>) <font color=red>Levels 0-4</font>


<b>Basic Temperature Compensation Parameters:</b>

 Tnom:    Reference temperature (default value = 300.15 K) <font color=red>Levels 0-4</font>

 EG:      Energy gap for temperature effect on saturation current at 0 K (default value = 1.11 Volt) <font color=red>Levels 0-4</font>


<b>Level=1 Electrical Parameters:</b>

 AD:      Drain diffusion area (default value = 0 m<sup>2</sup>) <font color=red>Levels 1-4</font>

 PD:      Drain perimeter width (default value = 0 m) <font color=red>Levels 1-4</font>

 AS:      Source diffusion area (default value = 0 m<sup>2</sup>) <font color=red>Levels 1-4</font>

 PS:      Source perimeter width (default value = 0 m) <font color=red>Levels 1-4</font>

 ISD:     Drain junction saturation current at reference temperature (default value = 0 Amp) <font color=red>Levels 1-4</font>

 ISS:     Source junction saturation current at reference temperature (default value = 0 Amp) <font color=red>Levels 1-4</font>

 JS:      Transport saturation current density (default value = 0 Amp/m<sup>2</sup>) <font color=red>Levels 1-4</font>

 PB:      Built-in potential at reference temperature (default value = 0.8 Volt) <font color=red>Levels 1-4</font>


<b>Level=1 Parasitic Resistor Parameters:</b>

 RD:      Drain Ohmic resistance (default value = 1 Ohm) <font color=red>Levels 1-4</font>

 RS:      Source Ohmic resistance (default value = 1 Ohm) <font color=red>Levels 1-4</font>


<b>Level=1 Junction Capacitance Parameters:</b>

 CJ:      Zero-bias bulk capacitance per square meter at reference temperature (default value = 0 F/m<sup>2</sup>) <font color=red>Levels 1-4</font>

 MJ:      Bulk junction grading coefficient (default value = 0.33) <font color=red>Levels 1-4</font>

 CJSW:    Zero-bias perimeter capacitance per meter at reference temperature (default value = 1e-9 F/m) <font color=red>Levels 1-4</font>

 MJSW:    Perimeter capacitance grading coefficient (default value = 0.33) <font color=red>Levels 1-4</font>

 FC:      Forward-bias depletion capacitance coefficient (default value = 0.5) <font color=red>Levels 1-4</font>

 CGB0:    Gate-bulk overlap capacitance per meter (default value = 0 F/m) <font color=red>Levels 1-4</font>

 CGD0:    Gate-drain overlap capacitance per meter (default value = 0 F/m) <font color=red>Levels 1-4</font>

 CGS0:    Gate-source overlap capacitance per meter (default value = 0 F/m) <font color=red>Levels 1-4</font>


<b>Level=1 Temperature Compensation Parameters:</b>

 TRD1:    Linear temperature coefficient of drain resistance (default value = 0 1/K) <font color=red>Levels 1-4</font>

 TRD2:    Quadratic temperature coefficient of drain resistance (default value = 0 1/K<sup>2</sup>) <font color=red>Levels 1-4</font>

 TRS1:    Linear temperature coefficient of source resistance (default value = 0 1/K) <font color=red>Levels 1-4</font>

 TRS2:    Quadratic temperature coefficient of source resistance (default value = 0 1/K<sup>2</sup>) <font color=red>Levels 1-4</font>


<b>Level=2 Process Parameters:</b>

 XJ:      Metallurgical Junction Depth (default value = 0 m) <font color=red>Levels 2-4</font>


<b>Level=2 Electrical Parameters:</b>

 UCRIT:   Critical electric field for mobility (default value = 1e6 V/m) <font color=red>Levels 2,4</font>

 UEXP:    Exponential coefficient for mobility (default value = 0) <font color=red>Levels 2,4</font>

 UTRA:    Transverse field coefficient (default value = 0) <font color=red>Levels 2,4</font>

 ECRIT:   Critical electric field for pinch-off (default value = 0 V/m) <font color=red>Levels 2,4</font>

 DELTA:   Width effect on threshold voltage (default value = 0) <font color=red>Levels 2-4</font>

 VMAX:    Maximum drift velocity of carriers (default value = 0 m/s) <font color=red>Levels 2-4</font>
            <font color=blue><b>VMAX = 0: Frohman-Grove model of drain current computation (Levels 2,4)</b></font>
            <font color=blue><b>VMAX &gt; 0: Baum-Beneking model of drain current computation (Levels 2,4)</b></font>

 NFS:     Surface fast state density (default value = 0 1/m<sup>2</sup>) <font color=red>Levels 2-4</font>

 NEFF:    Total channel charge coefficient (default value = 1) <font color=red>Levels 2,4</font>

 XQC:     Coefficient of channel charge share (default value = 0) <font color=red>Levels 2-4</font>
            <font color=blue><b>XQC &lt;= 0.5: Ward-Dutton model of gate capacitance computation</b></font>
            <font color=blue><b>XQC &gt;  0.5: Meyer model of gate capacitance computation</b></font>


<b>Level=3 Electrical Parameters:</b>

 KAPPA:   Saturation field factor (default value = 0.2) <font color=red>Level 3</font>

 ETA:     Static feedback on threshold voltage (default value = 0) <font color=red>Level 3</font>

 THETA:   Mobility modulation (default value = 0 1/Volt) <font color=red>Level 3</font>


<b>Numerical Parameters:</b>

 EMin:    if x &lt; EMin, the exp(x) function is linearized (default value = -100) <font color=red>Levels 1-4</font>

 EMax:    if x &gt; EMax, the exp(x) function is linearized (default value = 40) <font color=red>Levels 1-4</font>

 GminDC:  Leakage conductance (default value = 1e-12 mho) <font color=red>Levels 0-4</font>
</pre> <p>

<hr> <p>

<b>References:</b> <p>

<ol>
<li> Cellier, F.E. (1991), <i>Continuous System Modeling</i>, Springer-Verlag, New York.
<li> Massobrio, G. and P. Antognetti (1993), <i>Semiconductor Device Modeling with Spice</i>,
     2<sup>nd</sup> edition, McGraw Hill, New York.
</ol>
</html>
"),     Icon(
          Text(
            extent=[-80,110; 80,70],
            style(color=3, rgbcolor={0,0,255}),
            string="%name"),
            Line(points=[0,100; 0,-100],
                                       style(
                color=9,
                rgbcolor={149,164,171},
                thickness=2,
                fillColor=74,
                rgbfillColor={0,0,185},
                fillPattern=1)),
            Text(
              extent=[-114,92; 116,-96],
              style(color=0),
            string="NMOS"),
            Line(points=[-20,-80; 0,-100],style(
                color=9,
                rgbcolor={149,164,171},
                thickness=2,
                fillColor=74,
                rgbfillColor={0,0,185},
                fillPattern=1)),
            Line(points=[0,-100; 20,-80],style(
                color=9,
                rgbcolor={149,164,171},
                thickness=2,
                fillColor=74,
                rgbfillColor={0,0,185},
                fillPattern=1)),
            Line(points=[10,50; 108,50], style(
                color=9,
                rgbcolor={149,164,171},
                thickness=2,
                fillColor=74,
                rgbfillColor={0,0,185},
                fillPattern=1)),
            Line(points=[10,50; 40,60],  style(
                color=9,
                rgbcolor={149,164,171},
                thickness=2,
                fillColor=74,
                rgbfillColor={0,0,185},
                fillPattern=1)),
            Line(points=[10,50; 40,40],  style(
                color=9,
                rgbcolor={149,164,171},
                thickness=2,
                fillColor=74,
                rgbfillColor={0,0,185},
                fillPattern=1)),
          Text(
            extent=[4,126; 26,112],
            style(color=10, rgbcolor={135,135,135}),
            string="D"),
          Text(
            extent=[110,20; 132,6],
            style(color=10, rgbcolor={135,135,135}),
            string="B"),
          Text(
            extent=[2,-114; 24,-128],
            style(color=10, rgbcolor={135,135,135}),
            string="S"),
          Text(
            extent=[-90,-114; -68,-128],
            style(color=10, rgbcolor={135,135,135}),
            string="Heat"),
          Text(
            extent=[-130,20; -108,6],
            style(color=10, rgbcolor={135,135,135}),
            string="G")));
      BondLib.Spice.Utilities.NMOSint NMOSint(
        Level=Level,
        Tnom=Tnom,
        NSS=NSS,
        NFS=NFS,
        NEFF=NEFF,
        XJ=XJ,
        AD=AD,
        Type=1,
        VT0=VT0eff,
        VFB=VFBeff,
        KP=KPeff,
        GAMMA=GAMMAeff,
        PHI=PHIeff,
        LAMBDA=LAMBDAeff,
        TOX=TOXeff,
        COX=COXeff,
        NSUB=NSUBeff,
        L=Leff,
        W=Weff,
        LD=LDeff,
        PD=PD,
        AS=AS,
        PS=PS,
        XD=XD,
        TPG=TPG,
        U0=U0eff,
        UCRIT=UCRIT,
        UEXP=UEXP,
        UTRA=UTRA,
        VMAX=VMAX,
        ECRIT=ECRIT,
        XQC=XQC,
        DELTA=DELTA,
        KAPPA=KAPPA,
        ETA=ETA,
        THETA=THETA,
        VBP=VBP,
        ISD=ISDeff,
        ISS=ISSeff,
        PB=PB,
        CJ=CJeff,
        MJ=MJ,
        CJSW=CJSW,
        MJSW=MJSW,
        FC=FC,
        GminDC=GminDC,
        EG=EGeff,
        EMin=EMin,
        EMax=EMax) annotation (extent=[-20,-20; 20,20]);
      BondLib.Junctions.J0p4 J0p4_2 annotation (extent=[90,-10; 110,10]);
      BondLib.Junctions.J0p4 J0p4_1 annotation (extent=[-10,80; 10,100]);
      BondLib.Bonds.Bond B1 annotation (extent=[-40,-10; -20,10],rotation=0);
      BondLib.Bonds.Bond B2 annotation (extent=[-10,20; 10,40],  rotation=-90);
      BondLib.Bonds.Bond B3 annotation (extent=[-10,-40; 10,-20],rotation=-90);
      BondLib.Bonds.Bond B4 annotation (extent=[20,-10; 40,10],  rotation=-180);
      BondLib.Junctions.J1p3 J1p3_1
        annotation (extent=[10,40; -10,60],rotation=90);
      BondLib.Bonds.Bond B5 annotation (extent=[-10,60; 10,80],  rotation=-90);
      BondLib.Junctions.J1p3 J1p3_2
        annotation (extent=[10,-60; -10,-40],
                                           rotation=90);
      BondLib.Bonds.Bond B6 annotation (extent=[-10,-80; 10,-60],rotation=-90);
      BondLib.Bonds.Bond B7 annotation (extent=[-30,40; -10,60], rotation=-180);
      BondLib.Bonds.Bond B8 annotation (extent=[-30,-60; -10,-40],
                                                                 rotation=-180);
      BondLib.Spice.Utilities.RM Rd(
        Tnom=Tnom,
        R=RD,
        Level=Level,
        TR1=TRD1,
        TR2=TRD2) annotation (extent=[-30,40; -50,60]);
      BondLib.Spice.Utilities.RM Rs(
        Tnom=Tnom,
        Level=Level,
        R=RS,
        TR1=TRS1,
        TR2=TRS2) annotation (extent=[-30,-60; -50,-40]);
      BondLib.Bonds.Bond B9 annotation (extent=[-30,80; -10,100],rotation=0);
      BondLib.Junctions.J1p3 J1p3_3
        annotation (extent=[-50,80; -30,100],
                                           rotation=180);
      BondLib.Bonds.Bond B10
                            annotation (extent=[-70,80; -50,100],rotation=0);
      BondLib.Bonds.Bond B11
                            annotation (extent=[-50,100; -30,120],
                                                                 rotation=90);
      BondLib.Spice.Utilities.Cgd Cgd1(
        Tnom=Tnom,
        Level=Level,
        Type=1,
        VFB=VFBeff,
        KP=KPeff,
        GAMMA=GAMMAeff,
        PHI=PHIeff,
        COX=COXeff,
        NFS=NFS,
        XJ=XJ,
        L=Leff,
        W=Weff,
        LD=LDeff,
        XD=XD,
        XQC=XQC,
        DELTA=DELTA,
        ETA=ETA,
        CGD0=CGD0eff,
        EG=EGeff) annotation (extent=[-50,132; -90,108]);
      BondLib.Bonds.Bond B12
                            annotation (extent=[-100,20; -80,40],rotation=90);
      BondLib.Junctions.J1p3 J1p3_4
        annotation (extent=[-100,40; -80,60],
                                           rotation=270);
      BondLib.Bonds.Bond B13
                            annotation (extent=[-100,60; -80,80],rotation=90);
      BondLib.Bonds.Bond B14
                            annotation (extent=[-120,40; -100,60],
                                                                 rotation=-180);
      BondLib.Spice.Utilities.Cgb Cgb1(
        Tnom=Tnom,
        Level=Level,
        Type=1,
        VFB=VFBeff,
        KP=KPeff,
        GAMMA=GAMMAeff,
        PHI=PHIeff,
        COX=COXeff,
        NFS=NFS,
        XJ=XJ,
        L=Leff,
        W=Weff,
        LD=LDeff,
        XD=XD,
        XQC=XQC,
        DELTA=DELTA,
        ETA=ETA,
        CGB0=CGB0eff,
        EG=EGeff) annotation (extent=[-108,54; -132,94], rotation=90);
      BondLib.Junctions.J0p5 J0p5_1
        annotation (extent=[-10,-100; 10,-80], rotation=90);
      BondLib.Bonds.Bond B15
                            annotation (extent=[20,-90; 40,-70], rotation=-180);
      BondLib.Junctions.J1p3 J1p3_5
        annotation (extent=[40,-90; 60,-70],
                                           rotation=180);
      BondLib.Bonds.Bond B16
                            annotation (extent=[60,-90; 80,-70], rotation=-180);
      BondLib.Bonds.fBond B17
        annotation (extent=[40,-70; 60,-50], rotation=90);
      BondLib.Sensors.De Vbs annotation (extent=[40,-50; 60,-30], rotation=90);
      BondLib.Bonds.Bond B18
                            annotation (extent=[10,80; 30,100],  rotation=0);
      BondLib.Junctions.J1p3 J1p3_6
        annotation (extent=[30,100; 50,80],rotation=180);
      BondLib.Bonds.Bond B19
                            annotation (extent=[50,80; 70,100],  rotation=0);
      BondLib.Bonds.fBond B20
                             annotation (extent=[30,60; 50,80], rotation=-90);
      BondLib.Sensors.De Vds
        annotation (extent=[30,40; 50,60],         rotation=-90);
      BondLib.Junctions.J0p5 J0p5_2
        annotation (extent=[-96,10; -76,-10],  rotation=0);
      BondLib.Bonds.Bond B21
                            annotation (extent=[-96,-30; -76,-10],
                                                                 rotation=-90);
      BondLib.Junctions.J1p3 J1p3_7
        annotation (extent=[-76,-50; -96,-30],
                                           rotation=90);
      BondLib.Bonds.Bond B22
                            annotation (extent=[-96,-70; -76,-50],
                                                                 rotation=-90);
      BondLib.Bonds.Bond B23
                            annotation (extent=[-116,-50; -96,-30],
                                                                 rotation=-180);
      BondLib.Spice.Utilities.Cgs Cgs1(
        Tnom=Tnom,
        Level=Level,
        Type=1,
        VFB=VFBeff,
        KP=KPeff,
        GAMMA=GAMMAeff,
        PHI=PHIeff,
        COX=COXeff,
        NFS=NFS,
        XJ=XJ,
        L=Leff,
        W=Weff,
        LD=LDeff,
        XD=XD,
        XQC=XQC,
        DELTA=DELTA,
        ETA=ETA,
        CGS0=CGS0eff,
        EG=EGeff) annotation (extent=[-132,-90; -108,-50], rotation=-90);
      BondLib.Junctions.J0p6 J0p6_1
        annotation (extent=[-70,-106; -50,-86],  rotation=90);
      Bonds.eBond B24
        annotation (extent=[-130,-110; -110,-90],
                                                rotation=-90);
      Bonds.eBond B25
        annotation (extent=[-70,40; -50,60],    rotation=-180);
      Bonds.eBond B26
        annotation (extent=[-70,-60; -50,-40],  rotation=-180);
      Bonds.eBond B27
        annotation (extent=[-30,-30; -10,-10],  rotation=-180);
      Bonds.eBond B28
        annotation (extent=[-110,110; -90,130], rotation=-180);
      Bonds.eBond B29
        annotation (extent=[-130,94; -110,114], rotation=-270);
      Junctions.J0p3 J0p3_1         annotation (extent=[-50,-30; -30,-10]);
      Bonds.eBond B30
        annotation (extent=[-64,-46; -44,-26],  rotation=-180);
      algorithm
        assert(NSUBeff > XNI, "NSUB must be larger than intrinsic carrier concentration");
        assert(LAMBDAeff < 0.2, "LAMBDA value too large, leads to numerical instability");
        assert(Leff > 0, "The effective channel length must be larger than zero");

      equation
      connect(J0p4_2.BondCon2, B)        annotation (points=[110,0; 120,0], style(color=8,
            rgbcolor={192,192,192}));
      connect(J0p4_1.BondCon4, D)        annotation (points=[0,100; 0,120],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p4_2.BondCon1, B4.BondCon1) annotation (points=[90,0; 54.9,0;
            54.9,1.20011e-015; 39.8,1.20011e-015], style(color=8, rgbcolor={192,
              192,192}));
      connect(J1p3_1.BondCon1, B2.BondCon1) annotation (points=[6.12303e-016,40;
            -6.00057e-016,40; -6.00057e-016,39.8], style(color=8, rgbcolor={192,
              192,192}));
      connect(B5.BondCon2, J1p3_1.BondCon2) annotation (points=[6.12303e-016,60;
            -6.12303e-016,60], style(color=8, rgbcolor={192,192,192}));
      connect(J0p4_1.BondCon3, B5.BondCon1) annotation (points=[0,80; 0,79.8;
            -6.00057e-016,79.8], style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_2.BondCon2, B3.BondCon2) annotation (points=[-6.12303e-016,
            -40; 6.12303e-016,-40], style(color=8, rgbcolor={192,192,192}));
      connect(B6.BondCon1, J1p3_2.BondCon1) annotation (points=[-6.00057e-016,
            -60.2; -6.00057e-016,-63.1; 6.12303e-016,-63.1; 6.12303e-016,-60],
          style(color=8, rgbcolor={192,192,192}));
      connect(B7.BondCon1, J1p3_1.BondCon3) annotation (points=[-10.2,50; -10,
            50], style(color=8, rgbcolor={192,192,192}));
      connect(B8.BondCon1, J1p3_2.BondCon3) annotation (points=[-10.2,-50; -10,
            -50], style(color=8, rgbcolor={192,192,192}));
      connect(Rd.BondCon1, B7.BondCon2) annotation (points=[-30,50; -30,50],
          style(color=8, rgbcolor={192,192,192}));
      connect(Rs.BondCon1, B8.BondCon2) annotation (points=[-30,-50; -30,-50],
          style(color=8, rgbcolor={192,192,192}));
      connect(B9.BondCon2, J0p4_1.BondCon1) annotation (points=[-10,90; -10,90],
          style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_3.BondCon1, B9.BondCon1) annotation (points=[-30,90; -29.8,
            90], style(color=8, rgbcolor={192,192,192}));
      connect(B10.BondCon2, J1p3_3.BondCon2) annotation (points=[-50,90; -50,90],
          style(color=8, rgbcolor={192,192,192}));
      connect(B11.BondCon1, J1p3_3.BondCon3) annotation (points=[-40,100.2; -40,
            100], style(color=8, rgbcolor={192,192,192}));
      connect(Cgd1.BondCon1, B11.BondCon2) annotation (points=[-50,120; -40,120],
          style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_4.BondCon2, B12.BondCon2) annotation (points=[-90,40; -90,40],
          style(color=8, rgbcolor={192,192,192}));
      connect(B13.BondCon1, J1p3_4.BondCon1) annotation (points=[-90,60.2; -90,
            60], style(color=8, rgbcolor={192,192,192}));
      connect(B14.BondCon1, J1p3_4.BondCon3) annotation (points=[-100.2,50;
            -100,50], style(color=8, rgbcolor={192,192,192}));
      connect(B14.BondCon2, Cgb1.BondCon1) annotation (points=[-120,50; -120,54],
          style(color=8, rgbcolor={192,192,192}));
      connect(B13.BondCon2, J0p4_2.BondCon4) annotation (points=[-90,80; -90,
            104; 100,104; 100,10], style(color=8, rgbcolor={192,192,192}));
      connect(B6.BondCon2, J0p5_1.BondCon2) annotation (points=[6.12303e-016,
            -80; 6.12303e-016,-80], style(color=8, rgbcolor={192,192,192}));
      connect(J0p5_1.BondCon1, S)        annotation (points=[-6.12303e-016,-100;
            0,-100; 0,-120], style(color=8, rgbcolor={192,192,192}));
      connect(B15.BondCon1, J1p3_5.BondCon2) annotation (points=[39.8,-80; 40,
            -80], style(color=8, rgbcolor={192,192,192}));
      connect(J0p5_1.BondCon4, B15.BondCon2) annotation (points=[10,-86; 10,-80;
            20,-80], style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_5.BondCon1, B16.BondCon2) annotation (points=[60,-80; 60,-80],
          style(color=8, rgbcolor={192,192,192}));
      connect(B17.eBondCon1, J1p3_5.BondCon3) annotation (points=[50,-70; 50,
            -70], style(color=8, rgbcolor={192,192,192}));
      connect(Vbs.BondCon1, B17.fBondCon1) annotation (points=[50,-50; 50,-50],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p4_2.BondCon3, B16.BondCon1) annotation (points=[100,-10; 100,
            -80; 79.8,-80], style(color=8, rgbcolor={192,192,192}));
      connect(J0p4_1.BondCon2, B18.BondCon1) annotation (points=[10,90; 10.2,90],
          style(color=8, rgbcolor={192,192,192}));
      connect(B18.BondCon2, J1p3_6.BondCon2) annotation (points=[30,90; 30,90],
          style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_6.BondCon1, B19.BondCon1) annotation (points=[50,90; 50.2,90],
          style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_6.BondCon3, B20.eBondCon1) annotation (points=[40,80; 40,80],
          style(color=8, rgbcolor={192,192,192}));
      connect(B20.fBondCon1, Vds.BondCon1) annotation (points=[40,60; 40,60],
          style(color=8, rgbcolor={192,192,192}));
      connect(B19.BondCon2, J0p5_1.BondCon3) annotation (points=[70,90; 130,90;
            130,-94; 10,-94], style(color=8, rgbcolor={192,192,192}));
      connect(J0p5_2.BondCon2, B1.BondCon1) annotation (points=[-76,0; -39.8,0],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p5_2.BondCon1, G)        annotation (points=[-96,0; -120,0],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p5_2.BondCon3, B12.BondCon1) annotation (points=[-90,10; -90,
            20.2], style(color=8, rgbcolor={192,192,192}));
      connect(J0p5_2.BondCon4, B10.BondCon1) annotation (points=[-82,10; -76,10;
            -76,90; -69.8,90], style(color=8, rgbcolor={192,192,192}));
      connect(J0p5_2.BondCon5, B21.BondCon1) annotation (points=[-86,-10; -86,
            -10.2], style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_7.BondCon2, B21.BondCon2) annotation (points=[-86,-30; -86,
            -30], style(color=8, rgbcolor={192,192,192}));
      connect(B22.BondCon1, J1p3_7.BondCon1) annotation (points=[-86,-50.2; -86,
            -50], style(color=8, rgbcolor={192,192,192}));
      connect(B23.BondCon1, J1p3_7.BondCon3) annotation (points=[-96.2,-40; -96,
            -40], style(color=8, rgbcolor={192,192,192}));
      connect(Cgs1.BondCon1, B23.BondCon2) annotation (points=[-120,-50; -120,
            -40; -116,-40], style(color=8, rgbcolor={192,192,192}));
      connect(B22.BondCon2, J0p5_1.BondCon5) annotation (points=[-86,-70; -86,
            -76; -10,-76; -10,-90], style(color=8, rgbcolor={192,192,192}));
      connect(Vds.OutPort1, Cgd1.u1) annotation (points=[40,39; 40,30; 80,30;
            80,138; -58,138; -58,132], style(color=74, rgbcolor={0,0,127}));
      connect(Vbs.OutPort1, Cgd1.u2) annotation (points=[50,-29; 50,28; 82,28;
            82,140; -82,140; -82,132], style(color=74, rgbcolor={0,0,127}));
      connect(Cgb1.u2, Cgd1.u2) annotation (points=[-132,86; -140,86; -140,140;
            -82,140; -82,132], style(color=74, rgbcolor={0,0,127}));
      connect(Cgb1.u1, Cgd1.u1) annotation (points=[-132,62; -138,62; -138,138;
            -58,138; -58,132], style(color=74, rgbcolor={0,0,127}));
      connect(Cgs1.u1, Cgb1.u1) annotation (points=[-132,-58; -138,-58; -138,62;
            -132,62], style(color=74, rgbcolor={0,0,127}));
      connect(Cgs1.u2, Cgb1.u2) annotation (points=[-132,-82; -140,-82; -140,86;
            -132,86], style(color=74, rgbcolor={0,0,127}));
      connect(NMOSint.y1, Cgd1.u)  annotation (points=[-10,20; -10,32; -62,32;
            -62,98; -70,98; -70,108], style(color=74, rgbcolor={0,0,127}));
      connect(Cgb1.u, Cgd1.u) annotation (points=[-108,74; -104,74; -104,98;
            -70,98; -70,108], style(color=74, rgbcolor={0,0,127}));
      connect(Cgb1.u, Cgs1.u) annotation (points=[-108,74; -104,74; -104,-70;
            -108,-70], style(color=74, rgbcolor={0,0,127}));
      connect(J0p6_1.BondCon1, H)        annotation (points=[-60,-106; -60,-120],
          style(color=8, rgbcolor={192,192,192}));
      connect(Cgs1.BondCon2, B24.fBondCon1) annotation (points=[-120,-90; -120,
            -90], style(color=8, rgbcolor={192,192,192}));
      connect(Rd.BondCon2, B25.fBondCon1) annotation (points=[-50,50; -50,50],
          style(color=8, rgbcolor={192,192,192}));
      connect(B26.fBondCon1, Rs.BondCon2) annotation (points=[-50,-50; -50,-50],
          style(color=8, rgbcolor={192,192,192}));
      connect(B28.fBondCon1, Cgd1.BondCon2) annotation (points=[-90,120; -90,
            120], style(color=8, rgbcolor={192,192,192}));
      connect(B29.fBondCon1, Cgb1.BondCon2) annotation (points=[-120,94; -120,
            94], style(color=8, rgbcolor={192,192,192}));
      connect(B26.eBondCon1, J0p6_1.BondCon3) annotation (points=[-70,-50; -74,
            -50; -74,-66; -40,-66; -40,-100; -50,-100], style(color=8, rgbcolor=
             {192,192,192}));
      connect(B25.eBondCon1, J0p6_1.BondCon4) annotation (points=[-70,50; -74,
            50; -74,10; -68,10; -68,-38; -76,-38; -76,-68; -50,-68; -50,-92],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_1.BondCon2, B27.eBondCon1) annotation (points=[-30,-20; -30,
            -20], style(color=8, rgbcolor={192,192,192}));
      connect(B30.fBondCon1, J0p3_1.BondCon3) annotation (points=[-44,-36; -40,
            -36; -40,-30], style(color=8, rgbcolor={192,192,192}));
      connect(B30.eBondCon1, J0p6_1.BondCon2) annotation (points=[-64,-36; -78,
            -36; -78,-70; -60,-70; -60,-86], style(color=8, rgbcolor={192,192,
              192}));
      connect(B24.eBondCon1, J0p3_1.BondCon1) annotation (points=[-120,-110;
            -120,-114; -102,-114; -102,-20; -50,-20], style(color=8, rgbcolor={
              192,192,192}));
      connect(J0p6_1.BondCon6, B29.eBondCon1) annotation (points=[-70,-92; -100,
            -92; -100,-116; -142,-116; -142,118; -120,118; -120,114], style(
            color=8, rgbcolor={192,192,192}));
      connect(J0p6_1.BondCon5, B28.eBondCon1) annotation (points=[-70,-100; -98,
            -100; -98,-118; -144,-118; -144,120; -110,120], style(color=8,
            rgbcolor={192,192,192}));
      connect(B1.BondCon2, NMOSint.Gint) annotation (points=[-20,0; -20,0],
          style(color=8, rgbcolor={192,192,192}));
      connect(B3.BondCon1, NMOSint.Sint) annotation (points=[-6.00057e-016,
            -20.2; 0,-20.2; 0,-20], style(color=8, rgbcolor={192,192,192}));
      connect(NMOSint.H, B27.fBondCon1) annotation (points=[-10,-20; -10,-20],
          style(color=8, rgbcolor={192,192,192}));
      connect(B4.BondCon2, NMOSint.Bint) annotation (points=[20,-1.22461e-015;
            20,0], style(color=8, rgbcolor={192,192,192}));
      connect(B2.BondCon2, NMOSint.Dint) annotation (points=[6.12303e-016,20; 0,
            20], style(color=8, rgbcolor={192,192,192}));
      end NMOS;

      model PMOS "Spice-style PMOS transistor in bond graph technology"
        constant Real pi=Modelica.Constants.pi;
        constant Modelica.SIunits.Entropy k=Modelica.Constants.k
        "Boltzmann's constant";
        constant Modelica.SIunits.Permittivity e0=Modelica.Constants.epsilon_0
        "Permittivity of free space";
        constant Modelica.SIunits.ElectricCharge q=1.6021892e-19
        "Electron charge";
        constant Modelica.SIunits.ElectronNumberDensity ni=1.45e16
        "Intrinsic carrier concentration at 300 K";
        constant Real kox=3.9 "Dielectric constant of SiO2";
        constant Real ks=11.8 "Dielectric constant of Si";
        constant Real GapC1(unit="V/K")=7.02E-4
        "Sze's first bandgap correction factor Silicon";
        constant Modelica.SIunits.Temperature GapC2=1108.0
        "Sze's second bandgap correction factor Silicon";
        constant Modelica.SIunits.Voltage Fn0=3.25
        "Work function constant for computation of flat band voltage";
        constant Modelica.SIunits.Length TOXdef = 1e-7
        "Default oxide thickness";

        /* General parameters */
        parameter Integer Level(min=0,max=4) = 1 "Level of MOSFET model";

        /* Basic Geometric parameters */
        parameter Modelica.SIunits.Length L(min=0) = 1e-4 "Channel length"
          annotation(Dialog(group="Geometric parameters"));
        parameter Modelica.SIunits.Length W(min=0) = 1e-4 "Channel width"
          annotation(Dialog(group="Geometric parameters"));

        /* Basic Process parameters */
        parameter Real TPG(min=1,max=2) = 1 "Type of gate material"
          annotation(Dialog(group="Process parameters"));
        parameter Modelica.SIunits.Length TOX=TOXdef "Thin oxide thickness"
          annotation(Dialog(group="Process parameters"));
        parameter Real COX(unit="F/m2") = 0 "Specific capacitance of SiO2"
          annotation(Dialog(group="Process parameters"));
        parameter Modelica.SIunits.ElectronNumberDensity NSUB= 0
        "Substrate doping"
          annotation(Dialog(group="Process parameters"));

        /* Basic Electrical parameters */
        parameter Real U0(unit="m2/(V.s)") = 0
        "Surface mobility at reference temperature"
          annotation(Dialog(group="Electrical parameters"));
        parameter Modelica.SIunits.Voltage VT0=0
        "Zero-bias threshold voltage (VT0<0: enhancement type, VT0>0: depletion type)"
          annotation(Dialog(group="Electrical parameters", enable=(Level==0 or Level==1)));
        parameter Modelica.SIunits.InversePotential LAMBDA(min=0) = 0
        "Channel length modulation"
          annotation(Dialog(group="Electrical parameters"));
        parameter Modelica.SIunits.Length LD(min=0) = 0 "Lateral diffusion"
          annotation(Dialog(group="Electrical parameters"));
        parameter Modelica.SIunits.Length WD(min=0) = 0 "Width diffusion"
          annotation(Dialog(group="Electrical parameters"));
        parameter Modelica.SIunits.Transconductance KP=0
        "Transconductance parameter at reference temperature"
          annotation(Dialog(group="Electrical parameters"));
        parameter Real GAMMA(unit="V^0.5") = 0 "Body-effect parameter"
          annotation(Dialog(group="Electrical parameters"));
        parameter Modelica.SIunits.Voltage PHI=0
        "Surface inversion potential at reference temperature"
          annotation(Dialog(group="Electrical parameters"));
        parameter Real NSS(unit="1/m2") = 0 "Surface state density"
          annotation(Dialog(group="Electrical parameters"));

        /* Basic temperature compensation parameters */
        parameter Modelica.SIunits.Temperature Tnom=300.15
        "Reference temperature"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Modelica.SIunits.Voltage EG=1.11
        "Energy gap for temperature effect on saturation current at 0 K"
          annotation(Dialog(group="Temperature compensation parameters"));

        /* Level=1 electrical parameters */
        parameter Modelica.SIunits.Area AD=0 "Drain diffusion area"
          annotation(Dialog(tab="Level=1", group="Electrical parameters", enable=Level>0));
        parameter Modelica.SIunits.Length PD=0 "Drain perimeter width"
          annotation(Dialog(tab="Level=1", group="Electrical parameters", enable=Level>0));
        parameter Modelica.SIunits.Area AS=0 "Source diffusion area"
          annotation(Dialog(tab="Level=1", group="Electrical parameters", enable=Level>0));
        parameter Modelica.SIunits.Length PS=0 "Source perimeter width"
          annotation(Dialog(tab="Level=1", group="Electrical parameters", enable=Level>0));
        parameter Modelica.SIunits.Current ISD(min=0) = 0
        "Drain junction saturation current at reference temperature"
          annotation(Dialog(tab="Level=1", group="Electrical parameters", enable=Level>0));
        parameter Modelica.SIunits.Current ISS(min=0) = 0
        "Source junction saturation current at reference temperature"
          annotation(Dialog(tab="Level=1", group="Electrical parameters", enable=Level>0));
        parameter Modelica.SIunits.CurrentDensity JS(min=0) = 0
        "Saturation current density at reference temperature"
          annotation(Dialog(tab="Level=1", group="Electrical parameters", enable=Level>0));
        parameter Modelica.SIunits.Voltage PB(min=0.2) = 0.8
        "Built-in potential at reference temperature"
          annotation(Dialog(tab="Level=1", group="Electrical parameters", enable=Level>0));

        /* Level=1 parasitic resistance parameters */
        parameter Modelica.SIunits.Resistance RD(min=0) = 0.1
        "Drain Ohmic resistance"
          annotation(Dialog(tab="Level=1", group="Parasitic resistance parameters", enable=Level>0));
        parameter Modelica.SIunits.Resistance RS(min=0) = 0.1
        "Source Ohmic resistance"
          annotation(Dialog(tab="Level=1", group="Parasitic resistance parameters", enable=Level>0));

        /* Level=1 junction capacitance parameters */
        parameter Real CJ(unit="F/m2",min=0) = 0
        "Zero-bias bulk capacitance per square meter at reference temperature"
          annotation(Dialog(tab="Level=1", group="Junction capacitance parameters", enable=Level>0));
        parameter Real MJ(min=0.3,max=0.5) = 0.33
        "Bulk junction grading coefficient"
          annotation(Dialog(tab="Level=1", group="Junction capacitance parameters", enable=Level>0));
        parameter Real CJSW(unit="F/m",min=0) = 1e-9
        "Zero-bias perimeter capacitance per meter at reference temperature"
          annotation(Dialog(tab="Level=1", group="Junction capacitance parameters", enable=Level>0));
        parameter Real MJSW(min=0.3,max=0.5) = 0.33
        "Perimeter capacitance grading coefficient"
          annotation(Dialog(tab="Level=1", group="Junction capacitance parameters", enable=Level>0));
        parameter Real FC(min=0,max=0.95)= 0.5
        "Forward-bias depletion capacitance coefficient"
          annotation(Dialog(tab="Level=1", group="Junction capacitance parameters", enable=Level>0));
        parameter Real CGB0(unit="F/m",min=0) = 0
        "Gate-bulk overlap capacitance per meter"
          annotation(Dialog(tab="Level=1", group="Junction capacitance parameters", enable=Level>0));
        parameter Real CGD0(unit="F/m",min=0) = 0
        "Gate-drain overlap capacitance per meter"
          annotation(Dialog(tab="Level=1", group="Junction capacitance parameters", enable=Level>0));
        parameter Real CGS0(unit="F/m",min=0) = 0
        "Gate-source overlap capacitance per meter"
          annotation(Dialog(tab="Level=1", group="Junction capacitance parameters", enable=Level>0));

        /* Level=1 temperature compensation parameters */
        parameter Real TRD1(unit="1/K") = 0
        "Linear temperature coefficient of drain resistance"
          annotation(Dialog(tab="Level=1", group="Temperature compensation parameters", enable=Level>0));
        parameter Real TRD2(unit="1/K2") = 0
        "Quadratic temperature coefficient of drain resistance"
          annotation(Dialog(tab="Level=1", group="Temperature compensation parameters", enable=Level>0));
        parameter Real TRS1(unit="1/K") = 0
        "Linear temperature coefficient of source resistance"
          annotation(Dialog(tab="Level=1", group="Temperature compensation parameters", enable=Level>0));
        parameter Real TRS2(unit="1/K2") = 0
        "Quadratic temperature coefficient of source resistance"
          annotation(Dialog(tab="Level=1", group="Temperature compensation parameters", enable=Level>0));

        /* Level=2 process parameters */
        parameter Modelica.SIunits.Length XJ=0 "Metallurgical junction depth"
          annotation(Dialog(tab="Level=2", group="Process parameters", enable=Level>1));

        /* Level=2 electrical parameters */
        parameter Modelica.SIunits.ElectricFieldStrength UCRIT=1e6
        "Critical electric field for mobility"
          annotation(Dialog(tab="Level=2", group="Electrical parameters", enable=(Level==2 or Level==4)));
        parameter Real UEXP=0 "Exponential coefficient for mobility"
          annotation(Dialog(tab="Level=2", group="Electrical parameters", enable=(Level==2 or Level==4)));
        parameter Real UTRA(min=0,max=0.5) = 0 "Transverse field coefficient"
          annotation(Dialog(tab="Level=2", group="Electrical parameters", enable=(Level==2 or Level==4)));
        parameter Modelica.SIunits.ElectricFieldStrength ECRIT=0
        "Critical electric field for pinch-off"
          annotation(Dialog(tab="Level=2", group="Electrical parameters", enable=(Level==2 or Level==4)));
        parameter Real DELTA=0 "Width effect on threshold voltage"
          annotation(Dialog(tab="Level=2", group="Electrical parameters", enable=Level>1));
        parameter Modelica.SIunits.Velocity VMAX=0
        "Maximum drift velocity of carriers (VMAX=0: Frohman-Grove model, VMAX>0: Baum-Beneking model)"
          annotation(Dialog(tab="Level=2", group="Electrical parameters", enable=Level>1));
        parameter Real NFS(unit="1/m2") = 0 "Surface fast state density"
          annotation(Dialog(tab="Level=2", group="Electrical parameters", enable=Level>1));
        parameter Real NEFF(min=Modelica.Constants.small) = 1
        "Total channel charge coefficient"
          annotation(Dialog(tab="Level=2", group="Electrical parameters", enable=Level>1));
        parameter Real XQC=0
        "Coefficient of channel charge share (XQC<=0.5: Ward-Dutton model, XQC>0.5: Meyer model)"
          annotation(Dialog(tab="Level=2", group="Electrical parameters", enable=Level>1));

        /* Level=3 electrical parameters */
        parameter Real KAPPA(min=Modelica.Constants.small) = 0.2
        "Saturation field factor"
          annotation(Dialog(tab="Level=3", group="Electrical parameters", enable=Level == 3));
        parameter Real ETA=0 "Static feedback on threshold voltage"
          annotation(Dialog(tab="Level=3", group="Electrical parameters", enable=Level == 3));
        parameter Modelica.SIunits.InversePotential THETA=0
        "Mobility modulation"
          annotation(Dialog(tab="Level=3", group="Electrical parameters", enable=Level == 3));

        /* Numerical parameters */
        parameter Real EMin=-100
        "if x < EMin, the exp(x) function is linearized"
          annotation(Dialog(tab="Advanced", group="Numerical parameters", enable=Level>0));
        parameter Real EMax=40 "if x > EMax, the exp(x) function is linearized"
          annotation(Dialog(tab="Advanced", group="Numerical parameters", enable=Level>0));
        parameter Modelica.SIunits.Conductance GminDC=1e-12
        "Leakage conductance"
          annotation(Dialog(tab="Advanced", group="Numerical parameters"));

        /* Implement Spice-style MODCHK */
    protected
        parameter Integer Type = -1 "Type=1 for NMOS; Type=-1 for PMOS";
        parameter Modelica.SIunits.Permittivity es = ks*e0
        "Permittivity of silicon";
        parameter Modelica.SIunits.Permittivity eox = kox*e0
        "Permittivity of SiO2";
        parameter Modelica.SIunits.Length LDeff = LD;
        parameter Modelica.SIunits.Length Leff = L - 2*LDeff;
        parameter Modelica.SIunits.Length Weff = W - 2*WD;
        parameter Modelica.SIunits.Length TOXeff = if TOX == TOXdef and COX > 0 then eox/COX else TOX;
        parameter Real COXeff(unit="F/m2") = if COX > 0 then COX else eox/TOX;
        parameter Real CGB0eff(unit="F/m") = CGB0;
        parameter Real CGD0eff(unit="F/m") = CGD0;
        parameter Real CGS0eff(unit="F/m") = CGS0;
        parameter Real U0def(unit="m2/(V.s)") = 0.06;
        parameter Real U0tmp(unit="m2/(V.s)") = if KP == 0 and U0 == 0 then U0def else U0;
        parameter Modelica.SIunits.Transconductance KPeff = if KP > 0 then KP else U0tmp*COXeff;
        parameter Real U0eff(unit="m2/(V.s)") = if KP > 0 and U0 == 0 then KP/COXeff else U0tmp;
        parameter Real GAMMAtmp(unit="V^0.5") = if GAMMA == 0 and NSUB == 0 then 0.35 else GAMMA;
        parameter Modelica.SIunits.ElectronNumberDensity NSUBeff = if NSUB > 0 then NSUB else (GAMMAtmp*COXeff)^2/(2*es*q);
        parameter Real GAMMAeff(unit="V^0.5") = if GAMMA == 0 and NSUB > 0 then sqrt(2*es*q*NSUB)/COXeff else GAMMAtmp;
        parameter Modelica.SIunits.InversePotential LAMBDAeff = if LAMBDA > 0 then LAMBDA else
                                                                if Level <> 2 then sqrt(2*es/(q*NSUBeff)) else LAMBDA;
        parameter Modelica.SIunits.Voltage EGref = EG - GapC1*Tnom*Tnom/(Tnom + GapC2)
        "Energy gap for temperature effect on saturation current at reference temperature";
        parameter Modelica.SIunits.ElectronNumberDensity XNI = ni*(Tnom/300)^1.5 * exp(EGref*q/(2*k)*(1/300 - 1/Tnom))
        "Intrinsic carrier concentration at reference temperature";
        parameter Modelica.SIunits.Voltage VTref = k*Tnom/q
        "Thermal voltage at reference temperature";
        parameter Modelica.SIunits.Voltage PHIeff = if PHI > 0 then PHI else if NSUB > 0 then max({2*VTref*ln(NSUB/XNI),0.1}) else 0.;
        parameter Modelica.SIunits.Voltage FermiG = Type*TPG*EGref/2;
        parameter Modelica.SIunits.Voltage Fn1 = if TPG == 0 then 3.2 else Fn0 + EGref/2 - FermiG;
        parameter Modelica.SIunits.Voltage FermiS = Type*PHIeff/2;
        parameter Modelica.SIunits.Voltage Fn2 = Fn1 - Fn0 - EGref/2 - FermiS;
        parameter Modelica.SIunits.Voltage VFB = Fn2 - NSS*q/COXeff;
        parameter Modelica.SIunits.Voltage Vcorr = Type*(GAMMAeff*sqrt(PHIeff) + PHIeff);
        parameter Modelica.SIunits.Voltage VT0eff = if Type*VT0 > 0 then VT0 else VFB + Vcorr;
        parameter Modelica.SIunits.Voltage VFBeff = if Type*VT0 > 0 then VT0 - Vcorr else VFB;
        parameter Real XD(unit="m/V^0.5") = sqrt(2*es/(q*NSUBeff))
        "Depletion factor";
        parameter Real CJeff(unit="F/m2") = if CJ > 0 then CJ else sqrt(es*q*NSUBeff/(2*PB));
        parameter Modelica.SIunits.Voltage EGeff = if Level == 3 and EG == 0 then 0.2 else EGref;
        parameter Modelica.SIunits.Voltage VBP = UCRIT*es/COXeff
        "Gate to channel critical voltage";
        parameter Modelica.SIunits.Current ISDeff = if ISD > 0 then ISD else if AD*JS > 0 then AD*JS else 1e-15;
        parameter Modelica.SIunits.Current ISSeff = if ISS > 0 then ISS else if AS*JS > 0 then AS*JS else 1e-15;
      annotation (
        Coordsys(extent=[-120,-120; 120,120]),
        Diagram(
          Text(
            extent=[-90,-114; -68,-128],
            style(color=10, rgbcolor={135,135,135}),
            string="Heat"),
          Text(
            extent=[2,-114; 24,-128],
            style(color=10, rgbcolor={135,135,135}),
            string="S"),
          Text(
            extent=[110,20; 132,6],
            style(color=10, rgbcolor={135,135,135}),
            string="B"),
          Text(
            extent=[4,126; 26,112],
            style(color=10, rgbcolor={135,135,135}),
            string="D"),
          Text(
            extent=[-130,20; -108,6],
            style(color=10, rgbcolor={135,135,135}),
            string="G"),
            Text(
              extent=[-138,60; -124,54],
              style(color=10, rgbcolor={135,135,135}),
            string="vsd"),
            Text(
              extent=[-138,-50; -124,-56],
              style(color=10, rgbcolor={135,135,135}),
            string="vsd"),
            Text(
              extent=[-54,136; -40,130],
              style(color=10, rgbcolor={135,135,135}),
            string="vsd"),
            Text(
              extent=[-98,136; -84,130],
              style(color=10, rgbcolor={135,135,135}),
            string="vsb"),
            Text(
              extent=[-138,94; -124,88],
              style(color=10, rgbcolor={135,135,135}),
            string="vsb"),
            Text(
              extent=[-140,-84; -126,-90],
              style(color=10, rgbcolor={135,135,135}),
            string="vsb")),
        Documentation(info="<html>
The Spice-style p-channel MOSFET model contains the internal MOSFET plus the three external gate capacitances and the drain and source resistances. <p>

<hr> <p>

<pre>
<b>Parameters:</b>

 <font color=red><b>Level:   MOSFET modeling level (default value = 3)
            Level=0: Static injection model
            Level=1: Shichman-Hodges model
            Level=2: Grove-Frohman model
            Level=3: Empirical model
            Level=4: Simplified Grove-Frohman model</b></font>


<b>Basic Geometric Parameters:</b>

 L:       Channel length (default value = 1e-4 m) <font color=red>Levels 0-4</font>

 W:       Channel width (default value = 1e-4 m) <font color=red>Levels 0-4</font>


<b>Basic Process Parameters:</b>

 TPG:     Type of gate material (default value = 1) <font color=red>Levels 0-4</font>
            <font color=blue><b>TPG = 0:  Aluminum gate</b></font>
            <font color=blue><b>TPG = 1:  Polysilicon gate</b></font>

 TOX:     Thin oxide thickness (default value = 1e-7 m) <font color=red>Levels 0-4</font>

 COX:     Specific capacitance of SiO2 (default value = 0 F/m<sup>2</sup>) <font color=red>Levels 0-4</font>

 NSUB:    Substrate doping (default value = 0 1/m<sup>3</sup>) <font color=red>Levels 0-4</font>


<b>Basic Electrical Parameters:</b>

 U0:      Surface mobility at reference temperature (default value = 0 m<sup>2</sup>/(V*s)) <font color=red>Levels 0-4</font>

 VT0:     Zero-bias threshold voltage (default value = 0 Volt) <font color=red>Levels 0,1</font>
            <font color=blue><b>VT0 &lt; 0 for enhancement MOSFET
            VT0 &gt; 0 for depletion MOSFET</b></font>

 LAMBDA:  Channel length modulation  (default value = 0 1/Volt) <font color=red>Levels 0-4</font>

 LD:      Lateral diffusion (default value = 0 m) <font color=red>Levels 0-4</font>

 WD:      Width diffusion (default value = 0 m) <font color=red>Levels 0-4</font>

 KP:      Transconductance parameter at reference temperature (default value = 0 Amp/Volt<sup>2</sup>) <font color=red>Levels 0-4</font>

 GAMMA:   Body-effect parameter (default value = 0 Volt<sup>0.5</sup>) <font color=red>Levels 0-4</font>

 PHI:     Surface inversion potential at reference temperature (default value = 0 Volt) <font color=red>Levels 0-4</font>

 NSS:     Surface state density (default value = 0 1/m<sup>2</sup>) <font color=red>Levels 0-4</font>


<b>Basic Temperature Compensation Parameters:</b>

 Tnom:    Reference temperature (default value = 300.15 K) <font color=red>Levels 0-4</font>

 EG:      Energy gap for temperature effect on saturation current at 0 K (default value = 1.11 Volt) <font color=red>Levels 0-4</font>


<b>Level=1 Electrical Parameters:</b>

 AD:      Drain diffusion area (default value = 0 m<sup>2</sup>) <font color=red>Levels 1-4</font>

 PD:      Drain perimeter width (default value = 0 m) <font color=red>Levels 1-4</font>

 AS:      Source diffusion area (default value = 0 m<sup>2</sup>) <font color=red>Levels 1-4</font>

 PS:      Source perimeter width (default value = 0 m) <font color=red>Levels 1-4</font>

 ISD:     Drain junction saturation current at reference temperature (default value = 0 Amp) <font color=red>Levels 1-4</font>

 ISS:     Source junction saturation current at reference temperature (default value = 0 Amp) <font color=red>Levels 1-4</font>

 JS:      Transport saturation current density (default value = 0 Amp/m<sup>2</sup>) <font color=red>Levels 1-4</font>

 PB:      Built-in potential at reference temperature (default value = 0.8 Volt) <font color=red>Levels 1-4</font>


<b>Level=1 Parasitic Resistor Parameters:</b>

 RD:      Drain Ohmic resistance (default value = 1 Ohm) <font color=red>Levels 1-4</font>

 RS:      Source Ohmic resistance (default value = 1 Ohm) <font color=red>Levels 1-4</font>


<b>Level=1 Junction Capacitance Parameters:</b>

 CJ:      Zero-bias bulk capacitance per square meter at reference temperature (default value = 0 F/m<sup>2</sup>) <font color=red>Levels 1-4</font>

 MJ:      Bulk junction grading coefficient (default value = 0.33) <font color=red>Levels 1-4</font>

 CJSW:    Zero-bias perimeter capacitance per meter at reference temperature (default value = 1e-9 F/m) <font color=red>Levels 1-4</font>

 MJSW:    Perimeter capacitance grading coefficient (default value = 0.33) <font color=red>Levels 1-4</font>

 FC:      Forward-bias depletion capacitance coefficient (default value = 0.5) <font color=red>Levels 1-4</font>

 CGB0:    Gate-bulk overlap capacitance per meter (default value = 0 F/m) <font color=red>Levels 1-4</font>

 CGD0:    Gate-drain overlap capacitance per meter (default value = 0 F/m) <font color=red>Levels 1-4</font>

 CGS0:    Gate-source overlap capacitance per meter (default value = 0 F/m) <font color=red>Levels 1-4</font>


<b>Level=1 Temperature Compensation Parameters:</b>

 TRD1:    Linear temperature coefficient of drain resistance (default value = 0 1/K) <font color=red>Levels 1-4</font>

 TRD2:    Quadratic temperature coefficient of drain resistance (default value = 0 1/K<sup>2</sup>) <font color=red>Levels 1-4</font>

 TRS1:    Linear temperature coefficient of source resistance (default value = 0 1/K) <font color=red>Levels 1-4</font>

 TRS2:    Quadratic temperature coefficient of source resistance (default value = 0 1/K<sup>2</sup>) <font color=red>Levels 1-4</font>


<b>Level=2 Process Parameters:</b>

 XJ:      Metallurgical Junction Depth (default value = 0 m) <font color=red>Levels 2-4</font>


<b>Level=2 Electrical Parameters:</b>

 UCRIT:   Critical electric field for mobility (default value = 1e6 V/m) <font color=red>Levels 2,4</font>

 UEXP:    Exponential coefficient for mobility (default value = 0) <font color=red>Levels 2,4</font>

 UTRA:    Transverse field coefficient (default value = 0) <font color=red>Levels 2,4</font>

 ECRIT:   Critical electric field for pinch-off (default value = 0 V/m) <font color=red>Levels 2,4</font>

 DELTA:   Width effect on threshold voltage (default value = 0) <font color=red>Levels 2-4</font>

 VMAX:    Maximum drift velocity of carriers (default value = 0 m/s) <font color=red>Levels 2-4</font>
            <font color=blue><b>VMAX = 0: Frohman-Grove model of drain current computation (Levels 2,4)</b></font>
            <font color=blue><b>VMAX &gt; 0: Baum-Beneking model of drain current computation (Levels 2,4)</b></font>

 NFS:     Surface fast state density (default value = 0 1/m<sup>2</sup>) <font color=red>Levels 2-4</font>

 NEFF:    Total channel charge coefficient (default value = 1) <font color=red>Levels 2,4</font>

 XQC:     Coefficient of channel charge share (default value = 0) <font color=red>Levels 2-4</font>
            <font color=blue><b>XQC &lt;= 0.5: Ward-Dutton model of gate capacitance computation</b></font>
            <font color=blue><b>XQC &gt;  0.5: Meyer model of gate capacitance computation</b></font>


<b>Level=3 Electrical Parameters:</b>

 KAPPA:   Saturation field factor (default value = 0.2) <font color=red>Level 3</font>

 ETA:     Static feedback on threshold voltage (default value = 0) <font color=red>Level 3</font>

 THETA:   Mobility modulation (default value = 0 1/Volt) <font color=red>Level 3</font>


<b>Numerical Parameters:</b>

 EMin:    if x &lt; EMin, the exp(x) function is linearized (default value = -100) <font color=red>Levels 1-4</font>

 EMax:    if x &gt; EMax, the exp(x) function is linearized (default value = 40) <font color=red>Levels 1-4</font>

 GminDC:  Leakage conductance (default value = 1e-12 mho) <font color=red>Levels 0-4</font>
</pre> <p>

<hr> <p>

<b>References:</b> <p>

<ol>
<li> Cellier, F.E. (1991), <i>Continuous System Modeling</i>, Springer-Verlag, New York.
<li> Massobrio, G. and P. Antognetti (1993), <i>Semiconductor Device Modeling with Spice</i>,
     2<sup>nd</sup> edition, McGraw Hill, New York.
</ol>
</html>
"),     Icon(
          Text(
            extent=[-78,110; 82,70],
            style(color=3, rgbcolor={0,0,255}),
            string="%name"),
            Line(points=[0,100; 0,-100],
                                       style(
                color=9,
                rgbcolor={149,164,171},
                thickness=2,
                fillColor=74,
                rgbfillColor={0,0,185},
                fillPattern=1)),
            Text(
              extent=[-114,92; 116,-96],
              style(color=0),
            string="PMOS"),
            Line(points=[0,100; 20,80],   style(
                color=9,
                rgbcolor={149,164,171},
                thickness=2,
                fillColor=74,
                rgbfillColor={0,0,185},
                fillPattern=1)),
            Line(points=[-20,80; 0,100], style(
                color=9,
                rgbcolor={149,164,171},
                thickness=2,
                fillColor=74,
                rgbfillColor={0,0,185},
                fillPattern=1)),
            Line(points=[10,50; 108,50], style(
                color=9,
                rgbcolor={149,164,171},
                thickness=2,
                fillColor=74,
                rgbfillColor={0,0,185},
                fillPattern=1)),
            Line(points=[80,40; 110,50], style(
                color=9,
                rgbcolor={149,164,171},
                thickness=2,
                fillColor=74,
                rgbfillColor={0,0,185},
                fillPattern=1)),
            Line(points=[80,60; 110,50], style(
                color=9,
                rgbcolor={149,164,171},
                thickness=2,
                fillColor=74,
                rgbfillColor={0,0,185},
                fillPattern=1)),
          Text(
            extent=[-90,-114; -68,-128],
            style(color=10, rgbcolor={135,135,135}),
            string="Heat"),
          Text(
            extent=[2,-114; 24,-128],
            style(color=10, rgbcolor={135,135,135}),
            string="S"),
          Text(
            extent=[-130,20; -108,6],
            style(color=10, rgbcolor={135,135,135}),
            string="G"),
          Text(
            extent=[4,126; 26,112],
            style(color=10, rgbcolor={135,135,135}),
            string="D"),
          Text(
            extent=[110,20; 132,6],
            style(color=10, rgbcolor={135,135,135}),
            string="B")));
    public
      BondLib.Interfaces.BondCon G "Gate"
        annotation (extent=[-130,-10; -110,10]);
      BondLib.Interfaces.BondCon D "Drain"
        annotation (extent=[-10,110; 10,130]);
      BondLib.Interfaces.BondCon B "Bulk"
        annotation (extent=[110,-10; 130,10]);
      BondLib.Interfaces.BondCon S "Source"
        annotation (extent=[-10,-130; 10,-110]);
      BondLib.Interfaces.BondCon H "Heat"
        annotation (extent=[-70,-130; -50,-110]);
      BondLib.Junctions.J0p4 J0p4_2 annotation (extent=[90,-10; 110,10]);
      BondLib.Junctions.J0p4 J0p4_1 annotation (extent=[-10,80; 10,100]);
      BondLib.Bonds.Bond B1 annotation (extent=[-40,-10; -20,10],rotation=180);
      BondLib.Bonds.Bond B2 annotation (extent=[-10,20; 10,40],  rotation=90);
      BondLib.Bonds.Bond B3 annotation (extent=[-10,-40; 10,-20],rotation=90);
      BondLib.Bonds.Bond B4 annotation (extent=[20,-10; 40,10],  rotation=0);
      BondLib.Junctions.J1p3 J1p3_1
        annotation (extent=[10,40; -10,60],rotation=90);
      BondLib.Bonds.Bond B5 annotation (extent=[-10,60; 10,80],  rotation=90);
      BondLib.Junctions.J1p3 J1p3_2
        annotation (extent=[10,-60; -10,-40],
                                           rotation=90);
      BondLib.Bonds.Bond B6 annotation (extent=[-10,-80; 10,-60],rotation=90);
      BondLib.Bonds.Bond B7 annotation (extent=[-30,40; -10,60], rotation=-180);
      BondLib.Bonds.Bond B8 annotation (extent=[-30,-60; -10,-40],
                                                                 rotation=-180);
      BondLib.Spice.Utilities.RM Rd(
        Tnom=Tnom,
        R=RD,
        Level=Level,
        TR1=TRD1,
        TR2=TRD2) annotation (extent=[-30,40; -50,60]);
      BondLib.Spice.Utilities.RM Rs(
        Tnom=Tnom,
        Level=Level,
        R=RS,
        TR1=TRS1,
        TR2=TRS2) annotation (extent=[-30,-60; -50,-40]);
      BondLib.Bonds.Bond B9 annotation (extent=[-30,80; -10,100],rotation=180);
      BondLib.Junctions.J1p3 J1p3_3
        annotation (extent=[-50,80; -30,100],
                                           rotation=180);
      BondLib.Bonds.Bond B10
                            annotation (extent=[-70,80; -50,100],rotation=180);
      BondLib.Bonds.Bond B11
                            annotation (extent=[-50,100; -30,120],
                                                                 rotation=90);
      BondLib.Spice.Utilities.Cgd Cdg(
        Tnom=Tnom,
        Level=Level,
        VFB=VFBeff,
        KP=KPeff,
        GAMMA=GAMMAeff,
        PHI=PHIeff,
        COX=COXeff,
        NFS=NFS,
        XJ=XJ,
        L=Leff,
        W=Weff,
        LD=LDeff,
        XD=XD,
        XQC=XQC,
        DELTA=DELTA,
        ETA=ETA,
        CGD0=CGD0eff,
        EG=EGeff,
        Type=-1) annotation (extent=[-50,132; -90,108]);
      BondLib.Bonds.Bond B12
                            annotation (extent=[-100,20; -80,40],rotation=270);
      BondLib.Junctions.J1p3 J1p3_4
        annotation (extent=[-100,40; -80,60],
                                           rotation=270);
      BondLib.Bonds.Bond B13
                            annotation (extent=[-100,60; -80,80],rotation=270);
      BondLib.Bonds.Bond B14
                            annotation (extent=[-120,40; -100,60],
                                                                 rotation=-180);
      BondLib.Spice.Utilities.Cgb Cbg(
        Tnom=Tnom,
        Level=Level,
        VFB=VFBeff,
        KP=KPeff,
        GAMMA=GAMMAeff,
        PHI=PHIeff,
        COX=COXeff,
        NFS=NFS,
        XJ=XJ,
        L=Leff,
        W=Weff,
        LD=LDeff,
        XD=XD,
        XQC=XQC,
        DELTA=DELTA,
        ETA=ETA,
        CGB0=CGB0eff,
        EG=EGeff,
        Type=-1) annotation (extent=[-108,54; -132,94], rotation=90);
      BondLib.Junctions.J0p5 J0p5_1
        annotation (extent=[-10,-100; 10,-80], rotation=90);
      BondLib.Bonds.Bond B15
                            annotation (extent=[20,-90; 40,-70], rotation=0);
      BondLib.Junctions.J1p3 J1p3_5
        annotation (extent=[40,-90; 60,-70],
                                           rotation=180);
      BondLib.Bonds.Bond B16
                            annotation (extent=[60,-90; 80,-70], rotation=0);
      BondLib.Bonds.fBond B17
        annotation (extent=[40,-70; 60,-50], rotation=90);
      BondLib.Sensors.De Vsb annotation (extent=[40,-50; 60,-30], rotation=90);
      BondLib.Bonds.Bond B18
                            annotation (extent=[10,80; 30,100],  rotation=180);
      BondLib.Junctions.J1p3 J1p3_6
        annotation (extent=[30,100; 50,80],rotation=180);
      BondLib.Bonds.Bond B19
                            annotation (extent=[50,80; 70,100],  rotation=180);
      BondLib.Bonds.fBond B20
                             annotation (extent=[30,60; 50,80], rotation=-90);
      BondLib.Sensors.De Vsd
        annotation (extent=[30,40; 50,60],         rotation=-90);
      BondLib.Junctions.J0p5 J0p5_2
        annotation (extent=[-96,10; -76,-10],  rotation=0);
      BondLib.Bonds.Bond B21
                            annotation (extent=[-96,-30; -76,-10],
                                                                 rotation=90);
      BondLib.Junctions.J1p3 J1p3_7
        annotation (extent=[-76,-50; -96,-30],
                                           rotation=90);
      BondLib.Bonds.Bond B22
                            annotation (extent=[-96,-70; -76,-50],
                                                                 rotation=90);
      BondLib.Bonds.Bond B23
                            annotation (extent=[-116,-50; -96,-30],
                                                                 rotation=-180);
      BondLib.Spice.Utilities.Cgs Csg(
        Tnom=Tnom,
        Level=Level,
        Type=-1,
        VFB=VFBeff,
        KP=KPeff,
        GAMMA=GAMMAeff,
        PHI=PHIeff,
        COX=COXeff,
        NFS=NFS,
        XJ=XJ,
        L=Leff,
        W=Weff,
        LD=LDeff,
        XD=XD,
        XQC=XQC,
        DELTA=DELTA,
        ETA=ETA,
        CGS0=CGS0eff,
        EG=EGeff) annotation (extent=[-132,-90; -108,-50], rotation=-90);
      BondLib.Junctions.J0p6 J0p6_1
        annotation (extent=[-70,-106; -50,-86],  rotation=90);
      BondLib.Bonds.eBond B24
        annotation (extent=[-130,-110; -110,-90],
                                                rotation=-90);
      BondLib.Bonds.eBond B25
        annotation (extent=[-70,40; -50,60],    rotation=-180);
      BondLib.Bonds.eBond B26
        annotation (extent=[-70,-60; -50,-40],  rotation=-180);
      BondLib.Bonds.eBond B27
        annotation (extent=[-30,-30; -10,-10],  rotation=-180);
      BondLib.Bonds.eBond B28
        annotation (extent=[-110,110; -90,130], rotation=-180);
      BondLib.Bonds.eBond B29
        annotation (extent=[-130,94; -110,114], rotation=-270);
      BondLib.Junctions.J0p3 J0p3_1 annotation (extent=[-50,-30; -30,-10]);
      BondLib.Bonds.eBond B30
        annotation (extent=[-64,-46; -44,-26],  rotation=-180);
      BondLib.Spice.Utilities.PMOSint PMOSint(
        Level=Level,
        Type=-1,
        Tnom=Tnom,
        VT0=VT0eff,
        VFB=VFBeff,
        KP=KPeff,
        GAMMA=GAMMAeff,
        PHI=PHIeff,
        LAMBDA=LAMBDAeff,
        TOX=TOXeff,
        COX=COXeff,
        NSUB=NSUBeff,
        NSS=NSS,
        NFS=NFS,
        NEFF=NEFF,
        L=Leff,
        W=Weff,
        XJ=XJ,
        LD=LDeff,
        AD=AD,
        PD=PD,
        AS=AS,
        PS=PS,
        XD=XD,
        TPG=TPG,
        U0=U0eff,
        UCRIT=UCRIT,
        UEXP=UEXP,
        UTRA=UTRA,
        VMAX=VMAX,
        ECRIT=ECRIT,
        XQC=XQC,
        DELTA=DELTA,
        KAPPA=KAPPA,
        ETA=ETA,
        THETA=THETA,
        VBP=VBP,
        ISD=ISDeff,
        ISS=ISSeff,
        PB=PB,
        CJ=CJeff,
        MJ=MJ,
        CJSW=CJSW,
        MJSW=MJSW,
        FC=FC,
        GminDC=GminDC,
        EG=EGeff,
        EMin=EMin,
        EMax=EMax) annotation (extent=[-20,-20; 20,20]);
      algorithm
        assert(NSUBeff > XNI, "NSUB must be larger than intrinsic carrier concentration");
        assert(LAMBDAeff < 0.2, "LAMBDA value too large, leads to numerical instability");
        assert(Leff > 0, "The effective channel length must be larger than zero");

      equation
      connect(J0p4_2.BondCon2, B)        annotation (points=[110,0; 120,0], style(color=8,
            rgbcolor={192,192,192}));
      connect(J0p4_1.BondCon4, D)        annotation (points=[0,100; 0,120],
          style(color=8, rgbcolor={192,192,192}));
      connect(B7.BondCon1, J1p3_1.BondCon3) annotation (points=[-10.2,50; -10,
            50], style(color=8, rgbcolor={192,192,192}));
      connect(B8.BondCon1, J1p3_2.BondCon3) annotation (points=[-10.2,-50; -10,
            -50], style(color=8, rgbcolor={192,192,192}));
      connect(Rd.BondCon1, B7.BondCon2) annotation (points=[-30,50; -30,50],
          style(color=8, rgbcolor={192,192,192}));
      connect(Rs.BondCon1, B8.BondCon2) annotation (points=[-30,-50; -30,-50],
          style(color=8, rgbcolor={192,192,192}));
      connect(B11.BondCon1, J1p3_3.BondCon3) annotation (points=[-40,100.2; -40,
            100], style(color=8, rgbcolor={192,192,192}));
      connect(Cdg.BondCon1, B11.BondCon2) annotation (points=[-50,120; -40,120],
          style(color=8, rgbcolor={192,192,192}));
      connect(B14.BondCon1, J1p3_4.BondCon3) annotation (points=[-100.2,50;
            -100,50], style(color=8, rgbcolor={192,192,192}));
      connect(B14.BondCon2, Cbg.BondCon1) annotation (points=[-120,50; -120,54],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p5_1.BondCon1, S)        annotation (points=[-6.12303e-016,-100;
            0,-100; 0,-120], style(color=8, rgbcolor={192,192,192}));
      connect(B17.eBondCon1, J1p3_5.BondCon3) annotation (points=[50,-70; 50,
            -70], style(color=8, rgbcolor={192,192,192}));
      connect(Vsb.BondCon1, B17.fBondCon1) annotation (points=[50,-50; 50,-50],
          style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_6.BondCon3, B20.eBondCon1) annotation (points=[40,80; 40,80],
          style(color=8, rgbcolor={192,192,192}));
      connect(B20.fBondCon1, Vsd.BondCon1) annotation (points=[40,60; 40,60],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p5_2.BondCon1, G)        annotation (points=[-96,0; -120,0],
          style(color=8, rgbcolor={192,192,192}));
      connect(B23.BondCon1, J1p3_7.BondCon3) annotation (points=[-96.2,-40; -96,
            -40], style(color=8, rgbcolor={192,192,192}));
      connect(Csg.BondCon1, B23.BondCon2) annotation (points=[-120,-50; -120,
            -40; -116,-40], style(color=8, rgbcolor={192,192,192}));
      connect(Vsd.OutPort1, Cdg.u1) annotation (points=[40,39; 40,30; 80,30; 80,
            138; -58,138; -58,132], style(color=74, rgbcolor={0,0,127}));
      connect(Vsb.OutPort1, Cdg.u2) annotation (points=[50,-29; 50,28; 82,28;
            82,140; -82,140; -82,132], style(color=74, rgbcolor={0,0,127}));
      connect(Cbg.u2, Cdg.u2) annotation (points=[-132,86; -140,86; -140,140;
            -82,140; -82,132], style(color=74, rgbcolor={0,0,127}));
      connect(Cbg.u1, Cdg.u1) annotation (points=[-132,62; -138,62; -138,138;
            -58,138; -58,132], style(color=74, rgbcolor={0,0,127}));
      connect(Csg.u1, Cbg.u1) annotation (points=[-132,-58; -138,-58; -138,62;
            -132,62], style(color=74, rgbcolor={0,0,127}));
      connect(Csg.u2, Cbg.u2) annotation (points=[-132,-82; -140,-82; -140,86;
            -132,86], style(color=74, rgbcolor={0,0,127}));
      connect(Cbg.u, Cdg.u) annotation (points=[-108,74; -104,74; -104,98; -70,
            98; -70,108], style(color=74, rgbcolor={0,0,127}));
      connect(Cbg.u, Csg.u) annotation (points=[-108,74; -104,74; -104,-70;
            -108,-70], style(color=74, rgbcolor={0,0,127}));
      connect(J0p6_1.BondCon1, H)        annotation (points=[-60,-106; -60,-120],
          style(color=8, rgbcolor={192,192,192}));
      connect(Csg.BondCon2, B24.fBondCon1) annotation (points=[-120,-90; -120,
            -90], style(color=8, rgbcolor={192,192,192}));
      connect(Rd.BondCon2, B25.fBondCon1) annotation (points=[-50,50; -50,50],
          style(color=8, rgbcolor={192,192,192}));
      connect(B26.fBondCon1, Rs.BondCon2) annotation (points=[-50,-50; -50,-50],
          style(color=8, rgbcolor={192,192,192}));
      connect(B28.fBondCon1, Cdg.BondCon2) annotation (points=[-90,120; -90,120],
          style(color=8, rgbcolor={192,192,192}));
      connect(B29.fBondCon1, Cbg.BondCon2) annotation (points=[-120,94; -120,94],
          style(color=8, rgbcolor={192,192,192}));
      connect(B26.eBondCon1, J0p6_1.BondCon3) annotation (points=[-70,-50; -74,
            -50; -74,-66; -40,-66; -40,-100; -50,-100], style(color=8, rgbcolor=
             {192,192,192}));
      connect(B25.eBondCon1, J0p6_1.BondCon4) annotation (points=[-70,50; -74,
            50; -74,10; -68,10; -68,-38; -76,-38; -76,-68; -50,-68; -50,-92],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p3_1.BondCon2, B27.eBondCon1) annotation (points=[-30,-20; -30,
            -20], style(color=8, rgbcolor={192,192,192}));
      connect(B30.fBondCon1, J0p3_1.BondCon3) annotation (points=[-44,-36; -40,
            -36; -40,-30], style(color=8, rgbcolor={192,192,192}));
      connect(B30.eBondCon1, J0p6_1.BondCon2) annotation (points=[-64,-36; -78,
            -36; -78,-70; -60,-70; -60,-86], style(color=8, rgbcolor={192,192,
              192}));
      connect(B24.eBondCon1, J0p3_1.BondCon1) annotation (points=[-120,-110;
            -120,-114; -102,-114; -102,-20; -50,-20], style(color=8, rgbcolor={
              192,192,192}));
      connect(J0p6_1.BondCon6, B29.eBondCon1) annotation (points=[-70,-92; -100,
            -92; -100,-116; -142,-116; -142,118; -120,118; -120,114], style(
            color=8, rgbcolor={192,192,192}));
      connect(J0p6_1.BondCon5, B28.eBondCon1) annotation (points=[-70,-100; -98,
            -100; -98,-118; -144,-118; -144,120; -110,120], style(color=8,
            rgbcolor={192,192,192}));
      connect(B2.BondCon2, J1p3_1.BondCon1) annotation (points=[6.12303e-016,40;
            6.12303e-016,40], style(color=8, rgbcolor={192,192,192}));
      connect(B4.BondCon2, J0p4_2.BondCon1) annotation (points=[40,0; 90,0],
          style(color=8, rgbcolor={192,192,192}));
      connect(B1.BondCon2, J0p5_2.BondCon2) annotation (points=[-40,
            1.22461e-015; -66,1.22461e-015; -66,0; -76,0], style(color=8,
            rgbcolor={192,192,192}));
      connect(J0p4_1.BondCon3, B5.BondCon2) annotation (points=[0,80;
            6.12303e-016,80], style(color=8, rgbcolor={192,192,192}));
      connect(B5.BondCon1, J1p3_1.BondCon2) annotation (points=[-6.00057e-016,
            60.2; -6.00057e-016,63.1; -6.12303e-016,63.1; -6.12303e-016,60],
          style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_2.BondCon1, B6.BondCon2) annotation (points=[6.12303e-016,
            -60; 6.12303e-016,-60], style(color=8, rgbcolor={192,192,192}));
      connect(J0p5_1.BondCon2, B6.BondCon1) annotation (points=[6.12303e-016,
            -80; -6.00057e-016,-80; -6.00057e-016,-79.8], style(color=8,
            rgbcolor={192,192,192}));
      connect(B15.BondCon2, J1p3_5.BondCon2) annotation (points=[40,-80; 40,-80],
          style(color=8, rgbcolor={192,192,192}));
      connect(J0p5_1.BondCon4, B15.BondCon1) annotation (points=[10,-86; 10,-80;
            20.2,-80], style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_5.BondCon1, B16.BondCon1) annotation (points=[60,-80; 60.2,
            -80], style(color=8, rgbcolor={192,192,192}));
      connect(B16.BondCon2, J0p4_2.BondCon3) annotation (points=[80,-80; 100,
            -80; 100,-10], style(color=8, rgbcolor={192,192,192}));
      connect(B18.BondCon2, J0p4_1.BondCon2) annotation (points=[10,90; 10,90],
          style(color=8, rgbcolor={192,192,192}));
      connect(B18.BondCon1, J1p3_6.BondCon2) annotation (points=[29.8,90; 30,90],
          style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_6.BondCon1, B19.BondCon2) annotation (points=[50,90; 50,90],
          style(color=8, rgbcolor={192,192,192}));
      connect(B19.BondCon1, J0p5_1.BondCon3) annotation (points=[69.8,90; 130,
            90; 130,-94; 10,-94], style(color=8, rgbcolor={192,192,192}));
      connect(B9.BondCon2, J1p3_3.BondCon1) annotation (points=[-30,90; -30,90],
          style(color=8, rgbcolor={192,192,192}));
      connect(B9.BondCon1, J0p4_1.BondCon1) annotation (points=[-10.2,90; -10,
            90], style(color=8, rgbcolor={192,192,192}));
      connect(B10.BondCon1, J1p3_3.BondCon2) annotation (points=[-50.2,90; -50,
            90], style(color=8, rgbcolor={192,192,192}));
      connect(B10.BondCon2, J0p5_2.BondCon4) annotation (points=[-70,90; -76,90;
            -76,10; -82,10], style(color=8, rgbcolor={192,192,192}));
      connect(B12.BondCon1, J1p3_4.BondCon2) annotation (points=[-90,39.8; -90,
            40], style(color=8, rgbcolor={192,192,192}));
      connect(B12.BondCon2, J0p5_2.BondCon3) annotation (points=[-90,20; -90,10],
          style(color=8, rgbcolor={192,192,192}));
      connect(B13.BondCon2, J1p3_4.BondCon1) annotation (points=[-90,60; -90,60],
          style(color=8, rgbcolor={192,192,192}));
      connect(B13.BondCon1, J0p4_2.BondCon4) annotation (points=[-90,79.8; -90,
            104; 100,104; 100,10], style(color=8, rgbcolor={192,192,192}));
      connect(B22.BondCon2, J1p3_7.BondCon1) annotation (points=[-86,-50; -86,
            -50], style(color=8, rgbcolor={192,192,192}));
      connect(B22.BondCon1, J0p5_1.BondCon5) annotation (points=[-86,-69.8; -86,
            -76; -10,-76; -10,-90], style(color=8, rgbcolor={192,192,192}));
      connect(J0p5_2.BondCon5, B21.BondCon2) annotation (points=[-86,-10; -86,
            -10], style(color=8, rgbcolor={192,192,192}));
      connect(J1p3_7.BondCon2, B21.BondCon1) annotation (points=[-86,-30; -86,
            -29.8], style(color=8, rgbcolor={192,192,192}));
      connect(PMOSint.y1, Cdg.u)  annotation (points=[-10,20; -10,32; -62,32;
            -62,98; -70,98; -70,108], style(color=74, rgbcolor={0,0,127}));
      connect(B1.BondCon1, PMOSint.Gint) annotation (points=[-20.2,
            -1.20011e-015; -20.2,0; -20,0],
                             style(color=8, rgbcolor={192,192,192}));
      connect(PMOSint.H, B27.fBondCon1) annotation (points=[-10,-20; -10,-20],
          style(color=8, rgbcolor={192,192,192}));
      connect(B3.BondCon2, PMOSint.Sint) annotation (points=[6.12303e-016,-20;
            0,-20], style(color=8, rgbcolor={192,192,192}));
      connect(B3.BondCon1, J1p3_2.BondCon2) annotation (points=[-6.00057e-016,
            -39.8; 0,-39.8; 0,-40; -6.12303e-016,-40], style(color=8, rgbcolor=
              {192,192,192}));
      connect(PMOSint.Bint, B4.BondCon1) annotation (points=[20,0; 20.2,0],
          style(color=8, rgbcolor={192,192,192}));
      connect(B2.BondCon1, PMOSint.Dint) annotation (points=[-6.00057e-016,20.2;
            0,20.2; 0,20], style(color=8, rgbcolor={192,192,192}));
      end PMOS;

      model NFET "Spice-style n-channel JFET in bond graph technology"

        /* General parameters */
        parameter Integer Level(min=1,max=3) = 1
        "JFET modeling level (check documentation layer for details)";
        parameter Real Area=1 "Relative area occupied by capacitor";

        /* Basic Electrical parameters */
        parameter Modelica.SIunits.Voltage VT0=0
        "Threshold voltage at reference temperature (VT0>0: enhancement type, VT0<0: depletion type)"
          annotation(Dialog(group="Electrical parameters"));
        parameter Modelica.SIunits.InversePotential LAMBDA(min=0) = 0
        "Channel length modulation"
          annotation(Dialog(group="Electrical parameters"));
        parameter Modelica.SIunits.Transconductance BETA=0
        "Transconductance parameter at reference temperature"
          annotation(Dialog(group="Electrical parameters"));
        parameter Real N=1 "Current emission coefficient"
          annotation(Dialog(group="Electrical parameters"));
        parameter Modelica.SIunits.Current IS(min=Modelica.Constants.small) = 1.e-14
        "Transport saturation current at reference temperature"
          annotation(Dialog(group="Electrical parameters"));
        parameter Modelica.SIunits.Voltage PB=0.8
        "Built-in potential at reference temperature"
          annotation(Dialog(group="Electrical parameters"));

        /* Parasitic resistance parameters */
        parameter Modelica.SIunits.Resistance RS(min=0) = 0 "Source resistance"
          annotation(Dialog(group="Parasitic resistance parameters"));
        parameter Modelica.SIunits.Resistance RD(min=0) = 0 "Drain resistance"
          annotation(Dialog(group="Parasitic resistance parameters"));
        parameter Modelica.SIunits.Resistance RG(min=0) = 0 "Gate resistance"
          annotation(Dialog(group="Parasitic resistance parameters"));

        /* Junction capacitance parameters */
        parameter Modelica.SIunits.Capacitance CGS(min=Modelica.Constants.small) = 0.5e-12
        "Zero-bias gate-source depletion capacitance at reference temperature"
          annotation(Dialog(group="Junction capacitance parameters"));
        parameter Modelica.SIunits.Capacitance CGD(min=Modelica.Constants.small) = 0.5e-12
        "Zero-bias gate-drain depletion capacitance at reference temperature"
          annotation(Dialog(group="Junction capacitance parameters"));
        parameter Real M(min=0.1,max=0.9) = if Level==3 then 0.5 else 0.33
        "Junction grading coefficient"
          annotation(Dialog(group="Junction capacitance parameters"));
        parameter Real FC(min=0,max=0.95)= 0.5
        "Factor for linearization of depletion capacitance formula"
          annotation(Dialog(group="Junction capacitance parameters", enable=Level<3));

        /* Transit time parameters */
        parameter Modelica.SIunits.Time TT(min=0) = 5e-9 "Ideal transit time"
          annotation(Dialog(group="Transit time parameters", enable=Level<3));

        /* Temperature compensation parameters */
        parameter Modelica.SIunits.Temperature Tnom=300.15
        "Reference temperature"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Real XTI = if Level==3 then 2 else 3
        "Saturation current temperature exponent"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Real BEX=0 "Mobility temperature exponent"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Modelica.SIunits.Voltage EG(min=0.1) = if Level==3 then 0.73 else 1.11
        "Energy gap for temperature effect on saturation current at 0 K"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Real TCV(unit="V/K") = 0
        "Linear temperature coefficient of threshold voltage"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Real TRS(unit="1/K") = 0
        "Linear temperature coefficient of source resistance"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Real TRD(unit="1/K") = 0
        "Linear temperature coefficient of drain resistance"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Real TRG(unit="1/K") = 0
        "Linear temperature coefficient of gate resistance"
          annotation(Dialog(group="Temperature compensation parameters"));

        /* Level=2 electrical parameters */
        parameter Modelica.SIunits.InversePotential LAM1=0
        "Channel length modulation gate voltage parameter"
          annotation(Dialog(tab="Level=2", group="Electrical parameters", enable=Level == 2));

        /* Level=3 electrical parameters */
        parameter Modelica.SIunits.InversePotential ALPHA=1
        "Hyperbolic tangient fitting parameter"
          annotation(Dialog(tab="Level=3", group="Electrical parameters", enable=Level == 3));
        parameter Modelica.SIunits.InversePotential B=0
        "Measure of doping profile"
          annotation(Dialog(tab="Level=3", group="Electrical parameters", enable=Level == 3));
        parameter Modelica.SIunits.Voltage DELTA=0.2
        "Voltage range for transition"
          annotation(Dialog(tab="Level=3", group="Electrical parameters", enable=Level == 3));
        parameter Modelica.SIunits.Voltage VMAX=0.5
        "Limit voltage for carrier velocity saturation"
          annotation(Dialog(tab="Level=3", group="Electrical parameters", enable=Level == 3));

        /* Numerical parameters */
        parameter Real EMin=-100
        "if x < EMin, the exp(x) function is linearized"
          annotation(Dialog(tab="Advanced", group="Numerical parameters"));
        parameter Real EMax=40 "if x > EMax, the exp(x) function is linearized"
          annotation(Dialog(tab="Advanced", group="Numerical parameters"));
        parameter Modelica.SIunits.Conductance GminDC=1e-16
        "Leakage conductance"
          annotation(Dialog(tab="Advanced", group="Numerical parameters"));
      Junctions.J0p2 J0p2_1 annotation (extent=[-100,-10; -80,10]);
      BondLib.Interfaces.BondCon G "Gate"
        annotation (extent=[-130,-10; -110,10]);
      BondLib.Interfaces.BondCon H "Heat" annotation (extent=[110,-10; 130,10]);
      BondLib.Interfaces.BondCon S "Source"
        annotation (extent=[-10,-130; 10,-110]);
      BondLib.Interfaces.BondCon D "Drain"
                                          annotation (extent=[-10,110; 10,130]);
      annotation (
        Coordsys(extent=[-120,-120; 120,120]),
        Diagram(
          Text(
            extent=[-130,24; -108,10],
            style(color=10, rgbcolor={135,135,135}),
            string="G"),
          Text(
            extent=[4,126; 26,112],
            style(color=10, rgbcolor={135,135,135}),
            string="D"),
          Text(
            extent=[4,-114; 26,-128],
            style(color=10, rgbcolor={135,135,135}),
            string="S"),
          Text(
            extent=[108,-8; 132,-26],
            style(
              color=10,
              rgbcolor={135,135,135},
              fillColor=3,
              rgbfillColor={0,0,255},
              fillPattern=1),
            string="Heat")),
        Icon(
          Text(
            extent=[-130,24; -108,10],
            style(color=10, rgbcolor={135,135,135}),
            string="G"),
          Text(
            extent=[4,126; 26,112],
            style(color=10, rgbcolor={135,135,135}),
            string="D"),
          Text(
            extent=[4,-114; 26,-128],
            style(color=10, rgbcolor={135,135,135}),
            string="S"),
          Text(
            extent=[108,-6; 132,-24],
            style(
              color=10,
              rgbcolor={135,135,135},
              fillColor=3,
              rgbfillColor={0,0,255},
              fillPattern=1),
            string="Heat"),
          Text(
            extent=[-74,120; 86,80],
            style(color=3, rgbcolor={0,0,255}),
            string="%name"),
            Line(points=[0,-100; 20,-80],style(
                color=9,
                rgbcolor={149,164,171},
                thickness=2,
                fillColor=74,
                rgbfillColor={0,0,185},
                fillPattern=1)),
            Line(points=[-20,-80; 0,-100],style(
                color=9,
                rgbcolor={149,164,171},
                thickness=2,
                fillColor=74,
                rgbfillColor={0,0,185},
                fillPattern=1)),
            Line(points=[0,78; 0,-100],style(
                color=9,
                rgbcolor={149,164,171},
                thickness=2,
                fillColor=74,
                rgbfillColor={0,0,185},
                fillPattern=1)),
          Line(points=[-100,-60; 100,-60; 80,-50], style(color=3, rgbcolor={0,0,
                  255})),
            Text(
              extent=[-110,92; 120,-96],
              style(color=0),
            string="NFET"),
          Line(points=[100,-60; 80,-70], style(
              color=3,
              rgbcolor={0,0,255},
              fillColor=3,
              rgbfillColor={0,0,255},
              fillPattern=1))),
        Documentation(info="<html>
The Spice-style n-channel JFET model contains an internal n-channel JFET together with the three external resistances. <p>

<hr> <p>

<pre>
<b>Parameters:</b>

 <font color=red><b>Level:   JFET modeling level (default value = 1)
            Level=1: Shichman-Hodges (Si) JFET model
            Level=2: Improved Shichman-Hodges (Si) JFET model [1]
            Level=3: Statz-Curtice (GaAs) MESFET model [2,3]<b></font>

 Area:    Relative area occupied by the diode (default value = 1) <font color=red>Levels 1-3</font>


<b>Basic Electrical Parameters:</b>

 VT0:     Threshold voltage at reference temperature (default value = 0 Volt) <font color=red>Levels 1-3</font>
            <font color=blue><b>VT0 &gt; 0 for enhancement JFET
            VT0 &lt; 0 for depletion JFET</b></font>

 LAMBDA:  Channel length modulation (default value = 0 1/Volt) <font color=red>Levels 1-3</font>

 BETA:    Transconductance parameter at reference temperature (default value = 0 Amp/Volt<sup>2</sup>) <font color=red>Levels 1-3</font>

 N:       Current emission coefficient (default value = 1) <font color=red>Levels 1-3</font>

 IS:      Transport saturation current (default value = 1e-14 Amp) <font color=red>Levels 1-3</font>

 PB:      Built-in potential at reference temperature (default value = 0.8 Volt) <font color=red>Levels 1-3</font>


<b>Parasitic Resistor Parameters:</b>

 RS:      Source resistance (default value = 0 Ohm) <font color=red>Levels 1-3</font>

 RD:      Drain resistance (default value = 0 Ohm) <font color=red>Levels 1-3</font>

 RG:      Gate resistance (default value = 0 Ohm) <font color=red>Levels 1-3</font>


<b>Junction Capacitance Parameters:</b>

 CGS:     Zero-bias gate-source depletion capacitance at reference temperature (default value = 0.5e-12 F) <font color=red>Levels 1-3</font>

 CGD:     Zero-bias gate-drain depletion capacitance at reference temperature (default value = 0.5e-12 F) <font color=red>Levels 1-3</font>

 M:       Junction grading coefficient <font color=red>Levels 1-3</font>
            <font color=blue><b>default value = 0.33 for Levels=1,2 (Si)
            default value = 0.5  for Level=3 (GaAs)</b></font>

 FC:      Forward-bias depletion capacitance coefficient (default value = 0.5) <font color=red>Levels 1,2</font>


<b>Transit Time Parameters:</b>

 TT:      Ideal transit time (default value = 5e-9 sec) <font color=red>Levels 1,2</font>


<b>Temperature Compensation Parameters:</b>

 TCV:     Linear temperature coefficient of threshold voltage (default value = 0 Volt/K) <font color=red>Levels 1-3</font>

 TRS:     Linear temperature coefficient of source resistance (default value = 0 1/K) <font color=red>Levels 1-3</font>

 TRD:     Linear temperature coefficient of drain resistance (default value = 0 1/K) <font color=red>Levels 1-3</font>

 TRG:     Linear temperature coefficient of gate resistance (default value = 0 1/K) <font color=red>Levels 1-3</font>

 Tnom:    Reference temperature (default value = 300.15 K) <font color=red>Levels 1-3</font>

 XTI:     Saturation current temperature exponent <font color=red>Levels 1-3</font>
            <font color=blue><b>default value = 3 for Levels=1,2 (Si)
            default value = 2 for Level=3 (GaAs)</b></font>

 BEX:     Mobility temperature exponent (default value = 0) <font color=red>Levels 1-3</font>

 EG:      Energy gap for temperature effect on saturation current at 0 K <font color=red>Levels 1-3</font>
            <font color=blue><b>default value = 1.11 Volt for Levels=1,2 (Si)
            default value = 0.73 Volt for Level=3 (GaAs)</b></font>


<b>Level=2 Electrical Parameters:</b>

 LAM1:    Channel length modulation gate voltage parameter (default value = 0 1/Volt) <font color=red>Level 2</font>


<b>Level=3 Electrical Parameters:</b>

 ALPHA:   Hyperbolic tangient fitting parameter (default value = 1) <font color=red>Level 3</font>

 B:       Measure of doping profile (default value = 0 1/Volt) <font color=red>Level 3</font>

 DELTA:   Voltage range for transition (default value = 0.2 Volt) <font color=red>Level 3</font>

 VMAX:    Limit voltage for carrier velocity saturation (default value = 0.5 Volt) <font color=red>Level 3</font>


<b>Numerical Parameters:</b>

 EMin:    if x &lt; EMin, the exp(x) function is linearized (default value = -100) <font color=red>Levels 1-3</font>

 EMax:    if x &gt; EMax, the exp(x) function is linearized (default value = 40) <font color=red>Levels 1-3</font>

 GminDC:  Leakage conductance (default value = 1e-15 Mho) <font color=red>Levels 1-3</font>
</pre> <p>

<hr> <p>

<b>References:</b> <p>

<ol>
<li> Massobrio, G. and P. Antognetti (1993), <i>Semiconductor Device Modeling with Spice</i>,
     2<sup>nd</sup> edition, McGraw Hill, New York, p.153.
<li> Massobrio, G. and P. Antognetti (1993), <i>Semiconductor Device Modeling with Spice</i>,
     2<sup>nd</sup> edition, McGraw Hill, New York, p.382.
<li> Statz, H., P. Newman, I.W. Smith, R.A. Pucel, and H.A. Hans (1987), \"GaAs FET Device and Circuit Simulation in Spice,\"
     <i>IEEE Trans. Electron Devices</i>, <b>34</b>(2), pp. 160-169.
</ol>
</html>
"));
      BondLib.Bonds.Bond B1 annotation (extent=[-80,-10; -60,10]);
      BondLib.Junctions.J1p3 J1p3_1 annotation (extent=[-60,10; -40,-10]);
      BondLib.Bonds.Bond B2 annotation (extent=[-40,-10; -20,10]);
      BondLib.Spice.Utilities.NFETint NFETint1(
        Level=Level,
        Tnom=Tnom,
        VT0=VT0,
        IS=IS,
        BETA=BETA,
        LAMBDA=LAMBDA,
        TCV=TCV,
        EG=EG,
        N=N,
        XTI=XTI,
        PB=PB,
        CGS=CGS,
        CGD=CGD,
        M=M,
        FC=FC,
        TT=TT,
        BEX=BEX,
        LAM1=LAM1,
        ALPHA=ALPHA,
        B=B,
        DELTA=DELTA,
        VMAX=VMAX,
        GminDC=GminDC,
        Area=Area,
        EMin=EMin,
        EMax=EMax) annotation (extent=[-20,-20; 20,20]);
      BondLib.Bonds.eBond B3 annotation (extent=[20,-10; 40,10]);
      BondLib.Junctions.J0p5 J0p5_1 annotation (extent=[80,10; 100,-10]);
      Junctions.J0p2 J0p2_2 annotation (extent=[-10,80; 10,100], rotation=90);
      BondLib.Bonds.Bond B4 annotation (extent=[-10,60; 10,80], rotation=-90);
      BondLib.Junctions.J1p3 J1p3_2
        annotation (extent=[10,60; -10,40], rotation=-90);
      BondLib.Bonds.Bond B5 annotation (extent=[-10,20; 10,40], rotation=-90);
      BondLib.Bonds.Bond B6 annotation (extent=[-10,-40; 10,-20], rotation=-90);
      BondLib.Junctions.J1p3 J1p3_3
        annotation (extent=[10,-40; -10,-60], rotation=-90);
      BondLib.Bonds.Bond B7 annotation (extent=[-10,-80; 10,-60], rotation=-90);
      Junctions.J0p2 J0p2_3 annotation (extent=[-10,-100; 10,-80], rotation=90);
      BondLib.Bonds.Bond B8 annotation (extent=[-60,10; -40,30], rotation=90);
      BondLib.Bonds.Bond B9 annotation (extent=[10,40; 30,60]);
      BondLib.Bonds.Bond B10 annotation (extent=[10,-60; 30,-40]);
      BondLib.Spice.RS Rd(
        Tnom=Tnom,
        R=RD,
        TR1=TRD,
        TR2=0,
        Area=Area) annotation (extent=[30,36; 70,64]);
      BondLib.Spice.RS Rs(
        Tnom=Tnom,
        TR2=0,
        Area=Area,
        R=RS,
        TR1=TRS) annotation (extent=[30,-64; 70,-36]);
      BondLib.Spice.RS Rg(
        Tnom=Tnom,
        TR2=0,
        Area=Area,
        R=RG,
        TR1=TRG) annotation (extent=[-64,30; -36,70], rotation=90);
      BondLib.Bonds.eBond B11 annotation (extent=[-60,70; -40,90], rotation=90);
      BondLib.Bonds.eBond B12 annotation (extent=[70,40; 90,60]);
      BondLib.Bonds.eBond B13 annotation (extent=[70,-60; 90,-40]);
      equation
      connect(J0p2_1.BondCon1, G)        annotation (points=[-100,0; -120,0],
          style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(J0p2_1.BondCon2, B1.BondCon1) annotation (points=[-80,0; -79.8,0],
          style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(B1.BondCon2, J1p3_1.BondCon1) annotation (points=[-60,0; -60,0],
          style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(J1p3_1.BondCon2, B2.BondCon1) annotation (points=[-40,0; -39.8,0],
          style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(J0p5_1.BondCon2, H)        annotation (points=[100,0; 120,0],
          style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(B3.eBondCon1, J0p5_1.BondCon1) annotation (points=[40,0; 80,0],
          style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(D, J0p2_2.BondCon2)        annotation (points=[0,120; 0,100;
            6.12303e-016,100], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(J0p2_2.BondCon1, B4.BondCon1) annotation (points=[-6.12303e-016,
            80; -6.00057e-016,80; -6.00057e-016,79.8], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(B4.BondCon2, J1p3_2.BondCon2) annotation (points=[6.12303e-016,60;
            -6.12303e-016,60], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(J1p3_2.BondCon1, B5.BondCon1) annotation (points=[6.12303e-016,40;
            -6.00057e-016,40; -6.00057e-016,39.8], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(J1p3_3.BondCon1, B7.BondCon1) annotation (points=[6.12303e-016,
            -60; -6.00057e-016,-60; -6.00057e-016,-60.2], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(B7.BondCon2, J0p2_3.BondCon2) annotation (points=[6.12303e-016,
            -80; 6.12303e-016,-80], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(J0p2_3.BondCon1, S)        annotation (points=[-6.12303e-016,-100;
            -6.12303e-016,-110; 0,-110; 0,-120], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(B8.BondCon1, J1p3_1.BondCon3) annotation (points=[-50,10.2; -50,
            10], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(J1p3_2.BondCon3, B9.BondCon1) annotation (points=[10,50; 10.2,50],
          style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(J1p3_3.BondCon3, B10.BondCon1) annotation (points=[10,-50; 10.2,
            -50], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(B9.BondCon2, Rd.BondCon1) annotation (points=[30,50; 30,50],
          style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(B10.BondCon2, Rs.BondCon1) annotation (points=[30,-50; 30,-50],
          style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(Rg.BondCon1, B8.BondCon2) annotation (points=[-50,30; -50,30],
          style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(B11.fBondCon1, Rg.BondCon2) annotation (points=[-50,70; -50,70],
          style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(Rd.BondCon2, B12.fBondCon1) annotation (points=[70,50; 70,50],
          style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(Rs.BondCon2, B13.fBondCon1) annotation (points=[70,-50; 70,-50],
          style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(B13.eBondCon1, J0p5_1.BondCon5) annotation (points=[90,-50; 100,
            -50; 100,-20; 90,-20; 90,-10], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(B12.eBondCon1, J0p5_1.BondCon3) annotation (points=[90,50; 100,50;
            100,30; 86,30; 86,10], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(B11.eBondCon1, J0p5_1.BondCon4) annotation (points=[-50,90; -50,
            106; 110,106; 110,20; 94,20; 94,10], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(NFETint1.Gint, B2.BondCon2) annotation (points=[-20,0; -20,0],
          style(color=8, rgbcolor={192,192,192}));
      connect(B6.BondCon2, J1p3_3.BondCon2) annotation (points=[6.12303e-016,
            -40; -6.12303e-016,-40], style(color=8, rgbcolor={192,192,192}));
      connect(B6.BondCon1, NFETint1.Sint) annotation (points=[-6.00057e-016,
            -20.2; 0,-20.2; 0,-20], style(color=8, rgbcolor={192,192,192}));
      connect(B3.fBondCon1, NFETint1.H) annotation (points=[20,0; 20,0], style(
            color=8, rgbcolor={192,192,192}));
      connect(B5.BondCon2, NFETint1.Dint) annotation (points=[6.12303e-016,20;
            0,20], style(color=8, rgbcolor={192,192,192}));
      end NFET;

      model PFET "Spice-style p-channel JFET in bond graph technology"

        /* General parameters */
        parameter Integer Level(min=1,max=3) = 1
        "JFET modeling level (check documentation layer for details)";
        parameter Real Area=1 "Relative area occupied by capacitor";

        /* Basic Electrical parameters */
        parameter Modelica.SIunits.Voltage VT0=0
        "Threshold voltage at reference temperature (VT0<0: enhancement type, VT0>0: depletion type)"
          annotation(Dialog(group="Electrical parameters"));
        parameter Modelica.SIunits.InversePotential LAMBDA(min=0) = 0
        "Channel length modulation"
          annotation(Dialog(group="Electrical parameters"));
        parameter Modelica.SIunits.Transconductance BETA=0
        "Transconductance parameter at reference temperature"
          annotation(Dialog(group="Electrical parameters"));
        parameter Real N=1 "Current emission coefficient"
          annotation(Dialog(group="Electrical parameters"));
        parameter Modelica.SIunits.Current IS(min=Modelica.Constants.small) = 1.e-14
        "Transport saturation current at reference temperature"
          annotation(Dialog(group="Electrical parameters"));
        parameter Modelica.SIunits.Voltage PB=0.8
        "Built-in potential at reference temperature"
          annotation(Dialog(group="Electrical parameters"));

        /* Parasitic resistance parameters */
        parameter Modelica.SIunits.Resistance RS(min=0) = 0 "Source resistance"
          annotation(Dialog(group="Parasitic resistance parameters"));
        parameter Modelica.SIunits.Resistance RD(min=0) = 0 "Drain resistance"
          annotation(Dialog(group="Parasitic resistance parameters"));
        parameter Modelica.SIunits.Resistance RG(min=0) = 0 "Gate resistance"
          annotation(Dialog(group="Parasitic resistance parameters"));

        /* Junction capacitance parameters */
        parameter Modelica.SIunits.Capacitance CGS(min=Modelica.Constants.small) = 0.5e-12
        "Zero-bias gate-source depletion capacitance at reference temperature"
          annotation(Dialog(group="Junction capacitance parameters"));
        parameter Modelica.SIunits.Capacitance CGD(min=Modelica.Constants.small) = 0.5e-12
        "Zero-bias gate-drain depletion capacitance at reference temperature"
          annotation(Dialog(group="Junction capacitance parameters"));
        parameter Real M(min=0.1,max=0.9) = if Level==3 then 0.5 else 0.33
        "Junction grading coefficient"
          annotation(Dialog(group="Junction capacitance parameters"));
        parameter Real FC(min=0,max=0.95)= 0.5
        "Factor for linearization of depletion capacitance formula"
          annotation(Dialog(group="Junction capacitance parameters", enable=Level<3));

        /* Transit time parameters */
        parameter Modelica.SIunits.Time TT(min=0) = 5e-9 "Ideal transit time"
          annotation(Dialog(group="Transit time parameters", enable=Level<3));

        /* Temperature compensation parameters */
        parameter Modelica.SIunits.Temperature Tnom=300.15
        "Reference temperature"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Real XTI = if Level==3 then 2 else 3
        "Saturation current temperature exponent"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Real BEX=0 "Mobility temperature exponent"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Modelica.SIunits.Voltage EG(min=0.1) = if Level==3 then 0.73 else 1.11
        "Energy gap for temperature effect on saturation current at 0 K"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Real TCV(unit="V/K") = 0
        "Linear temperature coefficient of threshold voltage"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Real TRS(unit="1/K") = 0
        "Linear temperature coefficient of source resistance"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Real TRD(unit="1/K") = 0
        "Linear temperature coefficient of drain resistance"
          annotation(Dialog(group="Temperature compensation parameters"));
        parameter Real TRG(unit="1/K") = 0
        "Linear temperature coefficient of gate resistance"
          annotation(Dialog(group="Temperature compensation parameters"));

        /* Level=2 electrical parameters */
        parameter Modelica.SIunits.InversePotential LAM1=0
        "Channel length modulation gate voltage parameter"
          annotation(Dialog(tab="Level=2", group="Electrical parameters", enable=Level == 2));

        /* Level=3 electrical parameters */
        parameter Modelica.SIunits.InversePotential ALPHA=1
        "Hyperbolic tangient fitting parameter"
          annotation(Dialog(tab="Level=3", group="Electrical parameters", enable=Level == 3));
        parameter Modelica.SIunits.InversePotential B=0
        "Measure of doping profile"
          annotation(Dialog(tab="Level=3", group="Electrical parameters", enable=Level == 3));
        parameter Modelica.SIunits.Voltage DELTA=0.2
        "Voltage range for transition"
          annotation(Dialog(tab="Level=3", group="Electrical parameters", enable=Level == 3));
        parameter Modelica.SIunits.Voltage VMAX=0.5
        "Limit voltage for carrier velocity saturation"
          annotation(Dialog(tab="Level=3", group="Electrical parameters", enable=Level == 3));

        /* Numerical parameters */
        parameter Real EMin=-100
        "if x < EMin, the exp(x) function is linearized"
          annotation(Dialog(tab="Advanced", group="Numerical parameters"));
        parameter Real EMax=40 "if x > EMax, the exp(x) function is linearized"
          annotation(Dialog(tab="Advanced", group="Numerical parameters"));
        parameter Modelica.SIunits.Conductance GminDC=1e-16
        "Leakage conductance"
          annotation(Dialog(tab="Advanced", group="Numerical parameters"));
      annotation (
        Coordsys(extent=[-120,-120; 120,120]),
        Diagram(
          Text(
            extent=[-130,24; -108,10],
            style(color=10, rgbcolor={135,135,135}),
            string="G"),
          Text(
            extent=[4,126; 26,112],
            style(color=10, rgbcolor={135,135,135}),
            string="D"),
          Text(
            extent=[4,-114; 26,-128],
            style(color=10, rgbcolor={135,135,135}),
            string="S"),
          Text(
            extent=[108,-8; 132,-26],
            style(
              color=10,
              rgbcolor={135,135,135},
              fillColor=3,
              rgbfillColor={0,0,255},
              fillPattern=1),
            string="Heat")),
        Icon(
            Line(points=[-20,60; 0,80],  style(
                color=9,
                rgbcolor={149,164,171},
                thickness=2,
                fillColor=74,
                rgbfillColor={0,0,185},
                fillPattern=1)),
            Line(points=[0,80; 20,60],    style(
                color=9,
                rgbcolor={149,164,171},
                thickness=2,
                fillColor=74,
                rgbfillColor={0,0,185},
                fillPattern=1)),
            Line(points=[0,78; 0,-100],style(
                color=9,
                rgbcolor={149,164,171},
                thickness=2,
                fillColor=74,
                rgbfillColor={0,0,185},
                fillPattern=1)),
          Line(points=[-100,-60; 100,-60; 80,-50], style(color=3, rgbcolor={0,0,
                  255})),
            Text(
              extent=[-112,92; 118,-96],
              style(color=0),
            string="PFET"),
          Text(
            extent=[108,-6; 132,-24],
            style(
              color=10,
              rgbcolor={135,135,135},
              fillColor=3,
              rgbfillColor={0,0,255},
              fillPattern=1),
            string="Heat"),
          Text(
            extent=[-74,120; 86,80],
            style(color=3, rgbcolor={0,0,255}),
            string="%name"),
          Text(
            extent=[4,126; 26,112],
            style(color=10, rgbcolor={135,135,135}),
            string="D"),
          Text(
            extent=[-130,24; -108,10],
            style(color=10, rgbcolor={135,135,135}),
            string="G"),
          Text(
            extent=[4,-114; 26,-128],
            style(color=10, rgbcolor={135,135,135}),
            string="S"),
          Line(points=[100,-60; 80,-70], style(
              color=3,
              rgbcolor={0,0,255},
              fillColor=3,
              rgbfillColor={0,0,255},
              fillPattern=1))),
        Documentation(info="<html>
The Spice-style p-channel JFET model contains an internal p-channel JFET together with the three external resistances. <p>

<hr> <p>

<pre>
<b>Parameters:</b>

 <font color=red><b>Level:   JFET modeling level (default value = 1)
            Level=1: Shichman-Hodges (Si) JFET model
            Level=2: Improved Shichman-Hodges (Si) JFET model [1]
            Level=3: Statz-Curtice (GaAs) MESFET model [2,3]<b></font>

 Area:    Relative area occupied by the diode (default value = 1) <font color=red>Levels 1-3</font>


<b>Basic Electrical Parameters:</b>

 VT0:     Threshold voltage at reference temperature (default value = 0 Volt) <font color=red>Levels 1-3</font>
            <font color=blue><b>VT0 &gt; 0 for enhancement JFET
            VT0 &lt; 0 for depletion JFET</b></font>

 LAMBDA:  Channel length modulation (default value = 0 1/Volt) <font color=red>Levels 1-3</font>

 BETA:    Transconductance parameter at reference temperature (default value = 0 Amp/Volt<sup>2</sup>) <font color=red>Levels 1-3</font>

 N:       Current emission coefficient (default value = 1) <font color=red>Levels 1-3</font>

 IS:      Transport saturation current (default value = 1e-14 Amp) <font color=red>Levels 1-3</font>

 PB:      Built-in potential at reference temperature (default value = 0.8 Volt) <font color=red>Levels 1-3</font>


<b>Parasitic Resistor Parameters:</b>

 RS:      Source resistance (default value = 0 Ohm) <font color=red>Levels 1-3</font>

 RD:      Drain resistance (default value = 0 Ohm) <font color=red>Levels 1-3</font>

 RG:      Gate resistance (default value = 0 Ohm) <font color=red>Levels 1-3</font>


<b>Junction Capacitance Parameters:</b>

 CGS:     Zero-bias gate-source depletion capacitance at reference temperature (default value = 0.5e-12 F) <font color=red>Levels 1-3</font>

 CGD:     Zero-bias gate-drain depletion capacitance at reference temperature (default value = 0.5e-12 F) <font color=red>Levels 1-3</font>

 M:       Junction grading coefficient <font color=red>Levels 1-3</font>
            <font color=blue><b>default value = 0.33 for Levels=1,2 (Si)
            default value = 0.5  for Level=3 (GaAs)</b></font>

 FC:      Forward-bias depletion capacitance coefficient (default value = 0.5) <font color=red>Levels 1,2</font>


<b>Transit Time Parameters:</b>

 TT:      Ideal transit time (default value = 5e-9 sec) <font color=red>Levels 1,2</font>


<b>Temperature Compensation Parameters:</b>

 TCV:     Linear temperature coefficient of threshold voltage (default value = 0 Volt/K) <font color=red>Levels 1-3</font>

 TRS:     Linear temperature coefficient of source resistance (default value = 0 1/K) <font color=red>Levels 1-3</font>

 TRD:     Linear temperature coefficient of drain resistance (default value = 0 1/K) <font color=red>Levels 1-3</font>

 TRG:     Linear temperature coefficient of gate resistance (default value = 0 1/K) <font color=red>Levels 1-3</font>

 Tnom:    Reference temperature (default value = 300.15 K) <font color=red>Levels 1-3</font>

 XTI:     Saturation current temperature exponent <font color=red>Levels 1-3</font>
            <font color=blue><b>default value = 3 for Levels=1,2 (Si)
            default value = 2 for Level=3 (GaAs)</b></font>

 BEX:     Mobility temperature exponent (default value = 0) <font color=red>Levels 1-3</font>

 EG:      Energy gap for temperature effect on saturation current at 0 K <font color=red>Levels 1-3</font>
            <font color=blue><b>default value = 1.11 Volt for Levels=1,2 (Si)
            default value = 0.73 Volt for Level=3 (GaAs)</b></font>


<b>Level=2 Electrical Parameters:</b>

 LAM1:    Channel length modulation gate voltage parameter (default value = 0 1/Volt) <font color=red>Level 2</font>


<b>Level=3 Electrical Parameters:</b>

 ALPHA:   Hyperbolic tangient fitting parameter (default value = 1) <font color=red>Level 3</font>

 B:       Measure of doping profile (default value = 0 1/Volt) <font color=red>Level 3</font>

 DELTA:   Voltage range for transition (default value = 0.2 Volt) <font color=red>Level 3</font>

 VMAX:    Limit voltage for carrier velocity saturation (default value = 0.5 Volt) <font color=red>Level 3</font>


<b>Numerical Parameters:</b>

 EMin:    if x &lt; EMin, the exp(x) function is linearized (default value = -100) <font color=red>Levels 1-3</font>

 EMax:    if x &gt; EMax, the exp(x) function is linearized (default value = 40) <font color=red>Levels 1-3</font>

 GminDC:  Leakage conductance (default value = 1e-15 Mho) <font color=red>Levels 1-3</font>
</pre> <p>

<hr> <p>

<b>References:</b> <p>

<ol>
<li> Massobrio, G. and P. Antognetti (1993), <i>Semiconductor Device Modeling with Spice</i>,
     2<sup>nd</sup> edition, McGraw Hill, New York, p.153.
<li> Massobrio, G. and P. Antognetti (1993), <i>Semiconductor Device Modeling with Spice</i>,
     2<sup>nd</sup> edition, McGraw Hill, New York, p.382.
<li> Statz, H., P. Newman, I.W. Smith, R.A. Pucel, and H.A. Hans (1987), \"GaAs FET Device and Circuit Simulation in Spice,\"
     <i>IEEE Trans. Electron Devices</i>, <b>34</b>(2), pp. 160-169.
</ol>
</html>
"));
      Junctions.J0p2 J0p2_1 annotation (extent=[-100,-10; -80,10]);
      BondLib.Interfaces.BondCon G "Gate"
        annotation (extent=[-130,-10; -110,10]);
      BondLib.Interfaces.BondCon H "Heat" annotation (extent=[110,-10; 130,10]);
      BondLib.Interfaces.BondCon S "Source"
        annotation (extent=[-10,-130; 10,-110]);
      BondLib.Interfaces.BondCon D "Drain"
                                          annotation (extent=[-10,110; 10,130]);
      BondLib.Bonds.Bond B1 annotation (extent=[-80,-10; -60,10], rotation=180);
      BondLib.Junctions.J1p3 J1p3_1 annotation (extent=[-60,10; -40,-10]);
      BondLib.Bonds.Bond B2 annotation (extent=[-40,-10; -20,10], rotation=180);
      BondLib.Bonds.eBond B3 annotation (extent=[20,-10; 40,10]);
      BondLib.Junctions.J0p5 J0p5_1 annotation (extent=[80,10; 100,-10]);
      BondLib.Junctions.J0p2 J0p2_2
        annotation (extent=[-10,80; 10,100], rotation=90);
      BondLib.Bonds.Bond B4 annotation (extent=[-10,60; 10,80], rotation=90);
      BondLib.Junctions.J1p3 J1p3_2
        annotation (extent=[10,60; -10,40], rotation=-90);
      BondLib.Bonds.Bond B5 annotation (extent=[-10,20; 10,40], rotation=90);
      BondLib.Bonds.Bond B6 annotation (extent=[-10,-40; 10,-20], rotation=90);
      BondLib.Junctions.J1p3 J1p3_3
        annotation (extent=[10,-40; -10,-60], rotation=-90);
      BondLib.Bonds.Bond B7 annotation (extent=[-10,-80; 10,-60], rotation=90);
      BondLib.Junctions.J0p2 J0p2_3
        annotation (extent=[-10,-100; 10,-80], rotation=90);
      BondLib.Bonds.Bond B8 annotation (extent=[-60,10; -40,30], rotation=90);
      BondLib.Bonds.Bond B9 annotation (extent=[10,40; 30,60]);
      BondLib.Bonds.Bond B10 annotation (extent=[10,-60; 30,-40]);
      BondLib.Spice.RS Rd(
        Tnom=Tnom,
        R=RD,
        TR1=TRD,
        TR2=0,
        Area=Area) annotation (extent=[30,36; 70,64]);
      BondLib.Spice.RS Rs(
        Tnom=Tnom,
        TR2=0,
        Area=Area,
        R=RS,
        TR1=TRS) annotation (extent=[30,-64; 70,-36]);
      BondLib.Spice.RS Rg(
        Tnom=Tnom,
        TR2=0,
        Area=Area,
        R=RG,
        TR1=TRG) annotation (extent=[-64,30; -36,70], rotation=90);
      BondLib.Bonds.eBond B11 annotation (extent=[-60,70; -40,90], rotation=90);
      BondLib.Bonds.eBond B12 annotation (extent=[70,40; 90,60]);
      BondLib.Bonds.eBond B13 annotation (extent=[70,-60; 90,-40]);
      BondLib.Spice.Utilities.PFETint PFETint1(
        Level=Level,
        Tnom=Tnom,
        VT0=VT0,
        IS=IS,
        BETA=BETA,
        LAMBDA=LAMBDA,
        TCV=TCV,
        EG=EG,
        N=N,
        XTI=XTI,
        PB=PB,
        CGS=CGS,
        CGD=CGD,
        M=M,
        FC=FC,
        TT=TT,
        BEX=BEX,
        LAM1=LAM1,
        ALPHA=ALPHA,
        B=B,
        DELTA=DELTA,
        VMAX=VMAX,
        GminDC=GminDC,
        Area=Area,
        EMin=EMin,
        EMax=EMax) annotation (extent=[-20,-20; 20,20]);
      equation
      connect(J0p2_1.BondCon1, G)        annotation (points=[-100,0; -120,0],
          style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(J0p5_1.BondCon2, H)        annotation (points=[100,0; 120,0],
          style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(B3.eBondCon1, J0p5_1.BondCon1) annotation (points=[40,0; 80,0],
          style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(D, J0p2_2.BondCon2)        annotation (points=[0,120; 0,100;
            6.12303e-016,100], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(J0p2_3.BondCon1, S)        annotation (points=[-6.12303e-016,-100;
            -6.12303e-016,-112; 0,-110; 0,-120], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(B8.BondCon1, J1p3_1.BondCon3) annotation (points=[-50,10.2; -50,
            10], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(J1p3_2.BondCon3, B9.BondCon1) annotation (points=[10,50; 10.2,50],
          style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(J1p3_3.BondCon3, B10.BondCon1) annotation (points=[10,-50; 10.2,
            -50], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(B9.BondCon2, Rd.BondCon1) annotation (points=[30,50; 30,50],
          style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(B10.BondCon2, Rs.BondCon1) annotation (points=[30,-50; 30,-50],
          style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(Rg.BondCon1, B8.BondCon2) annotation (points=[-50,30; -50,30],
          style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(B11.fBondCon1, Rg.BondCon2) annotation (points=[-50,70; -50,70],
          style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(Rd.BondCon2, B12.fBondCon1) annotation (points=[70,50; 70,50],
          style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(Rs.BondCon2, B13.fBondCon1) annotation (points=[70,-50; 70,-50],
          style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(B13.eBondCon1, J0p5_1.BondCon5) annotation (points=[90,-50; 100,
            -50; 100,-20; 90,-20; 90,-10], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(B12.eBondCon1, J0p5_1.BondCon3) annotation (points=[90,50; 100,50;
            100,30; 86,30; 86,10], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(B11.eBondCon1, J0p5_1.BondCon4) annotation (points=[-50,90; -50,
            106; 110,106; 110,20; 94,20; 94,10], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(J1p3_2.BondCon1, B5.BondCon2) annotation (points=[6.12303e-016,40;
            6.12303e-016,40], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(B4.BondCon1, J1p3_2.BondCon2) annotation (points=[-6.00057e-016,
            60.2; -6.00057e-016,63.1; -6.12303e-016,63.1; -6.12303e-016,60],
          style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(J0p2_2.BondCon1, B4.BondCon2) annotation (points=[-6.12303e-016,
            80; -6.12303e-016,85; 6.12303e-016,85; 6.12303e-016,80], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(J1p3_3.BondCon2, B6.BondCon1) annotation (points=[-6.12303e-016,
            -40; -6.00057e-016,-40; -6.00057e-016,-39.8], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(B7.BondCon2, J1p3_3.BondCon1) annotation (points=[6.12303e-016,
            -60; 6.12303e-016,-60], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(J0p2_3.BondCon2, B7.BondCon1) annotation (points=[6.12303e-016,
            -80; -6.00057e-016,-80; -6.00057e-016,-79.8], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(B2.BondCon2, J1p3_1.BondCon2) annotation (points=[-40,
            1.22461e-015; -40,0], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(B1.BondCon1, J1p3_1.BondCon1) annotation (points=[-60.2,
            -1.20011e-015; -66.1,-1.20011e-015; -66.1,0; -60,0], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(J0p2_1.BondCon2, B1.BondCon2) annotation (points=[-80,0; -80,
            1.22461e-015], style(
          color=8,
          rgbcolor={192,192,192},
          fillColor=3,
          rgbfillColor={0,0,255},
          fillPattern=1));
      connect(PFETint1.Gint, B2.BondCon1) annotation (points=[-20,0; -20,
            -1.20011e-015; -20.2,-1.20011e-015], style(color=8, rgbcolor={192,
              192,192}));
      connect(B6.BondCon2, PFETint1.Sint) annotation (points=[6.12303e-016,-20;
            0,-20], style(color=8, rgbcolor={192,192,192}));
      connect(B3.fBondCon1, PFETint1.H) annotation (points=[20,0; 20,0], style(
            color=8, rgbcolor={192,192,192}));
      connect(B5.BondCon1, PFETint1.Dint) annotation (points=[-6.00057e-016,
            20.2; 0,20.2; 0,20], style(color=8, rgbcolor={192,192,192}));
      end PFET;

      package Utilities "Utility models of Spice circuit element models"
      annotation (preferedView="info",
          Icon(
            Rectangle(extent=[-100,-100; 80,50], style(
                color=42,
                rgbcolor={127,0,0},
                fillColor=30,
                rgbfillColor={235,235,235},
                fillPattern=1)),
            Polygon(points=[-100,50; -80,70; 100,70; 80,50; -100,50], style(
                color=42,
                rgbcolor={127,0,0},
                fillColor=30,
                rgbfillColor={235,235,235},
                fillPattern=1)),
            Polygon(points=[100,70; 100,-80; 80,-100; 80,50; 100,70], style(
                color=42,
                rgbcolor={127,0,0},
                fillColor=30,
                rgbfillColor={235,235,235},
                fillPattern=1)),
            Text(
              extent=[-85,35; 65,-85],
              string="Library",
              style(color=42, rgbcolor={127,0,0})),
            Text(
              extent=[-120,122; 120,73],
              string="%name",
              style(color=1))), Documentation(info="<html>
This package contains some utility models used within the Spice-style circuit element models.
</html>
"));
        block nlC "Non-linear capacitance computation of Gummel-Poon BJT"
          constant Modelica.SIunits.Entropy k=Modelica.Constants.k
          "Boltzmann's constant";
          constant Modelica.SIunits.ElectricCharge q=1.6021892e-19
          "Electron charge";
          constant Real GapC1(unit="V/K")=7.02E-4
          "First bandgap correction factor Silicon";
          constant Modelica.SIunits.Temperature GapC2=1108.0
          "Second bandgap correction factor Silicon";
          parameter Modelica.SIunits.Temperature Tnom=300.15
          "Reference temperature";
          parameter Modelica.SIunits.Current IS=1.e-16
          "Transport saturation current";
          parameter Modelica.SIunits.Voltage EG=1.11
          "Energy gap for temperature effect on saturation current at 0 K";
          parameter Real N=1 "Current emission coefficient";
          parameter Real XTI=3 "Saturation current temperature exponent";
          parameter Modelica.SIunits.Voltage VJ=0.8
          "Built-in potential at reference temperature";
          parameter Modelica.SIunits.Capacitance CJ=0.5e-12
          "Zero-bias depletion capacitance at reference temperature";
          parameter Real MJ=0.333 "Junction grading coefficient";
          parameter Real FC=0.5
          "Factor for linearization of depletion capacitance formula";
          parameter Modelica.SIunits.Time Tau=5e-9 "Ideal transit time";
          parameter Modelica.SIunits.Conductance GminDC=1e-15
          "Leakage conductance";
          parameter Real Area=1 "Relative area occupied by capacitor";
          parameter Integer Level=2
          "Transistor modeling level (Ebers-Moll = 1; Gummel-Poon = 2)";
          parameter Boolean BE=true "True if base-emitter junction";
          parameter Real EMin=-100
          "if x < EMin, the exp(x) function is linearized";
          parameter Real EMax=40
          "if x > EMax, the exp(x) function is linearized";
          output Modelica.SIunits.Voltage Vt "Thermal voltage";
          output Modelica.SIunits.Current ISval
          "Saturation current at device temperature";
          output Modelica.SIunits.Voltage VJval
          "Built-in potential at device temperature";
          output Modelica.SIunits.Capacitance CJval
          "Depletion capacitance at device temperature";
          output Modelica.SIunits.Voltage EGval
          "Energy gap at device temperature";
          output Modelica.SIunits.Voltage EGnom
          "Energy gap at reference temperature";

      protected
          parameter Real ExMin = exp(EMin);
          parameter Real ExMax = exp(EMax);
          Modelica.SIunits.Temperature DTemp
          "Difference between circuit temperature and reference temperature";
          Real RTemp "Temperature quotient";
          Modelica.SIunits.Capacitance Cdiff1
          "Part of diffusion capacitance at device temperature";
          Real et;
        annotation (Diagram(
                    Text(
                extent=[-78,70; 76,-58],
                string="nl C",
                style(color=0, rgbcolor={0,0,0})), Text(
                extent=[-82,-46; 82,-70],
                style(color=0, rgbcolor={0,0,0}),
              string="Gummel-Poon BJT")), Icon(
            Rectangle(extent=[-100,100; 100,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255})),
                    Text(
                extent=[-78,70; 76,-58],
                string="nl C",
                style(color=0, rgbcolor={0,0,0})), Text(
                extent=[-82,-46; 82,-70],
                style(color=0, rgbcolor={0,0,0}),
              string="Gummel-Poon BJT")));
      public
        Modelica.Blocks.Interfaces.RealInput qb "Base charge"
          annotation (extent=[-60,-110; -40,-90], rotation=90);
        Modelica.Blocks.Interfaces.RealInput C0(redeclare type SignalType =
                Modelica.SIunits.Capacitance) "Base capacitance"
          annotation (extent=[40,-110; 60,-90], rotation=90);
      public
        Modelica.Blocks.Interfaces.RealInput Tdev(redeclare type SignalType =
                Modelica.SIunits.Temperature) "Device temperature"
          annotation (extent=[-10,90; 10,110],    rotation=270);
        Modelica.Blocks.Interfaces.RealInput v(redeclare type SignalType =
                Modelica.SIunits.Voltage) "Voltage across junction"
          annotation (extent=[-110,-10; -90,10]);
        Modelica.Blocks.Interfaces.RealOutput Cdiff(redeclare type SignalType
            =   Modelica.SIunits.Capacitance) "Diffusion capacitance"
          annotation (extent=[100,40; 120,60]);
        Modelica.Blocks.Interfaces.RealOutput Cdepl(redeclare type SignalType
            =   Modelica.SIunits.Capacitance) "Depletion capacitance"
          annotation (extent=[100,-60; 120,-40]);
        equation
          /* Compute thermal voltage as function of temperature */
          Vt = k*Tdev/q;
          et = v/(N*Vt);

          /* Compute temperature dependence of saturation current and junction capacitance */
          DTemp = Tdev - Tnom;
          RTemp = Tdev/Tnom;
          if Level==2 then
            /* Gummel-Poon model */
            EGval = EG - GapC1*Tdev*Tdev/(Tdev + GapC2);
            EGnom = EG - GapC1*Tnom*Tnom/(Tnom + GapC2);
            VJval = VJ*RTemp + EGval - EGnom*RTemp - 3*Vt*ln(RTemp);
            CJval = CJ*(1 + MJ*(1 - VJval/VJ + 4e-4*DTemp));
          else
            /* Ebers-Moll model */
            EGval = EG;
            EGnom = EG;
            VJval = VJ;
            CJval = CJ;
          end if;
          ISval = IS*exp((RTemp - 1)*EGval/Vt + XTI*ln(RTemp));

          /* Compute Ebers-Moll diffusion capacitance */
          Cdiff1 = Tau*ISval/(N*Vt)*(if et < EMin then ExMin*(et - EMin + 1) else
                                     if et > EMax then ExMax*(et - EMax + 1) else exp(et));

          /* Compute diffusion and depletion capacitance values */
          if Level==2 then
            /* Gummel-Poon model */
            Cdiff = Area*(Tau*GminDC + (if BE then Cdiff1/qb - C0 else Cdiff1));
            Cdepl = Area*CJval*(if v > FC*VJval then (1 - FC*(1 + MJ) + MJ*v/VJval)/((1 - FC)^(1 + MJ)) else
                                                     1/noEvent((abs(1 - v/VJval))^MJ));
          else
            /* Ebers-Moll model */
            Cdiff = Tau*GminDC + Cdiff1;
            Cdepl = CJval*(if v/VJval > 0 then 1 + MJ*v/VJval else 1/noEvent((abs(1 - v/VJval))^MJ));
          end if;
        end nlC;

        block BJTvars
        "Space charge and injected currents of bipolar transistors"
          constant Modelica.SIunits.Entropy k=Modelica.Constants.k
          "Boltzmann's constant";
          constant Modelica.SIunits.ElectricCharge q=1.6021892e-19
          "Electron charge";
          constant Real GapC1(unit="V/K")=7.02E-4
          "First bandgap correction factor Silicon";
          constant Modelica.SIunits.Temperature GapC2=1108.0
          "Second bandgap correction factor Silicon";
          parameter Modelica.SIunits.Temperature Tnom=300.15
          "Reference temperature";
          parameter Real BF=50
          "Maximum forward current gain at reference temperature";
          parameter Real BR=0.1
          "Maximum reverse current gain at reference temperature";
          parameter Modelica.SIunits.Current IS=1e-16
          "Saturation current at reference temperature";
          parameter Modelica.SIunits.Current ISS=IS
          "Saturation current used for current injection";
          parameter Modelica.SIunits.Time TF=0.12e-9
          "Ideal forward transit time";
          parameter Modelica.SIunits.Voltage EG=1.11
          "Energy gap for temperature effect on saturation current";
          parameter Modelica.SIunits.Voltage VAF=9e30 "Forward early voltage";
          parameter Modelica.SIunits.Voltage VAR=9e30 "Reverse early voltage";

          /* High current beta degradation effect parameters */
          parameter Modelica.SIunits.Current IKF=9e30
          "Corner for forward beta high-current roll-off";
          parameter Modelica.SIunits.Current IKR=9e30
          "Corner for reverse beta high-current roll-off";
          parameter Modelica.SIunits.Current ISC=1e-16
          "Base-collector leakage saturation current at reference temperature (ISC = C4*IS)";
          parameter Modelica.SIunits.Current ISE=1e-16
          "Base-emitter leakage saturation current at reference temperature (ISE = C2*IS)";
          parameter Real NC=2
          "Low-current base-collector leakage emission coefficient";
          parameter Real NE=1.5
          "Low-current base-emitter leakage emission coefficient";
          parameter Real XTI=3 "Saturation current temperature exponent";
          parameter Real XTB=0
          "Forward and reverse beta temperature coefficient";
          parameter Real Area=1 "Relative area occupied by resistor";
          parameter Integer Level=2
          "Transistor modeling level (Ebers-Moll = 1; Gummel-Poon = 2)";
          output Modelica.SIunits.Voltage Vt "Thermal voltage";
          output Modelica.SIunits.Voltage EGval
          "Energy gap at device temperature";
          output Modelica.SIunits.Current ISCval
          "Collector saturation current at device temperature";
          output Modelica.SIunits.Current ISEval
          "Emitter saturation current at device temperature";
          output Modelica.SIunits.Current icn
          "Beta degradation collector current";
          output Modelica.SIunits.Current ien
          "Beta degradation emitter current";

      protected
          Real RTemp "Temperature quotient";
          Real Rxtb "Exponentiated temperature quotient";
          Real q1 "Primary base charge";
          Real q2 "Secondary base charge";
          Real dq1 "Derivative of q1";
          Real dq2 "Derivative of q2";
          Real dqb "Derivative of qb";
          Real BFval "Forward beta coefficient at device temperature";
          Real BRval "Backward beta coefficient at device temperature";
          Real facln
          "Logarithm of temperature dependence of saturation current";
      public
        Modelica.Blocks.Interfaces.RealInput vbc(redeclare type SignalType =
                Modelica.SIunits.Voltage) "Base-collector voltage"
          annotation (extent=[-120,50; -100,70]);
        Modelica.Blocks.Interfaces.RealInput vbe(redeclare type SignalType =
                Modelica.SIunits.Voltage) "Base-emitter voltage"
          annotation (extent=[-120,-30; -100,-10]);
        Modelica.Blocks.Interfaces.RealInput ibc(redeclare type SignalType =
                Modelica.SIunits.Current) "Base-collector current"
          annotation (extent=[-120,10; -100,30]);
        annotation (Diagram(
            Rectangle(extent=[-100,100; 100,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255})),
            Text(
              extent=[-72,58; 76,20],
              style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1),
              string="Injected currents"),
            Text(
              extent=[-18,22; 16,-2],
              style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1),
              string="and"),
            Text(
              extent=[-68,-6; 70,-28],
              style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1),
              string="base charge")), Icon(
            Rectangle(extent=[-100,100; 100,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255})),
            Text(
              extent=[-70,58; 78,20],
              style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1),
              string="Injected currents"),
            Text(
              extent=[-18,22; 16,-2],
              style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1),
              string="and"),
            Text(
              extent=[-68,-6; 70,-28],
              style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1),
              string="base charge")));
        Modelica.Blocks.Interfaces.RealInput ibe(redeclare type SignalType =
                Modelica.SIunits.Current) "Base-emitter current"
          annotation (extent=[-120,-70; -100,-50]);
        Modelica.Blocks.Interfaces.RealOutput qb "Base charge"
          annotation (extent=[100,-30; 120,-10]);
        Modelica.Blocks.Interfaces.RealOutput C0(redeclare type SignalType =
                Modelica.SIunits.Capacitance) "Base capacitance"
          annotation (extent=[100,-70; 120,-50]);
        Modelica.Blocks.Interfaces.RealOutput ib0(redeclare type SignalType =
                Modelica.SIunits.Current) "Injected base current"
          annotation (extent=[100,50; 120,70]);
        Modelica.Blocks.Interfaces.RealOutput ic0(redeclare type SignalType =
                Modelica.SIunits.Current) "Injected collector current"
          annotation (extent=[100,10; 120,30]);
        Modelica.Blocks.Interfaces.RealInput Tdev(redeclare type SignalType =
                Modelica.SIunits.Temperature) "Device temperature"
          annotation (extent=[-10,-120; 10,-100], rotation=90);
        equation
          RTemp = Tdev/Tnom;
          Rxtb = RTemp^XTB;

          /* Compute thermal voltage as function of temperature */
          Vt = k*Tdev/q;

          /* Compute base charge */
          if Level==2 then
            /* Gummel-Poon model */
            q1 = 1/(1 - vbc/VAF - vbe/VAR);
            q2 = (ISS/IS) * (ibc/IKR + ibe/IKF) / Area;
            qb = q1*0.5*(1 + sqrt(1 + 4*q2));
            dq1 = 1/(VAR*q1*q1);
            dq2 = (ien + ISEval)/(Vt*NC*IKF);
            dqb = qb*dq1/q1 + q1*dq2/sqrt(1+4*q2);
            C0 = ibe*TF*dqb/(qb*qb);
          else
            /* Ebers-Moll model */
            q1 = 1/(1 - vbc/VAF);
            q2 = 0;
            qb = q1;
            dq1 = 0;
            dq2 = 0;
            dqb = 0;
            C0 = 0;
          end if;

          /* Compute non-linear current sources */
          BFval = BF*Rxtb;
          BRval = BR*Rxtb;
          EGval = EG - GapC1*Tdev*Tdev/(Tdev + GapC2);
          facln  = (RTemp - 1)*EGval/Vt + XTI*ln(RTemp);
          ISCval = ISC*exp(facln/NC)/Rxtb;
          ISEval = ISE*exp(facln/NE)/Rxtb;
          icn = Area*ISCval*(exp(vbc/(Vt*NC)) - 1);
          ien = Area*ISEval*(exp(vbe/(Vt*NE)) - 1);

          if Level==2 then
            /* Gummel-Poon model */
            ic0 = (ibe - ibc)/qb - ibc/BRval - icn;
            ib0 = ibe/BFval + ibc/BRval + ien + icn;
          else
            /* Ebers-Moll model */
            ic0 = (ibe - ibc)/qb - ibc/BRval;
            ib0 = ibe/BFval + ibc/BRval;
          end if;
        end BJTvars;

        model RS2 "Spice-style resistor model for BJTs"
          extends Interfaces.SpiceTwoPort;
          parameter Modelica.SIunits.Resistance R=0
          "Resistance at reference temperature";
          parameter Real TR1(unit="1/K") = 0 "Linear temperature coefficient";
          parameter Real TR2(unit="1/K2") = 0
          "Quadratic temperature coefficient";
          parameter Real Area=1 "Relative area occupied by resistor";
          parameter Integer Level=2
          "Transistor modeling level (Ebers-Moll = 1; Gummel-Poon = 2)";
          Modelica.SIunits.Resistance Rval
          "Resistance value at circuit temperature";
        equation
          if Level==2 and R > 0 then
          /* Gummel-Poon model */
            Rval = R*(1 + TR1*DTemp + TR2*DTempSq)/Area;
          else
          /* Ebers-Moll model */
            Rval = 0;
          end if;
          e1 = Rval*f1;
          e1*f1 = e2*f2;
        annotation (
          Diagram(
              Text(
                extent=[-82,78; 28,-78],
                string="RS",
                style(color=0))),
          Icon(
              Text(
                extent=[-82,78; 28,-78],
                string="RS",
                style(color=0)),
              Text(extent=[-72,-58; 66,-86],   string="R=%R")),
          Documentation(info="<html>
This is the regular Spice-style resistive source <b>RS</b> model, except that it looks at the modeling level.  If the level is not equal to 2, the resistance is set equal to zero.
</html>
"));
        end RS2;

        model RBS "Spice-style base resistor model of bipolar transistors"
          extends Interfaces.SpiceTwoPort;
          constant Real pi=Modelica.Constants.pi;
          parameter Modelica.SIunits.Resistance RB=0
          "Resistance at reference temperature";
          parameter Modelica.SIunits.Resistance RBM=0
          "Modified resistance at reference temperature";
          parameter Modelica.SIunits.Current IRB=0
          "Reference base current at reference temperature";
          parameter Real TRB1(unit="1/K") = 0 "Linear temperature coefficient";
          parameter Real TRB2(unit="1/K2") = 0
          "Quadratic temperature coefficient";
          parameter Real TRM1(unit="1/K") = 0 "Linear temperature coefficient";
          parameter Real TRM2(unit="1/K2") = 0
          "Quadratic temperature coefficient";
          parameter Real Area=1 "Relative area occupied by resistor";
          parameter Integer Level=2
          "Transistor modeling level (Ebers-Moll = 1; Gummel-Poon = 2)";
          Modelica.SIunits.Resistance RBval
          "Base resistance value at circuit temperature";
          Modelica.SIunits.Resistance RBMval
          "Modified base resistance value at circuit temperature";
          Modelica.SIunits.Resistance Rval
          "Resistance value at circuit temperature";

      protected
          parameter Real piSq = pi*pi;
          Real z;
          Real tz;
        annotation (
          Icon(
              Text(
                extent=[-82,76; 28,-80],
                style(color=0),
              string="RBS"),
              Text(extent=[-72,-58; 66,-86], string="R=%RB")),
          Diagram(
              Text(
                extent=[-82,76; 28,-80],
                style(color=0),
              string="RBS")),
          Documentation(info="<html>
The base resistor is the most important of the parasitic resistors of a bipolar transistor.  For this reason, Spice models this resistor using a more sophisticated model with additional parameters. <p>

<hr> <p>

<pre>
Parameters:

 RB:     Reference zero-bias base resistance (default value = 0 Ohm)

 RBM:    Reference minimum base resistance (default value = 0 Ohm)

 IRB:    Current where base resistance falls halfway to minimum value (default value = 0 Amp)

 Tnom:   Reference temperature (default value = 300.15 <sup>o</sup>K)

 TRB1:   Linear temperature variation coefficient of RB (default value = 0 Ohm/<sup>o</sup>K)

 TRB2:   Quadratic temperature variation coefficient of RB (default value = 0 Ohm/<sup>o</sup>K<sup>2</sup>)

 TRM1:   Linear temperature variation coefficient of RBM (default value = 0 Ohm/<sup>o</sup>K)

 TRM2:   Quadratic temperature variation coefficient of RBM (default value = 0 Ohm/<sup>o</sup>K<sup>2</sup>)

 Area:   Relative area occupied by resistor (default value = 1)

 Level:  Transistor modeling level (Ebers-Moll = 1; Gummel-Poon = 2) (default value = 2)
</pre> <p>

<hr> <p>

Equations: <p>

<b>DTemp = e2 - Tnom</b> <br>
<b>RBval = RB*(1 + TRB1*DTemp + TRB2*DTemp*DTemp)/Area</b> <br>
<b>RBMval = RBM*(1 + TRM1*DTemp + TRM2*DTemp*DTemp)/Area</b> <br>
<b>z = if IRB &gt; 0 then (-1 + sqrt(1 + 144*f1/(pi*pi*IRB*Area)))/(24*sqrt(f1/(IRB*Area))/(pi*pi)) else 0</b> <br>
<b>tz = if IRB &gt; 0 then tan(z) else 0</b> <br>
<b>Rval = if IRB &gt; 0 then (Rbmval + 3*(Rbval - Rbmval)*(tz-z)/(z*tz*tz)) else (Rbmval + (Rbval - Rbmval)/qb)</b> <br>
<b>e1 = Rval*f1</b> <br>
<b>f2 = (e1/e2)*f1</b>
</html>
"));
      public
        Modelica.Blocks.Interfaces.RealInput qb "Base charge"
          annotation (extent=[-10,90; 10,110], rotation=-90);
        equation
          RBval = RB*(1 + TRB1*DTemp + TRB2*DTempSq)/Area;
          RBMval = RBM*(1 + TRM1*DTemp + TRM2*DTempSq)/Area;

          if Level==2 and (RB > 0 or RBM > 0) then
            /* Gummel-Poon model */
            if IRB > 0 and IRB < 1e10 then
              z = (-1 + sqrt(1 + 144*f1/(piSq*IRB*Area)))/(24*sqrt(f1/(IRB*Area))/piSq);
              tz = tan(z);
              Rval = RBMval + 3*(RBval - RBMval)*(tz-z)/(z*tz*tz);
            else
              z = 0;
              tz = 0;
              Rval = RBMval + (RBval - RBMval)/qb;
            end if;
          else
            /* Ebers-Moll model */
            z = 0;
            tz = 0;
            Rval = 0;
          end if;

          e1 = Rval*f1;
          e1*f1 = e2*f2;
        end RBS;

        model Gj "Spice-style leakage conductance model"
          extends BondLib.Interfaces.PassiveOnePort;
          parameter Modelica.SIunits.Conductance GminDC=1e-16
          "Leakage conductance";
          parameter Real Area=1 "Relative area occupied by device";
          parameter Integer Level=2
          "Transistor modeling level (Ebers-Moll = 1; Gummel-Poon = 2)";
        annotation (
          Icon(
              Text(
                extent=[-84,64; 12,-68],
                style(color=0),
              string="Gj"),
            Text(
              extent=[0,-6; 84,-34],
              string="Spice",
              style(color=0, rgbcolor={0,0,0})),
                                 Text(extent=[-68,92; 72,52],   string="%name")),
          Diagram(
              Text(
                extent=[-84,64; 12,-68],
                style(color=0),
              string="Gj"), Text(
              extent=[0,-6; 84,-34],
              string="Spice",
              style(color=0, rgbcolor={0,0,0}))),
          Documentation(info="<html>
The Spice-style leakage conductance model is a conductor element without temperature variation.  The <b>Level</b> parameter is used to acknowledge the additional <b>Area</b> parameter of the Gummel-Poon transistor model. No heat port is needed here, as the model is used only as a mathematical construct to compute the currents of the non-linear current sources. The heat generation is accounted for in the non-linear sources. <p>

<hr> <p>

<pre>
Parameters:

 GminDC:   Leakage conductance (default value = 1e-16 Mho)

 Area:     Relative area occupied by resistor (default value = 1)

 Level:    Transistor modeling level (Ebers-Moll = 1; Gummel-Poon = 2) (default value = 2)
</pre> <p>

<hr> <p>

Equations: <p>

<b>f = Area*GminDC*e</b>
</html>
"));
        equation
          if Level==2 then
            /* Gummel-Poon model */
            f = Area*GminDC*e;
          else
            /* Ebers-Moll model */
            f = GminDC*e;
          end if;
        end Gj;

        model mC3 "Bondgraphic modulated capacitor element for Spice"
          extends BondLib.Interfaces.PassiveOnePort;
          parameter Integer Level=2
          "Transistor modeling level (Ebers-Moll = 1; Gummel-Poon = 2)";
          parameter Real XCJ=1
          "Fraction of depletion capacitance connected to internal base node";
          Real C "Bondgraphic Capacitance";

      protected
          Real du "Derivative of voltage across capacitor";
        annotation (
          Diagram(
                 Text(
                extent=[-76,78; 44,-82],
                style(color=0),
                string="mC"),
                 Text(
                extent=[40,-2; 70,-54],
                style(color=0),
              string="3")),
          Icon(                  Text(extent=[-70,92; 70,52],   string="%name"),
                 Text(
                extent=[-76,78; 44,-82],
                style(color=0),
                string="mC"),
                 Text(
                extent=[40,-2; 70,-54],
                style(color=0),
              string="3")),
          Documentation(info="<html>
This model presents a generalization of the <b>mC2</b> model presented in the semiconductor section, generalized for Spice-style transistor models.  It computes also the external current, <b>ix</b>, which doesn't pass through the junction, but in parallel to it through the <b>Cbx</b> capacitor, which is realized as a current source to prevent an unnecessary higher index DAE problem. <p>

<hr> <p>

<pre>
Potential variables:

 e:      Bondgraphic effort variable

 f:      Bondgraphic flow variable, normalized positive for flows into the model

 Cdiff:  Diffusion capacitance (modulating input signal)

 Cdepl:  Depletion capacitance (modulating input signal)

 ix:     External depletion capacitive current


Parameters:

 XCJC:   Fraction of depletion capacitance connected to internal base node (default value = 1)

 Level:  Transistor modeling level (Ebers-Moll = 1; Gummel-Poon = 2) (default value = 2)
</pre> <p>

<hr> <p>

Equations: <p>

<b>C = Cdiff + XCJ*Cdepl;</b> <br>
<b>du = f/C;</b> <br>
<b>ix = (1 - XCJ)*Cdepl*du;</b> <br>
<b>der(e) = du;</b>
</html>
"));
      public
        Modelica.Blocks.Interfaces.RealInput Cdiff(redeclare type SignalType =
                Modelica.SIunits.Capacitance) "Diffusion capacitance"
          annotation (extent=[-60,-110; -40,-90], rotation=90);
        Modelica.Blocks.Interfaces.RealInput Cdepl(redeclare type SignalType =
                Modelica.SIunits.Capacitance) "Depletion capacitance"
          annotation (extent=[40,-110; 60,-90], rotation=90);
        Modelica.Blocks.Interfaces.RealOutput ix(redeclare type SignalType =
                Modelica.SIunits.Current)
          "External base-collector capacitance current"
          annotation (extent=[-10,90; 10,110], rotation=90);
        equation
          if Level==2 then
          /* Gummel-Poon model */
            C = Cdiff + XCJ*Cdepl;
            du = f/C;
            ix = (1 - XCJ)*Cdepl*du;
            der(e) = du;
          else
          /* Ebers-Moll model */
            C = Cdiff + Cdepl;
            du = f/C;
            ix = 0;
            der(e) = du;
          end if;
        end mC3;

        model CjS
        "Spice-style junction capacitor model for bipolar transistors"
          extends BondLib.Interfaces.TwoPort;
          parameter Modelica.SIunits.Temperature Tnom=300.15
          "Reference temperature";
          parameter Modelica.SIunits.Current IS=1e-16
          "Saturation current at reference temperature";
          parameter Modelica.SIunits.Conductance GminDC=1e-15
          "Leakage conductance";
          parameter Real N=1 "Current emission coefficient";
          parameter Modelica.SIunits.Capacitance CJ=0.5e-12
          "Zero-bias depletion capacitance at reference temperature";
          parameter Real MJ=0.333 "Junction grading coefficient";
          parameter Modelica.SIunits.Voltage VJ=0.8
          "Built-in potential at reference temperature";
          parameter Real XCJ=1
          "Fraction of depletion capacitance connected to internal base node";
          parameter Real FC=0.5
          "Factor for linearization of depletion capacitance formula";
          parameter Modelica.SIunits.Time Tau=1e-9 "Ideal transit time";
          parameter Real XTI=3 "Saturation current temperature exponent";
          parameter Real XTB=0
          "Forward and reverse beta temperature coefficient";
          parameter Modelica.SIunits.Voltage EG=1.11
          "Energy gap for temperature effect on saturation current";
          parameter Real Area=1 "Relative area occupied by device";
          parameter Integer Level=2
          "Transistor modeling level (Ebers-Moll = 1; Gummel-Poon = 2)";
          parameter Boolean BE=true "True if base-emitter junction";
          parameter Real EMin=-100
          "if x < EMin, the exp(x) function is linearized";
          parameter Real EMax=40
          "if x > EMax, the exp(x) function is linearized";
        annotation (
          Icon(
              Text(
                extent=[-82,76; 28,-80],
                style(color=0),
              string="CjS"),
            Text(
              extent=[14,-4; 102,-32],
              string="Spice",
              style(color=0, rgbcolor={0,0,0})),
                                 Text(extent=[-70,88; 70,48],   string="%name")),
          Diagram,
          Documentation(info="<html>
The Spice-style junction capacitance computes both the depletion capacitance value and the diffusion capacitance value of a junction used in a bipolar transistor.  Different Spice dialects vary in the formulae they use for this purpose.  Many Spice dialects actually don't use a formula for the junction capacitance at all, but rather compute the electric charge stored in the junction directly, which is conceptually cleaner.  However, that approach is computationally cumbersome, as it leads to an awkward algebraic loop [1].  Thus, we chose to compute the junction capacitance, and use a (physically incorrect) approximate non-linear capacitor model (just as in the case of the simpler Ebers-Moll model).  The numerical error should remain small, as the time constants associated with temperature change are much larger than those associated with electrical phenomena. <p>

This particular model uses the space charge formula advocated in [2].  Since: <p>

<b>f = der(q)</b> <p>

the space charge, <b>q</b>, was symbolically differentiated.  In the differentiation, only electrical signals, i.e., the voltage, <b>e</b>, were considered time-varying.  The temperature gradients were assumed to be negligible.  This produced an equation of the form: <p>

<b>f = C(e)*der(e)</b> <p>

which is the formula that was used to compute the non-linear capacitance <b>C(e)</b> in the model. <p>

<hr> <p>

<pre>
Parameters:

 Is:      Transport saturation current (default value = 1e-16 Amp)

 EG:      Energy gap for temperature effect on saturation current (default value = 1.11 Volt)

 N:       Current emission coefficient (default value = 1)

 XTI:     Saturation current temperature exponent (default value = 3)

 VJ:      Built-in potential at reference temperature (default value = 0.75 Volt)

 CJ:      Zero-bias depletion capacitance at reference temperature (default value = 0.5e-12 F)

 XCJ:     Fraction of base-collector depletion capacitance connected to internal base node (default value = 1)

 MJ:      Junction grading coefficient (default value = 0.333)

 Tau:     Ideal transit time (default value = 1e-9 sec)

 GminDC:  Leakage conductance (default value = 1e-15 Mho)

 Area:    Relative area occupied by the diode (default value = 1)

 Level:  Transistor modeling level (Ebers-Moll = 1; Gummel-Poon = 2) (default value = 2)

 BE:      True if base-emitter junction (default value = true)
</pre> <p>

<hr> <p>

<b>References:</b> <p>

<ol>
<li> Cellier, F.E. (1991), <i>Continuous System Modeling</i>, Springer-Verlag, New York, pp. 224-225.
<li> Massobrio, G. and P. Antognetti (1993), <i>Semiconductor Device Modeling with Spice</i>,
     2<sup>nd</sup> edition, McGraw Hill, New York, p.69.
</ol>
</html>
"));
      public
        Modelica.Blocks.Interfaces.RealInput qb "Base charge"
          annotation (extent=[40,-110; 60,-90],   rotation=90);
        Modelica.Blocks.Interfaces.RealInput C0(redeclare type SignalType =
                Modelica.SIunits.Capacitance) "Base capacitance"
          annotation (extent=[-60,-110; -40,-90],
                                                rotation=90);
        Modelica.Blocks.Interfaces.RealOutput ix(redeclare type SignalType =
                Modelica.SIunits.Current)
          "External base-collector capacitance current"
          annotation (extent=[-60,90; -40,110],rotation=90);
        Modelica.Blocks.Interfaces.RealOutput id(redeclare type SignalType =
                Modelica.SIunits.Current) "Diode current"
          annotation (extent=[40,90; 60,110],  rotation=90);
          Junctions.J0p3 J0p3_1 annotation (extent=[-90,-10; -70,10]);
          Bonds.fBond B1 annotation (extent=[-70,-10; -50,10]);
          Sensors.De vbi annotation (extent=[-50,-10; -30,10]);
          Bonds.eBond B2 annotation (extent=[-90,-30; -70,-10],
                                                              rotation=270);
          Sources.mSe Vbi annotation (extent=[0,-10; -20,10]);
          Bonds.fBond B3 annotation (extent=[0,-10; 20,10]);
          Junctions.J1p3 J1p3_1 annotation (extent=[20,-10; 40,10], rotation=
              -90);
          Bonds.eBond B4 annotation (extent=[20,10; 40,30],  rotation=90);
          Sensors.Df id1
                        annotation (extent=[20,30; 40,50], rotation=90);
          Bonds.fBond B5 annotation (extent=[20,-30; 40,-10], rotation=270);
          Junctions.J0p3 J0p3_2 annotation (extent=[20,-30; 40,-50]);
          Bonds.fBond B6 annotation (extent=[40,-50; 60,-30], rotation=0);
          Bonds.fBond B7 annotation (extent=[0,-50; 20,-30], rotation=180);
        nlC nlC1(
          Tnom=Tnom,
          IS=IS,
          EG=EG,
          N=N,
          XTI=XTI,
          VJ=VJ,
          CJ=CJ,
          MJ=MJ,
          FC=FC,
          Tau=Tau,
          GminDC=GminDC,
          Area=Area,
          Level=Level,
          BE=BE,
          EMin=EMin,
          EMax=EMax) annotation (extent=[-40,-50; -60,-30]);
          Bonds.fBond B8 annotation (extent=[50,10; 70,30], rotation=90);
          Sensors.De Tdev
                         annotation (extent=[50,30; 70,50], rotation=90);
        DjS DjS1(
          Tnom=Tnom,
          IS=IS,
          EG=EG,
          N=N,
          XTI=XTI,
          Area=Area,
          Level=Level,
          EMin=EMin,
          EMax=EMax) annotation (extent=[60,-52; 100,-28]);
        mC3 Cj1(Level=Level, XCJ=XCJ)
          annotation (extent=[-70,-50; -90,-30], rotation=-90);
          Junctions.J0p3 J0p3_3 annotation (extent=[70,-10; 90,10]);
        Gj Gj1(
          GminDC=GminDC,
          Area=Area,
          Level=Level)
               annotation (extent=[0,-50; -28,-30]);
          Bonds.eBond B9 annotation (extent=[90,-36; 110,-16],
                                                             rotation=90);
        equation
        connect(J0p3_1.BondCon1, BondCon1) annotation (points=[-90,0; -100,0],
            style(color=8, rgbcolor={192,192,192}));
        connect(J0p3_1.BondCon3, B2.fBondCon1) annotation (points=[-80,-10; -80,
              -10], style(color=8, rgbcolor={192,192,192}));
        connect(B2.eBondCon1, Cj1.BondCon1) annotation (points=[-80,-30; -80,
              -30], style(color=8, rgbcolor={192,192,192}));
        connect(nlC1.Cdiff, Cj1.Cdiff) annotation (points=[-61,-35; -70,-35],
            style(color=74, rgbcolor={0,0,127}));
        connect(nlC1.Cdepl, Cj1.Cdepl) annotation (points=[-61,-45; -70,-45],
            style(color=74, rgbcolor={0,0,127}));
        connect(C0, nlC1.C0) annotation (points=[-50,-100; -50,-70; -55,-70;
              -55,-50], style(color=74, rgbcolor={0,0,127}));
        connect(qb, nlC1.qb) annotation (points=[50,-100; 50,-70; -44,-70; -44,
              -50; -45,-50], style(color=74, rgbcolor={0,0,127}));
        connect(J0p3_1.BondCon2, B1.eBondCon1) annotation (points=[-70,0; -70,0],
            style(color=8, rgbcolor={192,192,192}));
        connect(Cj1.ix, ix) annotation (points=[-90,-40; -112,-40; -112,70; -50,
              70; -50,100], style(color=74, rgbcolor={0,0,127}));
        connect(vbi.BondCon1, B1.fBondCon1) annotation (points=[-50,0; -50,0],
            style(color=8, rgbcolor={192,192,192}));
        connect(Vbi.BondCon1, B3.eBondCon1) annotation (points=[0,0; 0,0],
            style(color=8, rgbcolor={192,192,192}));
        connect(id1.BondCon1, B4.eBondCon1) annotation (points=[30,30; 30,30],
            style(color=8, rgbcolor={192,192,192}));
        connect(id1.OutPort1, id) annotation (points=[30,51; 30,70; 50,70; 50,
              100], style(color=74, rgbcolor={0,0,127}));
        connect(Tdev.OutPort1, nlC1.Tdev) annotation (points=[60,51; 60,60; -56,
              60; -56,-20; -50,-20; -50,-30], style(color=74, rgbcolor={0,0,127}));
        connect(J1p3_1.BondCon3, B3.fBondCon1) annotation (points=[20,
              -6.12303e-016; 20,0], style(color=8, rgbcolor={192,192,192}));
        connect(J1p3_1.BondCon1, B4.fBondCon1) annotation (points=[30,10; 30,10],
            style(color=8, rgbcolor={192,192,192}));
        connect(B5.eBondCon1, J1p3_1.BondCon2) annotation (points=[30,-10; 30,
              -10], style(color=8, rgbcolor={192,192,192}));
        connect(B5.fBondCon1, J0p3_2.BondCon3) annotation (points=[30,-30; 30,
              -30], style(color=8, rgbcolor={192,192,192}));
        connect(J0p3_2.BondCon2, B6.eBondCon1) annotation (points=[40,-40; 40,
              -40], style(color=8, rgbcolor={192,192,192}));
        connect(B7.eBondCon1, J0p3_2.BondCon1) annotation (points=[20,-40; 20,
              -40], style(color=8, rgbcolor={192,192,192}));
        connect(nlC1.v, vbi.OutPort1) annotation (points=[-40,-40; -34,-40; -34,
              -12; -24,-12; -24,0; -29,0], style(color=74, rgbcolor={0,0,127}));
        connect(Tdev.BondCon1, B8.fBondCon1) annotation (points=[60,30; 60,30],
            style(color=8, rgbcolor={192,192,192}));
        connect(J0p3_3.BondCon2, BondCon2) annotation (points=[90,0; 100,0],
            style(color=8, rgbcolor={192,192,192}));
        connect(J0p3_3.BondCon1, B8.eBondCon1) annotation (points=[70,0; 60,0;
              60,10], style(color=8, rgbcolor={192,192,192}));
        connect(Gj1.BondCon1, B7.fBondCon1) annotation (points=[0,-40; 0,-40],
            style(color=8, rgbcolor={192,192,192}));
        connect(B6.fBondCon1, DjS1.BondCon1) annotation (points=[60,-40; 60,-40],
            style(color=8, rgbcolor={192,192,192}));
        connect(B9.fBondCon1, DjS1.BondCon2) annotation (points=[100,-36; 100,
              -40], style(color=8, rgbcolor={192,192,192}));
        connect(B9.eBondCon1, J0p3_3.BondCon3) annotation (points=[100,-16; 100,
              -14; 80,-14; 80,-10], style(color=8, rgbcolor={192,192,192}));
        connect(vbi.OutPort1, Vbi.s) annotation (points=[-29,0; -19,0], style(
              color=74, rgbcolor={0,0,127}));
        end CjS;

        model CbxS
        "Spice-style external base-collector capacitance model for bipolar transistors"
          extends BondLib.Interfaces.TwoPort;
        annotation (
          Icon(
              Text(
                extent=[-84,76; 26,-80],
                style(color=0),
              string="CbxS"), Text(
              extent=[12,2; 100,-26],
              string="Spice",
              style(color=0, rgbcolor={0,0,0})),
            Text(
              extent=[-60,96; 60,64],
              style(color=3, rgbcolor={0,0,255}),
              string="%name")),
          Diagram,
          Documentation(info="<html>
The Spice-style external base-collector capacitance computes the external depletion capacitance value of a bipolar transistor.  The external base-collector capacitance leads invariably to a higher-index DAE.  In order to prevent the Pantelides algorithm from having to be used constantly and unnecessarily, the higher index problem was reduced manually.  This leads to an implementation, whereby the external base-collector capacitance is being realized internally as a modulated current source.
</html>
"));
      public
        Modelica.Blocks.Interfaces.RealInput ix(redeclare type SignalType =
                Modelica.SIunits.Current)
          "External base-collector capacitance current"
          annotation (extent=[-10,-110; 10,-90],  rotation=90);
        Sources.mSf dbx annotation (extent=[-40,-10; -20,10]);
        Junctions.J0p2 J0p2_1 annotation (extent=[-80,-10; -60,10]);
        Bonds.eBond B1 annotation (extent=[-60,-10; -40,10], rotation=180);
        Modelica.Blocks.Math.Gain Gain1(k=-1)
          annotation (extent=[-10,-60; 10,-40], rotation=90);
        equation
          /* Compute heat flow */
          f2 = 0;
        connect(J0p2_1.BondCon1, BondCon1) annotation (points=[-80,0; -100,0],
            style(color=8, rgbcolor={192,192,192}));
        connect(dbx.BondCon1, B1.fBondCon1) annotation (points=[-40,0; -40,
              -1.22461e-015], style(color=8, rgbcolor={192,192,192}));
        connect(ix, Gain1.u) annotation (points=[0,-100; 0,-81; 0,-62;
              -7.34764e-016,-62], style(color=74, rgbcolor={0,0,127}));
        connect(J0p2_1.BondCon2, B1.eBondCon1) annotation (points=[-60,0; -60,
              1.22461e-015], style(color=8, rgbcolor={192,192,192}));
        connect(Gain1.y, dbx.s) annotation (points=[6.73533e-016,-39;
              6.73533e-016,0; -21,0], style(color=74, rgbcolor={0,0,127}));
        end CbxS;

        model DjS "Spice-style junction diode model for bipolar transistors"
          extends Interfaces.SpiceTwoPort;
          constant Modelica.SIunits.Entropy k=Modelica.Constants.k
          "Boltzmann's constant";
          constant Modelica.SIunits.ElectricCharge q=1.6021892e-19
          "Electron charge";
          constant Real GapC1(unit="V/K")=7.02E-4
          "First bandgap correction factor Silicon";
          constant Modelica.SIunits.Temperature GapC2=1108.0
          "Second bandgap correction factor Silicon";
          parameter Modelica.SIunits.Current IS=1e-16
          "Saturation current at reference temperature";
          parameter Modelica.SIunits.Voltage EG=1.11
          "Energy gap for temperature effect on saturation current";
          parameter Real N=1 "Current emission coefficient";
          parameter Real XTI=3 "Saturation current temperature exponent";
          parameter Real Area=1 "Relative area occupied by diode";
          parameter Integer Level=2
          "Transistor modeling level (Ebers-Moll = 1; Gummel-Poon = 2)";
          parameter Real EMin=-100
          "if x < EMin, the exp(x) function is linearized";
          parameter Real EMax=40
          "if x > EMax, the exp(x) function is linearized";
          Modelica.SIunits.Voltage Vt "Thermal voltage";
          Modelica.SIunits.Current ISval
          "Saturation current at device temperature";
          Modelica.SIunits.Voltage EGval
          "Activation energy at device temperature";

      protected
          parameter Real ExMin = exp(EMin);
          parameter Real ExMax = exp(EMax);
          Real RTemp "Temperature quotient";
          Real et;
        annotation (
          Icon(
              Text(
                extent=[-84,76; 26,-82],
                style(color=0),
              string="DjS")),
          Diagram(
              Line(points=[-80,0; 80,0],   style(color=10)),
              Polygon(points=[70,4; 80,0; 70,-4; 70,4],     style(color=10,
                    fillColor=10)),
              Line(points=[0,80; 0,-80],   style(
                  color=10,
                  fillColor=10,
                  fillPattern=1)),
              Polygon(points=[-4,70; 0,80; 4,70; -4,70],     style(
                  color=10,
                  fillColor=10,
                  fillPattern=1)),
              Text(
                extent=[4,82; 14,72],
                style(
                  color=10,
                  fillColor=10,
                  fillPattern=1),
                string="f"),
              Text(
                extent=[70,-4; 80,-14],
                style(
                  color=10,
                  fillColor=10,
                  fillPattern=1),
                string="e"),
              Line(points=[-80,-4; -28,-4; -6,-2; 0,0],         style(
                  color=0,
                  thickness=2,
                  fillColor=10,
                  fillPattern=1)),
              Text(
                extent=[-66,-6; -36,-14],
                style(
                  color=10,
                  pattern=3,
                  fillColor=10,
                  fillPattern=1),
                string="blocking"),
              Text(
                extent=[16,50; 48,42],
                style(
                  color=10,
                  pattern=3,
                  fillColor=10,
                  fillPattern=1),
                string="conducting"),
              Line(points=[0,0; 6,10; 12,28; 16,66],            style(
                  color=0,
                  thickness=2,
                  fillColor=10,
                  fillPattern=1))),
          Documentation(info="<html>
The Spice-style junction diode for BJTs is a regular diode without heat generation. A separate model was needed to implement the <b>Level</b> parameter, as the Ebers-Moll model doesn't recognize the <b>Area</b> parameter. A heat port was still needed, as the model makes use of the device temperature that it imports through the heat port.  However, the heat flow itself is zero, as the model is used only as a mathematical construct to compute the currents of the non-linear current sources. The heat generation is accounted for in the non-linear sources. <p>

<hr>

<pre>
Parameters:

 IS:     Transport saturation current (default value = 1e-16 Amp)

 EG:     Energy gap for temperature effect on saturation current (default value = 1.11 Volt)

 N:      Current emission coefficient (default value = 1)

 XTI:    Saturation current temperature exponent (default value = 3)

 Area:   Relative area occupied by the diode (default value = 1)

 Level:  Transistor modeling level (Ebers-Moll = 1; Gummel-Poon = 2) (default = 2)

 EMin:   if x &lt; EMin, the exp(x) function is linearized (default value = -100)

 EMax:   if x &gt; EMax, the exp(x) function is linearized (default value = 40)
</pre>
</html>
"));
        equation
          /* Compute thermal voltage as function of temperature */
          Vt = k*e2/q;
          et = e1/(N*Vt);

          /* Compute temperature dependence of saturation current */
          RTemp = e2/Tnom;
          EGval = EG - GapC1*e2*e2/(e2 + GapC2);
          ISval = IS*exp((RTemp - 1)*EGval/Vt + XTI*ln(RTemp));

          /* Compute diode characteristic */
          if Level==2 then
            /* Gummel-Poon model */
            f1 = ISval*Area*(if et < EMin then ExMin*(et - EMin + 1) - 1 else
                             if et > EMax then ExMax*(et - EMax + 1) - 1 else exp(et) - 1);
          else
            /* Ebers-Moll model */
            f1 = ISval*(if et < EMin then ExMin*(et - EMin + 1) - 1 else
                        if et > EMax then ExMax*(et - EMax + 1) - 1 else exp(et) - 1);
          end if;

          /* Compute heat flow */
          f2 = 0;
        end DjS;

        model NPNint
        "Spice-style NPN transistor model without parasitic resistors and substrate diode"
          /* DC model parameters */
          constant Real inf=Modelica.Constants.inf;
          parameter Real BF=100
          "Maximum forward current gain at reference temperature";
          parameter Real BR=1
          "Maximum reverse current gain at reference temperature";
          parameter Modelica.SIunits.Current IS=1e-16
          "Saturation current at reference temperature";
          parameter Modelica.SIunits.Current ISS=IS
          "Saturation current used for current injection";
          parameter Real NF=1 "Forward current emission coefficient";
          parameter Real NR=1 "Reverse current emission coefficient";
          parameter Modelica.SIunits.Conductance GminDC(min=0) = 1e-19
          "Leakage conductance";

          /* Low current beta degradation effect parameters */
          parameter Modelica.SIunits.Current ISC(min=0) = 0
          "Base-collector leakage saturation current at reference temperature (ISC = C4*IS)";
          parameter Modelica.SIunits.Current ISE(min=0) = 0
          "Base-emitter leakage saturation current at reference temperature (ISE = C2*IS)";
          parameter Real NC=2
          "Low-current base-collector leakage emission coefficient";
          parameter Real NE=1.5
          "Low-current base-emitter leakage emission coefficient";

            /* Base width modulation parameters */
          parameter Modelica.SIunits.Voltage VAF=inf "Forward early voltage";
          parameter Modelica.SIunits.Voltage VAR=inf "Reverse early voltage";

          /* High current beta degradation effect parameters */
          parameter Modelica.SIunits.Current IKF=inf
          "Corner for forward beta high-current roll-off";
          parameter Modelica.SIunits.Current IKR=inf
          "Corner for reverse beta high-current roll-off";

          /* Junction capacitor parameters */
          parameter Modelica.SIunits.Capacitance CJC(min=Modelica.Constants.small) = 1e-12
          "Zero-bias base-collector depletion capacitance at reference temperature";
          parameter Real MJC(min=0.3,max=0.5) = 0.33
          "Base-collector junction grading coefficient";
          parameter Modelica.SIunits.Voltage VJC=0.8
          "Base-collector built-in potential at reference temperature";
          parameter Modelica.SIunits.Capacitance CJE(min=Modelica.Constants.small) = 1e-12
          "Zero-bias base-emitter depletion capacitance at reference temperature";
          parameter Real MJE(min=0.3,max=0.5) = 0.33
          "Base-emitter junction grading coefficient";
          parameter Modelica.SIunits.Voltage VJE=0.75
          "Base-emitter built-in potential at reference temperature";
          parameter Real XCJC(min=0,max=1) = 1
          "Fraction of base-collector depletion capacitance connected to internal base node";
          parameter Real FC(min=0,max=1) = 0.5
          "Depletion capacitance factor for linearization";

          /* Transit time parameters */
          parameter Modelica.SIunits.Time TF(min=0) = 0
          "Ideal forward transit time";
          parameter Modelica.SIunits.Time TR(min=0) = 0
          "Ideal reverse transit time";

          /* Temperature compensation and area parameters */
          parameter Modelica.SIunits.Temperature Tnom=300.15
          "Reference temperature";
          parameter Real XTI=3 "Saturation current temperature exponent";
          parameter Real XTB=0
          "Forward and reverse beta temperature coefficient";
          parameter Modelica.SIunits.Voltage EG=1.11
          "Energy gap for temperature effect on saturation current";
          parameter Real Area=1 "Relative area occupied by device";
          parameter Integer Level(min=1,max=2) = 2
          "Transistor modeling level (Ebers-Moll = 1; Gummel-Poon = 2)";

          /* Numerical parameters */
          parameter Real EMin=-100
          "if x < EMin, the exp(x) function is linearized";
          parameter Real EMax=40
          "if x > EMax, the exp(x) function is linearized";
        annotation (Coordsys(extent=[-140,-140; 140,140]), Diagram(
            Text(
              extent=[-152,28; -126,12],
              style(color=10, rgbcolor={135,135,135}),
              string="Bint"),
            Text(
              extent=[10,148; 36,130],
              style(color=10, rgbcolor={135,135,135}),
              string="Cint"),
            Text(
              extent=[10,-132; 36,-150],
              style(color=10, rgbcolor={135,135,135}),
              string="Eint"),
            Text(
              extent=[126,-2; 158,-28],
              style(color=10, rgbcolor={135,135,135}),
              string="Heat"),
            Text(
              extent=[-90,22; -78,10],
              style(color=10, rgbcolor={135,135,135}),
              string="vbc"),
            Text(
              extent=[-98,-12; -86,-24],
              style(color=10, rgbcolor={135,135,135}),
              string="vbe"),
            Text(
              extent=[-126,-38; -114,-50],
              style(color=10, rgbcolor={135,135,135}),
              string="ibe"),
            Text(
              extent=[-124,122; -112,110],
              style(color=10, rgbcolor={135,135,135}),
              string="ibc"),
            Text(
              extent=[92,-124; 110,-140],
              style(color=10, rgbcolor={135,135,135}),
              string="Tdev"),
            Text(
              extent=[136,40; 154,24],
              style(color=10, rgbcolor={135,135,135}),
              string="Tdev"),
            Text(
              extent=[134,-60; 146,-74],
              style(color=10, rgbcolor={135,135,135}),
              string="IB0"),
            Text(
              extent=[-22,-114; -10,-128],
              style(color=10, rgbcolor={135,135,135}),
              string="IB0"),
            Text(
              extent=[100,-20; 112,-34],
              style(color=10, rgbcolor={135,135,135}),
              string="IC0"),
            Text(
              extent=[134,-100; 144,-112],
              style(color=10, rgbcolor={135,135,135}),
              string="qb"),
            Text(
              extent=[-112,-106; -102,-118],
              style(color=10, rgbcolor={135,135,135}),
              string="qb"),
            Text(
              extent=[106,-140; 120,-156],
              style(color=10, rgbcolor={135,135,135}),
              string="C0"),
            Text(
              extent=[-78,-108; -66,-120],
              style(color=10, rgbcolor={135,135,135}),
              string="C0")),
          Icon(
              Text(
                extent=[-112,90; 96,-84],
                style(color=0),
                string="NPN"),
              Line(points=[0,100; 0,-100],
                                         style(
                  color=9,
                  rgbcolor={149,164,171},
                  thickness=2,
                  fillColor=74,
                  rgbfillColor={0,0,185},
                  fillPattern=1)),
              Line(points=[-20,-80; 0,-100],style(
                  color=9,
                  rgbcolor={149,164,171},
                  thickness=2,
                  fillColor=74,
                  rgbfillColor={0,0,185},
                  fillPattern=1)),
              Line(points=[0,-100; 20,-80],style(
                  color=9,
                  rgbcolor={149,164,171},
                  thickness=2,
                  fillColor=74,
                  rgbfillColor={0,0,185},
                  fillPattern=1)),
              Text(
                extent=[8,-126; 38,-152],
                style(color=9, rgbcolor={149,164,171}),
              string="Eint"),
              Text(
                extent=[-156,28; -124,4],
                style(color=9, rgbcolor={149,164,171}),
              string="Bint"),
              Text(
                extent=[10,150; 38,126],
                style(color=9, rgbcolor={149,164,171}),
              string="Cint"),
              Text(
                extent=[122,30; 158,6],
                style(color=9, rgbcolor={149,164,171}),
              string="Heat"),
              Text(
                extent=[142,-66; 164,-86],
                style(color=9, rgbcolor={149,164,171}),
              string="qb"),
            Line(points=[-100,-60; 100,-60], style(color=3, rgbcolor={0,0,255})),
            Line(points=[80,-40; 100,-60], style(color=3, rgbcolor={0,0,255})),
            Line(points=[80,-80; 100,-60], style(color=3, rgbcolor={0,0,255})),
              Text(
                extent=[88,4; 128,-38],
                style(color=0),
              string="int"),
            Text(
              extent=[-80,120; 80,80],
              style(color=3, rgbcolor={0,0,255}),
              string="%name")),
          Documentation(info="<html>
The NPN element of the Spice bond graph library implements a full-fledges Spice-stye Gummel-Poon model of the NPN bipolar transistor [1-3].  <b>NPNint</b> is a partial model that implements only the internal nodes of the bipolar transistor.  The external parasitic resistances, the external capacitor between base and collector, and the substrate are being added from the outside. <p>

Notice the use of the two internal modulated current sources.  They are connected on the primary side to the <b>Heat</b> port.  As shown in [4,5], these current sources are in fact non-linear resistors, as voltage and current through these sources are always pointing in the same direction.  Thus, they generate heat. <p>

The NPN bipolar transistor is a directed FourPort.  The direction of positive power flow is assumed into the model at the base, <b>B</b>, and at the collector, <b>C</b>, whereas it is assumed out of the model at the emitter, <b>E</b>, and at the <b>Heat</b> port. <p>

The causality of the NPN model is free. <p>

<hr>

<pre>
<b>Parameters:</b>

 Level:   Transistor modeling level (default value = 2)
            Level = 1: Ebers-Moll model
            Level = 2: Gummel-Poon model


<b>DC Model Parameters:</b>

 BF:      Maximum forward current gain at reference temperature (default value = 100)

 BR:      Maximum reverse current gain at reference temperature (default value = 1)

 IS:      Saturation current at reference temperature (default value = 1e-16 Amp)

 ISS:     Saturation current for injection (default value = IS Amp)

 NF:      Forward current emission coefficient (default value = 1)

 NR:      Reverse current emission coefficient (default value = 1)

 GminDC:  Leakage conductance (default value = 1e-19 Mho)


<b>Low Current Beta Degradation Effect Parameters:</b>

 ISC:     Base-collector leakage saturation current at reference temperature (default value = 0 Amp)

 ISE:     Base-emitter leakage saturation current at reference temperature (default value = 0 Amp)

 NC:      Low-current base-collector leakage emission coefficient (default value = 2)

 NE:      Low-current base-emitter leakage emission coefficient (default value = 1.5)


<b>Base Width Modulation Parameters:</b>

 VAF:     Forward early voltage (default value = 9e30 Volt)

 VAR:     Reverse early voltage (default value = 9e30 Volt)


<b>High Current Beta Degradation Effect Parameters:</b>

 IKF:     Corner for forward beta high-current roll-off (default value = 9e30 Amp)

 IKR:     Corner for reverse beta high-current roll-off (default value = 9e30 Amp)


<b>Junction Capacitor Parameters:</b>

 CJC:     Zero-bias base-collector depletion capacitance at reference temperature (default value = 1e-12 F)

 MJC:     Base-collector junction grading coefficient (default value = 0.33)

 VJC:     Base-collector built-in potential at reference temperature (default value = 0.75 Volt)

 CJE:     Zero-bias base-emitter depletion capacitance at reference temperature (default value = 1e-12 F)

 MJE:     Base-emitter junction grading coefficient (default value = 0.33)

 VJE:     Base-emitter built-in potential at reference temperature (default value = 0.75 Volt)

 XCJC:    Fraction of base-collector depletion capacitance connected to internal base node (default value = 1)

 FC:      Depletion capacitance factor for linearization (default value = 0.5)


<b>Transit Time Parameters:</b>

 TF:      Ideal forward transit time (default value = 0 sec)

 TR:      Ideal reverse transit time (default value = 0 sec)


<b>Temperature Compensation and Area Parameters:</b>

 Tnom:    Reference temperature (default value = 300.15 K)

 XTI:     Saturation current temperature exponent (default value = 3)

 XTB:     Forward and reverse beta temperature coefficient (default value = 0)

 EG:      Energy gap for temperature effect on saturation current (default value = 1.11 Volt)

 Area:    Relative area occupied by device (default value = 1)


<b>Numerical Parameters:</b>

 EMin:    Minimum exponent for linearization of junction current (default value = -100)

 EMax:    Maximum exponent for linearization of junction current (default value = 40)
</pre> <p>

<hr> <p>

<b>References:</b> <p>

<ol>
<li>Cellier, F.E. (1991), <i>Continuous System Modeling</i>, Springer-Verlag, New York.
<li>Hild, D.R. and F.E. Cellier (1994), \"Object-oriented electronic circuit modeling using Dymola,\"
    <i>Proc. OOS'94, SCS Object Oriented Simulation Conference</i>, Tempe, AZ, pp.68-75.
<li>Hild, D.R. (1993), <i>Circuit Modeling in Dymola</i>, MS Thesis, Dept. of Electr. &amp; Comp. Engr.,
    University of Arizona, Tucson.
<li> Massobrio, G. and P. Antognetti (1993), <i>Semiconductor Device Modeling with Spice</i>,
     2<sup>nd</sup> edition, McGraw Hill, New York.
<li>Schweisguth, M.C. and F.E. Cellier (1999), \"<a href=\"http://www.inf.ethz.ch/~fcellier/Publikationen/BJT.pdf\">A
    bond graph model of the bipolar junction transistor</a>,\" <i>Proc. SCS Intl. Conf. on Bond Graph Modeling</i>,
    San Francisco, CA, pp.344-349.
<li>Schweisguth, M.C. (1997), <a href=\"http://www.inf.ethz.ch/~fcellier/Publikationen/schweisguth_ms.pdf\"><i>Semiconductor
    Modeling with Bondgraphs</i></a>, MS Thesis, Dept. of Electr. &amp; Comp. Engr., University of Arizona, Tucson.
</ol>
</html>
"));
        BondLib.Interfaces.BondCon Bint "Internal base"
          annotation (extent=[-150,-10; -130,10]);
        BondLib.Interfaces.BondCon Eint "Internal emitter"
          annotation (extent=[-10,-150; 10,-130]);
        BondLib.Interfaces.BondCon H "Heat"
          annotation (extent=[130,-10; 150,10]);
        BondLib.Interfaces.BondCon Cint "Internal collector"
          annotation (extent=[-10,130; 10,150]);
          BondLib.Junctions.J0p3 J0p3_1
          annotation (extent=[-10,-10; 10,10], rotation=-90);
          BondLib.Bonds.Bond B1 annotation (extent=[-10,10; 10,70], rotation=90);
          BondLib.Junctions.J1p3 J1p3_1
          annotation (extent=[-10,70; 10,90], rotation=-90);
          BondLib.Bonds.Bond B2
          annotation (extent=[-10,-70; 10,-10], rotation=270);
          BondLib.Junctions.J1p3 J1p3_2
          annotation (extent=[-10,-90; 10,-70], rotation=-90);
          BondLib.Bonds.Bond B3 annotation (extent=[-30,70; -10,90], rotation=180);
          BondLib.Bonds.Bond B4
          annotation (extent=[-30,-90; -10,-70], rotation=180);
          BondLib.Junctions.J0p3 J0p3_2
          annotation (extent=[-50,70; -30,90], rotation=0);
          BondLib.Bonds.fBond B5
          annotation (extent=[-50,50; -30,70], rotation=-90);
          BondLib.Sensors.De Vbc
          annotation (extent=[-50,24; -30,50], rotation=-90);
          BondLib.Junctions.J0p4 J0p4_1 annotation (extent=[-50,-90; -30,-70]);
          BondLib.Bonds.fBond B6
          annotation (extent=[-50,-70; -30,-50], rotation=90);
          BondLib.Sensors.De Vbe
          annotation (extent=[-50,-50; -30,-24], rotation=90);
          BondLib.Bonds.Bond B7 annotation (extent=[-10,90; 10,110], rotation=90);
          BondLib.Bonds.Bond B8
          annotation (extent=[-10,-110; 10,-90], rotation=270);
          BondLib.Junctions.J0p3 J0p3_3
          annotation (extent=[10,-130; -10,-110], rotation=-90);
          BondLib.Junctions.J1p3 J1p3_5
          annotation (extent=[24,10; 44,-10], rotation=90);
          BondLib.Bonds.Bond B9 annotation (extent=[24,10; 44,70], rotation=270);
          BondLib.Bonds.Bond B10 annotation (extent=[24,-68; 44,-8], rotation=270);
          BondLib.Bonds.eBond B11
          annotation (extent=[44,-10; 64,10], rotation=180);
          BondLib.Bonds.Bond B13
          annotation (extent=[-70,70; -50,90], rotation=180);
          BondLib.Bonds.eBond B14
          annotation (extent=[-130,70; -110,90], rotation=180);
          BondLib.Bonds.Bond B15
          annotation (extent=[-70,-90; -50,-70], rotation=180);
          BondLib.Bonds.eBond B16
          annotation (extent=[-130,-90; -110,-70], rotation=180);
          BondLib.Bonds.eBond B12
          annotation (extent=[-50,-110; -30,-90], rotation=90);
          BondLib.Junctions.J0p3 J0p3_4
          annotation (extent=[10,110; -10,130], rotation=-90);
          BondLib.Spice.Utilities.BJTvars BJTvars1(
          Tnom=Tnom,
          BF=BF,
          BR=BR,
          TF=TF,
          EG=EG,
          XTI=XTI,
          VAF=VAF,
          VAR=VAR,
          IKF=IKF,
          IKR=IKR,
          ISC=ISC,
          ISE=ISE,
          NC=NC,
          NE=NE,
          XTB=XTB,
          Area=Area,
          Level=Level,
          IS=IS,
          ISS=ISS)
          annotation (extent=[60,-120; 120,-60]);
          BondLib.Junctions.J0p6 J0p6_1 annotation (extent=[80,80; 120,120]);
          Modelica.Blocks.Interfaces.RealOutput qb "Base charge"
          annotation (extent=[140,-106; 160,-86]);
        Sources.mSf_int IC0 annotation (extent=[94,-14; 64,14]);
        Bonds.fBond B17 annotation (extent=[94,-10; 114,10],  rotation=180);
        Sources.mSf_int IB0
          annotation (extent=[-56,-140; -24,-110], rotation=90);
        Bonds.fBond B18 annotation (extent=[-70,-150; -50,-130], rotation=0);
        Sensors.De Tdev annotation (extent=[96,36; 120,60], rotation=-90);
        Modelica.Blocks.Sources.Constant qb1(k=1)
          annotation (extent=[-118,44; -106,56]);
        Modelica.Blocks.Sources.Constant C0(k=0)
          annotation (extent=[-62,44; -74,56]);
        Modelica.Blocks.Interfaces.RealOutput ix(redeclare type SignalType =
                Modelica.SIunits.Current)
          "External base-collector capacitance current"
          annotation (extent=[-80,130; -60,150], rotation=90);
        Modelica.Blocks.Math.Gain Gain1(k=-1)
          annotation (extent=[150,-166; 170,-146], rotation=-90);
        CjS Dbc(
          Tnom=Tnom,
          IS=IS,
          GminDC=GminDC,
          N=NR,
          CJ=CJC,
          MJ=MJC,
          VJ=VJC,
          XCJ=XCJC,
          FC=FC,
          Tau=TR,
          XTI=XTI,
          XTB=XTB,
          EG=EG,
          Area=Area,
          Level=Level,
          BE=false,
          EMin=EMin,
          EMax=EMax) annotation (extent=[-70,66; -110,94]);
        CjS Dbe(
          Tnom=Tnom,
          IS=IS,
          GminDC=GminDC,
          FC=FC,
          XTI=XTI,
          XTB=XTB,
          EG=EG,
          Area=Area,
          Level=Level,
          EMin=EMin,
          EMax=EMax,
          N=NF,
          CJ=CJE,
          MJ=MJE,
          VJ=VJE,
          XCJ=1,
          Tau=TF,
          BE=true) annotation (extent=[-70,-94; -110,-66]);
          BondLib.Bonds.fBond B19
          annotation (extent=[98,60; 118,80],  rotation=-90);
        equation
          connect(J1p3_2.BondCon1,B2. BondCon2) annotation (points=[
              -6.12303e-016,-70; -1.83691e-015,-70], style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(B3.BondCon1,J1p3_1. BondCon3) annotation (points=[-10.2,80;
              -10,80],
                   style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(B4.BondCon1,J1p3_2. BondCon3) annotation (points=[-10.2,-80;
              -10,-80], style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(J0p3_2.BondCon2,B3. BondCon2) annotation (points=[-30,80; -30,
              80], style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(B5.eBondCon1,J0p3_2. BondCon3) annotation (points=[-40,70;
              -40,70],
                   style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(Vbc.BondCon1,B5. fBondCon1) annotation (points=[-40,50; -40,
              50],
            style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(J0p4_1.BondCon2,B4. BondCon2) annotation (points=[-30,-80;
              -30,-80],
                    style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(J0p4_1.BondCon4,B6. eBondCon1) annotation (points=[-40,-70;
              -40,-70],
                    style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(B6.fBondCon1,Vbe. BondCon1) annotation (points=[-40,-50; -40,
              -50], style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(J1p3_1.BondCon1,B7. BondCon1) annotation (points=[
              -6.12303e-016,90; -6.12303e-016,90.2; -6.00057e-016,90.2],
                                                         style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(B9.BondCon2,J1p3_5. BondCon1) annotation (points=[34,10; 34,
              10],
            style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(J1p3_5.BondCon2,B10. BondCon1) annotation (points=[34,-10; 34,
              -8.6], style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(B15.BondCon1,J0p4_1. BondCon1) annotation (points=[-50.2,-80;
              -50,-80], style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(B12.eBondCon1,J0p4_1. BondCon3) annotation (points=[-40,-90;
              -40,-90], style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(B7.BondCon2,J0p3_4. BondCon2) annotation (points=[
              6.12303e-016,110; -6.12303e-016,110],
                                       style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(J0p3_4.BondCon3,B9. BondCon1) annotation (points=[10,120; 34,
              120; 34,69.4], style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(B8.BondCon1,J1p3_2. BondCon2) annotation (points=[
              1.80017e-015,-90.2; 1.80017e-015,-98.1; 0,-90; 6.12303e-016,-90],
            style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(J0p3_3.BondCon1,B8. BondCon2) annotation (points=[
              6.12303e-016,-110; -1.83691e-015,-110],
                                         style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(J0p3_3.BondCon3,B10. BondCon2) annotation (points=[10,-120;
              34,-120; 34,-68],
                             style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(J0p3_1.BondCon1,B1. BondCon1) annotation (points=[
              -6.12303e-016,10; -6.12303e-016,10.6; -6.00057e-016,10.6],
                                                         style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(B1.BondCon2,J1p3_1. BondCon2) annotation (points=[
              6.12303e-016,70; 6.12303e-016,70],
                                    style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(Vbc.OutPort1,BJTvars1. vbc) annotation (points=[-40,22.7; -40,
              12; -152,12; -152,-160; 38,-160; 38,-72; 57,-72],       style(
            color=74,
            rgbcolor={0,0,127},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(Vbe.OutPort1,BJTvars1. vbe) annotation (points=[-40,-22.7;
              -40,-12; -160,-12; -160,-168; 46,-168; 46,-96; 57,-96],   style(
            color=74,
            rgbcolor={0,0,127},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(BJTvars1.qb,qb)  annotation (points=[123,-96; 150,-96],
                                   style(
            color=74,
            rgbcolor={0,0,127},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(J0p3_4.BondCon1, Cint)   annotation (points=[6.12303e-016,130;
              0,130; 0,140], style(color=8, rgbcolor={192,192,192}));
          connect(J0p3_1.BondCon3, Bint)   annotation (points=[-10,
              -6.12303e-016; -74,-6.12303e-016; -74,0; -140,0], style(color=8,
              rgbcolor={192,192,192}));
          connect(J0p3_3.BondCon2, Eint)   annotation (points=[-6.12303e-016,
              -130; 0,-130; 0,-140], style(color=8, rgbcolor={192,192,192}));
          connect(J0p6_1.BondCon2, H)      annotation (points=[120,100; 128,100;
              128,0; 140,0], style(color=8, rgbcolor={192,192,192}));
        connect(IB0.BondCon2, B12.fBondCon1) annotation (points=[-40,-110; -40,
              -110], style(color=8, rgbcolor={192,192,192}));
        connect(B17.eBondCon1, J0p6_1.BondCon3) annotation (points=[114,
              -1.22461e-015; 114,20; 92,20; 92,80], style(color=8, rgbcolor={
                192,192,192}));
        connect(B18.fBondCon1, IB0.BondCon1) annotation (points=[-50,-140; -40,
              -140], style(color=8, rgbcolor={192,192,192}));
        connect(B16.eBondCon1, J0p6_1.BondCon5) annotation (points=[-130,-80;
              -168,-80; -168,156; 92,156; 92,120], style(color=8, rgbcolor={192,
                192,192}));
        connect(B14.eBondCon1, J0p6_1.BondCon1) annotation (points=[-130,80;
              -164,80; -164,152; 80,152; 80,100], style(color=8, rgbcolor={192,
                192,192}));
        connect(B18.eBondCon1, J0p6_1.BondCon6) annotation (points=[-70,-140;
              -172,-140; -172,160; 108,160; 108,120], style(color=8, rgbcolor={
                192,192,192}));
        connect(Tdev.OutPort1, BJTvars1.Tdev) annotation (points=[108,34.8; 108,
              28; 164,28; 164,-136; 90,-136; 90,-123],   style(color=74,
              rgbcolor={0,0,127}));
        connect(BJTvars1.ib0, Gain1.u) annotation (points=[123,-72; 160,-72;
              160,-144],        style(color=74, rgbcolor={0,0,127}));
        connect(IC0.BondCon2, B11.fBondCon1) annotation (points=[64,
              1.77636e-015; 64,-1.22461e-015], style(color=8, rgbcolor={192,192,
                192}));
        connect(B17.fBondCon1, IC0.BondCon1) annotation (points=[94,
              1.22461e-015; 94,1.77636e-015], style(color=8, rgbcolor={192,192,
                192}));
        connect(B11.eBondCon1, J1p3_5.BondCon3) annotation (points=[44,
              1.22461e-015; 44,6.12303e-016], style(color=8, rgbcolor={192,192,
                192}));
        connect(Dbc.ix, ix) annotation (points=[-80,94; -80,120; -70,120; -70,
              140], style(color=74, rgbcolor={0,0,127}));
        connect(Dbc.id, BJTvars1.ibc) annotation (points=[-100,94; -100,110;
              -156,110; -156,-164; 42,-164; 42,-84; 57,-84], style(color=74,
              rgbcolor={0,0,127}));
        connect(C0.y, Dbc.C0) annotation (points=[-74.6,50; -80,50; -80,66],
            style(color=74, rgbcolor={0,0,127}));
        connect(qb1.y, Dbc.qb) annotation (points=[-105.4,50; -100,50; -100,66],
            style(color=74, rgbcolor={0,0,127}));
        connect(Dbe.id, BJTvars1.ibe) annotation (points=[-100,-66; -100,-48;
              -164,-48; -164,-172; 50,-172; 50,-108; 57,-108], style(color=74,
              rgbcolor={0,0,127}));
        connect(BJTvars1.C0, Dbe.C0) annotation (points=[123,-108; 128,-108;
              128,-152; -80,-152; -80,-94], style(color=74, rgbcolor={0,0,127}));
        connect(BJTvars1.qb, Dbe.qb) annotation (points=[123,-96; 132,-96; 132,
              -156; -100,-156; -100,-94], style(color=74, rgbcolor={0,0,127}));
        connect(B13.BondCon1, J0p3_2.BondCon1) annotation (points=[-50.2,80; -50,80],
            style(color=8, rgbcolor={192,192,192}));
        connect(Dbc.BondCon1, B13.BondCon2) annotation (points=[-70,80; -70,80],
            style(color=8, rgbcolor={192,192,192}));
        connect(B14.fBondCon1, Dbc.BondCon2) annotation (points=[-110,80; -110,
              80], style(color=8, rgbcolor={192,192,192}));
        connect(Dbe.BondCon1, B15.BondCon2) annotation (points=[-70,-80; -70,
              -80], style(color=8, rgbcolor={192,192,192}));
        connect(B16.fBondCon1, Dbe.BondCon2) annotation (points=[-110,-80; -110,
              -80], style(color=8, rgbcolor={192,192,192}));
        connect(J0p3_1.BondCon2, B2.BondCon1) annotation (points=[6.12303e-016,
              -10; 1.80017e-015,-10; 1.80017e-015,-10.6], style(color=8,
              rgbcolor={192,192,192}));
        connect(B19.eBondCon1, J0p6_1.BondCon4) annotation (points=[108,80; 108,
              80], style(color=8, rgbcolor={192,192,192}));
        connect(Tdev.BondCon1, B19.fBondCon1) annotation (points=[108,60; 108,
              60], style(color=8, rgbcolor={192,192,192}));
          connect(BJTvars1.ic0, IC0.s) annotation (points=[123,-84; 130,-84;
              130,-30; 79,-30; 79,-12.6], style(color=74, rgbcolor={0,0,127}));
          connect(Gain1.y, IB0.s) annotation (points=[160,-167; 160,-176; -16,
              -176; -16,-125; -25.6,-125], style(color=74, rgbcolor={0,0,127}));
        end NPNint;

        model PNPint
        "Spice-style PNP transistor model without parasitic resistors and substrate diode"
          /* DC model parameters */
          constant Real inf=Modelica.Constants.inf;
          parameter Real BF=100
          "Maximum forward current gain at reference temperature";
          parameter Real BR=1
          "Maximum reverse current gain at reference temperature";
          parameter Modelica.SIunits.Current IS=1e-16
          "Saturation current at reference temperature";
          parameter Modelica.SIunits.Current ISS=IS
          "Saturation current used for current injection";
          parameter Real NF=1 "Forward current emission coefficient";
          parameter Real NR=1 "Reverse current emission coefficient";
          parameter Modelica.SIunits.Conductance GminDC(min=0) = 1e-19
          "Leakage conductance";

          /* Low current beta degradation effect parameters */
          parameter Modelica.SIunits.Current ISC(min=0) = 0
          "Base-collector leakage saturation current at reference temperature (ISC = C4*IS)";
          parameter Modelica.SIunits.Current ISE(min=0) = 0
          "Base-emitter leakage saturation current at reference temperature (ISE = C2*IS)";
          parameter Real NC=2
          "Low-current base-collector leakage emission coefficient";
          parameter Real NE=1.5
          "Low-current base-emitter leakage emission coefficient";

            /* Base width modulation parameters */
          parameter Modelica.SIunits.Voltage VAF=inf "Forward early voltage";
          parameter Modelica.SIunits.Voltage VAR=inf "Reverse early voltage";

          /* High current beta degradation effect parameters */
          parameter Modelica.SIunits.Current IKF=inf
          "Corner for forward beta high-current roll-off";
          parameter Modelica.SIunits.Current IKR=inf
          "Corner for reverse beta high-current roll-off";

          /* Junction capacitor parameters */
          parameter Modelica.SIunits.Capacitance CJC(min=Modelica.Constants.small) = 1e-12
          "Zero-bias base-collector depletion capacitance at reference temperature";
          parameter Real MJC(min=0.3,max=0.5) = 0.33
          "Base-collector junction grading coefficient";
          parameter Modelica.SIunits.Voltage VJC=0.8
          "Base-collector built-in potential at reference temperature";
          parameter Modelica.SIunits.Capacitance CJE(min=Modelica.Constants.small) = 1e-12
          "Zero-bias base-emitter depletion capacitance at reference temperature";
          parameter Real MJE(min=0.3,max=0.5) = 0.33
          "Base-emitter junction grading coefficient";
          parameter Modelica.SIunits.Voltage VJE=0.75
          "Base-emitter built-in potential at reference temperature";
          parameter Real XCJC(min=0,max=1) = 1
          "Fraction of base-collector depletion capacitance connected to internal base node";
          parameter Real FC(min=0,max=1) = 0.5
          "Depletion capacitance factor for linearization";

          /* Transit time parameters */
          parameter Modelica.SIunits.Time TF(min=0) = 0
          "Ideal forward transit time";
          parameter Modelica.SIunits.Time TR(min=0) = 0
          "Ideal reverse transit time";

          /* Temperature compensation and area parameters */
          parameter Modelica.SIunits.Temperature Tnom=300.15
          "Reference temperature";
          parameter Real XTI=3 "Saturation current temperature exponent";
          parameter Real XTB=0
          "Forward and reverse beta temperature coefficient";
          parameter Modelica.SIunits.Voltage EG=1.11
          "Energy gap for temperature effect on saturation current";
          parameter Real Area=1 "Relative area occupied by device";
          parameter Integer Level(min=1,max=2) = 2
          "Transistor modeling level (Ebers-Moll = 1; Gummel-Poon = 2)";

          /* Numerical parameters */
          parameter Real EMin=-100
          "if x < EMin, the exp(x) function is linearized";
          parameter Real EMax=40
          "if x > EMax, the exp(x) function is linearized";
        annotation (
          Coordsys(extent=[-140,-140; 140,140]),
          Diagram(
            Text(
              extent=[-152,28; -126,12],
              style(color=10, rgbcolor={135,135,135}),
              string="Bint"),
            Text(
              extent=[10,148; 36,130],
              style(color=10, rgbcolor={135,135,135}),
              string="Cint"),
            Text(
              extent=[10,-132; 36,-150],
              style(color=10, rgbcolor={135,135,135}),
              string="Eint"),
            Text(
              extent=[126,-2; 158,-28],
              style(color=10, rgbcolor={135,135,135}),
              string="Heat"),
            Text(
              extent=[-90,22; -78,10],
              style(color=10, rgbcolor={135,135,135}),
              string="vcb"),
            Text(
              extent=[-98,-12; -86,-24],
              style(color=10, rgbcolor={135,135,135}),
              string="veb"),
            Text(
              extent=[-126,-38; -114,-50],
              style(color=10, rgbcolor={135,135,135}),
              string="ieb"),
            Text(
              extent=[-124,122; -112,110],
              style(color=10, rgbcolor={135,135,135}),
              string="icb"),
            Text(
              extent=[92,-124; 110,-140],
              style(color=10, rgbcolor={135,135,135}),
              string="Tdev"),
            Text(
              extent=[136,40; 154,24],
              style(color=10, rgbcolor={135,135,135}),
              string="Tdev"),
            Text(
              extent=[134,-60; 146,-74],
              style(color=10, rgbcolor={135,135,135}),
              string="IB0"),
            Text(
              extent=[-22,-114; -10,-128],
              style(color=10, rgbcolor={135,135,135}),
              string="IB0"),
            Text(
              extent=[100,-20; 112,-34],
              style(color=10, rgbcolor={135,135,135}),
              string="IC0"),
            Text(
              extent=[134,-104; 144,-116],
              style(color=10, rgbcolor={135,135,135}),
              string="qb"),
            Text(
              extent=[-112,-104; -102,-116],
              style(color=10, rgbcolor={135,135,135}),
              string="qb"),
            Text(
              extent=[100,-142; 112,-154],
              style(color=10, rgbcolor={135,135,135}),
              string="C0"),
            Text(
              extent=[-78,-106; -66,-118],
              style(color=10, rgbcolor={135,135,135}),
              string="C0")),
          Icon(
            Text(
              extent=[-80,120; 80,80],
              style(color=3, rgbcolor={0,0,255}),
              string="%name"),
              Line(points=[0,100; 0,-100],
                                         style(
                  color=9,
                  rgbcolor={149,164,171},
                  thickness=2,
                  fillColor=74,
                  rgbfillColor={0,0,185},
                  fillPattern=1)),
              Line(points=[0,100; 20,80],   style(
                  color=9,
                  rgbcolor={149,164,171},
                  thickness=2,
                  fillColor=74,
                  rgbfillColor={0,0,185},
                  fillPattern=1)),
              Line(points=[-20,80; 0,100], style(
                  color=9,
                  rgbcolor={149,164,171},
                  thickness=2,
                  fillColor=74,
                  rgbfillColor={0,0,185},
                  fillPattern=1)),
            Line(points=[-100,-60; 100,-60], style(color=3, rgbcolor={0,0,255})),
            Line(points=[80,-40; 100,-60], style(color=3, rgbcolor={0,0,255})),
            Line(points=[80,-80; 100,-60], style(color=3, rgbcolor={0,0,255})),
              Text(
                extent=[-112,90; 96,-84],
                style(color=0),
              string="PNP"),
              Text(
                extent=[88,4; 128,-38],
                style(color=0),
              string="int"),
              Text(
                extent=[-156,28; -124,4],
                style(color=9, rgbcolor={149,164,171}),
              string="Bint"),
              Text(
                extent=[10,150; 38,126],
                style(color=9, rgbcolor={149,164,171}),
              string="Cint"),
              Text(
                extent=[122,30; 158,6],
                style(color=9, rgbcolor={149,164,171}),
              string="Heat"),
              Text(
                extent=[8,-126; 38,-152],
                style(color=9, rgbcolor={149,164,171}),
              string="Eint"),
              Text(
                extent=[142,-66; 164,-86],
                style(color=9, rgbcolor={149,164,171}),
              string="qb")),
          Documentation(info="<html>
The PNP element of the Spice bond graph library implements a full-fledges Spice-stye Gummel-Poon model of the PNP bipolar transistor [1-3].  <b>PNPint</b> is a partial model that implements only the internal nodes of the bipolar transistor.  The external parasitic resistances, the external capacitor between base and collector, and the substrate are being added from the outside. <p>

Notice the use of the two internal modulated current sources.  They are connected on the primary side to the <b>Heat</b> port.  As shown in [4,5], these current sources are in fact non-linear resistors, as voltage and current through these sources are always pointing in the same direction.  Thus, they generate heat. <p>

The PNP bipolar transistor is a directed FourPort.  The direction of positive power flow is assumed into the model at the emitter, <b>E</b>, whereas it is assumed out of the model at the base, <b>B</b>, at the collector, <b>C</b>, and at the <b>Heat</b> port. <p>

The causality of the PNP model is free. <p>

<hr>

<pre>
<b>Parameters:</b>

 Level:   Transistor modeling level (default value = 2)
            Level = 1: Ebers-Moll model
            Level = 2: Gummel-Poon model


<b>DC Model Parameters:</b>

 BF:      Maximum forward current gain at reference temperature (default value = 100)

 BR:      Maximum reverse current gain at reference temperature (default value = 1)

 IS:      Saturation current at reference temperature (default value = 1e-16 Amp)

 ISS:     Saturation current for injection (default value = IS Amp)

 NF:      Forward current emission coefficient (default value = 1)

 NR:      Reverse current emission coefficient (default value = 1)

 GminDC:  Leakage conductance (default value = 1e-19 Mho)


<b>Low Current Beta Degradation Effect Parameters:</b>

 ISC:     Base-collector leakage saturation current at reference temperature (default value = 0 Amp)

 ISE:     Base-emitter leakage saturation current at reference temperature (default value = 0 Amp)

 NC:      Low-current base-collector leakage emission coefficient (default value = 2)

 NE:      Low-current base-emitter leakage emission coefficient (default value = 1.5)


<b>Base Width Modulation Parameters:</b>

 VAF:     Forward early voltage (default value = 9e30 Volt)

 VAR:     Reverse early voltage (default value = 9e30 Volt)


<b>High Current Beta Degradation Effect Parameters:</b>

 IKF:     Corner for forward beta high-current roll-off (default value = 9e30 Amp)

 IKR:     Corner for reverse beta high-current roll-off (default value = 9e30 Amp)


<b>Junction Capacitor Parameters:</b>

 CJC:     Zero-bias base-collector depletion capacitance at reference temperature (default value = 1e-12 F)

 MJC:     Base-collector junction grading coefficient (default value = 0.33)

 VJC:     Base-collector built-in potential at reference temperature (default value = 0.75 Volt)

 CJE:     Zero-bias base-emitter depletion capacitance at reference temperature (default value = 1e-12 F)

 MJE:     Base-emitter junction grading coefficient (default value = 0.33)

 VJE:     Base-emitter built-in potential at reference temperature (default value = 0.75 Volt)

 XCJC:    Fraction of base-collector depletion capacitance connected to internal base node (default value = 1)

 FC:      Depletion capacitance factor for linearization (default value = 0.5)


<b>Transit Time Parameters:</b>

 TF:      Ideal forward transit time (default value = 0 sec)

 TR:      Ideal reverse transit time (default value = 0 sec)


<b>Temperature Compensation and Area Parameters:</b>

 Tnom:    Reference temperature (default value = 300.15 K)

 XTI:     Saturation current temperature exponent (default value = 3)

 XTB:     Forward and reverse beta temperature coefficient (default value = 0)

 EG:      Energy gap for temperature effect on saturation current (default value = 1.11 Volt)

 Area:    Relative area occupied by device (default value = 1)


<b>Numerical Parameters:</b>

 EMin:    Minimum exponent for linearization of junction current (default value = -100)

 EMax:    Maximum exponent for linearization of junction current (default value = 40)
</pre> <p>

<hr> <p>

<b>References:</b> <p>

<ol>
<li>Cellier, F.E. (1991), <i>Continuous System Modeling</i>, Springer-Verlag, New York.
<li>Hild, D.R. and F.E. Cellier (1994), \"Object-oriented electronic circuit modeling using Dymola,\"
    <i>Proc. OOS'94, SCS Object Oriented Simulation Conference</i>, Tempe, AZ, pp.68-75.
<li>Hild, D.R. (1993), <i>Circuit Modeling in Dymola</i>, MS Thesis, Dept. of Electr. &amp; Comp. Engr.,
    University of Arizona, Tucson.
<li> Massobrio, G. and P. Antognetti (1993), <i>Semiconductor Device Modeling with Spice</i>,
     2<sup>nd</sup> edition, McGraw Hill, New York.
<li>Schweisguth, M.C. and F.E. Cellier (1999), \"<a href=\"http://www.inf.ethz.ch/~fcellier/Publikationen/BJT.pdf\">A
    bond graph model of the bipolar junction transistor</a>,\" <i>Proc. SCS Intl. Conf. on Bond Graph Modeling</i>,
    San Francisco, CA, pp.344-349.
<li>Schweisguth, M.C. (1997), <a href=\"http://www.inf.ethz.ch/~fcellier/Publikationen/schweisguth_ms.pdf\"><i>Semiconductor
    Modeling with Bondgraphs</i></a>, MS Thesis, Dept. of Electr. &amp; Comp. Engr., University of Arizona, Tucson.
</ol>
</html>
"));
        BondLib.Interfaces.BondCon Bint "Internal base"
          annotation (extent=[-150,-10; -130,10]);
        BondLib.Interfaces.BondCon Eint "Internal emitter"
          annotation (extent=[-10,-150; 10,-130]);
        BondLib.Interfaces.BondCon H "Heat"
          annotation (extent=[130,-10; 150,10]);
        BondLib.Interfaces.BondCon Cint "Internal collector"
          annotation (extent=[-10,130; 10,150]);
          BondLib.Junctions.J0p3 J0p3_1
          annotation (extent=[-10,-10; 10,10], rotation=-90);
          BondLib.Bonds.Bond B1 annotation (extent=[-10,10; 10,70], rotation=
              270);
          BondLib.Junctions.J1p3 J1p3_1
          annotation (extent=[-10,70; 10,90], rotation=-90);
          BondLib.Bonds.Bond B2
          annotation (extent=[-10,-70; 10,-10], rotation=90);
          BondLib.Junctions.J1p3 J1p3_2
          annotation (extent=[-10,-90; 10,-70], rotation=-90);
          BondLib.Bonds.Bond B3 annotation (extent=[-30,70; -10,90], rotation=180);
          BondLib.Bonds.Bond B4
          annotation (extent=[-30,-90; -10,-70], rotation=180);
          BondLib.Junctions.J0p3 J0p3_2
          annotation (extent=[-50,70; -30,90], rotation=0);
          BondLib.Bonds.fBond B5
          annotation (extent=[-50,50; -30,70], rotation=-90);
          BondLib.Sensors.De Vcb
          annotation (extent=[-50,24; -30,50], rotation=-90);
          BondLib.Junctions.J0p4 J0p4_1 annotation (extent=[-50,-90; -30,-70]);
          BondLib.Bonds.fBond B6
          annotation (extent=[-50,-70; -30,-50], rotation=90);
          BondLib.Sensors.De Veb
          annotation (extent=[-50,-50; -30,-24], rotation=90);
          BondLib.Bonds.Bond B7 annotation (extent=[-10,90; 10,110], rotation=
              270);
          BondLib.Bonds.Bond B8
          annotation (extent=[-10,-110; 10,-90], rotation=90);
          BondLib.Junctions.J0p3 J0p3_3
          annotation (extent=[10,-130; -10,-110], rotation=-90);
          BondLib.Junctions.J1p3 J1p3_5
          annotation (extent=[24,10; 44,-10], rotation=90);
          BondLib.Bonds.Bond B9 annotation (extent=[24,10; 44,70], rotation=90);
          BondLib.Bonds.Bond B10 annotation (extent=[24,-70; 44,-10],rotation=
              90);
          BondLib.Bonds.eBond B11
          annotation (extent=[44,-10; 64,10], rotation=180);
          BondLib.Bonds.Bond B13
          annotation (extent=[-70,70; -50,90], rotation=180);
          BondLib.Bonds.eBond B14
          annotation (extent=[-130,70; -110,90], rotation=180);
          BondLib.Bonds.Bond B15
          annotation (extent=[-70,-90; -50,-70], rotation=180);
          BondLib.Bonds.eBond B16
          annotation (extent=[-130,-90; -110,-70], rotation=180);
          BondLib.Bonds.eBond B12
          annotation (extent=[-50,-110; -30,-90], rotation=90);
          BondLib.Junctions.J0p3 J0p3_4
          annotation (extent=[10,110; -10,130], rotation=-90);
          BondLib.Spice.Utilities.BJTvars BJTvars1(
          Tnom=Tnom,
          BF=BF,
          BR=BR,
          TF=TF,
          EG=EG,
          XTI=XTI,
          VAF=VAF,
          VAR=VAR,
          IKF=IKF,
          IKR=IKR,
          ISC=ISC,
          ISE=ISE,
          NC=NC,
          NE=NE,
          XTB=XTB,
          Area=Area,
          IS=IS,
          ISS=ISS,
          Level=Level)
          annotation (extent=[60,-120; 120,-60]);
          BondLib.Junctions.J0p6 J0p6_1 annotation (extent=[80,80; 120,120]);
          Modelica.Blocks.Interfaces.RealOutput qb "Base charge"
          annotation (extent=[140,-106; 160,-86]);
        BondLib.Sources.mSf_int IC0
                            annotation (extent=[94,-14; 64,14]);
        BondLib.Bonds.fBond B17
                        annotation (extent=[94,-10; 114,10],  rotation=180);
        BondLib.Sources.mSf_int IB0
          annotation (extent=[-56,-140; -24,-110], rotation=90);
        BondLib.Bonds.fBond B18
                        annotation (extent=[-70,-150; -50,-130], rotation=0);
        BondLib.Sensors.De Tdev
                        annotation (extent=[96,36; 120,60], rotation=-90);
        Modelica.Blocks.Sources.Constant qb1(k=1)
          annotation (extent=[-118,44; -106,56]);
        Modelica.Blocks.Sources.Constant C0(k=0)
          annotation (extent=[-62,44; -74,56]);
        Modelica.Blocks.Interfaces.RealOutput ix(redeclare type SignalType =
                Modelica.SIunits.Current)
          "External base-collector capacitance current"
          annotation (extent=[-80,130; -60,150], rotation=90);
        Modelica.Blocks.Math.Gain Gain1(k=-1)
          annotation (extent=[150,-166; 170,-146], rotation=-90);
        CjS Dbc(
          Tnom=Tnom,
          IS=IS,
          GminDC=GminDC,
          N=NR,
          CJ=CJC,
          MJ=MJC,
          VJ=VJC,
          XCJ=XCJC,
          FC=FC,
          Tau=TR,
          XTI=XTI,
          XTB=XTB,
          EG=EG,
          Area=Area,
          Level=Level,
          BE=false,
          EMin=EMin,
          EMax=EMax) annotation (extent=[-70,66; -110,94]);
        CjS Dbe(
          Tnom=Tnom,
          IS=IS,
          GminDC=GminDC,
          FC=FC,
          XTI=XTI,
          XTB=XTB,
          EG=EG,
          Area=Area,
          Level=Level,
          EMin=EMin,
          EMax=EMax,
          N=NF,
          CJ=CJE,
          MJ=MJE,
          VJ=VJE,
          XCJ=1,
          Tau=TF,
          BE=true) annotation (extent=[-70,-94; -110,-66]);
          BondLib.Bonds.fBond B19
          annotation (extent=[98,60; 118,80],  rotation=-90);
        equation
          connect(B3.BondCon1,J1p3_1. BondCon3) annotation (points=[-10.2,80;
              -10,80],
                   style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(B4.BondCon1,J1p3_2. BondCon3) annotation (points=[-10.2,-80;
              -10,-80], style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(J0p3_2.BondCon2,B3. BondCon2) annotation (points=[-30,80; -30,
              80], style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(B5.eBondCon1,J0p3_2. BondCon3) annotation (points=[-40,70;
              -40,70],
                   style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(Vcb.BondCon1,B5. fBondCon1) annotation (points=[-40,50; -40,
              50],
            style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(J0p4_1.BondCon2,B4. BondCon2) annotation (points=[-30,-80;
              -30,-80],
                    style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(J0p4_1.BondCon4,B6. eBondCon1) annotation (points=[-40,-70;
              -40,-70],
                    style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(B6.fBondCon1,Veb. BondCon1) annotation (points=[-40,-50; -40,
              -50], style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(B13.BondCon1,J0p3_2. BondCon1) annotation (points=[-50.2,80;
              -50,80], style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(B15.BondCon1,J0p4_1. BondCon1) annotation (points=[-50.2,-80;
              -50,-80], style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(B12.eBondCon1,J0p4_1. BondCon3) annotation (points=[-40,-90;
              -40,-90], style(
            color=8,
            rgbcolor={192,192,192},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(Vcb.OutPort1,BJTvars1. vbc) annotation (points=[-40,22.7; -40,
              12; -152,12; -152,-160; 38,-160; 38,-72; 57,-72],       style(
            color=74,
            rgbcolor={0,0,127},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(Veb.OutPort1,BJTvars1. vbe) annotation (points=[-40,-22.7;
              -40,-12; -160,-12; -160,-168; 46,-168; 46,-96; 57,-96],   style(
            color=74,
            rgbcolor={0,0,127},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
          connect(BJTvars1.qb,qb)  annotation (points=[123,-96; 150,-96],
                                   style(
            color=74,
            rgbcolor={0,0,127},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(J0p3_4.BondCon1, Cint)     annotation (points=[6.12303e-016,130;
              0,130; 0,140], style(color=8, rgbcolor={192,192,192}));
        connect(J0p3_1.BondCon3, Bint)     annotation (points=[-10,
              -6.12303e-016; -74,-6.12303e-016; -74,0; -140,0], style(color=8,
              rgbcolor={192,192,192}));
        connect(J0p3_3.BondCon2, Eint)     annotation (points=[-6.12303e-016,
              -130; 0,-130; 0,-140], style(color=8, rgbcolor={192,192,192}));
        connect(J0p6_1.BondCon2, H)        annotation (points=[120,100; 128,100;
              128,0; 140,0], style(color=8, rgbcolor={192,192,192}));
        connect(IB0.BondCon2,B12. fBondCon1) annotation (points=[-40,-110; -40,
              -110], style(color=8, rgbcolor={192,192,192}));
        connect(B17.eBondCon1,J0p6_1. BondCon3) annotation (points=[114,
              -1.22461e-015; 114,20; 92,20; 92,80], style(color=8, rgbcolor={
                192,192,192}));
        connect(B18.fBondCon1,IB0. BondCon1) annotation (points=[-50,-140; -40,
              -140], style(color=8, rgbcolor={192,192,192}));
        connect(B16.eBondCon1,J0p6_1. BondCon5) annotation (points=[-130,-80;
              -168,-80; -168,156; 92,156; 92,120], style(color=8, rgbcolor={192,
                192,192}));
        connect(B14.eBondCon1,J0p6_1. BondCon1) annotation (points=[-130,80;
              -164,80; -164,152; 80,152; 80,100], style(color=8, rgbcolor={192,
                192,192}));
        connect(B18.eBondCon1,J0p6_1. BondCon6) annotation (points=[-70,-140;
              -172,-140; -172,160; 108,160; 108,120], style(color=8, rgbcolor={
                192,192,192}));
        connect(Tdev.OutPort1,BJTvars1. Tdev) annotation (points=[108,34.8; 108,
              28; 164,28; 164,-136; 90,-136; 90,-123],   style(color=74,
              rgbcolor={0,0,127}));
        connect(J1p3_1.BondCon1, B7.BondCon2) annotation (points=[-6.12303e-016,
              90; -1.83691e-015,90], style(color=8, rgbcolor={192,192,192}));
        connect(J0p3_4.BondCon2, B7.BondCon1) annotation (points=[-6.12303e-016,
              110; 1.80017e-015,110; 1.80017e-015,109.8], style(color=8,
              rgbcolor={192,192,192}));
        connect(J0p3_1.BondCon1, B1.BondCon2) annotation (points=[-6.12303e-016,
              10; -1.83691e-015,10], style(color=8, rgbcolor={192,192,192}));
        connect(J1p3_1.BondCon2, B1.BondCon1) annotation (points=[6.12303e-016,
              70; 1.80017e-015,70; 1.80017e-015,69.4], style(color=8, rgbcolor=
                {192,192,192}));
        connect(J0p3_1.BondCon2, B2.BondCon2) annotation (points=[6.12303e-016,
              -10; 6.12303e-016,-10], style(color=8, rgbcolor={192,192,192}));
        connect(B2.BondCon1, J1p3_2.BondCon1) annotation (points=[-6.00057e-016,
              -69.4; -6.00057e-016,-68.7; -6.12303e-016,-68.7; -6.12303e-016,
              -70], style(color=8, rgbcolor={192,192,192}));
        connect(J1p3_2.BondCon2, B8.BondCon2) annotation (points=[6.12303e-016,
              -90; 6.12303e-016,-90], style(color=8, rgbcolor={192,192,192}));
        connect(B8.BondCon1, J0p3_3.BondCon1) annotation (points=[-6.00057e-016,
              -109.8; -6.00057e-016,-107.9; 6.12303e-016,-107.9; 6.12303e-016,
              -110], style(color=8, rgbcolor={192,192,192}));
        connect(B10.BondCon2, J1p3_5.BondCon2) annotation (points=[34,-10; 34,
              -10], style(color=8, rgbcolor={192,192,192}));
        connect(J0p3_3.BondCon3, B10.BondCon1) annotation (points=[10,-120; 34,
              -120; 34,-69.4], style(color=8, rgbcolor={192,192,192}));
        connect(J1p3_5.BondCon1, B9.BondCon1) annotation (points=[34,10; 34,
              10.6], style(color=8, rgbcolor={192,192,192}));
        connect(J0p3_4.BondCon3, B9.BondCon2) annotation (points=[10,120; 34,
              120; 34,70], style(color=8, rgbcolor={192,192,192}));
        connect(BJTvars1.ib0, Gain1.u) annotation (points=[123,-72; 160,-72;
              160,-144],        style(color=74, rgbcolor={0,0,127}));
        connect(B11.eBondCon1, J1p3_5.BondCon3) annotation (points=[44,
              1.22461e-015; 44,6.12303e-016], style(color=8, rgbcolor={192,192,
                192}));
        connect(IC0.BondCon2, B11.fBondCon1) annotation (points=[64,
              1.77636e-015; 64,-1.22461e-015], style(color=8, rgbcolor={192,192,
                192}));
        connect(IC0.BondCon1, B17.fBondCon1) annotation (points=[94,
              1.77636e-015; 94,1.22461e-015], style(color=8, rgbcolor={192,192,
                192}));
        connect(Dbc.BondCon1, B13.BondCon2) annotation (points=[-70,80; -70,80],
            style(color=8, rgbcolor={192,192,192}));
        connect(B14.fBondCon1, Dbc.BondCon2) annotation (points=[-110,80; -110,
              80], style(color=8, rgbcolor={192,192,192}));
        connect(Dbc.ix, ix) annotation (points=[-80,94; -80,120; -70,120; -70,
              140], style(color=74, rgbcolor={0,0,127}));
        connect(Dbc.id, BJTvars1.ibc) annotation (points=[-100,94; -100,110;
              -156,110; -156,-164; 42,-164; 42,-84; 57,-84], style(color=74,
              rgbcolor={0,0,127}));
        connect(qb1.y, Dbc.qb) annotation (points=[-105.4,50; -100,50; -100,66],
            style(color=74, rgbcolor={0,0,127}));
        connect(C0.y, Dbc.C0) annotation (points=[-74.6,50; -80,50; -80,66],
            style(color=74, rgbcolor={0,0,127}));
        connect(Dbe.BondCon1, B15.BondCon2) annotation (points=[-70,-80; -70,
              -80], style(color=8, rgbcolor={192,192,192}));
        connect(B16.fBondCon1, Dbe.BondCon2) annotation (points=[-110,-80; -110,
              -80], style(color=8, rgbcolor={192,192,192}));
        connect(Dbe.id, BJTvars1.ibe) annotation (points=[-100,-66; -100,-48;
              -164,-48; -164,-172; 50,-172; 50,-108; 57,-108], style(color=74,
              rgbcolor={0,0,127}));
        connect(BJTvars1.C0, Dbe.C0) annotation (points=[123,-108; 128,-108;
              128,-152; -80,-152; -80,-94], style(color=74, rgbcolor={0,0,127}));
        connect(BJTvars1.qb, Dbe.qb) annotation (points=[123,-96; 132,-96; 132,
              -156; -100,-156; -100,-94], style(color=74, rgbcolor={0,0,127}));
        connect(J0p6_1.BondCon4, B19.eBondCon1) annotation (points=[108,80; 108,
              80], style(color=8, rgbcolor={192,192,192}));
        connect(B19.fBondCon1, Tdev.BondCon1) annotation (points=[108,60; 108,
              60], style(color=8, rgbcolor={192,192,192}));
          connect(BJTvars1.ic0, IC0.s) annotation (points=[123,-84; 130,-84;
              130,-30; 79,-30; 79,-12.6], style(color=74, rgbcolor={0,0,127}));
          connect(Gain1.y, IB0.s) annotation (points=[160,-167; 160,-176; -16,
              -176; -16,-125; -25.6,-125], style(color=74, rgbcolor={0,0,127}));
        end PNPint;

        function MOSeq1
        "Computation of injected drain-source current for Levels = 0,1"
          input Modelica.SIunits.Voltage vds "Drain-source voltage";
          input Modelica.SIunits.Voltage vgs "Gate-source voltage";
          input Modelica.SIunits.Voltage vbs "Bulk-source voltage";
          input Modelica.SIunits.Temperature Tdev "Device temperature";
          input Real sig "sign(vds)";
          input Integer Type "Type=1 for NMOS; Type=-1 for PMOS";
          input Modelica.SIunits.Temperature Tnom "Reference temperature";
          input Modelica.SIunits.Voltage VFB
          "Flat band voltage at reference temperature";
          input Modelica.SIunits.Transconductance KP
          "Transconductance parameter at reference temperature";
          input Real GAMMA(unit="V^0.5") "Body-effect parameter";
          input Modelica.SIunits.Voltage PHI
          "Surface inversion potential at reference temperature";
          input Modelica.SIunits.InversePotential LAMBDA
          "Channel length modulation";
          input Modelica.SIunits.Length L "Effective hannel length";
          input Modelica.SIunits.Length W "Effective channel width";
          input Modelica.SIunits.Voltage EG
          "Energy gap for temperature effect on saturation current at 0 K";
          output Modelica.SIunits.Current ids0 "Injected drain-source current";
          output Modelica.SIunits.Voltage Vfb
          "Flat band voltage at device temperature";
          output Modelica.SIunits.Voltage Vth "Threshold voltage";
          output Modelica.SIunits.Voltage vdsat "Saturation voltage";

      protected
          constant Modelica.SIunits.Entropy k=Modelica.Constants.k
          "Boltzmann's constant";
          constant Modelica.SIunits.ElectricCharge q=1.6021892e-19
          "Electron charge";
          constant Real GapC1(unit="V/K")=7.02E-4
          "First bandgap correction factor Silicon";
          constant Modelica.SIunits.Temperature GapC2=1108.0
          "Second bandgap correction factor Silicon";
          Real RTemp "Temperature quotient";
          Real facTemp "Temperature coefficient";
          Modelica.SIunits.Voltage Vt "Thermal voltage";
          Modelica.SIunits.Voltage EGref "Energy gap at reference temperature";
          Modelica.SIunits.Voltage EGval "Energy gap at device temperature";
          Modelica.SIunits.Voltage PHIval
          "Surface inversion potential at device temperature";
          Modelica.SIunits.Transconductance KPval
          "Transconductance parameter at device temperature";
          Modelica.SIunits.Transconductance BETA
          "Transconductance at device temperature";
          Modelica.SIunits.Transconductance Beta
          "Adjusted transconductance at device temperature";
          Real cdlin(unit="V2") "Linear region current factor";
          Real cdsat(unit="V2") "Saturation region current factor";

          /* Auxiliary variables */
          Real PHIroot(unit="V^0.5");
          Real Sarg(unit="V^0.5");
          Modelica.SIunits.Voltage Vgst "vgs - vth";
        annotation (Diagram(
            Polygon(points=[-100,100; 20,100; 100,0; 20,-100; -100,-100; -100,
                  100], style(
                color=58,
                rgbcolor={0,111,0},
                fillColor=7,
                rgbfillColor={255,255,255})),
              Text(
                extent=[-56,-18; 36,-46],
              style(color=42, rgbcolor={220,0,0}),
              string="Levels = 0,1"),
              Text(
                extent=[-70,8; 50,-22],
                string="of a MOS transistor",
              style(color=58, rgbcolor={0,111,0})),
              Text(
                extent=[-92,36; 72,-4],
                string="between drain and source",
              style(color=58, rgbcolor={0,111,0})),
              Text(
                extent=[-62,50; 38,22],
                string="Injected current",
              style(color=58, rgbcolor={0,111,0}))), Icon(
            Polygon(points=[-100,100; 20,100; 100,0; 20,-100; -100,-100; -100,
                  100], style(
                color=58,
                rgbcolor={0,111,0},
                fillColor=7,
                rgbfillColor={255,255,255})),
              Text(
                extent=[-70,8; 50,-22],
                string="of a MOS transistor",
              style(color=58, rgbcolor={0,111,0})),
              Text(
                extent=[-92,36; 72,-4],
                string="between drain and source",
              style(color=58, rgbcolor={0,111,0})),
              Text(
                extent=[-62,50; 38,22],
                string="Injected current",
              style(color=58, rgbcolor={0,111,0})),
              Text(
                extent=[-56,-18; 36,-46],
              style(color=42, rgbcolor={220,0,0}),
              string="Levels = 0,1")),
          Documentation(info="<html>
This function implements a Spice-style <b>Level-1</b> model of the current flowing between drain and source of a MOSFET in accrodance with the theory of Shichman and Hodges [1].  It is a simple model that computes the current in its linear and saturation regions only.  No channel modulation effects are considered, and the current is set equal to zero in the subthreshold region of the device. <p>

The current injection model is identical for Modelica-Spice <b>Level-0</b> and <b>Level-1</b>. <p>

The model comments the equations by providing the corresponding equation numbers from [2]. <p>

<hr> <p>

<pre>
Input variables:

 vds:     Drain-source voltage (Volt)

 vgs:     Gate-source voltage (Volt)

 vbs:     Bulk-source voltage (Volt)

 Tdev:    Device temperature (K)

 sig:     Indicator of operating mode
            sig = +1:  direct mode
            sig = -1:  reverse mode (source and drain are flipped)


Input parameters:

 Type:    Type of MOSFET (default value = 1)
            Type = +1:  NMOS
            Type = -1:  PMOS

 Tnom:    Reference temperature (default value = 300.15 K)

 VFB:     Flat band voltage at reference temperature (default value = 0 Volt)

 KP:      Transconductance parameter at reference temperature (default value = 0 Amp/Volt<sup>2</sup>)

 GAMMA:   Body-effect parameter (default value = 0 Volt<sup>0.5</sup>)

 PHI:     Surface inversion potential at reference temperature (default value = 0 Volt)

 LAMBDA:  Channel length modulation  (default value = 0 1/Volt)

 L:       Channel length (default value = 1e-4 m)

 W:       Channel width (default value = 1e-4 m)

 EG:      Energy gap for temperature effect on saturation current at 0 K (default value = 1.11 Volt)


Output variables:

 isd0:    Drain-source current (Amp)

 Vfb:     Flat band voltage at device temperature (Volt)

 Vth:     Threshold voltage (Volt)

 vdsat:   Saturation voltage (Volt)
</pre> <p>

<hr> <p>

<b>References:</b> <p>

<ol>
<li>Shichman, H. and D.A. Hodges (1968), \"Modeling and Simulation of Insulated-Gate Field-Effector Transistor Switching Circuits,\" <i>IEEE J. Solid-State Circuits</i>, <b>3</b>(3), pp. 285-289.
<li> Massobrio, G. and P. Antognetti (1993), <i>Semiconductor Device Modeling with Spice</i>,
     2<sup>nd</sup> edition, McGraw Hill, New York.
</ol>
</html>
"));
        algorithm
          /* Parameter computations and temperature variations (implements TMPUPD) */
          RTemp := Tdev/Tnom;
          facTemp := RTemp*sqrt(RTemp);
          Vt := k*Tdev/q;
          EGref := EG - GapC1*Tnom*Tnom/(Tnom + GapC2);
          EGval := EG - GapC1*Tdev*Tdev/(Tdev + GapC2);
          PHIval := RTemp*PHI + EGval - RTemp*EGref - 3*Vt*ln(RTemp);
          Vfb := VFB + Type*(PHIval - PHI)/2 + (EGref - EGval)/2;
          KPval := KP/facTemp;
          BETA := KPval*W/L;

          /* Compute drain-source injection current */
          PHIroot := sqrt(PHIval);
          Sarg := if vbs <= 0 then sqrt(PHIval - vbs) else max({0,PHIroot - vbs/(2*PHIroot)});
          Vth := Vfb + Type*(PHIval + GAMMA*Sarg);  /* (4-13) p.172 */
          Vgst := vgs - Type*Vth;
          vdsat := if Vgst > 0 then Vgst else 0;
          Beta := BETA*(1 + LAMBDA*vds);
          cdlin := vds*(Vgst - vds/2);  /* (4-29) p.174 */
          cdsat := Vgst*Vgst/2;  /* (4-31) p.175 */
          ids0 := sig*Beta*(if Vgst > 0 and vds <= vdsat then cdlin else
                            if Vgst > 0 and vds > vdsat then cdsat else 0);
        end MOSeq1;

        function MOSeq2
        "Computation of injected drain-source current for Level=2"
          input Modelica.SIunits.Voltage vds "Drain-source voltage";
          input Modelica.SIunits.Voltage vgs "Gate-source voltage";
          input Modelica.SIunits.Voltage vbs "Bulk-source voltage";
          input Modelica.SIunits.Temperature Tdev "Device temperature";
          input Real sig "sign(vds)";
          input Integer Level=2 "Level of MOS model";
          input Integer Type "Type=1 for NMOS; Type=-1 for PMOS";
          input Modelica.SIunits.Temperature Tnom "Reference temperature";
          input Modelica.SIunits.Voltage VFB
          "Flat band voltage at reference temperature";
          input Modelica.SIunits.Transconductance KP
          "Transconductance parameter at reference temperature";
          input Real GAMMA(unit="V^0.5") "Body-effect parameter";
          input Modelica.SIunits.Voltage PHI
          "Surface inversion potential at reference temperature";
          input Modelica.SIunits.InversePotential LAMBDA
          "Channel length modulation";
          input Real COX(unit="F/m2") "Specific capacitance of SiO2";
          input Real NFS(unit="1/m2") "Surface fast state density";
          input Real NEFF "Total channel charge coefficient";
          input Modelica.SIunits.Length XJ "Metallurgical junction depth";
          input Modelica.SIunits.Length L "Effective hannel length";
          input Modelica.SIunits.Length W "Effective channel width";
          input Real XD(unit="m/V^0.5") "Depletion factor";
          input Real U0(unit="m2/(V.s)")
          "Surface mobility at reference temperature";
          input Real UEXP "Exponential coefficient for mobility";
          input Real UTRA "Transverse field coefficient";
          input Modelica.SIunits.Velocity VMAX
          "Maximum drift velocity of carriers";
          input Modelica.SIunits.ElectricFieldStrength ECRIT
          "Critical electric field for pinch-off";
          input Real DELTA "Width effect on threshold voltage";
          input Modelica.SIunits.Voltage VBP "Gate to channel critical voltage";
          input Modelica.SIunits.Voltage PB
          "Built-in potential at reference temperature";
          input Modelica.SIunits.Voltage EG
          "Energy gap for temperature effect on saturation current at 0 K";
          input Real EMin "if x < EMin, the exp(x) function is linearized";
          input Real EMax "if x > EMax, the exp(x) function is linearized";
          output Modelica.SIunits.Current ids0 "Injected drain-source current";
          output Modelica.SIunits.Voltage Vfb
          "Effective flat band voltage at device temperature";
          output Modelica.SIunits.Voltage Vth "Threshold voltage";
          output Modelica.SIunits.Voltage vdsat "Saturation voltage";
          output Modelica.SIunits.Voltage vdsat1 "Adjusted saturation voltage";
          output Modelica.SIunits.Voltage vpof "Pinch-off voltage";

      protected
          constant Real pi=Modelica.Constants.pi;
          constant Modelica.SIunits.Entropy k=Modelica.Constants.k
          "Boltzmann's constant";
          constant Modelica.SIunits.Permittivity e0=Modelica.Constants.epsilon_0
          "Permittivity of free space";
          constant Modelica.SIunits.ElectricCharge q=1.6021892e-19
          "Electron charge";
          constant Real ks=11.8 "Dielectric constant of Si";
          constant Real GapC1(unit="V/K")=7.02E-4
          "First bandgap correction factor Silicon";
          constant Modelica.SIunits.Temperature GapC2=1108.0
          "Second bandgap correction factor Silicon";
          constant Integer Scat1[4] = {1,-1,1,-1} "First scattering vector";
          constant Integer Scat2[4] = {1,1,-1,-1} "Second scattering vector";
          parameter Modelica.SIunits.Permittivity es = ks*e0
          "Permittivity of silicon";
          parameter Real Fn = if Level==2 then es*DELTA*pi/(2*COX*W) else 0;
          parameter Real Fn2 = 1 + Fn;
          parameter Real ExMin = exp(EMin);
          parameter Real ExMax = exp(EMax);
          Real RTemp "Temperature quotient";
          Real facTemp "Temperature coefficient";
          Modelica.SIunits.Voltage EGref "Energy gap at reference temperature";
          Modelica.SIunits.Voltage EGval "Energy gap at device temperature";
          Modelica.SIunits.Voltage Vt "Thermal voltage";
          Modelica.SIunits.Voltage PHIval
          "Surface inversion potential at device temperature";
          Modelica.SIunits.Voltage Von "Adjusted threshold voltage";
          Modelica.SIunits.Voltage Vgst "vgs - vth";
          Modelica.SIunits.Voltage VGSval "Effective vgs";
          Modelica.SIunits.Voltage VDSAT "Initial saturation voltage";
          Modelica.SIunits.Voltage VDSATval "Adjusted saturation voltage";
          Modelica.SIunits.Voltage vdson "Subthreshold voltage";
          Modelica.SIunits.Current ids "Injected drain-source current";
          Real cdlin(unit="V2") "Linear region current factor";
          Real cdsat(unit="V2") "Saturation region current factor";
          Real cdson(unit="V2") "Subthreshold region current coefficient";
          Real cdsub(unit="V2") "Subthreshold region current factor";
          Real GAMMAval(unit="V^0.5") "Adjusted body effect parameter";
          Real GAMMAmod(unit="V^0.5") "Modified body effect parameter";
          Real DGAMMA(unit="1/V^0.5") "der(GAMMAval,vbs)";
          Real et "Subthreshold voltage coefficient";
          Modelica.SIunits.Transconductance KPval
          "Transconductance parameter at device temperature";
          Modelica.SIunits.Transconductance BETA;
          Real Ueff(unit="m2/(V.s)")
          "Effective surface mobility at reference temperature";
          Modelica.SIunits.Length WS "Source depletion region width";
          Modelica.SIunits.Length WD "Drain depletion region width";
          Modelica.SIunits.Length WB "Punch-through width";
          Modelica.SIunits.Length Leff "Modulated channel length";
          Modelica.SIunits.Length LEFFval "Effective channel length";
          Modelica.SIunits.Length LD "Channel length modulation";
          Modelica.SIunits.InversePotential LAMBDAval
          "Effective channel length modulation";
          Real CB "Specific capacitance of depleted region";
          Real N "Intrinsic carrier concentration at device temperature";

          /* Auxiliary variables */
          Modelica.SIunits.Voltage Vgamma;
          Modelica.SIunits.Voltage Vden;
          Modelica.SIunits.Voltage Vdiff;
          Modelica.SIunits.Voltage Vdiff2;
          Modelica.SIunits.Voltage Vdiff3;
          Modelica.SIunits.Voltage Vdiff4;
          Modelica.SIunits.Voltage Vdiff5;
          Modelica.SIunits.Voltage V1;
          Modelica.SIunits.Voltage V2;
          Modelica.SIunits.Voltage Vd;
          Modelica.SIunits.Voltage X1b;
          Modelica.SIunits.Voltage X2a;
          Modelica.SIunits.Voltage X4;
          Modelica.SIunits.Voltage X5b;
          Modelica.SIunits.Voltage X6b[4];
          Modelica.SIunits.Voltage X7;
          Modelica.SIunits.Voltage VQChan;
          Real VSq(unit="V2");
          Real X1d(unit="V2");
          Real X2b(unit="V2");
          Real X9[8](unit="V2");
          Real X2c(unit="V3");
          Real X3(unit="V3");
          Real Psq(unit="V3");
          Real P(unit="V6");
          Real Pabs(unit="V6");
          Real PHIroot(unit="V^0.5");
          Real Sarg(unit="V^0.5");
          Real Barg(unit="V^0.5");
          Real BARGsat(unit="V^0.5");
          Real gdsat(unit="V^0.5");
          Real LV(unit="V^0.5");
          Real LS(unit="V^0.5");
          Real X1a(unit="V^0.5");
          Real X5a(unit="V^0.5");
          Real X6a[4](unit="V^0.5");
          Real X8[8](unit="V^0.5");
          Real PHI3(unit="V^1.5");
          Real BS3(unit="V^1.5");
          Real BS3sat(unit="V^1.5");
          Real BS3eff(unit="V^1.5");
          Real X1c(unit="V^1.5");
          Real DSarg(unit="1/V^0.5") "der(Sarg,vbs)";
          Real DBarg(unit="1/V^0.5") "der(Barg,vbs)";
          Real Lfact(unit="1/V^1.5");
          Modelica.SIunits.Length DeltaL;
          Modelica.SIunits.Transconductance BETAval;
          Real DV(unit="m/V^0.5");
          Real WSfac;
          Real WDfac;
          Real Ufact;
          Real Lfact2;
          Real LFACTval;
          Integer icnt;
          Integer jcnt;
        annotation (Diagram(
            Polygon(points=[-100,100; 20,100; 100,0; 20,-100; -100,-100; -100,
                  100], style(
                color=58,
                rgbcolor={0,111,0},
                fillColor=7,
                rgbfillColor={255,255,255})),
              Text(
                extent=[-52,-20; 20,-40],
              style(color=42, rgbcolor={220,0,0}),
              string="Level = 2"),
              Text(
                extent=[-70,8; 50,-22],
                string="of a MOS transistor",
              style(color=58, rgbcolor={0,111,0})),
              Text(
                extent=[-92,36; 72,-4],
                string="between drain and source",
              style(color=58, rgbcolor={0,111,0})),
              Text(
                extent=[-62,50; 38,22],
                string="Injected current",
              style(color=58, rgbcolor={0,111,0}))), Icon(
            Polygon(points=[-100,100; 20,100; 100,0; 20,-100; -100,-100; -100,
                  100], style(
                color=58,
                rgbcolor={0,111,0},
                fillColor=7,
                rgbfillColor={255,255,255})),
              Text(
                extent=[-52,-20; 20,-40],
              style(color=42, rgbcolor={220,0,0}),
              string="Level = 2"),
              Text(
                extent=[-70,8; 50,-22],
                string="of a MOS transistor",
              style(color=58, rgbcolor={0,111,0})),
              Text(
                extent=[-92,36; 72,-4],
                string="between drain and source",
              style(color=58, rgbcolor={0,111,0})),
              Text(
                extent=[-62,50; 38,22],
                string="Injected current",
              style(color=58, rgbcolor={0,111,0}))),
          Documentation(info="<html>
This function implements a Spice-style <b>Level-2</b> model of the current flowing between drain and source of a MOSFET.  It is one of the most complex models currently available that computes the current in its linear, saturation, and subthreshold regions.  Various channel modulation effects are also considered.  The saturation voltage is computed either in accordance with the Grove-Frohman model [1,2] or using Baum's theory of scattering velocity saturation [3]. <p>

The model simulates very slowly, and is probably not useful except for simulating across a single switching event. <p>

The model comments the equations by providing the corresponding equation numbers from [4]. <p>

<hr> <p>

<pre>
Input variables:

 vds:     Drain-source voltage (Volt)

 vgs:     Gate-source voltage (Volt)

 vbs:     Bulk-source voltage (Volt)

 Tdev:    Device temperature (K)

 sig:     Indicator of operating mode
            sig = +1:  direct mode
            sig = -1:  reverse mode (source and drain are flipped)


Input parameters:

 Type:    Type of MOSFET (default value = 1)
            Type = +1:  NMOS
            Type = -1:  PMOS

 Tnom:    Reference temperature (default value = 300.15 K)

 VFB:     Flat band voltage at reference temperature (default value = 0 Volt)

 KP:      Transconductance parameter at reference temperature (default value = 0 Amp/Volt<sup>2</sup>)

 GAMMA:   Body-effect parameter (default value = 0 Volt<sup>0.5</sup>)

 PHI:     Surface inversion potential at reference temperature (default value = 0 Volt)

 LAMBDA:  Channel length modulation  (default value = 0 1/Volt)

 COX:     Specific capacitance of SiO2 (default value = 0 F/m<sup>2</sup>)

 NFS:     Surface fast state density (default value = 0 1/m<sup>2</sup>)

 NEFF:    Total channel charge coefficient (default value = 1)

 XJ:      Metallurgical junction depth (default value = 0 m)

 L:       Channel length (default value = 1e-4 m)

 W:       Channel width (default value = 1e-4 m)

 XD:      Depletion factor (default value = 0 m/Volt<sup>0.5</sup>)

 U0:      Surface mobility at reference temperature (default value = 0 m<sup>2</sup>/(V*s))

 UEXP:    Exponential coefficient for mobility (default value = 0)

 UTRA:    Transverse field coefficient (default value = 0)

 VMAX:    Maximum drift velocity of carriers (default value = 0 m/s)
            <font color=blue><b>VMAX = 0: Frohman-Grove model of drain current computation</b></font>
            <font color=blue><b>VMAX &gt; 0: Baum-Beneking model of drain current computation</b></font>

 ECRIT:   Critical electric field for pinch-off (default value = 0 V/m)

 DELTA:   Width effect on threshold voltage (default value = 0)

 VBP:     Gate to channel critical voltage (default value = 0 Volt)

 PB:      Built-in potential at reference temperature (default value = 0.8 Volt)

 EG:      Energy gap for temperature effect on saturation current at 0 K (default value = 1.11 Volt)

 EMin:    if x &lt; EMin, the exp(x) function is linearized (default value = -100) <font color=red>Levels 1-3</font>

 EMax:    if x &gt; EMax, the exp(x) function is linearized (default value = 40) <font color=red>Levels 1-3</font>


Output variables:

 isd0:    Drain-source current (Amp)

 Vfb:     Flat band voltage at device temperature (Volt)

 Vth:     Threshold voltage (Volt)

 vdsat:   Saturation voltage (Volt)

 vdsat1:  Adjusted saturation voltage (Volt)

 vpof:    Pinch-off voltage (Volt)
</pre> <p>

<hr> <p>

<b>References:</b> <p>

<ol>
<li>Frohman-Bentchkowsky, D. (1968), \"On the Effect of Mobility Variation on MOS Device Characteristics,\" <i>Proc. IEEE</i>, <b>56</b>, pp.217-218.
<li>Frohman-Bentchkowsky, D. and A.S. Grove (1969), \"Conductance of MOS Transistors in Saturation,\" <i>IEEE Trans. Electron Devices</i>, <b>ED-16</b>, pp. 108-113.
<li>Baum, G. and H. Beneking (1970), \"Drift Velocity Saturation in MOS Transistors,\" <i>IEEE Trans. Electron Devices</i>, <b>ED-17</b>, pp. 481-482.
<li> Massobrio, G. and P. Antognetti (1993), <i>Semiconductor Device Modeling with Spice</i>,
     2<sup>nd</sup> edition, McGraw Hill, New York.
</ol>
</html>
"));
        algorithm
          /* Parameter computations and temperature variations (implements TMPUPD ) */
          RTemp := Tdev/Tnom;
          facTemp := RTemp*sqrt(RTemp);
          Vt := k*Tdev/q;
          EGref := EG - GapC1*Tnom*Tnom/(Tnom + GapC2);
          EGval := EG - GapC1*Tdev*Tdev/(Tdev + GapC2);
          PHIval := RTemp*PHI + EGval - RTemp*EGref - 3*Vt*ln(RTemp);
          Vfb := VFB + Type*(PHIval - PHI)/2 + (EGref - EGval)/2;
          KPval := KP/facTemp;
          BETA := KPval*W/L;

          /* Compute Sarg and Barg and their derivatives */
          PHIroot := sqrt(PHIval);
          PHI3 := PHIval*PHIroot;
          Sarg := if vbs <= 0 then sqrt(PHIval - vbs) else PHIroot/(1 + vbs/(2*PHIval));
          DSarg := if vbs <= 0 then -0.5/Sarg else -0.5*Sarg*Sarg/PHI3;
          Barg := if vbs < vds then sqrt(PHIval + vds - vbs) else PHIroot/(1 + (vbs - vds)/(2*PHIval));
          DBarg := if vbs < vds then -0.5/Barg else -0.5*Barg*Barg/PHI3;
          BS3 := Barg^3 - Sarg^3;

          /* Compute threshold voltage (narrow channel effect) */
          WS := XD*Sarg;  /* (4-57) p.189 */
          WD := XD*Barg;  /* (4-58) p.189 */
          WSfac := if XJ > 0 and Level==2 then sqrt(1 + 2*WS/XJ) else 1;
          WDfac := if XJ > 0 and Level==2 then sqrt(1 + 2*WD/XJ) else 1;
          GAMMAval := GAMMA*(if XJ > 0 then 1 - XJ*(WSfac + WDfac - 2)/(2*L) else 1);  /* (4-56) p.189 */
          DGAMMA := GAMMA*(if XJ > 0 then -XD*(DSarg/WSfac + DBarg/WDfac)/(2*L) else 0);
          CB := Fn - GAMMAval*DSarg - DGAMMA*Sarg;  /* (4-48) p.183 */
          N := if NFS > 0 then 1 + q*NFS/COX + CB else 1;  /* (4-47) p.183 */
          Vth := Vfb + Type*(PHIval + (GAMMA - GAMMAval)*sqrt(PHIval) + GAMMAval*Sarg + Fn*Sarg*Sarg/2);  /* (4-62) p.195 */
          Von := if NFS > 0 then Type*Vth + N*Vt else Type*Vth;  /* (4-46) p.183 */
          Vgst := vgs - Von;

          /* Compute effective carrier mobility */
          Vden := vgs - Type*Vth;
          Ufact := if Vden > VBP then exp(UEXP*ln(VBP/Vden)) else 1;  /* (4-45) p.182 */
          Ueff := Ufact*U0;

          /* Determine saturation voltage */
          /* Use Grove-Frohman equation */
          GAMMAmod := GAMMAval/Fn2;
          Vgamma := GAMMAmod*GAMMAmod;
          if GAMMAmod > 0 then
            VGSval := if NFS > 0 then max({vgs,Von}) else vgs;
            Vdiff := (VGSval - Type*Vfb - PHIval)/Fn2;
            Vdiff2 := Vdiff + PHIval - vbs;
            if Vdiff2 > 0 then
              Vdiff3 := VGSval - Type*Vfb - PHIval + 0.5*GAMMAval^2*(1 - sqrt(1 + 4/GAMMAval^2*(VGSval - Type*Vfb - vbs)));  /* (4-42) p.181 */
              VDSAT := if Vdiff3 > 0 then Vdiff3 else 0;
              VSq := Vgamma*(vds + PHIval - vbs)/Fn2^2;
              vpof := if VSq < 0 then Type*Vth else Type*Vfb + Fn2*(vds + sqrt(VSq));
              Vdiff3 := (vpof - Type*Vfb - PHIval)/Fn2;
              Vdiff4 := (vpof - Type*Vfb - PHIval)/Fn2 + PHIval - vbs;
              Vdiff5 := Vdiff3 + Vgamma*(1 - sqrt(1 + 2*Vdiff4/Vgamma));
              vdsat1 := if Vdiff4 > 0 and Vdiff5 > 0 then Vdiff5 else 0;
            else
              VDSAT := 0;
              vpof := Type*Vth;
              vdsat1 := 0;
            end if;
          else
            VGSval := vgs;
            VDSAT := if Vdiff > 0 then Vdiff else 0;
            vpof := if Fn2*vds + Type*Vfb > 0 then Fn2*vds + Type*Vfb else 0;
            vdsat1 := if (vpof - Type*Vfb)/Fn2 > 0 then (vpof - Type*Vfb)/Fn2 else 0;
          end if;
          VDSATval := if ECRIT > 0 then VDSAT + ECRIT*L - sqrt(VDSAT*VDSAT + (ECRIT*L)^2) else VDSAT;

          /* Use Baum's theory of scattering velocity saturation */
          if VMAX > 0 then
            V1 := (VGSval - Type*Vfb)/Fn2 + PHIval - vbs;
            V2 := PHIval - vbs;
            Vd := VMAX*L/Ueff;
            X1a := 4*GAMMAmod/3;
            X1b := -2*(V1 + Vd);
            X1c := -2*GAMMAmod*Vd;
            X1d := 2*V1*(V2 + Vd) - V2*V2 - X1a*Sarg^3;
            X2a := -X1b;
            X2b := X1a*X1c - 4*X1d;
            X2c := -X1d*(X1a*X1a - 4*X1b) - X1c*X1c;
            X3 := 2*X2a^3/27 - X2a*X2b/3 + X2c;
            P := X3*X3/4 + (X2b - X2a*X2a/3)^3/27;
            Pabs := if P > 0 then P else -P;
            Psq := sqrt(Pabs);
            X4 := (if P < 0 then 2*exp(ln(sqrt(X3*X3/4 + Pabs))/3)*cos(Modelica.Math.atan(-2*Psq/X3)/3) else
                                 exp(ln(abs(-X3/2 + Psq))/3) + exp(ln(abs(-X3/2 - Psq))/3)) - X2a/3;
            X5a := sqrt(X1a*X1a/4 + X4 - X1b);
            X5b := sqrt(X4*X4/4 - X1d);
            icnt := 0;
            for i in 1:4 loop
              X6a[i] := X1a/2 + Scat1[i]*X5a;
              X6b[i] := X4/2 + Scat2[i]*X5b;
              X7 := X6a[i]*X6a[i]/4 - X6b[i];
              if X7 >= 0 then
                icnt := icnt + 1;
                X8[icnt] := -X6a[i]/2 + sqrt(X7);
                icnt := icnt + 1;
                X8[icnt] := -X6a[i]/2 - sqrt(X7);
              end if;
            end for;
            jcnt := 0;
            for i in 1:icnt loop
              if X8[i] > 0 then
                X9[i] := X1d + X8[i]*(X1c + X8[i]*(X1b + X8[i]*(X1a + X8[i])));
                gdsat := 0;
                if abs(X9[i]) < 1e-6 then
                  jcnt := jcnt + 1;
                  if jcnt < 2 or X8[i] <= gdsat then
                    gdsat := X8[i];
                  end if;
                end if;
              end if;
            end for;
            vdsat := if jcnt > 0 then gdsat*gdsat + vbs - PHIval else VDSATval;
          else
            vdsat := VDSATval;
          end if;

          /* Compute channel length modulation */
          if vds > 0 then
            BARGsat := if vbs < vdsat then sqrt(PHIval + vdsat - vbs) else PHIroot/(1 + (vbs - vdsat)/(2*PHIval));
            BS3sat := BARGsat^3 - Sarg^3;
            if LAMBDA == 0 then
              if VMAX == 0 then
                Lfact := XD/(L*vds);
                Vdiff3 := (vds - vdsat)/4;
                LAMBDAval := Lfact*sqrt(Vdiff3 + sqrt(1 + Vdiff3*Vdiff3));  /* (4-54) p.187 */
              else
                DV := XD/sqrt(NEFF);
                LV := VMAX*DV/(2*Ueff);
                VQChan := Vdiff - vdsat - GAMMAval*BARGsat;
                LS := sqrt(LV*LV + (if vds > vdsat then vds - vdsat else 0));
                Lfact := DV/(L*vds);
                LAMBDAval := Lfact*(LS - LV);  /* (4-60) p.192 */
              end if;
            else
              LAMBDAval := LAMBDA;
            end if;
          else
            BS3sat := BS3;
          end if;
          BS3eff := if Vgst <= 0 and vdsat > 0 then BS3sat else BS3;

          /* Compute channel shortening at punch-through */
          WB := XD*sqrt(PB);
          LD := L - WB;
          Lfact2 := 1 - LAMBDAval*vds;
          Leff := Lfact2*L;
          DeltaL := LAMBDAval*vds*L;
          if Leff < WB then
            LEFFval := WB/(1 + (DeltaL - LD)/WB);
            LFACTval := LEFFval/L;
          else
            LEFFval := Leff;
            LFACTval := Lfact2;
          end if;

          /* Compute effective BETA value */
          BETAval := BETA*Ufact/LFACTval;

          /* Compute drain-source current */
          /* Linear region */
          cdlin := vds*(vgs - Type*Vfb - PHIval - Fn2*vds/2) - 2*GAMMAval*BS3eff/3;  /* (4-43) p.182 */
          /* Saturation region */
          cdsat := vdsat*(vgs - Type*Vfb - PHIval - Fn2*vdsat/2) - 2*GAMMAval*BS3sat/3;
          /* Subthreshold region */
          vdson := if vdsat < vds then vdsat else vds;
          cdson := vdson*(Von - Type*Vfb - PHIval - Fn2*vdson/2) - 2*GAMMAval*BS3eff/3;
          et := (vgs - Von)/(N*Vt);
          cdsub := cdson*(if et < EMin then ExMin*(et - EMin + 1) else
                          if et > EMax then ExMax*(et - EMax + 1) else  exp(et));  /* (4-49) p.183 */
          ids := sig*BETAval*(if Vgst > 0 and vds <= vdsat then cdlin else
                              if Vgst > 0 and vds > vdsat then cdsat else
                              if Vgst <= 0 and vdsat > 0 then cdsub else 0);
          ids0 := if vds > 1e-8 then ids else 0;
        end MOSeq2;

        function MOSeq3
        "Computation of injected drain-source current for Level=3"
          input Modelica.SIunits.Voltage vds "Drain-source voltage";
          input Modelica.SIunits.Voltage vgs "Gate-source voltage";
          input Modelica.SIunits.Voltage vbs "Bulk-source voltage";
          input Modelica.SIunits.Temperature Tdev "Device temperature";
          input Real sig "sign(vds)";
          input Integer Type "Type=1 for NMOS; Type=-1 for PMOS";
          input Modelica.SIunits.Temperature Tnom "Reference temperature";
          input Modelica.SIunits.Voltage VFB
          "Flat band voltage at reference temperature";
          input Modelica.SIunits.Transconductance KP
          "Transconductance parameter at reference temperature";
          input Real GAMMA(unit="V^0.5") "Body-effect parameter";
          input Modelica.SIunits.Voltage PHI
          "Surface inversion potential at reference temperature";
          input Real COX(unit="F/m2") "Specific capacitance of SiO2";
          input Real NFS(unit="1/m2") "Surface fast state density";
          input Modelica.SIunits.Length XJ "Metallurgical junction depth";
          input Modelica.SIunits.Length L "Effective hannel length";
          input Modelica.SIunits.Length W "Effective channel width";
          input Modelica.SIunits.Length LD "Lateral diffusion";
          input Real XD(unit="m/V^0.5") "Depletion factor";
          input Real U0(unit="m2/(V.s)")
          "Surface mobility at reference temperature";
          input Modelica.SIunits.Velocity VMAX
          "Maximum drift velocity of carriers";
          input Real DELTA "Width effect on threshold voltage";
          input Real KAPPA "Saturation field factor";
          input Real ETA "Static feedback on threshold voltage";
          input Modelica.SIunits.InversePotential THETA "Mobility modulation";
          input Modelica.SIunits.Voltage EG
          "Energy gap for temperature effect on saturation current at 0 K";
          input Real EMin "if x < EMin, the exp(x) function is linearized";
          input Real EMax "if x > EMax, the exp(x) function is linearized";
          output Modelica.SIunits.Current ids0 "Injected drain-source current";
          output Modelica.SIunits.Voltage Vfb
          "Effective flat band voltage at device temperature";
          output Modelica.SIunits.Voltage Vth "Threshold voltage";
          output Modelica.SIunits.Voltage vdsat "Saturation voltage";
          output Modelica.SIunits.Voltage vpof "Pinch-off voltage";

      protected
          constant Real pi=Modelica.Constants.pi;
          constant Modelica.SIunits.Entropy k=Modelica.Constants.k
          "Boltzmann's constant";
          constant Modelica.SIunits.Permittivity e0=Modelica.Constants.epsilon_0
          "Permittivity of free space";
          constant Modelica.SIunits.ElectricCharge q=1.6021892e-19
          "Electron charge";
          constant Real ks=11.8 "Dielectric constant of Si";
          constant Real GapC1(unit="V/K")=7.02E-4
          "First bandgap correction factor Silicon";
          constant Modelica.SIunits.Temperature GapC2=1108.0
          "Second bandgap correction factor Silicon";
          constant Real c0=0.0631353;
          constant Real c1=0.8013292;
          constant Real c2=-0.01110777;
          parameter Modelica.SIunits.Permittivity es = ks*e0
          "Permittivity of silicon";
          Real Fn = es*DELTA*pi/(2*COX*W) "Working function";  /* (4-99) p.207 */
          Real SIGMA = 8.15e-22*ETA/(COX*L^3)
          "Static feedback on threshold voltage";                                      /* (4-94) p.205 */
          Real ExMin = exp(EMin);
          Real ExMax = exp(EMax);
          Real RTemp "Temperature quotient";
          Real facTemp "Temperature coefficient";
          Modelica.SIunits.Voltage EGref "Energy gap at reference temperature";
          Modelica.SIunits.Voltage EGval "Energy gap at device temperature";
          Modelica.SIunits.Voltage Vt "Thermal voltage";
          Modelica.SIunits.Voltage PHIval
          "Surface inversion potential at device temperature";
          Modelica.SIunits.Voltage Von "Adjusted threshold voltage";
          Modelica.SIunits.Voltage Vgst "vgs - vth";
          Modelica.SIunits.Current idlin "Linear current";
          Modelica.SIunits.Current idsat "Saturation current";
          Modelica.SIunits.Current ids "Drain-source current";
          Modelica.SIunits.Conductance gdsat "Saturation region conductance";
          Real cdlin(unit="V2") "Linear region current factor";
          Real cdsat(unit="V2") "Saturation region current factor";
          Real cdsub(unit="V2") "Subthreshold region current factor";
          Real et "Subthreshold voltage coefficient";
          Modelica.SIunits.Transconductance KPval
          "Transconductance parameter at device temperature";
          Modelica.SIunits.Transconductance BETA
          "Transconductance at device temperature";
          Modelica.SIunits.Transconductance BETAval
          "Adjusted transconductance at device temperature";
          Modelica.SIunits.Transconductance BETAeff
          "Adjusted transconductance at device temperature";
          Modelica.SIunits.Transconductance Beta
          "Final transconductance at device temperature";
          Real LAMBDAval "Channel length modulation";
          Modelica.SIunits.Length Wp
          "Width of depleted region in flat source junction";
          Modelica.SIunits.Length Wc
          "Width of depleted region in source substrate junction";
          Real WpXJ "WP/XJ";
          Real WcXJ "WC/XJ";
          Real Us(unit="m2/(V.s)")
          "Dependence of mobility on gate electric field";
          Real CB "Specific capacitance of depleted region";
          Real N "Intrinsic carrier concentration at device temperature";
          Modelica.SIunits.ElectricFieldStrength Ep
          "Lateral field at channel pinch-off";
          Real Fs "Short channel working function";
          Real FB "Body effect working function";

          /* Auxiliary variables */
          Modelica.SIunits.Voltage Vbody;
          Modelica.SIunits.Voltage vgsx;
          Modelica.SIunits.Voltage vdsx;
          Modelica.SIunits.Voltage Va;
          Modelica.SIunits.Voltage Vb;
          Modelica.SIunits.Length DL;
          Modelica.SIunits.Length DLval;
          Modelica.SIunits.Length DLfac;
          Real ALPHA(unit="m2/V");
          Real Sarg(unit="V^0.5");
          Real Lfact;
          Real Fd;
          Real FG;
        annotation (
          Documentation(info="<html>
This function implements a Spice-style <b>Level-3</b> model of the current flowing between drain and source of a MOSFET.  It is is an empirical model that computes the current in its linear, saturation, and weak inversion regions.  Various channel modulation effects are also considered. <p>

The model offers a compromise between the much more sophisticated <b>Level-2</b> model that, however, simulates very slowly, and the often too simple <b>Level-1</b> model that doesn't allow many questions to be answered, such as what is the highest possible switching frequency reachable, how much power is being dissipated at high frequency operation, what is the amount of substrate cross-talk, etc. <p>

The model comments the equations by providing the corresponding equation numbers from [1]. <p>

<hr> <p>

<pre>
Input variables:

 vds:     Drain-source voltage (Volt)

 vgs:     Gate-source voltage (Volt)

 vbs:     Bulk-source voltage (Volt)

 Tdev:    Device temperature (K)

 sig:     Indicator of operating mode
            sig = +1:  direct mode
            sig = -1:  reverse mode (source and drain are flipped)


Input parameters:

 Type:    Type of MOSFET (default value = 1)
            Type = +1:  NMOS
            Type = -1:  PMOS

 Tnom:    Reference temperature (default value = 300.15 K)

 VFB:     Flat band voltage at reference temperature (default value = 0 Volt)

 KP:      Transconductance parameter at reference temperature (default value = 0 Amp/Volt<sup>2</sup>)

 GAMMA:   Body-effect parameter (default value = 0 Volt<sup>0.5</sup>)

 PHI:     Surface inversion potential at reference temperature (default value = 0 Volt)

 COX:     Specific capacitance of SiO2 (default value = 0 F/m<sup>2</sup>)

 NFS:     Surface fast state density (default value = 0 1/m<sup>2</sup>)

 XJ:      Metallurgical junction depth (default value = 0 m)

 L:       Channel length (default value = 1e-4 m)

 W:       Channel width (default value = 1e-4 m)

 LD:      Lateral diffusion (default value = 0 m)

 XD:      Depletion factor (default value = 0 m/Volt<sup>0.5</sup>)

 U0:      Surface mobility at reference temperature (default value = 0 m<sup>2</sup>/(V*s))

 VMAX:    Maximum drift velocity of carriers (default value = 0 m/s)

 DELTA:   Width effect on threshold voltage (default value = 0)

 KAPPA:   Saturation field factor (default value = 0.2)

 ETA:     Static feedback on threshold voltage (default value = 0)

 THETA:   Mobility modulation (default value = 0 1/Volt)

 EG:      Energy gap for temperature effect on saturation current at 0 K (default value = 1.11 Volt)

 EMin:    if x &lt; EMin, the exp(x) function is linearized (default value = -100) <font color=red>Levels 1-3</font>

 EMax:    if x &gt; EMax, the exp(x) function is linearized (default value = 40) <font color=red>Levels 1-3</font>


Output variables:

 isd0:    Drain-source current (Amp)

 Vfb:     Flat band voltage at device temperature (Volt)

 Vth:     Threshold voltage (Volt)

 vdsat:   Saturation voltage (Volt)

 vpof:    Pinch-off voltage (Volt)
</pre> <p>

<hr> <p>

<b>References:</b> <p>

<ol>
<li> Massobrio, G. and P. Antognetti (1993), <i>Semiconductor Device Modeling with Spice</i>,
     2<sup>nd</sup> edition, McGraw Hill, New York.
</ol>
</html>
"),       Icon(
            Polygon(points=[-100,100; 20,100; 100,0; 20,-100; -100,-100; -100,
                  100], style(
                color=58,
                rgbcolor={0,111,0},
                fillColor=7,
                rgbfillColor={255,255,255})),
              Text(
                extent=[-52,-20; 20,-40],
              style(color=42, rgbcolor={220,0,0}),
              string="Level = 3"),
              Text(
                extent=[-70,8; 50,-22],
                string="of a MOS transistor",
              style(color=58, rgbcolor={0,111,0})),
              Text(
                extent=[-92,36; 72,-4],
                string="between drain and source",
              style(color=58, rgbcolor={0,111,0})),
              Text(
                extent=[-62,50; 38,22],
                string="Injected current",
              style(color=58, rgbcolor={0,111,0}))),
          Diagram(
            Polygon(points=[-100,100; 20,100; 100,0; 20,-100; -100,-100; -100,
                  100], style(
                color=58,
                rgbcolor={0,111,0},
                fillColor=7,
                rgbfillColor={255,255,255})),
              Text(
                extent=[-52,-20; 20,-40],
              style(color=42, rgbcolor={220,0,0}),
              string="Level = 3"),
              Text(
                extent=[-70,8; 50,-22],
                string="of a MOS transistor",
              style(color=58, rgbcolor={0,111,0})),
              Text(
                extent=[-92,36; 72,-4],
                string="between drain and source",
              style(color=58, rgbcolor={0,111,0})),
              Text(
                extent=[-62,50; 38,22],
                string="Injected current",
              style(color=58, rgbcolor={0,111,0}))));
        algorithm
          /* Parameter computations and temperature variations (Implements TMPUPD) */
          RTemp := Tdev/Tnom;
          facTemp := RTemp*sqrt(RTemp);
          Vt := k*Tdev/q;
          EGref := EG - GapC1*Tnom*Tnom/(Tnom + GapC2);
          EGval := EG - GapC1*Tdev*Tdev/(Tdev + GapC2);
          PHIval := RTemp*PHI + EGval - RTemp*EGref - 3*Vt*ln(RTemp);
          Vfb := VFB + Type*((PHIval - PHI)/2 + (EGref - EGval)/2);
          KPval := KP/facTemp;
          BETA := KPval*W/L;

          /* Compute short channel effects */
          Sarg := if vbs <= 0 then sqrt(PHIval - vbs) else sqrt(PHIval)/(1 + vbs/(2*PHIval));
          if XJ > 0 then
            Wp := Sarg*XD;  /* (4-96) p.205 */
            WpXJ := Wp/XJ;
            WcXJ := c0 + c1*WpXJ + c2*WpXJ*WpXJ;  /* (4-98) p.206 */
            Wc := WcXJ*XJ;
            Fs := 1 - ((LD + Wc)*sqrt(1 - (Wp/(XJ + Wp))^2) - LD)/L;  /* (4-95) p.205 */
          else
            Fs := 1;
          end if;

          /* Compute body effects */
          FB := Fn + (if noEvent(Sarg > 0) then GAMMA*Fs/(4*Sarg) else 0);  /* (4-91) p.202 */
          Vbody := GAMMA*Fs*Sarg + Fn*Sarg*Sarg;
          Vth := Vfb + Type*(PHIval - SIGMA*vds + Vbody);  /* (4-93) p.204 */
          if NFS > 0 then
            CB := if noEvent(Vbody>0 and Sarg>0) then Vbody/(2*Sarg*Sarg) else 0;  /* (4-48) p.183 */
            N := 1 + q*NFS/COX + CB;  /* (4-47) p.183 */
            Von := Type*Vth + N*Vt;  /* (4-46) p.183 */
          else
            N := 1;
            Von := Type*Vth;
          end if;
          Vgst := vgs - Von;

          /* Compute mobility modulation by gate voltage */
          vgsx := if vgs > Von then vgs else Von;
          FG := 1/(1 + THETA*(vgsx - Type*Vth));
          Us :=FG*U0;  /* (4-92) p.203 */
          BETAval := FG*BETA;

          /* Compute saturation voltage */
          Va := (vgsx - Type*Vth)/(1 + FB);  /* (4-102) p.208 */
          Vb := VMAX*L/Us;  /* (4-103) p.209 */
          vdsat := if VMAX > 0 then Va + Vb - sqrt(Va*Va + Vb*Vb) else Va;  /* (4-101) p.208 */
          vpof := Va;

          /* Compute velocity saturation factor */
          vdsx := if vds < vdsat then vds else vdsat;
          Fd := if vdsx > 0 and VMAX > 0 then 1/(1 + vdsx/Vb) else 1;  /* (4-100) p.208 */
          BETAeff := Fd*BETAval;

          /* Begin with linear reagion */
          cdlin := vdsx*(vgsx - Type*Vth - (1 + FB)*vdsx/2);
          idlin := BETAeff*cdlin;  /* (4-90) p.202 */

          /* Compute channel length modulation */
          if vds > vdsat then
            if VMAX > 0 then
              idsat := idlin;
              gdsat := idsat*(1 - Fd)/Vb;
              Ep := idsat/(gdsat*L);  /* (4-105) p.209 */
              ALPHA := XD*XD;
              DLfac := Ep*ALPHA/2;
              DL := sqrt(DLfac*DLfac + KAPPA*ALPHA*(vds - vdsat)) - DLfac;  /* (4-104) p.209 */
            else
              DL := sqrt(KAPPA*ALPHA*(vds - vdsat));
            end if;
          else
            DL := 0;
          end if;

          /* Compute punch-through */
          DLval := if DL > 0.5*L then L - L*L/(4*DL) else DL;

          /* Look at saturation region */
          LAMBDAval := DLval/L;
          Lfact := 1/(1 - LAMBDAval);
          Beta := Lfact*BETAeff;
          cdsat := cdlin;

          /* Look at weak inversion region */
          et := (vgs - Von)/(N*Vt);
          cdsub := cdlin*(if et < EMin then ExMin*(et - EMin + 1) else
                          if et > EMax then ExMax*(et - EMax + 1) else  exp(et));  /* (4-49) p.183 */

          /* Compute drain-source current */
          ids := sig*Beta*(if vdsx > 0 and vds <= vdsat then cdlin else
                           if vdsx > 0 and vds > vdsat and Vgst >= 0 then cdsat else
                           if vdsx > 0 and vds > vdsat and Vgst < 0 then cdsub else 0);
          ids0 := if NFS == 0 and Vgst <= 0 then 0 else ids;
        end MOSeq3;

        block nlCM "Non-linear junction capacitance computation of MOSFET"
          extends Modelica.Blocks.Interfaces.SISO;
          constant Modelica.SIunits.Entropy k=Modelica.Constants.k
          "Boltzmann's constant";
          constant Modelica.SIunits.ElectricCharge q=1.6021892e-19
          "Electron charge";
          constant Real GapC1(unit="V/K")=7.02E-4
          "First bandgap correction factor Silicon";
          constant Modelica.SIunits.Temperature GapC2=1108.0
          "Second bandgap correction factor Silicon";
          parameter Integer Level=3
          "MOSFET modeling level (check documentation window for details)";
          parameter Modelica.SIunits.Temperature Tnom=300.15
          "Reference temperature";
          parameter Modelica.SIunits.Voltage PB=0.8
          "Built-in potential at reference temperature";
          parameter Real CJ(unit="F/m2") = 1e-4
          "Zero-bias bulk capacitance per square meter at reference temperature";
          parameter Real MJ=0.33 "Bulk junction grading coefficient";
          parameter Real CJSW(unit="F/m") = 1e-9
          "Zero-bias perimeter capacitance per meter at reference temperature";
          parameter Real MJSW=0.33 "Perimeter capacitance grading coefficient";
          parameter Real FC=0.5
          "Forward-bias depletion capacitance coefficient";
          parameter Modelica.SIunits.Area A=0 "Diffusion area";
          parameter Modelica.SIunits.Length P=0 "Perimeter width";
          parameter Modelica.SIunits.Voltage EG=1.11
          "Energy gap for temperature effect on saturation current at 0 K";

      protected
          Modelica.SIunits.Temperature DTemp
          "Difference between circuit temperature and reference temperature";
          Real RTemp "Temperature quotient";
          Modelica.SIunits.Voltage Vt "Thermal voltage";
          Modelica.SIunits.Voltage PBval
          "Built-in potential at device temperature";
          Real CJval(unit="F/m2")
          "Zero-bias bulk capacitance per square meter at device temperature";
          Real CJSWval(unit="F/m")
          "Zero-bias perimeter capacitance per meter at device temperature";
          Modelica.SIunits.Voltage EGval "Energy gap at device temperature";
          Modelica.SIunits.Voltage EGnom "Energy gap at reference temperature";
          Modelica.SIunits.Voltage v "Voltage across capacitance";
          Modelica.SIunits.Capacitance Cval
          "Non-linear junction capacitance value";
          Real Cbulk(unit="F/m2")
          "Bulk capacitance per square meter at device temperature";
          Real Cperi(unit="F/m")
          "Perimeter capacitance per meter at device temperature";
      public
        Modelica.Blocks.Interfaces.RealInput Tdev(redeclare type SignalType =
                Modelica.SIunits.Temperature) "Device temperature"
          annotation (extent=[-10,-110; 10,-90],  rotation=90);
        annotation (Icon(
                    Text(
                extent=[-78,72; 76,-56],
                string="nl C",
                style(color=0, rgbcolor={0,0,0})), Text(
                extent=[-82,-40; 82,-64],
                style(color=0, rgbcolor={0,0,0}),
              string="MOSFET")), Diagram(
                 Rectangle(extent=[-100,-100; 100,100],   style(
                color=74,
                rgbcolor={0,0,127},
                fillColor=7,
                rgbfillColor={255,255,255})),
                    Text(
                extent=[-78,72; 76,-56],
                string="nl C",
                style(color=0, rgbcolor={0,0,0})), Text(
                extent=[-82,-40; 82,-64],
                style(color=0, rgbcolor={0,0,0}),
              string="MOSFET")));
        equation
          v=u;

          /* Compute thermal voltage as function of temperature */
          Vt = k*Tdev/q;

          /* Compute temperature variation on junction capacitance */
          DTemp = Tdev - Tnom;
          RTemp = Tdev/Tnom;
          EGval = EG - GapC1*Tdev*Tdev/(Tdev + GapC2);
          EGnom = EG - GapC1*Tnom*Tnom/(Tnom + GapC2);
          PBval = PB*RTemp + EGval - EGnom*RTemp - 3*Vt*ln(RTemp);
          CJval = CJ*(1 + MJ*(1 - PBval/PB + 4e-4*DTemp));
          CJSWval = CJSW*(1 + MJSW*(1 - PBval/PB + 4e-4*DTemp));

          if Level==0 then
            Cbulk = 0;
            Cperi = 0;
            Cval = 1e-16;
          else
            Cbulk = CJval*(if v > FC*PBval then (1 - FC*(1 + MJ) + MJ*v/PBval)/((1 - FC)^(1 + MJ)) else
                                                1/noEvent((abs(1 - v/PBval))^MJ));
            Cperi = CJSWval*(if v > FC*PBval then (1 - FC*(1 + MJSW) + MJSW*v/PBval)/((1 - FC)^(1 + MJSW)) else
                                                  1/noEvent((abs(1 - v/PBval))^MJSW));
            Cval = if A*Cbulk + P*Cperi > 0 then A*Cbulk + P*Cperi else 1e-16;
          end if;

          y = Cval;
        end nlCM;

        block IDS0 "Injected current between drain and source"
          extends Modelica.Blocks.Interfaces.SI2SO;
          parameter Integer Level=3
          "Level of MOS model (check documentation window for details)";
          parameter Integer Type=1 "Type=1 for NMOS; Type=-1 for PMOS";
          parameter Modelica.SIunits.Temperature Tnom=300.15
          "Reference temperature";
          parameter Modelica.SIunits.Voltage VFB=0 "Flat band voltage";
          parameter Modelica.SIunits.Transconductance KP=0
          "Transconductance parameter at reference temperature";
          parameter Real GAMMA(unit="V^0.5") = 0 "Body-effect parameter";
          parameter Modelica.SIunits.Voltage PHI=0
          "Surface inversion potential at reference temperature";
          parameter Modelica.SIunits.InversePotential LAMBDA=0
          "Channel length modulation";
          parameter Real COX(unit="F/m2") = 0 "Specific capacitance of SiO2";
          parameter Real NFS(unit="1/m2") = 0 "Surface fast state density";
          parameter Real NEFF=1 "Total channel charge coefficient";
          parameter Modelica.SIunits.Length XJ=0 "Metallurgical junction depth";
          parameter Modelica.SIunits.Length L(min=0) = 1e-4 "Channel length";
          parameter Modelica.SIunits.Length W(min=0) = 1e-4 "Channel width";
          parameter Modelica.SIunits.Length LD=0 "Lateral diffusion";
          parameter Real XD(unit="m/V^0.5") = 0 "Depletion factor";
          parameter Real U0(unit="m2/(V.s)") = 0.06
          "Surface mobility at reference temperature";
          parameter Real UEXP=0 "Exponential coefficient for mobility";
          parameter Real UTRA=0 "Transverse field coefficient";
          parameter Modelica.SIunits.Velocity VMAX=0
          "Maximum drift velocity of carriers";
          parameter Modelica.SIunits.ElectricFieldStrength ECRIT=0
          "Critical electric field for pinch-off";
          parameter Real DELTA=0 "Width effect on threshold voltage";
          parameter Real KAPPA=0.2 "Saturation field factor";
          parameter Real ETA=0 "Static feedback on threshold voltage";
          parameter Modelica.SIunits.InversePotential THETA=0
          "Mobility modulation";
          parameter Modelica.SIunits.Voltage VBP=0
          "Gate to channel critical voltage";
          parameter Modelica.SIunits.Voltage PB=0.8
          "Built-in potential at reference temperature";
          parameter Modelica.SIunits.Voltage EG=1.11
          "Energy gap for temperature effect on saturation current";
          parameter Real EMin=-100
          "if x < EMin, the exp(x) function is linearized";
          parameter Real EMax=40
          "if x > EMax, the exp(x) function is linearized";
          output Modelica.SIunits.Voltage vds "Drain-source voltage";
          output Modelica.SIunits.Voltage vgs "Gate-source voltage";
          output Modelica.SIunits.Voltage vbs "Bulk-source voltage";
          output Modelica.SIunits.Current ids "Injected drain-source current";
          output Modelica.SIunits.Voltage Vfb "Flat band voltage";
          output Modelica.SIunits.Voltage Vth "Threshold voltage";
          output Modelica.SIunits.Voltage vdsat "Saturation voltage";
          output Modelica.SIunits.Voltage vdsat1 "Adjusted saturation voltage";
          output Modelica.SIunits.Voltage vpof "Punch-through voltage";

      protected
          Real sig "sign(vds)";
      public
          Modelica.Blocks.Interfaces.RealInput u3
          "Connector of Real input signal 3"
            annotation (extent=[-10,90; 10,110], rotation=-90);
          Modelica.Blocks.Interfaces.RealInput Tdev(redeclare type SignalType
            =   Modelica.SIunits.Temperature) "Device temperature"
            annotation (extent=[-10,-110; 10,-90], rotation=90);
        annotation (Icon(
              Text(
                extent=[-8,88; 12,76],
                style(color=3, rgbcolor={0,0,255}),
                string="vbs"),
              Text(
                extent=[-96,66; -76,54],
                style(color=3, rgbcolor={0,0,255}),
                string="vds"),
              Text(
                extent=[-96,-54; -76,-66],
                style(color=3, rgbcolor={0,0,255}),
                string="vgs"),
              Text(
                extent=[-10,-78; 10,-90],
                style(color=3, rgbcolor={0,0,255}),
                string="Tdev"),
              Text(
                extent=[-80,26; 84,-14],
                style(color=3, rgbcolor={0,0,255}),
                string="between drain and source"),
              Text(
                extent=[-50,40; 50,12],
                style(color=3, rgbcolor={0,0,255}),
                string="Injected current"),
              Text(
                extent=[-58,-2; 62,-32],
                style(color=3, rgbcolor={0,0,255}),
                string="of a MOS transistor")), Diagram(
                 Rectangle(extent=[-100,-100; 100,100],   style(
                color=74,
                rgbcolor={0,0,127},
                fillColor=7,
                rgbfillColor={255,255,255})),
              Text(
                extent=[-8,88; 12,76],
                style(color=3, rgbcolor={0,0,255}),
                string="vbs"),
              Text(
                extent=[-96,66; -76,54],
                style(color=3, rgbcolor={0,0,255}),
                string="vds"),
              Text(
                extent=[-96,-54; -76,-66],
                style(color=3, rgbcolor={0,0,255}),
                string="vgs"),
              Text(
                extent=[-10,-78; 10,-90],
                style(color=3, rgbcolor={0,0,255}),
                string="Tdev"),
              Text(
                extent=[-50,40; 50,12],
                style(color=3, rgbcolor={0,0,255}),
                string="Injected current"),
              Text(
                extent=[-80,26; 84,-14],
                style(color=3, rgbcolor={0,0,255}),
                string="between drain and source"),
              Text(
                extent=[-58,-2; 62,-32],
                style(color=3, rgbcolor={0,0,255}),
                string="of a MOS transistor")),
          Documentation(info="<html>
The <b>IDS0</b> block computes the drain-source current of a MOSFET.  Depending on the modeling level, a different algorithm is being used.  The equations for each of these algorithms are programmed in functions, called <b>MOSeq1</b> .. <b>MOSeq3</b> that implement the particular model in question.  These functions have been carefully documented to make it easier on future developers to add additional models of their own by providing corresponding <b>MOSeq_i</b> functions. <p>

<hr> <p>

<pre>
Parameters:

 <font color=red><b>Level:   MOSFET modeling level (default value = 3)
            Level=0: Static injection model
            Level=1: Shichman-Hodges model
            Level=2: Grove-Frohman model
            Level=3: Empirical model
            Level=4: Simplified Grove-Frohman model</b></font>

 Tnom:    Reference temperature (default value = 300.15 K) <font color=red>Levels 0-4</font>

 VFB:     Flat band voltage (default value = 0 Volt) <font color=red>Levels 0-4</font>

 KP:      Transconductance parameter at reference temperature (default value = 0 Amp/Volt<sup>2</sup>) <font color=red>Levels 0-4</font>

 GAMMA:   Body-effect parameter (default value = 0 Volt<sup>0.5</sup>) <font color=red>Levels 0-4</font>

 PHI:     Surface inversion potential at reference temperature (default value = 0 Volt) <font color=red>Levels 0-4</font>

 LAMBDA:  Channel length modulation  (default value = 0 1/Volt) <font color=red>Levels 0-4</font>

 COX:     Specific capacitance of SiO2 (default value = 0 F/m<sup>2</sup>) <font color=red>Levels 2-4</font>

 NFS:     Surface fast state density (default value = 0 1/m<sup>2</sup>) <font color=red>Levels 2-4</font>

 NEFF:    Total channel charge coefficient (default value = 1) <font color=red>Levels 2,4</font>

 XJ:      Metallurgical junction depth (default value = 0 m) <font color=red>Levels 2-4</font>

 L:       Channel length (default value = 1e-4 m) <font color=red>Levels 0-4</font>

 W:       Channel width (default value = 1e-4 m) <font color=red>Levels 0-4</font>

 LD:      Lateral diffusion (default value = 0 m) <font color=red>Level 3</font>

 XD:      Depletion factor (default value = 0 m/Volt<sup>0.5</sup>) <font color=red>Levels 2-4</font>

 U0:      Surface mobility at reference temperature (default value = 0 m<sup>2</sup>/(V*s)) <font color=red>Levels 2-4</font>

 UEXP:    Exponential coefficient for mobility (default value = 0) <font color=red>Levels 2,4</font>

 UTRA:    Transverse field coefficient (default value = 0) <font color=red>Levels 2,4</font>

 VMAX:    Maximum drift velocity of carriers (default value = 0 m/s) <font color=red>Levels 2-4</font>
            <font color=blue><b>VMAX = 0: Frohman-Grove model of drain current computation (Levels 2,4)</b></font>
            <font color=blue><b>VMAX &gt; 0: Baum-Beneking model of drain current computation (Levels 2,4)</b></font>

 ECRIT:   Critical electric field for pinch-off (default value = 0 V/m) <font color=red>Levels 2,4</font>

 DELTA:   Width effect on threshold voltage (default value = 0) <font color=red>Levels 2-4</font>

 KAPPA:   Saturation field factor (default value = 0.2) <font color=red>Level 3</font>

 ETA:     Static feedback on threshold voltage (default value = 0) <font color=red>Level 3</font>

 THETA:   Mobility modulation (default value = 0 1/Volt) <font color=red>Level 3</font>

 VBP:     Gate to channel critical voltage (default value = 0 Volt) <font color=red>Levels 2,4</font>

 PB:      Built-in potential at reference temperature (default value = 0.8 Volt) <font color=red>Levels 2,4</font>

 EG:      Energy gap for temperature effect on saturation current at 0 K (default value = 1.11 Volt) <font color=red>Levels 0-4</font>

 EMin:    if x &lt; EMin, the exp(x) function is linearized (default value = -100) <font color=red>Levels 2-4</font>

 EMax:    if x &gt; EMax, the exp(x) function is linearized (default value = 40) <font color=red>Levels 2-4</font>
</pre>
</html>

"));
          Modelica.Blocks.Interfaces.RealOutput y1[5]
          "Connector of Real output signals"
            annotation (extent=[100,-60; 120,-40]);
        equation
          /* Determine first whether vds > 0 */
          if noEvent(u1 > 0) then
            /* Keep drain and source as they are */
            sig = 1;
            vds = u1;
            vgs = u2;
            vbs = u3;
          else
            /* Switch drain and source */
            sig = -1;
            vds = -u1;
            vgs = u2 - u1;
            vbs = u3 - u1;
          end if;

          /* Implement the different levels of drain-source current injection models */
          if Level==0 or Level==1 then
            (ids,Vfb,Vth,vdsat) = MOSeq1(vds=vds,vgs=vgs,vbs=vbs,Tdev=Tdev,sig=sig,
                                         Type=Type,Tnom=Tnom,VFB=VFB,KP=KP,GAMMA=GAMMA,
                                         PHI=PHI,LAMBDA=LAMBDA,L=L,W=W,EG=EG);
            vdsat1 = 0;
            vpof = 0;
          elseif Level==2 or Level==4 then
            (ids,Vfb,Vth,vdsat,vdsat1,vpof) = MOSeq2(vds=vds,vgs=vgs,vbs=vbs,Tdev=Tdev,sig=sig,
                                                     Level=Level,Type=Type,Tnom=Tnom,VFB=VFB,KP=KP,
                                                     GAMMA=GAMMA,PHI=PHI,LAMBDA=LAMBDA,COX=COX,NFS=NFS,
                                                     NEFF=NEFF,XJ=XJ,L=L,W=W,XD=XD,
                                                     U0=U0,UEXP=UEXP,UTRA=UTRA,VMAX=VMAX,ECRIT=ECRIT,
                                                     DELTA=DELTA,VBP=VBP,PB=PB,EG=EG,EMin=EMin,
                                                     EMax=EMax);
          elseif Level==3 then
            (ids,Vfb,Vth,vdsat,vpof) = MOSeq3(vds=vds,vgs=vgs,vbs=vbs,Tdev=Tdev,sig=sig,
                                              Type=Type,Tnom=Tnom,VFB=VFB,KP=KP,GAMMA=GAMMA,
                                              PHI=PHI,COX=COX,NFS=NFS,XJ=XJ,L=L,
                                              W=W,LD=LD,XD=XD,U0=U0,VMAX=VMAX,
                                              DELTA=DELTA,KAPPA=KAPPA,ETA=ETA,THETA=THETA,EG=EG,
                                              EMin=EMin,EMax=EMax);
            vdsat1 = 0;
          end if;

          /* Assign outputs */
          y = ids;
          y1[1] = Vfb;
          y1[2] = Vth;
          y1[3] = vdsat;
          y1[4] = vdsat1;
          y1[5] = vpof;
        end IDS0;

        model RM "Spice-style resistor model for MOSFETs"
          extends Interfaces.SpiceTwoPort;
          parameter Modelica.SIunits.Resistance R=0
          "Resistance at reference temperature";
          parameter Real TR1(unit="1/K") = 0 "Linear temperature coefficient";
          parameter Real TR2(unit="1/K2") = 0
          "Quadratic temperature coefficient";
          Modelica.SIunits.Resistance Rval
          "Resistance value at circuit temperature";
          parameter Integer Level=3
          "MOSFET modeling level (check documentation window for details)";
        annotation (
          Diagram(
              Text(
                extent=[-82,76; 28,-80],
                string="RS",
                style(color=0))),
          Icon(
              Text(
                extent=[-82,76; 28,-80],
                string="RS",
                style(color=0))),
          Documentation(info="<html>
This is the regular Spice-style resistive source <b>RS</b> model, except that it looks at the modeling level.  If the level is equal to 0, the resistance is set equal to zero.
</html>
"));
        equation
          if Level==0 or not R > 0 then
          /* Static injection model */
            Rval = 0;
          else
          /* All other models */
            Rval = R*(1 + TR1*DTemp + TR2*DTempSq);
          end if;
          e1 = Rval*f1;
          e1*f1 = e2*f2;
        end RM;

        model CjM "Spice-style junction capacitor model for MOSFETs"
          extends BondLib.Interfaces.TwoPort;
          parameter Integer Level=3
          "MOSFET modeling level (check documentation window for details)";
          parameter Modelica.SIunits.Temperature Tnom=300.15
          "Reference temperature";
          parameter Modelica.SIunits.Current IS=0
          "Saturation current at reference temperature";
          parameter Modelica.SIunits.Voltage PB=0.8
          "Built-in potential at reference temperature";
          parameter Real CJ(unit="F/m2") = 1e-4
          "Zero-bias bulk capacitance per square meter at reference temperature";
          parameter Real MJ=0.33 "Bulk junction grading coefficient";
          parameter Real CJSW(unit="F/m") = 1e-9
          "Zero-bias perimeter capacitance per meter at reference temperature";
          parameter Real MJSW=0.33 "Perimeter capacitance grading coefficient";
          parameter Real FC=0.5
          "Forward-bias depletion capacitance coefficient";
          parameter Modelica.SIunits.Area A=0 "Diffusion area";
          parameter Modelica.SIunits.Length P=0 "Perimeter width";
          parameter Modelica.SIunits.Voltage EG=1.11
          "Energy gap for temperature effect on saturation current at 300 K";
          parameter Modelica.SIunits.Conductance GminDC=1e-12
          "Leakage conductance";
          parameter Real EMin=-100
          "if x < EMin, the exp(x) function is linearized";
          parameter Real EMax=40
          "if x > EMax, the exp(x) function is linearized";
        annotation (
          Icon(
              Text(
                extent=[-84,72; 30,-74],
                style(color=0),
              string="CjM"),
            Text(
              extent=[-52,96; 54,66],
              style(color=3, rgbcolor={0,0,255}),
              string="%name"),
            Text(
              extent=[4,-4; 112,-32],
              string="Spice",
              style(color=0, rgbcolor={0,0,0}))),
          Diagram,
          Documentation(info="<html>
The Spice-style junction capacitance for MOSFETs computes the depletion capacitance value of a junction used in a MOSFET.  Different Spice dialects vary in the formulae they use for this purpose.  Many Spice dialects actually don't use a formula for the junction capacitance at all, but rather compute the electric charge stored in the junction directly, which is conceptually cleaner.  However, that approach is computationally cumbersome, as it leads to an awkward algebraic loop [1].  Thus, we chose to compute the junction capacitance, and use a (physically incorrect) approximate non-linear capacitor model.  The numerical error should remain small, as the time constants associated with temperature variation are much larger than those associated with electrical phenomena. <p>

<hr> <p>

<pre>
Parameters:

 Level:   MOSFET modeling level (default value = 3)
            Level=0: Static injection model
            Level=1: Shichman-Hodges model
            Level=2: Grove-Frohman model
            Level=3: Empirical model
            Level=4: Simplified Grove-Frohman model

 IS:      Transport saturation current (default value = 0 Amp)

 PB:      Built-in potential at reference temperature (default value = 0.8 Volt)

 CJ:      Zero-bias bulk capacitance per square meter at reference temperature (default value = 1e-4 F/m<sup>2</sup>)

 MJ:      Bulk junction grading coefficient (default value = 0.33)

 CJSW:    Zero-bias perimeter capacitance per meter at reference temperature (default value = 1e-9 F/m)

 MJSW:    Perimeter capacitance grading coefficient (default value = 0.33)

 FC:      Forward-bias depletion capacitance coefficient (default value = 0.5)

 A:       Diffusion area (default value = 0 m<sup>2</sup>)

 P:       Perimeter width (default value = 0 m)

 EG:      Energy gap for temperature effect on saturation current (default value = 1.11 Volt)

 GminDC:  Leakage conductance (default value = 1e-12 Mho)

 EMin:    if x &lt; EMin, the exp(x) function is linearized (default value = -100)

 EMax:    if x &gt; EMax, the exp(x) function is linearized (default value = 40)
</pre> <p>

<hr> <p>

<b>References:</b> <p>

<ol>
<li> Cellier, F.E. (1991), <i>Continuous System Modeling</i>, Springer-Verlag, New York, pp. 224-225.
<li> Massobrio, G. and P. Antognetti (1993), <i>Semiconductor Device Modeling with Spice</i>,
     2<sup>nd</sup> edition, McGraw Hill, New York, p.200.
</ol>
</html>
"));
        Junctions.J0p5 J0p5_1 annotation (extent=[-80,-10; -60,10]);
          Bonds.fBond B1 annotation (extent=[-60,-10; -40,10]);
          Sensors.De vd  annotation (extent=[-40,-10; -20,10]);
          Bonds.eBond B2 annotation (extent=[-80,10; -60,30], rotation=90);
          Semiconductors.Utilities.mC2 Cj1
                  annotation (extent=[-80,30; -60,50], rotation=90);
        nlCM Cval(
          Tnom=Tnom,
          PB=PB,
          CJ=CJ,
          MJ=MJ,
          CJSW=CJSW,
          MJSW=MJSW,
          FC=FC,
          A=A,
          P=P,
          EG=EG,
          Level=Level) annotation (extent=[-30,70; -50,50]);
        Junctions.J0p4 J0p4_1 annotation (extent=[60,-10; 80,10]);
          Bonds.fBond B7 annotation (extent=[60,10; 80,30], rotation=90);
          Sensors.De Tdev
                         annotation (extent=[60,30; 80,50], rotation=90);
        DjM DjM1(
          Tnom=Tnom,
          IS=IS,
          EG=EG,
          Level=Level,
          EMin=-100,
          EMax=40) annotation (extent=[-20,-40; 20,-20]);
          Bonds.eBond B4 annotation (extent=[20,-40; 40,-20], rotation=0);
          Bonds.fBond B3 annotation (extent=[-40,-40; -20,-20]);
          Bonds.fBond B5 annotation (extent=[-40,-70; -20,-50]);
          Bonds.eBond B6 annotation (extent=[20,-70; 40,-50], rotation=0);
        RS Rd(
          Tnom=Tnom,
          R=1/GminDC,
          TR1=0,
          TR2=0,
          Area=1) annotation (extent=[-20,-70; 20,-50]);
        equation
        connect(J0p5_1.BondCon1, BondCon1) annotation (points=[-80,0; -100,0],
            style(color=8, rgbcolor={192,192,192}));
        connect(J0p5_1.BondCon2, B1.eBondCon1) annotation (points=[-60,0; -60,0],
            style(color=8, rgbcolor={192,192,192}));
        connect(B1.fBondCon1, vd.BondCon1) annotation (points=[-40,0; -40,0],
            style(color=8, rgbcolor={192,192,192}));
        connect(B2.fBondCon1, J0p5_1.BondCon5) annotation (points=[-70,10; -70,
              10], style(color=8, rgbcolor={192,192,192}));
        connect(Cj1.BondCon1, B2.eBondCon1) annotation (points=[-70,30; -70,30],
            style(color=8, rgbcolor={192,192,192}));
        connect(vd.OutPort1, Cval.u) annotation (points=[-19,0; -10,0; -10,60;
              -28,60], style(color=74, rgbcolor={0,0,127}));
        connect(J0p4_1.BondCon2, BondCon2) annotation (points=[80,0; 100,0],
            style(color=8, rgbcolor={192,192,192}));
        connect(B7.eBondCon1, J0p4_1.BondCon4) annotation (points=[70,10; 70,10],
            style(color=8, rgbcolor={192,192,192}));
        connect(Tdev.BondCon1, B7.fBondCon1) annotation (points=[70,30; 70,30],
            style(color=8, rgbcolor={192,192,192}));
        connect(Tdev.OutPort1, Cval.Tdev) annotation (points=[70,51; 70,80; -40,
              80; -40,70], style(color=74, rgbcolor={0,0,127}));
        connect(DjM1.BondCon2, B4.fBondCon1) annotation (points=[20,-30; 20,-30],
            style(color=8, rgbcolor={192,192,192}));
        connect(B3.fBondCon1, DjM1.BondCon1) annotation (points=[-20,-30; -20,
              -30], style(color=8, rgbcolor={192,192,192}));
        connect(J0p5_1.BondCon4, B3.eBondCon1) annotation (points=[-66,-10; -66,
              -30; -40,-30], style(color=8, rgbcolor={192,192,192}));
        connect(B4.eBondCon1, J0p4_1.BondCon1) annotation (points=[40,-30; 50,
              -30; 50,0; 60,0], style(color=8, rgbcolor={192,192,192}));
        connect(B5.eBondCon1, J0p5_1.BondCon3) annotation (points=[-40,-60; -74,
              -60; -74,-10], style(color=8, rgbcolor={192,192,192}));
        connect(B6.eBondCon1, J0p4_1.BondCon3) annotation (points=[40,-60; 70,
              -60; 70,-10], style(color=8, rgbcolor={192,192,192}));
        connect(Rd.BondCon1, B5.fBondCon1) annotation (points=[-20,-60; -20,-60],
            style(color=8, rgbcolor={192,192,192}));
        connect(Rd.BondCon2, B6.fBondCon1) annotation (points=[20,-60; 20,-60],
            style(color=8, rgbcolor={192,192,192}));
        connect(Cval.y, Cj1.s) annotation (points=[-51,60; -70,60; -70,49],
            style(color=74, rgbcolor={0,0,127}));
        end CjM;

        model Cgb
        "Spice-style gate capacitance between gate and bulk of MOSFETs"
          extends Interfaces.SpiceTwoPort;
          constant Real pi=Modelica.Constants.pi;
          constant Modelica.SIunits.Entropy k=Modelica.Constants.k
          "Boltzmann's constant";
          constant Modelica.SIunits.Permittivity e0=Modelica.Constants.epsilon_0
          "Permittivity of free space";
          constant Modelica.SIunits.ElectricCharge q=1.6021892e-19
          "Electron charge";
          constant Modelica.SIunits.ElectronNumberDensity ni=1.45e16
          "Intrinsic carrier concentration";
          constant Real ks=11.8 "Dielectric constant of Si";
          constant Real GapC1(unit="V/K")=7.02E-4
          "First bandgap correction factor Silicon";
          constant Modelica.SIunits.Temperature GapC2=1108.0
          "Second bandgap correction factor Silicon";
          constant Real c0=0.0631353;
          constant Real c1=0.8013292;
          constant Real c2=-0.01110777;
          parameter Integer Level=3
          "Level of MOS model (check documentation window for details)";
          parameter Integer Type=1 "Type=1 for NMOS; Type=-1 for PMOS";
          parameter Modelica.SIunits.Temperature Tnom=300.15
          "Reference temperature";
          parameter Modelica.SIunits.Voltage VFB=0
          "Flat band voltage at reference temperature";
          parameter Modelica.SIunits.Transconductance KP=0
          "Transconductance parameter at reference temperature";
          parameter Real GAMMA(unit="V^0.5") = 0 "Body-effect parameter";
          parameter Modelica.SIunits.Voltage PHI=0
          "Surface inversion potential at reference temperature";
          parameter Real COX(unit="F/m2") = 0 "Specific capacitance of SiO2";
          parameter Real NFS(unit="1/m2") = 0 "Surface fast state density";
          parameter Modelica.SIunits.Length XJ=0 "Metallurgical junction depth";
          parameter Modelica.SIunits.Length L(min=0) = 1e-4 "Channel length";
          parameter Modelica.SIunits.Length W(min=0) = 1e-4 "Channel width";
          parameter Modelica.SIunits.Length LD=0 "Lateral diffusion";
          parameter Real XD(unit="m/V^0.5") = 0 "Depletion factor";
          parameter Real XQC=0 "Coefficient of channel charge share";
          parameter Real DELTA=0 "Width effect on threshold voltage";
          parameter Real ETA=0 "Static feedback on threshold voltage";
          parameter Real CGB0(unit="F/m") = 1e-12
          "Gate-bulk overlap capacitance per meter";
          parameter Modelica.SIunits.Voltage EG=1.11
          "Energy gap for temperature effect on saturation current at 0 K";
          Modelica.SIunits.Voltage vds "Drain-source voltage";
          Modelica.SIunits.Voltage vgs "Gate-source voltage";
          Modelica.SIunits.Voltage vbs "Bulk-source voltage";
          Modelica.SIunits.Voltage Vt "Thermal voltage";
          Modelica.SIunits.Voltage EGref "Energy gap at reference temperature";
          Modelica.SIunits.Voltage EGval "Energy gap at device temperature";
          Modelica.SIunits.Voltage PHIval
          "Surface inversion potential at device temperature";
          Modelica.SIunits.Voltage Vfb
          "Flat band voltage at device temperature";
          Modelica.SIunits.Voltage Vth "Threshold voltage";
          Modelica.SIunits.Voltage Von "Adjusted threshold voltage";
          Modelica.SIunits.Capacitance Cval
          "Gate capacitance between gate and bulk";

      protected
          parameter Modelica.SIunits.Permittivity es = ks*e0
          "Permittivity of Si";
          parameter Modelica.SIunits.Capacitance COXval = W*L*COX
          "Capacitance of SiO2 layer";
          parameter Real SIGMA = 8.15e-22*ETA/(COX*L^3);
          parameter Real Fn = es*DELTA*pi/(2*COX*W);
          Real sig "sign(vds)";
          Real RTemp "Temperature quotient";
          Real facTemp "Temperature coefficient";
          Modelica.SIunits.Length WD "Width of drain depletion region";
          Modelica.SIunits.Length WS "Width of source depletion region";
          Modelica.SIunits.Length Wp
          "Width of depleted region in flat source junction";
          Modelica.SIunits.Length Wc
          "Width of depleted region in source substrate junction";
          Modelica.SIunits.Transconductance KPval
          "Transconductance parameter at device temperature";
          Modelica.SIunits.Transconductance BETA
          "Transconductance at device temperature";
          Real CB "Specific capacitance of depletion area";
          Real N "Current emission coefficient";
          Real GAMMAval(unit="V^0.5") "Adjusted body effect parameter";
          Real Fs "Short channel working function";

          /* Auxiliary variables */
          Modelica.SIunits.Voltage Vbody;
          Real PHIroot(unit="V^0.5");
          Real Sarg(unit="V^0.5");
          Real Barg(unit="V^0.5");
          Real PHI3(unit="V^1.5");
          Real DSarg(unit="1/V^0.5") "der(Sarg,vbs)";
          Real DBarg(unit="1/V^0.5") "der(Barg,vbs)";
          Real DGAMMA(unit="1/V^0.5") "der(GAMMAval,vbs)";
          Real WSfac;
          Real WDfac;
          Real WpXJ "WP/XJ";
          Real WcXJ "WC/XJ";
      public
          Modelica.Blocks.Interfaces.RealInput u2
          "Connector of Real input signal 2"
            annotation (extent=[50,-110; 70,-90],rotation=90);
      public
          Modelica.Blocks.Interfaces.RealInput u1
          "Connector of Real input signal 1"
            annotation (extent=[-70,-110; -50,-90],
                                                 rotation=90);
          Modelica.Blocks.Interfaces.RealInput u[5]
          "Connector of Real input signals"
            annotation (extent=[-10,90; 10,110],    rotation=270);
        equation
          /* Parameter computations and temperature variations */
          RTemp = e2/Tnom;
          facTemp = RTemp*sqrt(RTemp);
          Vt = k*e2/q;
          EGref = EG - GapC1*Tnom*Tnom/(Tnom + GapC2);
          EGval = EG - GapC1*e2*e2/(e2 + GapC2);
          PHIval = RTemp*PHI + EGval - RTemp*EGref - 3*Vt*ln(RTemp);
          Vfb = VFB + Type*((PHIval - PHI)/2 + (EGref - EGval)/2);
          KPval = KP/facTemp;
          BETA = KPval*W/L;

          /* Determine first whether vds > 0 */
          if noEvent(u1 > 0) then
            /* Keep drain and source as they are */
            sig = 1;
            vds = u1;
            vgs = e1 + u2;
            vbs = u2;
          else
            /* Switch drain and source */
            sig = -1;
            vds = -u1;
            vgs = e1 - u1 + u2;
            vbs = u2 - u1;
          end if;

          /* Compute some useful quantities */
          PHIroot = sqrt(PHIval);
          PHI3 = PHIval*PHIroot;

         /* Compute Sarg and Barg and their derivatives */
          if Level==0 then
            Sarg = 0;
            DSarg = 0;
            Barg = 0;
            DBarg = 0;
          elseif Level==1 then
            Sarg = if noEvent(vbs <= 0) then sqrt(PHIval - vbs) else max({0,PHIroot - vbs/(2*PHIroot)});
            DSarg = 0;
            Barg = 0;
            DBarg = 0;
          elseif Level==2 or Level==4 then
            Sarg = if vbs <= 0 then sqrt(PHIval - vbs) else PHIroot/(1 + vbs/(2*PHIval));
            DSarg = if vbs <= 0 then -0.5/Sarg else -0.5*Sarg*Sarg/PHI3;
            Barg = if vbs < vds then sqrt(PHIval + vds - vbs) else PHIroot/(1 + (vbs - vds)/(2*PHIval));
            DBarg = if vbs < vds then -0.5/Barg else -0.5*Barg*Barg/PHI3;
          elseif Level==3 then
            Sarg = if vbs <= 0 then sqrt(PHIval - vbs) else PHIroot/(1 + vbs/(2*PHIval));
            DSarg = 0;
            Barg = 0;
            DBarg = 0;
          end if;

          /* For Level=2 and Level=4 */
          WS = XD*Sarg;
          WD = XD*Barg;
          WSfac =  if XJ > 0 then sqrt(1 + 2*WS/XJ) else 0;
          WDfac =  if XJ > 0 then sqrt(1 + 2*WD/XJ) else 0;
          GAMMAval = GAMMA*(if XJ > 0 then 1 - 0.5*XJ*(WSfac + WDfac - 2)/L else 1);
          DGAMMA = GAMMA*(if XJ > 0 then -0.5*XD*(DSarg/WSfac + DBarg/WDfac)/L else 0);

          /* For Level=3 */
          Wp = Sarg*XD;
          WpXJ = if XJ > 0 then Wp/XJ else 0;
          WcXJ = c0 + c1*WpXJ + c2*WpXJ*WpXJ;
          Wc = WcXJ*XJ;
          Fs = if XJ > 0 then 1 - ((LD + Wc)*sqrt(1 - (Wp/(XJ + Wp))^2) - LD)/L else 1;
          Vbody =  GAMMA*Fs*Sarg + Fn*Sarg*Sarg;

         /* Compute Vth and Von */
          if Level==0 then
            Vth = 0;
            CB = 0;
            N = 0;
            Von = 0;
          elseif Level==1 then
            Vth = Vfb + Type*(PHIval + GAMMA*Sarg);
            CB = 0;
            N = 0;
            Von = Type*Vth;
          elseif Level==2 or Level==4 then
            Vth = Vfb + Type*(PHIval + (GAMMA - GAMMAval)*sqrt(PHIval) + GAMMAval*Sarg + Fn*Sarg*Sarg/2);
            CB = Fn - GAMMAval*DSarg - DGAMMA*Sarg;
            N = if NFS > 0 then 1 + q*NFS/COX + CB else 1;
            Von =  if NFS > 0 then Type*Vth + N*Vt else Type*Vth;
          elseif Level==3 then
            Vth =  Vfb + Type*(PHIval - SIGMA*vds + Vbody);
            CB = if noEvent(Vbody > 0 and Sarg > 0) then Vbody/(2*Sarg*Sarg) else 0;
            N = 1 + q*NFS/COX + CB;
            Von = if NFS > 0 then Type*Vth + N*Vt else Type*Vth;
          end if;

          /* Compute gate capacitance between gate and bulk */
          if Level==0 or not CGB0 > 0 then
            Cval = 1e-16;
          else
            Cval = L*CGB0 + (if noEvent(vgs < Von - PHIval) then COXval else
                             if noEvent(vgs >= Von - PHIval and vgs < Von) then COXval*(Von - vgs)/PHIval else 0);
          end if;

          /* Capacitance equation */
          f1 = Cval*der(e1);

          /* Computation of heat flow */
          f2 = 0;
        annotation (
          Icon(
              Text(
                extent=[-88,70; 26,-76],
                style(color=0),
              string="Cgb"),
              Text(
                extent=[-70,-76; -50,-88],
                style(color=3, rgbcolor={0,0,255}),
                string="vds"),
              Text(
                extent=[50,-76; 70,-88],
                style(color=3, rgbcolor={0,0,255}),
                string="vbs")),
          Diagram(
              Text(
                extent=[-88,72; 26,-74],
                style(color=0),
              string="Cgb")),
          Documentation(info="<html>
The Spice-style gate capacitances for MOSFETs are non-linear capacitors that use different formulae to compute the capacitance value in the four different regions: <i>accumulation</i>, <i>depletion</i>, <i>saturation</i>, and the <i>linear region</i>.  Different Spice dialects vary in the formulae they use for this purpose.  Many Spice dialects actually don't use a formula for the gate capacitances at all, but rather compute the electric charges stored in these regions directly, which is conceptually cleaner.  However, that approach is computationally cumbersome, as it leads to an awkward algebraic loop [1].  Thus, we chose to compute the gate capacitances directly, and use a (physically incorrect) approximate non-linear capacitor model.  The numerical error should remain small, as the time constants associated with temperature variation are much larger than those associated with electrical phenomena. <p>

This particular model uses the space charge formula advocated in [2]. <p>

<hr> <p>

<pre>
Parameters:

 Level:   MOSFET modeling level (default value = 3)
            Level=0: Static injection model
            Level=1: Shichman-Hodges model
            Level=2: Grove-Frohman model
            Level=3: Empirical model
            Level=4: Simplified Grove-Frohman model

 Type:    Type of MOSFET (default value = 1)
            Type = +1:  NMOS
            Type = -1:  PMOS

 Tnom:    Reference temperature (default value = 300.15 K)

 VFB:     Flat band voltage at reference temperature (default value = 0 Volt)

 KP:      Transconductance parameter at reference temperature (default value = 0 Amp/Volt<sup>2</sup>)

 GAMMA:   Body-effect parameter (default value = Volt<sup>0.5</sup>)

 PHI:     Surface inversion potential at reference temperature (default value = 0 Volt)

 COX:     Specific capacitance of SiO2 (default value = 0 F/m<sup>2</sup>)

 NFS:     Surface fast state density (default value = 0 1/m<sup>2</sup>)

 XJ:      Metallurgical junction depth (default value = 0 m)

 L:       Channel length (default value = 1e-4 m)

 W:       Channel width (default value = 1e-4 m)

 LD:      Lateral diffusion (default value = 0 m)

 XD:      Depletion factor (default value = 0 m/Volt<sup>0.5</sup>)

 XQC:     Coefficient of channel charge share (default value = 0)
            <font color=blue><b>XQC &lt;= 0.5: Ward-Dutton model of gate capacitance computation</b></font>
            <font color=blue><b>XQC &gt;  0.5: Meyer model of gate capacitance computation</b></font>

 DELTA:   Width effect on threshold voltage (default value = 0)

 ETA:     Static feedback on threshold voltage (default value = 1)

 CGB0:    Gate-bulk overlap capacitance per meter (default value = 1e-12 F/m)

 EG:      Energy gap for temperature effect on saturation current (default value = 1.11 Volt)
</pre> <p>

<hr> <p>

<b>References:</b> <p>

<ol>
<li> Cellier, F.E. (1991), <i>Continuous System Modeling</i>, Springer-Verlag, New York, pp. 224-225.
<li> Massobrio, G. and P. Antognetti (1993), <i>Semiconductor Device Modeling with Spice</i>,
     2<sup>nd</sup> edition, McGraw Hill, New York, pp.196-197.
</ol>
</html>
"));
        end Cgb;

        model Cgs
        "Spice-style gate capacitance between gate and source of MOSFETs"
          extends Interfaces.SpiceTwoPort;
          constant Real pi=Modelica.Constants.pi;
          constant Modelica.SIunits.Entropy k=Modelica.Constants.k
          "Boltzmann's constant";
          constant Modelica.SIunits.Permittivity e0=Modelica.Constants.epsilon_0
          "Permittivity of free space";
          constant Modelica.SIunits.ElectricCharge q=1.6021892e-19
          "Electron charge";
          constant Modelica.SIunits.ElectronNumberDensity ni=1.45e16
          "Intrinsic carrier concentration";
          constant Real ks=11.8 "Dielectric constant of Si";
          constant Real GapC1(unit="V/K")=7.02E-4
          "First bandgap correction factor Silicon";
          constant Modelica.SIunits.Temperature GapC2=1108.0
          "Second bandgap correction factor Silicon";
          constant Real c0=0.0631353;
          constant Real c1=0.8013292;
          constant Real c2=-0.01110777;
          parameter Integer Level=3
          "Level of MOS model (check documentation window for details)";
          parameter Integer Type=1 "Type=1 for NMOS; Type=-1 for PMOS";
          parameter Modelica.SIunits.Temperature Tnom=300.15
          "Reference temperature";
          parameter Modelica.SIunits.Voltage VFB=0
          "Flat band voltage at reference temperature";
          parameter Modelica.SIunits.Transconductance KP=0
          "Transconductance parameter at reference temperature";
          parameter Real GAMMA(unit="V^0.5") = 0 "Body-effect parameter";
          parameter Modelica.SIunits.Voltage PHI=0
          "Surface inversion potential at reference temperature";
          parameter Real COX(unit="F/m2") = 0 "Specific capacitance of SiO2";
          parameter Real NFS(unit="1/m2") = 0 "Surface fast state density";
          parameter Modelica.SIunits.Length XJ=0 "Metallurgical junction depth";
          parameter Modelica.SIunits.Length L(min=0) = 1e-4 "Channel length";
          parameter Modelica.SIunits.Length W(min=0) = 1e-4 "Channel width";
          parameter Modelica.SIunits.Length LD=0 "Lateral diffusion";
          parameter Real XD(unit="m/V^0.5") = 0 "Depletion factor";
          parameter Real XQC=0 "Coefficient of channel charge share";
          parameter Real DELTA=0 "Width effect on threshold voltage";
          parameter Real ETA=0 "Static feedback on threshold voltage";
          parameter Real CGS0(unit="F/m") = 1e-12
          "Gate-source overlap capacitance per meter";
          parameter Modelica.SIunits.Voltage EG=1.11
          "Energy gap for temperature effect on saturation current at 0 K";
          Modelica.SIunits.Voltage vds "Drain-source voltage";
          Modelica.SIunits.Voltage vgs "Gate-source voltage";
          Modelica.SIunits.Voltage vbs "Bulk-source voltage";
          Modelica.SIunits.Voltage Vt "Thermal voltage";
          Modelica.SIunits.Voltage EGref "Energy gap at reference temperature";
          Modelica.SIunits.Voltage EGval "Energy gap at device temperature";
          Modelica.SIunits.Voltage PHIval
          "Surface inversion potential at device temperature";
          Modelica.SIunits.Voltage Vfb
          "Flat band voltage at device temperature";
          Modelica.SIunits.Voltage Vth "Threshold voltage";
          Modelica.SIunits.Voltage Von "Adjusted threshold voltage";
          Modelica.SIunits.Capacitance Cval
          "Gate capacitance between gate and bulk";

      protected
          parameter Modelica.SIunits.Permittivity es = ks*e0
          "Permittivity of Si";
          parameter Modelica.SIunits.Capacitance COXval = W*L*COX
          "Capacitance of SiO2 layer";
          parameter Real SIGMA = 8.15e-22*ETA/(COX*L^3);
          parameter Real Fn = es*DELTA*pi/(2*COX*W);
          Real sig "sign(vds)";
          Real RTemp "Temperature quotient";
          Real facTemp "Temperature coefficient";
          Modelica.SIunits.Length WD "Width of drain depletion region";
          Modelica.SIunits.Length WS "Width of source depletion region";
          Modelica.SIunits.Length Wp
          "Width of depleted region in flat source junction";
          Modelica.SIunits.Length Wc
          "Width of depleted region in source substrate junction";
          Modelica.SIunits.Transconductance KPval
          "Transconductance parameter at device temperature";
          Modelica.SIunits.Transconductance BETA
          "Transconductance at device temperature";
          Real CB "Specific capacitance of depletion area";
          Real N "Current emission coefficient";
          Real GAMMAval(unit="V^0.5") "Adjusted body effect parameter";
          Real Fs "Short channel working function";

          /* Auxiliary variables */
          Modelica.SIunits.Voltage Vbody;
          Real PHIroot(unit="V^0.5");
          Real Sarg(unit="V^0.5");
          Real Barg(unit="V^0.5");
          Real PHI3(unit="V^1.5");
          Real DSarg(unit="1/V^0.5") "der(Sarg,vbs)";
          Real DBarg(unit="1/V^0.5") "der(Barg,vbs)";
          Real DGAMMA(unit="1/V^0.5") "der(GAMMAval,vbs)";
          Real WSfac;
          Real WDfac;
          Real WpXJ "WP/XJ";
          Real WcXJ "WC/XJ";
      public
          Modelica.Blocks.Interfaces.RealInput u1
          "Connector of Real input signal 1"
            annotation (extent=[-70,-110; -50,-90],
                                                 rotation=90);
      public
          Modelica.Blocks.Interfaces.RealInput u2
          "Connector of Real input signal 2"
            annotation (extent=[50,-110; 70,-90],rotation=90);
        annotation (
          Icon(
              Text(
                extent=[-88,70; 26,-76],
                style(color=0),
              string="Cgs"),
              Text(
                extent=[-70,-76; -50,-88],
                style(color=3, rgbcolor={0,0,255}),
                string="vds"),
              Text(
                extent=[50,-76; 70,-88],
                style(color=3, rgbcolor={0,0,255}),
                string="vbs")),
          Diagram(
              Text(
                extent=[-88,70; 26,-76],
                style(color=0),
              string="Cgs")),
          Documentation(info="<html>
The Spice-style gate capacitances for MOSFETs are non-linear capacitors that use different formulae to compute the capacitance value in the four different regions: <i>accumulation</i>, <i>depletion</i>, <i>saturation</i>, and the <i>linear region</i>.  Different Spice dialects vary in the formulae they use for this purpose.  Many Spice dialects actually don't use a formula for the gate capacitances at all, but rather compute the electric charges stored in these regions directly, which is conceptually cleaner.  However, that approach is computationally cumbersome, as it leads to an awkward algebraic loop [1].  Thus, we chose to compute the gate capacitances directly, and use a (physically incorrect) approximate non-linear capacitor model.  The numerical error should remain small, as the time constants associated with temperature variation are much larger than those associated with electrical phenomena. <p>

This particular model uses the space charge formula advocated in [2]. <p>

<hr> <p>

<pre>
Parameters:

 Level:   MOSFET modeling level (default value = 3)
            Level=0: Static injection model
            Level=1: Shichman-Hodges model
            Level=2: Grove-Frohman model
            Level=3: Empirical model
            Level=4: Simplified Grove-Frohman model

 Type:    Type of MOSFET (default value = 1)
            Type = +1:  NMOS
            Type = -1:  PMOS

 Tnom:    Reference temperature (default value = 300.15 K)

 VFB:     Flat band voltage at reference temperature (default value = 0 Volt)

 KP:      Transconductance parameter at reference temperature (default value = 0 Amp/Volt<sup>2</sup>)

 GAMMA:   Body-effect parameter (default value = Volt<sup>0.5</sup>)

 PHI:     Surface inversion potential at reference temperature (default value = 0 Volt)

 COX:     Specific capacitance of SiO2 (default value = 0 F/m<sup>2</sup>)

 NFS:     Surface fast state density (default value = 0 1/m<sup>2</sup>)

 XJ:      Metallurgical junction depth (default value = 0 m)

 L:       Channel length (default value = 1e-4 m)

 W:       Channel width (default value = 1e-4 m)

 LD:      Lateral diffusion (default value = 0 m)

 XD:      Depletion factor (default value = 0 m/Volt<sup>0.5</sup>)

 XQC:     Coefficient of channel charge share (default value = 0)
            <font color=blue><b>XQC &lt;= 0.5: Ward-Dutton model of gate capacitance computation</b></font>
            <font color=blue><b>XQC &gt;  0.5: Meyer model of gate capacitance computation</b></font>

 DELTA:   Width effect on threshold voltage (default value = 0)

 ETA:     Static feedback on threshold voltage (default value = 1)

 CGS0:    Gate-source overlap capacitance per meter (default value = 1e-12 F/m)

 EG:      Energy gap for temperature effect on saturation current (default value = 1.11 Volt)
</pre> <p>

<hr> <p>

<b>References:</b> <p>

<ol>
<li> Cellier, F.E. (1991), <i>Continuous System Modeling</i>, Springer-Verlag, New York, pp. 224-225.
<li> Massobrio, G. and P. Antognetti (1993), <i>Semiconductor Device Modeling with Spice</i>,
     2<sup>nd</sup> edition, McGraw Hill, New York, pp.196-197.
</ol>
</html>
"));
          Modelica.Blocks.Interfaces.RealInput u[5]
          "Connector of Real input signals"
            annotation (extent=[-10,90; 10,110],    rotation=270);
        equation
          /* Parameter computations and temperature variations */
          RTemp = e2/Tnom;
          facTemp = RTemp*sqrt(RTemp);
          Vt = k*e2/q;
          EGref = EG - GapC1*Tnom*Tnom/(Tnom + GapC2);
          EGval = EG - GapC1*e2*e2/(e2 + GapC2);
          PHIval = RTemp*PHI + EGval - RTemp*EGref - 3*Vt*ln(RTemp);
          Vfb = VFB + Type*((PHIval - PHI)/2 + (EGref - EGval)/2);
          KPval = KP/facTemp;
          BETA = KPval*W/L;

          /* Determine first whether vds > 0 */
          if noEvent(u1 > 0) then
            /* Keep drain and source as they are */
            sig = 1;
            vds = u1;
            vgs = e1 + u2;
            vbs = u2;
          else
            /* Switch drain and source */
            sig = -1;
            vds = -u1;
            vgs = e1 - u1;
            vbs = u2 - u1;
          end if;

          /* Compute some useful quantities */
          PHIroot = sqrt(PHIval);
          PHI3 = PHIval*PHIroot;

         /* Compute Sarg and Barg and their derivatives */
          if Level==0 then
            Sarg = 0;
            DSarg = 0;
            Barg = 0;
            DBarg = 0;
          elseif Level==1 then
            Sarg = if noEvent(vbs <= 0) then sqrt(PHIval - vbs) else max({0,PHIroot - vbs/(2*PHIroot)});
            DSarg = 0;
            Barg = 0;
            DBarg = 0;
          elseif Level==2 or Level==4 then
            Sarg = if vbs <= 0 then sqrt(PHIval - vbs) else PHIroot/(1 + vbs/(2*PHIval));
            DSarg = if vbs <= 0 then -0.5/Sarg else -0.5*Sarg*Sarg/PHI3;
            Barg = if vbs < vds then sqrt(PHIval + vds - vbs) else PHIroot/(1 + (vbs - vds)/(2*PHIval));
            DBarg = if vbs < vds then -0.5/Barg else -0.5*Barg*Barg/PHI3;
          elseif Level==3 then
            Sarg = if vbs <= 0 then sqrt(PHIval - vbs) else PHIroot/(1 + vbs/(2*PHIval));
            DSarg = 0;
            Barg = 0;
            DBarg = 0;
          end if;

          /* For Level=2 and Level=4 */
          WS = XD*Sarg;
          WD = XD*Barg;
          WSfac =  if XJ > 0 then sqrt(1 + 2*WS/XJ) else 0;
          WDfac =  if XJ > 0 then sqrt(1 + 2*WD/XJ) else 0;
          GAMMAval = GAMMA*(if XJ > 0 then 1 - 0.5*XJ*(WSfac + WDfac - 2)/L else 1);
          DGAMMA = GAMMA*(if XJ > 0 then -0.5*XD*(DSarg/WSfac + DBarg/WDfac)/L else 0);

          /* For Level=3 */
          Wp = Sarg*XD;
          WpXJ = if XJ > 0 then Wp/XJ else 0;
          WcXJ = c0 + c1*WpXJ + c2*WpXJ*WpXJ;
          Wc = WcXJ*XJ;
          Fs = if XJ > 0 then 1 - ((LD + Wc)*sqrt(1 - (Wp/(XJ + Wp))^2) - LD)/L else 1;
          Vbody =  GAMMA*Fs*Sarg + Fn*Sarg*Sarg;

         /* Compute Vth and Von */
          if Level==0 then
            Vth = 0;
            CB = 0;
            N = 0;
            Von = 0;
          elseif Level==1 then
            Vth = Vfb + Type*(PHIval + GAMMA*Sarg);
            CB = 0;
            N = 0;
            Von = Type*Vth;
          elseif Level==2 or Level==4 then
            Vth = Vfb + Type*(PHIval + (GAMMA - GAMMAval)*sqrt(PHIval) + GAMMAval*Sarg + Fn*Sarg*Sarg/2);
            CB = Fn - GAMMAval*DSarg - DGAMMA*Sarg;
            N = if NFS > 0 then 1 + q*NFS/COX + CB else 1;
            Von =  if NFS > 0 then Type*Vth + N*Vt else Type*Vth;
          elseif Level==3 then
            Vth =  Vfb + Type*(PHIval - SIGMA*vds + Vbody);
            CB = if noEvent(Vbody > 0 and Sarg > 0) then Vbody/(2*Sarg*Sarg) else 0;
            N = 1 + q*NFS/COX + CB;
            Von = if NFS > 0 then Type*Vth + N*Vt else Type*Vth;
          end if;

          /* Compute gate capacitance between gate and bulk */
          if Level==0 or not CGS0 > 0 then
            Cval = 1e-16;
          else
               if noEvent(sig > 0) then
                  Cval = W*CGS0 + (if noEvent(vgs < Von - PHIval) then 0 else
                                   if noEvent(vgs >= Von - PHIval and vgs < Von) then 2*COXval*(1 - (Von - vgs)/PHIval)/3 else
                                   if noEvent(vgs >= Von) and noEvent(vgs < Von + vds) then 2*COXval/3 else
                                      2*COXval*(1 - ((vgs - vds - Von)/(2*(vgs - Von) - vds))^2)/3);

               else
                  Cval = W*CGS0 + (if noEvent(vgs < Von + vds) then 0 else
                         2*COXval*(1 - ((vgs - Von)/(2*(vgs - Von) - vds))^2)/3);
               end if;
          end if;

          /* Capacitance equation */
          f1 = Cval*der(e1);

          /* Computation of heat flow */
          f2 = 0;
        end Cgs;

        model Cgd
        "Spice-style gate capacitance between gate and drain of MOSFETs"
          extends Interfaces.SpiceTwoPort;
          constant Real pi=Modelica.Constants.pi;
          constant Modelica.SIunits.Entropy k=Modelica.Constants.k
          "Boltzmann's constant";
          constant Modelica.SIunits.Permittivity e0=Modelica.Constants.epsilon_0
          "Permittivity of free space";
          constant Modelica.SIunits.ElectricCharge q=1.6021892e-19
          "Electron charge";
          constant Modelica.SIunits.ElectronNumberDensity ni=1.45e16
          "Intrinsic carrier concentration";
          constant Real ks=11.8 "Dielectric constant of Si";
          constant Real GapC1(unit="V/K")=7.02E-4
          "First bandgap correction factor Silicon";
          constant Modelica.SIunits.Temperature GapC2=1108.0
          "Second bandgap correction factor Silicon";
          constant Real c0=0.0631353;
          constant Real c1=0.8013292;
          constant Real c2=-0.01110777;
          parameter Integer Level=3
          "Level of MOS model (check documentation window for details)";
          parameter Integer Type=1 "Type=1 for NMOS; Type=-1 for PMOS";
          parameter Modelica.SIunits.Temperature Tnom=300.15
          "Reference temperature";
          parameter Modelica.SIunits.Voltage VFB=0
          "Flat band voltage at reference temperature";
          parameter Modelica.SIunits.Transconductance KP=0
          "Transconductance parameter at reference temperature";
          parameter Real GAMMA(unit="V^0.5") = 0 "Body-effect parameter";
          parameter Modelica.SIunits.Voltage PHI=0
          "Surface inversion potential at reference temperature";
          parameter Real COX(unit="F/m2") = 0 "Specific capacitance of SiO2";
          parameter Real NFS(unit="1/m2") = 0 "Surface fast state density";
          parameter Modelica.SIunits.Length XJ=0 "Metallurgical junction depth";
          parameter Modelica.SIunits.Length L(min=0) = 1e-4 "Channel length";
          parameter Modelica.SIunits.Length W(min=0) = 1e-4 "Channel width";
          parameter Modelica.SIunits.Length LD=0 "Lateral diffusion";
          parameter Real XD(unit="m/V^0.5") = 0 "Depletion factor";
          parameter Real XQC=0 "Coefficient of channel charge share";
          parameter Real DELTA=0 "Width effect on threshold voltage";
          parameter Real ETA=0 "Static feedback on threshold voltage";
          parameter Real CGD0(unit="F/m") = 1e-12
          "Gate-drain overlap capacitance per meter";
          parameter Modelica.SIunits.Voltage EG=1.11
          "Energy gap for temperature effect on saturation current at 0 K";
          Modelica.SIunits.Voltage vds "Drain-source voltage";
          Modelica.SIunits.Voltage vgs "Gate-source voltage";
          Modelica.SIunits.Voltage vbs "Bulk-source voltage";
          Modelica.SIunits.Voltage Vt "Thermal voltage";
          Modelica.SIunits.Voltage EGref "Energy gap at reference temperature";
          Modelica.SIunits.Voltage EGval "Energy gap at device temperature";
          Modelica.SIunits.Voltage PHIval
          "Surface inversion potential at device temperature";
          Modelica.SIunits.Voltage Vfb
          "Flat band voltage at device temperature";
          Modelica.SIunits.Voltage Vth "Threshold voltage";
          Modelica.SIunits.Voltage Von "Adjusted threshold voltage";
          Modelica.SIunits.Capacitance Cval
          "Gate capacitance between gate and bulk";

      protected
          parameter Modelica.SIunits.Permittivity es = ks*e0
          "Permittivity of Si";
          parameter Modelica.SIunits.Capacitance COXval = W*L*COX
          "Capacitance of SiO2 layer";
          parameter Real SIGMA = 8.15e-22*ETA/(COX*L^3);
          parameter Real Fn = es*DELTA*pi/(2*COX*W);
          Real sig "sign(vds)";
          Real RTemp "Temperature quotient";
          Real facTemp "Temperature coefficient";
          Modelica.SIunits.Length WD "Width of drain depletion region";
          Modelica.SIunits.Length WS "Width of source depletion region";
          Modelica.SIunits.Length Wp
          "Width of depleted region in flat source junction";
          Modelica.SIunits.Length Wc
          "Width of depleted region in source substrate junction";
          Modelica.SIunits.Transconductance KPval
          "Transconductance parameter at device temperature";
          Modelica.SIunits.Transconductance BETA
          "Transconductance at device temperature";
          Real CB "Specific capacitance of depletion area";
          Real N "Current emission coefficient";
          Real GAMMAval(unit="V^0.5") "Adjusted body effect parameter";
          Real Fs "Short channel working function";

          /* Auxiliary variables */
          Modelica.SIunits.Voltage Vbody;
          Real PHIroot(unit="V^0.5");
          Real Sarg(unit="V^0.5");
          Real Barg(unit="V^0.5");
          Real PHI3(unit="V^1.5");
          Real DSarg(unit="1/V^0.5") "der(Sarg,vbs)";
          Real DBarg(unit="1/V^0.5") "der(Barg,vbs)";
          Real DGAMMA(unit="1/V^0.5") "der(GAMMAval,vbs)";
          Real WSfac;
          Real WDfac;
          Real WpXJ "WP/XJ";
          Real WcXJ "WC/XJ";
        annotation (
          Documentation(info="<html>
The Spice-style gate capacitances for MOSFETs are non-linear capacitors that use different formulae to compute the capacitance value in the four different regions: <i>accumulation</i>, <i>depletion</i>, <i>saturation</i>, and the <i>linear region</i>.  Different Spice dialects vary in the formulae they use for this purpose.  Many Spice dialects actually don't use a formula for the gate capacitances at all, but rather compute the electric charges stored in these regions directly, which is conceptually cleaner.  However, that approach is computationally cumbersome, as it leads to an awkward algebraic loop [1].  Thus, we chose to compute the gate capacitances directly, and use a (physically incorrect) approximate non-linear capacitor model.  The numerical error should remain small, as the time constants associated with temperature variation are much larger than those associated with electrical phenomena. <p>

This particular model uses the space charge formula advocated in [2]. <p>

<hr> <p>

<pre>
Parameters:

 Level:   MOSFET modeling level (default value = 3)
            Level=0: Static injection model
            Level=1: Shichman-Hodges model
            Level=2: Grove-Frohman model
            Level=3: Empirical model
            Level=4: Simplified Grove-Frohman model

 Type:    Type of MOSFET (default value = 1)
            Type = +1:  NMOS
            Type = -1:  PMOS

 Tnom:    Reference temperature (default value = 300.15 K)

 VFB:     Flat band voltage at reference temperature (default value = 0 Volt)

 KP:      Transconductance parameter at reference temperature (default value = 0 Amp/Volt<sup>2</sup>)

 GAMMA:   Body-effect parameter (default value = Volt<sup>0.5</sup>)

 PHI:     Surface inversion potential at reference temperature (default value = 0 Volt)

 COX:     Specific capacitance of SiO2 (default value = 0 F/m<sup>2</sup>)

 NFS:     Surface fast state density (default value = 0 1/m<sup>2</sup>)

 XJ:      Metallurgical junction depth (default value = 0 m)

 L:       Channel length (default value = 1e-4 m)

 W:       Channel width (default value = 1e-4 m)

 LD:      Lateral diffusion (default value = 0 m)

 XD:      Depletion factor (default value = 0 m/Volt<sup>0.5</sup>)

 XQC:     Coefficient of channel charge share (default value = 0)
            <font color=blue><b>XQC &lt;= 0.5: Ward-Dutton model of gate capacitance computation</b></font>
            <font color=blue><b>XQC &gt;  0.5: Meyer model of gate capacitance computation</b></font>

 DELTA:   Width effect on threshold voltage (default value = 0)

 ETA:     Static feedback on threshold voltage (default value = 1)

 CGD0:    Gate-drain overlap capacitance per meter (default value = 1e-12 F/m)

 EG:      Energy gap for temperature effect on saturation current (default value = 1.11 Volt)
</pre> <p>

<hr> <p>

<b>References:</b> <p>

<ol>
<li> Cellier, F.E. (1991), <i>Continuous System Modeling</i>, Springer-Verlag, New York, pp. 224-225.
<li> Massobrio, G. and P. Antognetti (1993), <i>Semiconductor Device Modeling with Spice</i>,
     2<sup>nd</sup> edition, McGraw Hill, New York, pp.196-197.
</ol>
</html>
"),       Icon(
              Text(
                extent=[-88,70; 26,-76],
                style(color=0),
              string="Cgd"),
              Text(
                extent=[-70,-76; -50,-88],
                style(color=3, rgbcolor={0,0,255}),
                string="vds"),
              Text(
                extent=[50,-76; 70,-88],
                style(color=3, rgbcolor={0,0,255}),
                string="vbs")),
          Diagram(
              Text(
                extent=[-88,70; 26,-76],
                style(color=0),
              string="Cgd")));
      public
          Modelica.Blocks.Interfaces.RealInput u1
          "Connector of Real input signal 1"
            annotation (extent=[-70,-110; -50,-90],
                                                 rotation=90);
      public
          Modelica.Blocks.Interfaces.RealInput u2
          "Connector of Real input signal 2"
            annotation (extent=[50,-110; 70,-90],rotation=90);
          Modelica.Blocks.Interfaces.RealInput u[5]
          "Connector of Real input signals"
            annotation (extent=[-10,90; 10,110],    rotation=270);
        equation
          /* Parameter computations and temperature variations */
          RTemp = e2/Tnom;
          facTemp = RTemp*sqrt(RTemp);
          Vt = k*e2/q;
          EGref = EG - GapC1*Tnom*Tnom/(Tnom + GapC2);
          EGval = EG - GapC1*e2*e2/(e2 + GapC2);
          PHIval = RTemp*PHI + EGval - RTemp*EGref - 3*Vt*ln(RTemp);
          Vfb = VFB + Type*((PHIval - PHI)/2 + (EGref - EGval)/2);
          KPval = KP/facTemp;
          BETA = KPval*W/L;

          /* Determine first whether vds > 0 */
          if noEvent(u1 > 0) then
            /* Keep drain and source as they are */
            sig = 1;
            vds = u1;
            vgs = e1 + u1;
            vbs = u2;
          else
            /* Switch drain and source */
            sig = -1;
            vds = -u1;
            vgs = e1 - u1 + u2;
            vbs = u2 - u1;
          end if;

          /* Compute some useful quantities */
          PHIroot = sqrt(PHIval);
          PHI3 = PHIval*PHIroot;

         /* Compute Sarg and Barg and their derivatives */
          if Level==0 then
            Sarg = 0;
            DSarg = 0;
            Barg = 0;
            DBarg = 0;
          elseif Level==1 then
            Sarg = if noEvent(vbs <= 0) then sqrt(PHIval - vbs) else max({0,PHIroot - vbs/(2*PHIroot)});
            DSarg = 0;
            Barg = 0;
            DBarg = 0;
          elseif Level==2 or Level==4 then
            Sarg = if vbs <= 0 then sqrt(PHIval - vbs) else PHIroot/(1 + vbs/(2*PHIval));
            DSarg = if vbs <= 0 then -0.5/Sarg else -0.5*Sarg*Sarg/PHI3;
            Barg = if vbs < vds then sqrt(PHIval + vds - vbs) else PHIroot/(1 + (vbs - vds)/(2*PHIval));
            DBarg = if vbs < vds then -0.5/Barg else -0.5*Barg*Barg/PHI3;
          elseif Level==3 then
            Sarg = if vbs <= 0 then sqrt(PHIval - vbs) else PHIroot/(1 + vbs/(2*PHIval));
            DSarg = 0;
            Barg = 0;
            DBarg = 0;
          end if;

          /* For Level=2 and Level=4 */
          WS = XD*Sarg;
          WD = XD*Barg;
          WSfac =  if XJ > 0 then sqrt(1 + 2*WS/XJ) else 0;
          WDfac =  if XJ > 0 then sqrt(1 + 2*WD/XJ) else 0;
          GAMMAval = GAMMA*(if XJ > 0 then 1 - 0.5*XJ*(WSfac + WDfac - 2)/L else 1);
          DGAMMA = GAMMA*(if XJ > 0 then -0.5*XD*(DSarg/WSfac + DBarg/WDfac)/L else 0);

          /* For Level=3 */
          Wp = Sarg*XD;
          WpXJ = if XJ > 0 then Wp/XJ else 0;
          WcXJ = c0 + c1*WpXJ + c2*WpXJ*WpXJ;
          Wc = WcXJ*XJ;
          Fs = if XJ > 0 then 1 - ((LD + Wc)*sqrt(1 - (Wp/(XJ + Wp))^2) - LD)/L else 1;
          Vbody =  GAMMA*Fs*Sarg + Fn*Sarg*Sarg;

         /* Compute Vth and Von */
          if Level==0 then
            Vth = 0;
            CB = 0;
            N = 0;
            Von = 0;
          elseif Level==1 then
            Vth = Vfb + Type*(PHIval + GAMMA*Sarg);
            CB = 0;
            N = 0;
            Von = Type*Vth;
          elseif Level==2 or Level==4 then
            Vth = Vfb + Type*(PHIval + (GAMMA - GAMMAval)*sqrt(PHIval) + GAMMAval*Sarg + Fn*Sarg*Sarg/2);
            CB = Fn - GAMMAval*DSarg - DGAMMA*Sarg;
            N = if NFS > 0 then 1 + q*NFS/COX + CB else 1;
            Von =  if NFS > 0 then Type*Vth + N*Vt else Type*Vth;
          elseif Level==3 then
            Vth =  Vfb + Type*(PHIval - SIGMA*vds + Vbody);
            CB = if noEvent(Vbody > 0 and Sarg > 0) then Vbody/(2*Sarg*Sarg) else 0;
            N = 1 + q*NFS/COX + CB;
            Von = if NFS > 0 then Type*Vth + N*Vt else Type*Vth;
          end if;

          /* Compute gate capacitance between gate and bulk */
          if Level==0 or not CGD0 > 0 then
             Cval = 1e-16;
          else
             if noEvent(sig > 0) then
                   Cval = W*CGD0 + (if noEvent(vgs < Von + vds) then 0 else
                                    2*COXval*(1 - ((vgs - Von)/(2*(vgs - Von) - vds))^2)/3);
             else
                   Cval = W*CGD0 + (if noEvent(vgs < Von - PHIval) then 0 else
                                    if noEvent(vgs >= Von - PHIval and vgs < Von) then 2*COXval*(1 - (Von - vgs)/PHIval)/3 else
                                    if noEvent(vgs >= Von) and noEvent(vgs < Von + vds) then 2*COXval/3 else
                                    2*COXval*(1 - ((vgs - vds - Von)/(2*(vgs - Von) - vds))^2)/3);
             end if;
          end if;

          /* Capacitance equation */
          f1 = Cval*der(e1);

          /* Computation of heat flow */
          f2 = 0;
        end Cgd;

        model DjM "Spice-style junction diode model for MOSFETs"
          extends Interfaces.SpiceTwoPort;
          constant Modelica.SIunits.Entropy k=Modelica.Constants.k
          "Boltzmann's constant";
          constant Modelica.SIunits.ElectricCharge q=1.6021892e-19
          "Electron charge";
          constant Real GapC1(unit="V/K")=7.02E-4
          "First bandgap correction factor Silicon";
          constant Modelica.SIunits.Temperature GapC2=1108.0
          "Second bandgap correction factor Silicon";
          parameter Integer Level=3
          "MOSFET modeling level (check documentation window for details)";
          parameter Modelica.SIunits.Current IS=0
          "Saturation current at reference temperature";
          parameter Modelica.SIunits.Voltage EG=1.11
          "Energy gap for temperature effect on saturation current";
          parameter Real EMin=-100
          "if x < EMin, the exp(x) function is linearized";
          parameter Real EMax=40
          "if x > EMax, the exp(x) function is linearized";
          Modelica.SIunits.Voltage Vt "Thermal voltage";
          Modelica.SIunits.Current ISval
          "Saturation current at device temperature";
          Modelica.SIunits.Voltage EGnom
          "Activation energy at reference temperature";
          Modelica.SIunits.Voltage EGval
          "Activation energy at device temperature";

      protected
          parameter Real ExMin = exp(EMin);
          parameter Real ExMax = exp(EMax);
          Real RTemp "Temperature quotient";
          Real et;
        annotation (
          Diagram(
              Line(points=[-80,0; 80,0],   style(color=10)),
              Polygon(points=[70,4; 80,0; 70,-4; 70,4],     style(color=10,
                    fillColor=10)),
              Line(points=[0,80; 0,-80],   style(
                  color=10,
                  fillColor=10,
                  fillPattern=1)),
              Polygon(points=[-4,70; 0,80; 4,70; -4,70],     style(
                  color=10,
                  fillColor=10,
                  fillPattern=1)),
              Text(
                extent=[4,82; 14,72],
                style(
                  color=10,
                  fillColor=10,
                  fillPattern=1),
                string="f"),
              Text(
                extent=[70,-4; 80,-14],
                style(
                  color=10,
                  fillColor=10,
                  fillPattern=1),
                string="e"),
              Line(points=[-80,-4; -28,-4; -6,-2; 0,0],         style(
                  color=0,
                  thickness=2,
                  fillColor=10,
                  fillPattern=1)),
              Text(
                extent=[-66,-6; -36,-14],
                style(
                  color=10,
                  pattern=3,
                  fillColor=10,
                  fillPattern=1),
                string="blocking"),
              Text(
                extent=[16,50; 48,42],
                style(
                  color=10,
                  pattern=3,
                  fillColor=10,
                  fillPattern=1),
                string="conducting"),
              Line(points=[0,0; 6,10; 12,28; 16,66],            style(
                  color=0,
                  thickness=2,
                  fillColor=10,
                  fillPattern=1))),
          Icon(
              Text(
                extent=[-86,78; 28,-84],
                style(color=0),
              string="DjM")),
          Documentation(info="<html>
The Spice-style MOSFET junction diode is a regular exponential diode. A separate model was needed to implement the <b>Level</b> parameter of the MOSFET models. <p>

<hr> <p>

<pre>
Parameters:

 Level:  MOSFET modeling level (default value = 3)
           Level=0: Static injection model
           Level=1: Shichman-Hodges model
           Level=2: Grove-Frohman model
           Level=3: Empirical model
           Level=4: Simplified Grove-Frohman model

 IS:     Transport saturation current (default value = 0 Amp)

 EG:     Energy gap for temperature effect on saturation current (default value = 1.11 Volt)

 EMin:   if x &lt; EMin, the exp(x) function is linearized (default value = -100)

 EMax:   if x &gt; EMax, the exp(x) function is linearized (default value = 40)
</pre>
</html>
"));
        equation
          /* Compute thermal voltage as function of temperature */
          Vt = k*e2/q;

          /* Compute temperature dependence of saturation current */
          RTemp = e2/Tnom;
          EGval = EG - GapC1*e2*e2/(e2 + GapC2);
          EGnom = EG - GapC1*Tnom*Tnom/(Tnom + GapC2);
          ISval = IS*exp((EGval - EGnom*RTemp)/Vt);

          /* Compute diode characteristic */
          et = e1/Vt;
          if Level==0 then
            /* Static injection model */
            f1 = 0;
          else
            /* All other models */
            f1 = ISval*(if noEvent(et < EMin) then ExMin*(et - EMin + 1) - 1 else
                        if noEvent(et > EMax) then ExMax*(et - EMax + 1) - 1 else exp(et) - 1);
          end if;

          /* Compute heat flow */
          e2*f2 = e1*f1;
        end DjM;

        model NMOSint
        "Spice-style internal NMOS transistor in bond graph technology"
          parameter Integer Level=3
          "Level of MOS model (check documentation window for details)";
          parameter Integer Type=1 "Type=1 for NMOS; Type=-1 for PMOS";
          parameter Modelica.SIunits.Temperature Tnom=300.15
          "Reference temperature";
          parameter Modelica.SIunits.Voltage VT0=0
          "Zero-bias threshold voltage";
          parameter Modelica.SIunits.Voltage VFB=0 "Flat band voltage";
          parameter Modelica.SIunits.Transconductance KP=0
          "Transconductance parameter at reference temperature";
          parameter Real GAMMA(unit="V^0.5") = 0 "Body-effect parameter";
          parameter Modelica.SIunits.Voltage PHI=0
          "Surface inversion potential at reference temperature";
          parameter Modelica.SIunits.InversePotential LAMBDA=0
          "Channel length modulation";
          parameter Modelica.SIunits.Length TOX=1e-7 "Thin oxide thickness";
          parameter Real COX(unit="F/m2") = 0 "Specific capacitance of SiO2";
          parameter Modelica.SIunits.ElectronNumberDensity NSUB= 0
          "Substrate doping";
          parameter Real NSS(unit="1/m2") = 0 "Surface state density";
          parameter Real NFS(unit="1/m2") = 0 "Surface fast state density";
          parameter Real NEFF = 1 "Total channel charge coefficient";
          parameter Modelica.SIunits.Length L(min=0) = 1e-4 "Channel length";
          parameter Modelica.SIunits.Length W(min=0) = 1e-4 "Channel width";
          parameter Modelica.SIunits.Length XJ=0 "Metallurgical junction depth";
          parameter Modelica.SIunits.Length LD=0 "Lateral diffusion";
          parameter Modelica.SIunits.Area AD=0 "Drain diffusion area";
          parameter Modelica.SIunits.Length PD=0 "Drain perimeter width";
          parameter Modelica.SIunits.Area AS=0 "Source diffusion area";
          parameter Modelica.SIunits.Length PS=0 "Source perimeter width";
          parameter Real XD(unit="m/V^0.5") = 0 "Depletion factor";
          parameter Real TPG=1 "Type of gate material";
          parameter Real U0(unit="m2/(V.s)") = 0
          "Surface mobility at reference temperature";
          parameter Modelica.SIunits.ElectricFieldStrength UCRIT=1e6
          "Critical electric field for mobility";
          parameter Real UEXP=0 "Exponential coefficient for mobility";
          parameter Real UTRA=0 "Transverse field coefficient";
          parameter Modelica.SIunits.Velocity VMAX=0
          "Maximum drift velocity of carriers";
          parameter Modelica.SIunits.ElectricFieldStrength ECRIT=0
          "Critical electric field for pinch-off";
          parameter Real XQC=0 "Coefficient of channel charge share";
          parameter Real DELTA=0 "Width effect on threshold voltage";
          parameter Real KAPPA=0.2 "Saturation field factor";
          parameter Real ETA=0 "Static feedback on threshold voltage";
          parameter Modelica.SIunits.InversePotential THETA=0
          "Mobility modulation";
          parameter Modelica.SIunits.Voltage VBP=0
          "Gate to channel critical voltage";
          parameter Modelica.SIunits.Current ISD=0
          "Drain junction saturation current at reference temperature";
          parameter Modelica.SIunits.Current ISS=0
          "Source junction saturation current at reference temperature";
          parameter Modelica.SIunits.Voltage PB=0.8
          "Built-in potential at reference temperature";
          parameter Real CJ(unit="F/m2") = 0
          "Zero-bias bulk capacitance per square meter at reference temperature";
          parameter Real MJ=0.33 "Bulk junction grading coefficient";
          parameter Real CJSW(unit="F/m") = 1e-9
          "Zero-bias perimeter capacitance per meter at reference temperature";
          parameter Real MJSW=0.33 "Perimeter capacitance grading coefficient";
          parameter Real FC=0.5
          "Forward-bias depletion capacitance coefficient";
          parameter Modelica.SIunits.Conductance GminDC=1e-12
          "Leakage conductance";
          parameter Modelica.SIunits.Voltage EG=1.11
          "Energy gap for temperature effect on saturation current at 0 K";
          parameter Real EMin=-100
          "if x < EMin, the exp(x) function is linearized";
          parameter Real EMax=40
          "if x > EMax, the exp(x) function is linearized";
        annotation (Coordsys(extent=[-140,-140; 140,140]), Diagram(
            Text(
              extent=[130,20; 152,6],
              string="Bint",
              style(color=10, rgbcolor={135,135,135})),
            Text(
              extent=[-32,146; -10,132],
              style(color=10, rgbcolor={135,135,135}),
              string="Dint"),
            Text(
              extent=[10,-134; 32,-148],
              style(color=10, rgbcolor={135,135,135}),
              string="Sint"),
            Text(
              extent=[-152,-8; -130,-22],
              style(color=10, rgbcolor={135,135,135}),
              string="Gint"),
            Text(
              extent=[-60,-136; -38,-150],
              style(color=10, rgbcolor={135,135,135}),
              string="Heat"),
              Text(
                extent=[-110,94; -96,88],
                string="vds",
                style(color=10, rgbcolor={135,135,135})),
              Text(
                extent=[-110,78; -96,72],
                style(color=10, rgbcolor={135,135,135}),
                string="vgs"),
              Text(
                extent=[-74,108; -60,102],
                style(color=10, rgbcolor={135,135,135}),
                string="vbs"),
              Text(
                extent=[-90,64; -76,58],
                style(color=10, rgbcolor={135,135,135}),
                string="Tdev")),
          Icon(
            Text(
              extent=[-78,128; 82,88],
              style(color=3, rgbcolor={0,0,255}),
              string="%name"),
              Line(points=[0,100; 0,-100],
                                         style(
                  color=9,
                  rgbcolor={149,164,171},
                  thickness=2,
                  fillColor=74,
                  rgbfillColor={0,0,185},
                  fillPattern=1)),
              Text(
                extent=[-132,90; 98,-98],
                style(color=0),
              string="NMOS"),
              Line(points=[-20,-80; 0,-100],style(
                  color=9,
                  rgbcolor={149,164,171},
                  thickness=2,
                  fillColor=74,
                  rgbfillColor={0,0,185},
                  fillPattern=1)),
              Line(points=[0,-100; 20,-80],style(
                  color=9,
                  rgbcolor={149,164,171},
                  thickness=2,
                  fillColor=74,
                  rgbfillColor={0,0,185},
                  fillPattern=1)),
              Text(
                extent=[94,-2; 134,-44],
                style(color=0),
              string="int"),
            Text(
              extent=[128,22; 150,8],
              string="Bint",
              style(color=10, rgbcolor={135,135,135})),
            Text(
              extent=[12,146; 34,132],
              style(color=10, rgbcolor={135,135,135}),
              string="Dint"),
            Text(
              extent=[-152,22; -130,8],
              style(color=10, rgbcolor={135,135,135}),
              string="Gint"),
            Text(
              extent=[-60,-134; -38,-148],
              style(color=10, rgbcolor={135,135,135}),
              string="Heat"),
            Text(
              extent=[10,-134; 32,-148],
              style(color=10, rgbcolor={135,135,135}),
              string="Sint"),
              Line(points=[22,50; 120,50], style(
                  color=9,
                  rgbcolor={149,164,171},
                  thickness=2,
                  fillColor=74,
                  rgbfillColor={0,0,185},
                  fillPattern=1)),
              Line(points=[20,50; 50,60],  style(
                  color=9,
                  rgbcolor={149,164,171},
                  thickness=2,
                  fillColor=74,
                  rgbfillColor={0,0,185},
                  fillPattern=1)),
              Line(points=[20,50; 50,40],  style(
                  color=9,
                  rgbcolor={149,164,171},
                  thickness=2,
                  fillColor=74,
                  rgbfillColor={0,0,185},
                  fillPattern=1))),
          Documentation(info="<html>
The Spice-style internal n-channel MOSFET model contains the non-linear current source, the two junction diodes (including the junctions capacitances), as well as leakage conductances everywhere to keep the model together. <p>

<hr> <p>

<pre>
Parameters:

 Level:   MOSFET modeling level (default value = 3)
            Level=0: Static injection model
            Level=1: Shichman-Hodges model
            Level=2: Grove-Frohman model
            Level=3: Empirical model
            Level=4: Simplified Grove-Frohman model

 Tnom:    Reference temperature (default value = 300.15 K)

 VT0:     Zero-bias threshold voltage (default value = 0 Volt)
            <font color=blue><b>VT0 &gt; 0 for enhancement MOSFET
            VT0 &lt; 0 for depletion MOSFET</b></font>

 KP:      Transconductance parameter at reference temperature (default value = 0 Amp/Volt<sup>2</sup>)

 GAMMA:   Body-effect parameter (default value = Volt<sup>0.5</sup>)

 PHI:     Surface inversion potential at reference temperature (default value = 0 Volt)

 LAMBDA:  Channel length modulation  (default value = 0 1/Volt)

 TOX:     Thin oxide thickness (default value = 1e-7 m)

 COX:     Specific capacitance of SiO2 (default value = 0 F/m<sup>2</sup>)

 NSUB:    Substrate doping (default value = 0 1/m<sup>3</sup>)

 NSS:     Surface state density (default value = 0 1/m<sup>2</sup>)

 NFS:     Surface fast state density (default value = 0 1/m<sup>2</sup>)

 NEFF:    Total channel charge coefficient (default value = 1)

 XJ:      Metallurgical junction depth (default value = 0 m)

 L:       Channel length (default value = 1e-4 m)

 W:       Channel width (default value = 1e-4 m)

 LD:      Lateral diffusion (default value = 0 m)

 AD:      Drain diffusion area (default value = 0 m<sup>2</sup>)

 PD:      Drain perimeter width (default value = 0 m)

 AS:      Source diffusion area (default value = 0 m<sup>2</sup>)

 PS:      Source perimeter width (default value = 0 m)

 XD:      Depletion factor (default value = 0 m/Volt<sup>0.5</sup>)

 TPG:     Type of gate material (default value = 1)

 U0:      Surface mobility at reference temperature (default value = 0 m<sup>2</sup>/(V*s))

 UCRIT:   Critical electric field for mobility (default value = 1e6 V/m)

 UEXP:    Exponential coefficient for mobility (default value = 0)

 UTRA:    Transverse field coefficient (default value = 0)

 VMAX:    Maximum drift velocity of carriers (default value = 0 m/s)
            <font color=blue><b>VMAX = 0: Frohman-Grove model of drain current computation</b></font>
            <font color=blue><b>VMAX &gt; 0: Baum-Beneking model of drain current computation</b></font>

 ECRIT:   Critical electric field for pinch-off (default value = 0 V/m) <font color=red>Levels 2,3</font>

 XQC:     Coefficient of channel charge share (default value = 0) <font color=red>Levels 2,3</font>
            <font color=blue><b>XQC &lt;= 0.5: Ward-Dutton model of gate capacitance computation</b></font>
            <font color=blue><b>XQC &gt;  0.5: Meyer model of gate capacitance computation</b></font>

 DELTA:   Width effect on threshold voltage (default value = 0)

 KAPPA:   Saturation field factor (default value = 0.2)

 ETA:     Static feedback on threshold voltage (default value = 0)

 THETA:   Mobility modulation (default value = 0 1/Volt)

 VBP:     Gate to channel critical voltage (default value = 0 Volt)

 ISD:     Drain junction saturation current at reference temperature (default value = 0 Amp)

 ISS:     Source junction saturation current at reference temperature (default value = 0 Amp)

 PB:      Built-in potential at reference temperature (default value = 0.8 Volt)

 CJ:      Zero-bias bulk capacitance per square meter at reference temperature (default value = 1e-4 F/m<sup>2</sup>)

 MJ:      Bulk junction grading coefficient (default value = 0.33)

 CJSW:    Zero-bias perimeter capacitance per meter at reference temperature (default value = 1e-9 F/m)

 MJSW:    Perimeter capacitance grading coefficient (default value = 0.33)

 FC:      Forward-bias depletion capacitance coefficient (default value = 0.5)

 GminDC:  Leakage conductance (default value = 1e-12 mho)

 EG:      Energy gap for temperature effect on saturation current (default value = 1.11 Volt)

 EMin:    if x &lt; EMin, the exp(x) function is linearized (default value = -100)

 EMax:    if x &gt; EMax, the exp(x) function is linearized (default value = 40)
</pre> <p>

<hr> <p>

<b>References:</b> <p>

<ol>
<li> Cellier, F.E. (1991), <i>Continuous System Modeling</i>, Springer-Verlag, New York.
<li> Massobrio, G. and P. Antognetti (1993), <i>Semiconductor Device Modeling with Spice</i>,
     2<sup>nd</sup> edition, McGraw Hill, New York.
</ol>
</html>
"));
        BondLib.Interfaces.BondCon Bint "Internal bulk"
          annotation (extent=[130,-10; 150,10]);
        BondLib.Interfaces.BondCon Gint "Internal gate"
          annotation (extent=[-150,-10; -130,10]);
        BondLib.Interfaces.BondCon Sint "Internal source"
          annotation (extent=[-10,-150; 10,-130]);
        BondLib.Interfaces.BondCon Dint "Internal drain"
          annotation (extent=[-10,130; 10,150]);
        BondLib.Interfaces.BondCon H "Heat"
          annotation (extent=[-80,-150; -60,-130]);
        BondLib.Junctions.J0p4 J0p4_1 annotation (extent=[-130,-10; -110,10]);
        BondLib.Junctions.J0p4 J0p4_2 annotation (extent=[110,-10; 130,10]);
        BondLib.Junctions.J0p4 J0p4_3 annotation (extent=[-10,110; 10,130]);
        BondLib.Junctions.J0p4 J0p4_4 annotation (extent=[-10,-130; 10,-110]);
        BondLib.Bonds.Bond B1 annotation (extent=[40,20; 60,40], rotation=90);
        BondLib.Junctions.J1p3 J1p3_1
          annotation (extent=[40,40; 60,60], rotation=90);
        BondLib.Bonds.Bond B2 annotation (extent=[40,60; 60,80], rotation=90);
        BondLib.Bonds.Bond B3 annotation (extent=[60,40; 80,60], rotation=0);
        BondLib.Spice.Utilities.CjM Dbd(
          Level=Level,
          Tnom=Tnom,
          IS=ISD,
          PB=PB,
          CJ=CJ,
          MJ=MJ,
          CJSW=CJSW,
          MJSW=MJSW,
          FC=FC,
          A=AD,
          P=PD,
          EG=EG,
          GminDC=GminDC,
          EMin=EMin,
          EMax=EMax)     annotation (extent=[80,38; 110,62]);
        BondLib.Bonds.Bond B4 annotation (extent=[40,-40; 60,-20], rotation=270);
        BondLib.Junctions.J1p3 J1p3_2
          annotation (extent=[40,-60; 60,-40], rotation=90);
        BondLib.Bonds.Bond B5 annotation (extent=[40,-80; 60,-60], rotation=270);
        BondLib.Bonds.Bond B6 annotation (extent=[60,-60; 80,-40], rotation=0);
        BondLib.Junctions.J0p3 J0p3_1 annotation (extent=[80,-60; 100,-40]);
        BondLib.Bonds.fBond B7 annotation (extent=[100,-60; 120,-40]);
        BondLib.Sensors.De Vbs annotation (extent=[120,-60; 140,-40]);
        BondLib.Bonds.Bond B8
          annotation (extent=[80,-80; 100,-60], rotation=270);
        BondLib.Spice.Utilities.CjM Dbs(
          Level=Level,
          Tnom=Tnom,
          PB=PB,
          CJ=CJ,
          MJ=MJ,
          CJSW=CJSW,
          MJSW=MJSW,
          FC=FC,
          EG=EG,
          GminDC=GminDC,
          IS=ISS,
          A=AS,
          P=PS,
          EMin=EMin,
          EMax=EMax)
                annotation (extent=[78,-110; 102,-80], rotation=-90);
        BondLib.Bonds.eBond B9 annotation (extent=[110,40; 130,60]);
        BondLib.Bonds.eBond B10
          annotation (extent=[80,-130; 100,-110], rotation=-90);
        BondLib.Junctions.J1p3 J1p3_3
          annotation (extent=[30,-10; 10,10], rotation=90);
        BondLib.Bonds.Bond B11 annotation (extent=[10,10; 30,30], rotation=270);
        BondLib.Bonds.Bond B12
          annotation (extent=[10,-30; 30,-10], rotation=270);
        BondLib.Bonds.Bond B13
          annotation (extent=[-10,-10; 10,10], rotation=180);
        BondLib.Junctions.J0p4 J0p4_5 annotation (extent=[-30,-10; -10,10]);
        BondLib.Bonds.eBond B14 annotation (extent=[-50,-10; -30,10]);
        BondLib.Sources.mSf_int ISD0
                            annotation (extent=[-80,14; -50,-14]);
        BondLib.Bonds.fBond B15
          annotation (extent=[-30,10; -10,30], rotation=90);
        BondLib.Sensors.De Vds annotation (extent=[-30,30; -10,50], rotation=90);
        BondLib.Bonds.Bond B16
          annotation (extent=[-30,-30; -10,-10], rotation=270);
        BondLib.Spice.RS Rds(
          Tnom=Tnom,
          R=1/GminDC,
          TR1=0,
          TR2=0,
          Area=1) annotation (extent=[-32,-80; -8,-30],  rotation=-90);
        BondLib.Bonds.Bond B18
          annotation (extent=[-130,50; -110,70], rotation=90);
        BondLib.Junctions.J1p3 J1p3_4
          annotation (extent=[-130,30; -110,50], rotation=-90);
        BondLib.Bonds.Bond B17
          annotation (extent=[-120,4; -100,24],  rotation=90);
        BondLib.Bonds.Bond B19
          annotation (extent=[-150,50; -130,70], rotation=90);
        BondLib.Spice.RS Rdg(
          Tnom=Tnom,
          R=1/GminDC,
          TR1=0,
          TR2=0,
          Area=1) annotation (extent=[-152,70; -128,120], rotation=90);
        BondLib.Bonds.Bond B20
          annotation (extent=[-130,-30; -110,-10], rotation=270);
        BondLib.Junctions.J1p3 J1p3_5
          annotation (extent=[-120,-40; -100,-20], rotation=0);
        BondLib.Bonds.Bond B21
          annotation (extent=[-100,-40; -80,-20], rotation=0);
        BondLib.Bonds.Bond B22
          annotation (extent=[-120,-60; -100,-40], rotation=270);
        BondLib.Spice.RS Rgs(
          Tnom=Tnom,
          R=1/GminDC,
          TR1=0,
          TR2=0,
          Area=1) annotation (extent=[-152,-140; -128,-90], rotation=-90);
        BondLib.Junctions.J0p3 J0p3_2
          annotation (extent=[-120,-80; -100,-60], rotation=-90);
        BondLib.Bonds.Bond B24
          annotation (extent=[-150,-90; -130,-70], rotation=270);
        BondLib.Sensors.De Vgs
          annotation (extent=[-120,-120; -100,-100], rotation=-90);
        BondLib.Bonds.eBond B25
          annotation (extent=[-30,-100; -10,-80], rotation=-90);
        BondLib.Bonds.eBond B26 annotation (extent=[-130,-150; -110,-130]);
        BondLib.Bonds.eBond B27
          annotation (extent=[-150,120; -130,140], rotation=90);
        BondLib.Bonds.fBond B28 annotation (extent=[-100,-10; -80,10]);
          BondLib.Spice.Utilities.IDS0 IDS0(
          Tnom=Tnom,
          KP=KP,
          GAMMA=GAMMA,
          LAMBDA=LAMBDA,
          NFS=NFS,
          NEFF=NEFF,
          XJ=XJ,
          U0=U0,
          UEXP=UEXP,
          UTRA=UTRA,
          VMAX=VMAX,
          DELTA=DELTA,
          EG=EG,
          L=L,
          W=W,
          Level=Level,
          EMin=EMin,
          EMax=EMax,
          PHI=PHI,
          Type=Type,
          VFB=VFB,
          COX=COX,
          XD=XD,
          ECRIT=ECRIT,
          VBP=VBP,
          PB=PB,
          KAPPA=KAPPA,
          ETA=ETA,
          THETA=THETA,
          LD=LD)       annotation (extent=[-90,66; -60,94]);
          Modelica.Blocks.Interfaces.RealOutput y1[5]
          "Connector of Real output signals"
            annotation (extent=[-80,130; -60,150],rotation=90);
        Modelica.Blocks.Math.Gain Gain1(k=-1)
          annotation (extent=[-60,40; -40,60],     rotation=-90);
        BondLib.Junctions.J0p6 J0p6_1
          annotation (extent=[-80,-130; -60,-110], rotation=90);
        BondLib.Bonds.eBond B29
          annotation (extent=[-80,-110; -60,-90], rotation=-90);
        BondLib.Sensors.De Tdev
          annotation (extent=[-86,40; -64,60], rotation=90);
        BondLib.Bonds.fBond B30
          annotation (extent=[-86,20; -64,40], rotation=90);
        BondLib.Spice.RS Rbg(
          Tnom=Tnom,
          R=1/GminDC,
          TR1=0,
          TR2=0,
          Area=1) annotation (extent=[90,98; 138,122],    rotation=0);
        BondLib.Bonds.Bond B31
                              annotation (extent=[70,100; 90,120],
                                                                 rotation=0);
        BondLib.Bonds.eBond B32
                               annotation (extent=[130,80; 150,100], rotation=
              -90);
        Junctions.J0p5 J0p5_1
          annotation (extent=[-80,-90; -60,-70], rotation=90);
        BondLib.Junctions.J1p3 J1p3_6
          annotation (extent=[50,120; 70,140],   rotation=0);
        BondLib.Bonds.Bond B33
                              annotation (extent=[30,120; 50,140],
                                                                 rotation=0);
        BondLib.Bonds.Bond B34
                              annotation (extent=[70,120; 90,140],
                                                                 rotation=0);
        BondLib.Bonds.fBond B23
                               annotation (extent=[-120,-100; -100,-80],
            rotation=-90);
        equation
        connect(J0p4_1.BondCon1, Gint)     annotation (points=[-130,0; -140,0],
            style(color=8, rgbcolor={192,192,192}));
        connect(J0p4_2.BondCon2, Bint)     annotation (points=[130,0; 140,0],
            style(color=8, rgbcolor={192,192,192}));
        connect(J0p4_3.BondCon4, Dint)     annotation (points=[0,130; 0,140],
            style(color=8, rgbcolor={192,192,192}));
        connect(J0p4_4.BondCon3, Sint)     annotation (points=[0,-130; 0,-140],
            style(color=8, rgbcolor={192,192,192}));
        connect(J1p3_1.BondCon1, B1.BondCon2) annotation (points=[50,40; 50,40],
            style(color=8, rgbcolor={192,192,192}));
        connect(B2.BondCon1, J1p3_1.BondCon2) annotation (points=[50,60.2; 50,
              60], style(color=8, rgbcolor={192,192,192}));
        connect(J1p3_1.BondCon3, B3.BondCon1) annotation (points=[60,50; 60.2,
              50], style(color=8, rgbcolor={192,192,192}));
        connect(B3.BondCon2,Dbd. BondCon1) annotation (points=[80,50; 80,50],
            style(color=8, rgbcolor={192,192,192}));
        connect(B1.BondCon1, J0p4_2.BondCon1) annotation (points=[50,20.2; 50,0;
              110,0], style(color=8, rgbcolor={192,192,192}));
        connect(B4.BondCon2, J1p3_2.BondCon2) annotation (points=[50,-40; 50,
              -40], style(color=8, rgbcolor={192,192,192}));
        connect(J1p3_2.BondCon1, B5.BondCon1) annotation (points=[50,-60; 50,
              -60.2], style(color=8, rgbcolor={192,192,192}));
        connect(B5.BondCon2, J0p4_4.BondCon2) annotation (points=[50,-80; 50,
              -120; 10,-120], style(color=8, rgbcolor={192,192,192}));
        connect(J1p3_2.BondCon3, B6.BondCon1) annotation (points=[60,-50; 60.2,
              -50], style(color=8, rgbcolor={192,192,192}));
        connect(B6.BondCon2, J0p3_1.BondCon1) annotation (points=[80,-50; 80,
              -50], style(color=8, rgbcolor={192,192,192}));
        connect(J0p3_1.BondCon2, B7.eBondCon1) annotation (points=[100,-50; 100,
              -50], style(color=8, rgbcolor={192,192,192}));
        connect(B7.fBondCon1,Vbs. BondCon1) annotation (points=[120,-50; 120,
              -50], style(color=8, rgbcolor={192,192,192}));
        connect(J0p3_1.BondCon3, B8.BondCon1) annotation (points=[90,-60; 90,
              -60.2], style(color=8, rgbcolor={192,192,192}));
        connect(B8.BondCon2,Dbs. BondCon1) annotation (points=[90,-80; 90,-80],
            style(color=8, rgbcolor={192,192,192}));
        connect(B4.BondCon1, J0p4_2.BondCon3) annotation (points=[50,-20.2; 50,
              -10; 120,-10], style(color=8, rgbcolor={192,192,192}));
        connect(Dbd.BondCon2, B9.fBondCon1) annotation (points=[110,50; 110,50],
            style(color=8, rgbcolor={192,192,192}));
        connect(Dbs.BondCon2, B10.fBondCon1) annotation (points=[90,-110; 90,
              -110], style(color=8, rgbcolor={192,192,192}));
        connect(B11.BondCon2, J1p3_3.BondCon2) annotation (points=[20,10; 20,10],
            style(color=8, rgbcolor={192,192,192}));
        connect(J1p3_3.BondCon1, B12.BondCon1) annotation (points=[20,-10; 20,
              -10.1; 20,-10.2; 20,-10.2], style(color=8, rgbcolor={192,192,192}));
        connect(B11.BondCon1, J0p4_3.BondCon3) annotation (points=[20,29.8; 20,
              80; 0,80; 0,110],   style(color=8, rgbcolor={192,192,192}));
        connect(B12.BondCon2, J0p4_4.BondCon4) annotation (points=[20,-30; 20,
              -100; 0,-100; 0,-110], style(color=8, rgbcolor={192,192,192}));
        connect(B13.BondCon1, J1p3_3.BondCon3) annotation (points=[9.8,
              -1.20011e-015; 2.9,-1.20011e-015; 2.9,-6.12303e-016; 10,
              -6.12303e-016], style(color=8, rgbcolor={192,192,192}));
        connect(J0p4_5.BondCon2, B13.BondCon2) annotation (points=[-10,0; -16,0;
              -16,1.22461e-015; -10,1.22461e-015], style(color=8, rgbcolor={192,
                192,192}));
        connect(B14.eBondCon1, J0p4_5.BondCon1) annotation (points=[-30,0; -30,
              0], style(color=8, rgbcolor={192,192,192}));
        connect(B15.eBondCon1, J0p4_5.BondCon4) annotation (points=[-20,10; -20,
              10], style(color=8, rgbcolor={192,192,192}));
        connect(Vds.BondCon1, B15.fBondCon1) annotation (points=[-20,30; -20,30],
            style(color=8, rgbcolor={192,192,192}));
        connect(J0p4_5.BondCon3, B16.BondCon1) annotation (points=[-20,-10; -20,
              -10.1; -20,-10.2; -20,-10.2], style(color=8, rgbcolor={192,192,
                192}));
        connect(B16.BondCon2,Rds. BondCon1) annotation (points=[-20,-30; -20,
              -30], style(color=8, rgbcolor={192,192,192}));
        connect(J1p3_4.BondCon2, B17.BondCon2) annotation (points=[-120,30;
              -110,30; -110,24],
                        style(color=8, rgbcolor={192,192,192}));
        connect(B18.BondCon1, J1p3_4.BondCon1) annotation (points=[-120,50.2;
              -120,50], style(color=8, rgbcolor={192,192,192}));
        connect(J1p3_4.BondCon3, B19.BondCon1) annotation (points=[-130,40;
              -140,40; -140,50.2], style(color=8, rgbcolor={192,192,192}));
        connect(Rdg.BondCon1, B19.BondCon2) annotation (points=[-140,70; -140,
              70], style(color=8, rgbcolor={192,192,192}));
        connect(B18.BondCon2, J0p4_3.BondCon1) annotation (points=[-120,70;
              -120,120; -10,120], style(color=8, rgbcolor={192,192,192}));
        connect(J0p4_1.BondCon3, B20.BondCon1) annotation (points=[-120,-10;
              -120,-10.2], style(color=8, rgbcolor={192,192,192}));
        connect(B20.BondCon2, J1p3_5.BondCon1) annotation (points=[-120,-30;
              -120,-30], style(color=8, rgbcolor={192,192,192}));
        connect(J1p3_5.BondCon3, B22.BondCon1) annotation (points=[-110,-40;
              -110,-40.2], style(color=8, rgbcolor={192,192,192}));
        connect(J0p3_2.BondCon1, B22.BondCon2) annotation (points=[-110,-60;
              -110,-60], style(color=8, rgbcolor={192,192,192}));
        connect(J0p3_2.BondCon3, B24.BondCon1) annotation (points=[-120,-70;
              -130,-70; -130,-70.2; -140,-70.2], style(color=8, rgbcolor={192,
                192,192}));
        connect(B24.BondCon2, Rgs.BondCon1)  annotation (points=[-140,-90; -140,
              -90], style(color=8, rgbcolor={192,192,192}));
        connect(Rds.BondCon2, B25.fBondCon1) annotation (points=[-20,-80; -20,
              -80], style(color=8, rgbcolor={192,192,192}));
        connect(Rgs.BondCon2, B26.fBondCon1)  annotation (points=[-140,-140;
              -130,-140], style(color=8, rgbcolor={192,192,192}));
        connect(B27.fBondCon1,Rdg. BondCon2) annotation (points=[-140,120; -140,
              120], style(color=8, rgbcolor={192,192,192}));
        connect(J1p3_5.BondCon2, B21.BondCon1) annotation (points=[-100,-30;
              -99.8,-30], style(color=8, rgbcolor={192,192,192}));
        connect(B21.BondCon2, J0p4_4.BondCon1) annotation (points=[-80,-30; -34,
              -30; -34,-120; -10,-120], style(color=8, rgbcolor={192,192,192}));
        connect(IDS0.y, Gain1.u)   annotation (points=[-58.5,80; -50,80; -50,62],
            style(color=74, rgbcolor={0,0,127}));
        connect(IDS0.y1, y1)   annotation (points=[-58.5,73; -54,73; -54,120;
              -70,120; -70,140], style(color=74, rgbcolor={0,0,127}));
        connect(Vds.OutPort1, IDS0.u1)   annotation (points=[-20,51; -20,112;
              -110,112; -110,88.4; -93,88.4], style(color=74, rgbcolor={0,0,127}));
        connect(Vgs.OutPort1, IDS0.u2)   annotation (points=[-110,-121; -110,
              -126; -96,-126; -96,40; -110,40; -110,71.6; -93,71.6], style(
              color=74, rgbcolor={0,0,127}));
        connect(ISD0.BondCon2, B14.fBondCon1) annotation (points=[-50,
              -1.77636e-015; -50,0], style(color=8, rgbcolor={192,192,192}));
        connect(ISD0.BondCon1, B28.fBondCon1) annotation (points=[-80,
              -1.77636e-015; -80,0], style(color=8, rgbcolor={192,192,192}));
        connect(J0p6_1.BondCon1, H)        annotation (points=[-70,-130; -70,
              -140], style(color=8, rgbcolor={192,192,192}));
        connect(B29.eBondCon1, J0p6_1.BondCon2) annotation (points=[-70,-110;
              -70,-110], style(color=8, rgbcolor={192,192,192}));
        connect(IDS0.Tdev, Tdev.OutPort1)   annotation (points=[-75,66; -75,61],
            style(color=74, rgbcolor={0,0,127}));
        connect(Tdev.BondCon1, B30.fBondCon1) annotation (points=[-75,40; -75,
              40], style(color=8, rgbcolor={192,192,192}));
        connect(B26.eBondCon1, J0p6_1.BondCon6) annotation (points=[-110,-140;
              -92,-140; -92,-116; -80,-116], style(color=8, rgbcolor={192,192,
                192}));
        connect(B27.eBondCon1, J0p6_1.BondCon5) annotation (points=[-140,140;
              -158,140; -158,-152; -88,-152; -88,-124; -80,-124], style(color=8,
              rgbcolor={192,192,192}));
        connect(J0p5_1.BondCon1, B29.fBondCon1) annotation (points=[-70,-90;
              -70,-90], style(color=8, rgbcolor={192,192,192}));
        connect(J0p5_1.BondCon5, B28.eBondCon1) annotation (points=[-80,-80;
              -92,-80; -92,-16; -104,-16; -104,0; -100,0], style(color=8,
              rgbcolor={192,192,192}));
        connect(J0p5_1.BondCon2, B30.eBondCon1) annotation (points=[-70,-70;
              -70,-26; -44,-26; -44,16; -75,16; -75,20], style(color=8,
              rgbcolor={192,192,192}));
        connect(Vbs.OutPort1, IDS0.u3)   annotation (points=[141,-50; 158,-50;
              158,160; -88,160; -88,108; -75,108; -75,94], style(color=74,
              rgbcolor={0,0,127}));
        connect(B31.BondCon2,Rbg. BondCon1) annotation (points=[90,110; 90,110],
            style(color=8, rgbcolor={192,192,192}));
        connect(B2.BondCon2, J0p4_3.BondCon2) annotation (points=[50,80; 50,86;
              16,86; 16,120; 10,120], style(color=8, rgbcolor={192,192,192}));
        connect(Rbg.BondCon2, B32.fBondCon1) annotation (points=[138,110; 140,
              110; 140,100], style(color=8, rgbcolor={192,192,192}));
        connect(J1p3_6.BondCon3, B31.BondCon1) annotation (points=[60,120; 60,
              110; 70.2,110], style(color=8, rgbcolor={192,192,192}));
        connect(B33.BondCon2, J1p3_6.BondCon1) annotation (points=[50,130; 50,
              130], style(color=8, rgbcolor={192,192,192}));
        connect(J1p3_6.BondCon2, B34.BondCon1) annotation (points=[70,130; 70.2,
              130], style(color=8, rgbcolor={192,192,192}));
        connect(B34.BondCon2, J0p4_2.BondCon4) annotation (points=[90,130; 154,
              130; 154,30; 120,30; 120,10], style(color=8, rgbcolor={192,192,
                192}));
        connect(B17.BondCon1, J0p4_1.BondCon2) annotation (points=[-110,4.2;
              -110,0], style(color=8, rgbcolor={192,192,192}));
        connect(B33.BondCon1, J0p4_1.BondCon4) annotation (points=[30.2,130; 20,
              130; 20,164; -162,164; -162,20; -120,20; -120,10], style(color=8,
              rgbcolor={192,192,192}));
        connect(B25.eBondCon1, J0p5_1.BondCon4) annotation (points=[-20,-100;
              -20,-106; -42,-106; -42,-76; -60,-76], style(color=8, rgbcolor={
                192,192,192}));
        connect(B10.eBondCon1, J0p5_1.BondCon3) annotation (points=[90,-130; 90,
              -154; -16,-154; -16,-124; -46,-124; -46,-84; -60,-84], style(
              color=8, rgbcolor={192,192,192}));
        connect(B9.eBondCon1, J0p6_1.BondCon4) annotation (points=[130,50; 162,
              50; 162,-158; -20,-158; -20,-128; -50,-128; -50,-116; -60,-116],
            style(color=8, rgbcolor={192,192,192}));
        connect(B32.eBondCon1, J0p6_1.BondCon3) annotation (points=[140,80; 142,
              80; 142,66; 166,66; 166,-162; -24,-162; -24,-132; -54,-132; -54,
              -124; -60,-124], style(color=8, rgbcolor={192,192,192}));
        connect(Vgs.BondCon1, B23.fBondCon1) annotation (points=[-110,-100;
              -110,-100], style(color=8, rgbcolor={192,192,192}));
        connect(J0p3_2.BondCon2, B23.eBondCon1) annotation (points=[-110,-80;
              -110,-80], style(color=8, rgbcolor={192,192,192}));
        connect(Gain1.y, ISD0.s) annotation (points=[-50,39; -50,30; -65,30;
              -65,12.6], style(color=74, rgbcolor={0,0,127}));
        end NMOSint;

        model PMOSint
        "Spice-style internal PMOS transistor in bond graph technology"
          parameter Integer Level=3
          "Level of MOS model (check documentation window for details)";
          parameter Integer Type=1 "Type=1 for NMOS; Type=-1 for PMOS";
          parameter Modelica.SIunits.Temperature Tnom=300.15
          "Reference temperature";
          parameter Modelica.SIunits.Voltage VT0=0
          "Zero-bias threshold voltage";
          parameter Modelica.SIunits.Voltage VFB=0 "Flat band voltage";
          parameter Modelica.SIunits.Transconductance KP=0
          "Transconductance parameter at reference temperature";
          parameter Real GAMMA(unit="V^0.5") = 0 "Body-effect parameter";
          parameter Modelica.SIunits.Voltage PHI=0
          "Surface inversion potential at reference temperature";
          parameter Modelica.SIunits.InversePotential LAMBDA=0
          "Channel length modulation";
          parameter Modelica.SIunits.Length TOX=1e-7 "Thin oxide thickness";
          parameter Real COX(unit="F/m2") = 0 "Specific capacitance of SiO2";
          parameter Modelica.SIunits.ElectronNumberDensity NSUB= 0
          "Substrate doping";
          parameter Real NSS(unit="1/m2") = 0 "Surface state density";
          parameter Real NFS(unit="1/m2") = 0 "Surface fast state density";
          parameter Real NEFF = 1 "Total channel charge coefficient";
          parameter Modelica.SIunits.Length L(min=0) = 1e-4 "Channel length";
          parameter Modelica.SIunits.Length W(min=0) = 1e-4 "Channel width";
          parameter Modelica.SIunits.Length XJ=0 "Metallurgical junction depth";
          parameter Modelica.SIunits.Length LD=0 "Lateral diffusion";
          parameter Modelica.SIunits.Area AD=0 "Drain diffusion area";
          parameter Modelica.SIunits.Length PD=0 "Drain perimeter width";
          parameter Modelica.SIunits.Area AS=0 "Source diffusion area";
          parameter Modelica.SIunits.Length PS=0 "Source perimeter width";
          parameter Real XD(unit="m/V^0.5") = 0 "Depletion factor";
          parameter Real TPG=1 "Type of gate material";
          parameter Real U0(unit="m2/(V.s)") = 0
          "Surface mobility at reference temperature";
          parameter Modelica.SIunits.ElectricFieldStrength UCRIT=1e6
          "Critical electric field for mobility";
          parameter Real UEXP=0 "Exponential coefficient for mobility";
          parameter Real UTRA=0 "Transverse field coefficient";
          parameter Modelica.SIunits.Velocity VMAX=0
          "Maximum drift velocity of carriers";
          parameter Modelica.SIunits.ElectricFieldStrength ECRIT=0
          "Critical electric field for pinch-off";
          parameter Real XQC=0 "Coefficient of channel charge share";
          parameter Real DELTA=0 "Width effect on threshold voltage";
          parameter Real KAPPA=0.2 "Saturation field factor";
          parameter Real ETA=0 "Static feedback on threshold voltage";
          parameter Modelica.SIunits.InversePotential THETA=0
          "Mobility modulation";
          parameter Modelica.SIunits.Voltage VBP=0
          "Gate to channel critical voltage";
          parameter Modelica.SIunits.Current ISD=0
          "Drain junction saturation current at reference temperature";
          parameter Modelica.SIunits.Current ISS=0
          "Source junction saturation current at reference temperature";
          parameter Modelica.SIunits.Voltage PB=0.8
          "Built-in potential at reference temperature";
          parameter Real CJ(unit="F/m2") = 0
          "Zero-bias bulk capacitance per square meter at reference temperature";
          parameter Real MJ=0.33 "Bulk junction grading coefficient";
          parameter Real CJSW(unit="F/m") = 1e-9
          "Zero-bias perimeter capacitance per meter at reference temperature";
          parameter Real MJSW=0.33 "Perimeter capacitance grading coefficient";
          parameter Real FC=0.5
          "Forward-bias depletion capacitance coefficient";
          parameter Modelica.SIunits.Conductance GminDC=1e-12
          "Leakage conductance";
          parameter Modelica.SIunits.Voltage EG=1.11
          "Energy gap for temperature effect on saturation current at 0 K";
          parameter Real EMin=-100
          "if x < EMin, the exp(x) function is linearized";
          parameter Real EMax=40
          "if x > EMax, the exp(x) function is linearized";
        annotation (
          Coordsys(extent=[-140,-140; 140,140]),
          Diagram(
            Text(
              extent=[130,20; 152,6],
              string="Bint",
              style(color=10, rgbcolor={135,135,135})),
            Text(
              extent=[-32,146; -10,132],
              style(color=10, rgbcolor={135,135,135}),
              string="Dint"),
            Text(
              extent=[10,-134; 32,-148],
              style(color=10, rgbcolor={135,135,135}),
              string="Sint"),
            Text(
              extent=[-152,-8; -130,-22],
              style(color=10, rgbcolor={135,135,135}),
              string="Gint"),
            Text(
              extent=[-60,-136; -38,-150],
              style(color=10, rgbcolor={135,135,135}),
              string="Heat"),
              Text(
                extent=[-110,94; -96,88],
                style(color=10, rgbcolor={135,135,135}),
              string="vsd"),
              Text(
                extent=[-110,78; -96,72],
                style(color=10, rgbcolor={135,135,135}),
              string="vsg"),
              Text(
                extent=[-74,108; -60,102],
                style(color=10, rgbcolor={135,135,135}),
              string="vsb"),
              Text(
                extent=[-90,64; -76,58],
                style(color=10, rgbcolor={135,135,135}),
                string="Tdev")),
          Icon(
            Text(
              extent=[-78,128; 82,88],
              style(color=3, rgbcolor={0,0,255}),
              string="%name"),
              Line(points=[0,100; 0,-100],
                                         style(
                  color=9,
                  rgbcolor={149,164,171},
                  thickness=2,
                  fillColor=74,
                  rgbfillColor={0,0,185},
                  fillPattern=1)),
              Text(
                extent=[-132,92; 98,-96],
                style(color=0),
              string="PMOS"),
              Line(points=[0,100; 20,80],   style(
                  color=9,
                  rgbcolor={149,164,171},
                  thickness=2,
                  fillColor=74,
                  rgbfillColor={0,0,185},
                  fillPattern=1)),
              Line(points=[-20,80; 0,100], style(
                  color=9,
                  rgbcolor={149,164,171},
                  thickness=2,
                  fillColor=74,
                  rgbfillColor={0,0,185},
                  fillPattern=1)),
              Line(points=[22,50; 120,50], style(
                  color=9,
                  rgbcolor={149,164,171},
                  thickness=2,
                  fillColor=74,
                  rgbfillColor={0,0,185},
                  fillPattern=1)),
              Line(points=[90,40; 120,50], style(
                  color=9,
                  rgbcolor={149,164,171},
                  thickness=2,
                  fillColor=74,
                  rgbfillColor={0,0,185},
                  fillPattern=1)),
              Line(points=[90,60; 120,50], style(
                  color=9,
                  rgbcolor={149,164,171},
                  thickness=2,
                  fillColor=74,
                  rgbfillColor={0,0,185},
                  fillPattern=1)),
            Text(
              extent=[-152,22; -130,8],
              style(color=10, rgbcolor={135,135,135}),
              string="Gint"),
              Text(
                extent=[94,-2; 134,-44],
                style(color=0),
              string="int"),
            Text(
              extent=[128,22; 150,8],
              string="Bint",
              style(color=10, rgbcolor={135,135,135})),
            Text(
              extent=[-60,-134; -38,-148],
              style(color=10, rgbcolor={135,135,135}),
              string="Heat"),
            Text(
              extent=[10,-134; 32,-148],
              style(color=10, rgbcolor={135,135,135}),
              string="Sint"),
            Text(
              extent=[12,146; 34,132],
              style(color=10, rgbcolor={135,135,135}),
              string="Dint")),
          Documentation(info="<html>
The Spice-style internal p-channel MOSFET model contains the non-linear current source, the two junction diodes (including the junctions capacitances), as well as leakage conductances everywhere to keep the model together. <p>

<hr> <p>

<pre>
Parameters:

 Level:   MOSFET modeling level (default value = 3)
            Level=0: Static injection model
            Level=1: Shichman-Hodges model
            Level=2: Grove-Frohman model
            Level=3: Empirical model
            Level=4: Simplified Grove-Frohman model

 Tnom:    Reference temperature (default value = 300.15 K)

 VT0:     Zero-bias threshold voltage (default value = 0 Volt)
            <font color=blue><b>VT0 &lt; 0 for enhancement MOSFET
            VT0 &gt; 0 for depletion MOSFET</b></font>

 KP:      Transconductance parameter at reference temperature (default value = 0 Amp/Volt<sup>2</sup>)

 GAMMA:   Body-effect parameter (default value = Volt<sup>0.5</sup>)

 PHI:     Surface inversion potential at reference temperature (default value = 0 Volt)

 LAMBDA:  Channel length modulation  (default value = 0 1/Volt)

 TOX:     Thin oxide thickness (default value = 1e-7 m)

 COX:     Specific capacitance of SiO2 (default value = 0 F/m<sup>2</sup>)

 NSUB:    Substrate doping (default value = 0 1/m<sup>3</sup>)

 NSS:     Surface state density (default value = 0 1/m<sup>2</sup>)

 NFS:     Surface fast state density (default value = 0 1/m<sup>2</sup>)

 NEFF:    Total channel charge coefficient (default value = 1)

 XJ:      Metallurgical junction depth (default value = 0 m)

 L:       Channel length (default value = 1e-4 m)

 W:       Channel width (default value = 1e-4 m)

 LD:      Lateral diffusion (default value = 0 m)

 AD:      Drain diffusion area (default value = 0 m<sup>2</sup>)

 PD:      Drain perimeter width (default value = 0 m)

 AS:      Source diffusion area (default value = 0 m<sup>2</sup>)

 PS:      Source perimeter width (default value = 0 m)

 XD:      Depletion factor (default value = 0 m/Volt<sup>0.5</sup>)

 TPG:     Type of gate material (default value = 1)

 U0:      Surface mobility at reference temperature (default value = 0 m<sup>2</sup>/(V*s))

 UCRIT:   Critical electric field for mobility (default value = 1e6 V/m)

 UEXP:    Exponential coefficient for mobility (default value = 0)

 UTRA:    Transverse field coefficient (default value = 0)

 VMAX:    Maximum drift velocity of carriers (default value = 0 m/s)
            <font color=blue><b>VMAX = 0: Frohman-Grove model of drain current computation</b></font>
            <font color=blue><b>VMAX &gt; 0: Baum-Beneking model of drain current computation</b></font>

 ECRIT:   Critical electric field for pinch-off (default value = 0 V/m) <font color=red>Levels 2,3</font>

 XQC:     Coefficient of channel charge share (default value = 0) <font color=red>Levels 2,3</font>
            <font color=blue><b>XQC &lt;= 0.5: Ward-Dutton model of gate capacitance computation</b></font>
            <font color=blue><b>XQC &gt;  0.5: Meyer model of gate capacitance computation</b></font>

 DELTA:   Width effect on threshold voltage (default value = 0)

 KAPPA:   Saturation field factor (default value = 0.2)

 ETA:     Static feedback on threshold voltage (default value = 0)

 THETA:   Mobility modulation (default value = 0 1/Volt)

 VBP:     Gate to channel critical voltage (default value = 0 Volt)

 ISD:     Drain junction saturation current at reference temperature (default value = 0 Amp)

 ISS:     Source junction saturation current at reference temperature (default value = 0 Amp)

 PB:      Built-in potential at reference temperature (default value = 0.8 Volt)

 CJ:      Zero-bias bulk capacitance per square meter at reference temperature (default value = 1e-4 F/m<sup>2</sup>)

 MJ:      Bulk junction grading coefficient (default value = 0.33)

 CJSW:    Zero-bias perimeter capacitance per meter at reference temperature (default value = 1e-9 F/m)

 MJSW:    Perimeter capacitance grading coefficient (default value = 0.33)

 FC:      Forward-bias depletion capacitance coefficient (default value = 0.5)

 GminDC:  Leakage conductance (default value = 1e-12 mho)

 EG:      Energy gap for temperature effect on saturation current (default value = 1.11 Volt)

 EMin:    if x &lt; EMin, the exp(x) function is linearized (default value = -100)

 EMax:    if x &gt; EMax, the exp(x) function is linearized (default value = 40)
</pre> <p>

<hr> <p>

<b>References:</b> <p>

<ol>
<li> Cellier, F.E. (1991), <i>Continuous System Modeling</i>, Springer-Verlag, New York.
<li> Massobrio, G. and P. Antognetti (1993), <i>Semiconductor Device Modeling with Spice</i>,
     2<sup>nd</sup> edition, McGraw Hill, New York.
</ol>
</html>
"));
        BondLib.Interfaces.BondCon Bint "Internal bulk"
          annotation (extent=[130,-10; 150,10]);
        BondLib.Interfaces.BondCon Gint "Internal gate"
          annotation (extent=[-150,-10; -130,10]);
        BondLib.Interfaces.BondCon Sint "Internal source"
          annotation (extent=[-10,-150; 10,-130]);
        BondLib.Interfaces.BondCon Dint "Internal drain"
          annotation (extent=[-10,130; 10,150]);
        BondLib.Interfaces.BondCon H "Heat"
          annotation (extent=[-80,-150; -60,-130]);
        BondLib.Junctions.J0p4 J0p4_1 annotation (extent=[-130,-10; -110,10]);
        BondLib.Junctions.J0p4 J0p4_2 annotation (extent=[110,-10; 130,10]);
        BondLib.Junctions.J0p4 J0p4_3 annotation (extent=[-10,110; 10,130]);
        BondLib.Junctions.J0p4 J0p4_4 annotation (extent=[-10,-130; 10,-110]);
        BondLib.Bonds.Bond B1 annotation (extent=[40,20; 60,40], rotation=270);
        BondLib.Junctions.J1p3 J1p3_1
          annotation (extent=[40,40; 60,60], rotation=90);
        BondLib.Bonds.Bond B2 annotation (extent=[40,60; 60,80], rotation=270);
        BondLib.Bonds.Bond B3 annotation (extent=[60,40; 80,60], rotation=0);
        BondLib.Spice.Utilities.CjM Ddb(
          Level=Level,
          Tnom=Tnom,
          IS=ISD,
          PB=PB,
          CJ=CJ,
          MJ=MJ,
          CJSW=CJSW,
          MJSW=MJSW,
          FC=FC,
          A=AD,
          P=PD,
          EG=EG,
          GminDC=GminDC,
          EMin=EMin,
          EMax=EMax)     annotation (extent=[80,38; 110,62]);
        BondLib.Bonds.Bond B4 annotation (extent=[40,-40; 60,-20], rotation=90);
        BondLib.Junctions.J1p3 J1p3_2
          annotation (extent=[40,-60; 60,-40], rotation=90);
        BondLib.Bonds.Bond B5 annotation (extent=[40,-80; 60,-60], rotation=90);
        BondLib.Bonds.Bond B6 annotation (extent=[60,-60; 80,-40], rotation=0);
        BondLib.Junctions.J0p3 J0p3_1 annotation (extent=[80,-60; 100,-40]);
        BondLib.Bonds.fBond B7 annotation (extent=[100,-60; 120,-40]);
        BondLib.Sensors.De Vsb annotation (extent=[120,-60; 140,-40]);
        BondLib.Bonds.Bond B8
          annotation (extent=[80,-80; 100,-60], rotation=270);
        BondLib.Spice.Utilities.CjM Dsb(
          Level=Level,
          Tnom=Tnom,
          PB=PB,
          CJ=CJ,
          MJ=MJ,
          CJSW=CJSW,
          MJSW=MJSW,
          FC=FC,
          EG=EG,
          GminDC=GminDC,
          IS=ISS,
          A=AS,
          P=PS) annotation (extent=[78,-110; 102,-80], rotation=-90);
        BondLib.Bonds.eBond B9 annotation (extent=[110,40; 130,60]);
        BondLib.Bonds.eBond B10
          annotation (extent=[80,-130; 100,-110], rotation=-90);
        BondLib.Junctions.J1p3 J1p3_3
          annotation (extent=[30,-10; 10,10], rotation=90);
        BondLib.Bonds.Bond B11 annotation (extent=[10,10; 30,30], rotation=90);
        BondLib.Bonds.Bond B12
          annotation (extent=[10,-30; 30,-10], rotation=90);
        BondLib.Bonds.Bond B13
          annotation (extent=[-10,-10; 10,10], rotation=180);
        BondLib.Junctions.J0p4 J0p4_5 annotation (extent=[-30,-10; -10,10]);
        BondLib.Bonds.eBond B14 annotation (extent=[-50,-10; -30,10]);
        BondLib.Sources.mSf_int ISD0
                            annotation (extent=[-80,14; -50,-14]);
        BondLib.Bonds.fBond B15
          annotation (extent=[-30,10; -10,30], rotation=90);
        BondLib.Sensors.De Vsd annotation (extent=[-30,30; -10,50], rotation=90);
        BondLib.Bonds.Bond B16
          annotation (extent=[-30,-30; -10,-10], rotation=270);
        BondLib.Spice.RS Rsd(
          Tnom=Tnom,
          R=1/GminDC,
          TR1=0,
          TR2=0,
          Area=1) annotation (extent=[-32,-80; -8,-30],  rotation=-90);
        BondLib.Bonds.Bond B18
          annotation (extent=[-130,50; -110,70], rotation=90);
        BondLib.Junctions.J1p3 J1p3_4
          annotation (extent=[-130,30; -110,50], rotation=-90);
        BondLib.Bonds.Bond B17
          annotation (extent=[-120,4; -100,24],  rotation=90);
        BondLib.Bonds.Bond B19
          annotation (extent=[-150,50; -130,70], rotation=90);
        BondLib.Spice.RS Rgd(
          Tnom=Tnom,
          R=1/GminDC,
          TR1=0,
          TR2=0,
          Area=1) annotation (extent=[-152,70; -128,120], rotation=90);
        BondLib.Bonds.Bond B20
          annotation (extent=[-130,-30; -110,-10], rotation=90);
        BondLib.Junctions.J1p3 J1p3_5
          annotation (extent=[-120,-40; -100,-20], rotation=0);
        BondLib.Bonds.Bond B21
          annotation (extent=[-100,-40; -80,-20], rotation=180);
        BondLib.Bonds.Bond B22
          annotation (extent=[-120,-60; -100,-40], rotation=270);
        BondLib.Spice.RS Rsg(
          Tnom=Tnom,
          R=1/GminDC,
          TR1=0,
          TR2=0,
          Area=1) annotation (extent=[-152,-140; -128,-90], rotation=-90);
        BondLib.Junctions.J0p3 J0p3_2
          annotation (extent=[-120,-80; -100,-60], rotation=-90);
        BondLib.Bonds.Bond B23
          annotation (extent=[-120,-100; -100,-80], rotation=270);
        BondLib.Bonds.Bond B24
          annotation (extent=[-150,-90; -130,-70], rotation=270);
        BondLib.Sensors.De Vsg
          annotation (extent=[-120,-120; -100,-100], rotation=-90);
        BondLib.Bonds.eBond B25
          annotation (extent=[-30,-100; -10,-80], rotation=-90);
        BondLib.Bonds.eBond B26 annotation (extent=[-130,-150; -110,-130]);
        BondLib.Bonds.eBond B27
          annotation (extent=[-150,120; -130,140], rotation=90);
        BondLib.Bonds.fBond B28 annotation (extent=[-100,-10; -80,10]);
          BondLib.Spice.Utilities.IDS0 IDS0(
          Tnom=Tnom,
          KP=KP,
          GAMMA=GAMMA,
          LAMBDA=LAMBDA,
          NFS=NFS,
          NEFF=NEFF,
          XJ=XJ,
          U0=U0,
          UEXP=UEXP,
          UTRA=UTRA,
          VMAX=VMAX,
          DELTA=DELTA,
          EG=EG,
          L=L,
          W=W,
          Level=Level,
          EMin=EMin,
          EMax=EMax,
          PHI=PHI,
          Type=Type,
          VFB=VFB,
          COX=COX,
          XD=XD,
          ECRIT=ECRIT,
          VBP=VBP,
          PB=PB,
          KAPPA=KAPPA,
          ETA=ETA,
          THETA=THETA,
          LD=LD)       annotation (extent=[-90,66; -60,94]);
          Modelica.Blocks.Interfaces.RealOutput y1[5]
          "Connector of Real output signals"
            annotation (extent=[-80,130; -60,150],rotation=90);
        Modelica.Blocks.Math.Gain Gain1(k=-1)
          annotation (extent=[-60,40; -40,60],     rotation=-90);
        BondLib.Junctions.J0p6 J0p6_1
          annotation (extent=[-80,-130; -60,-110], rotation=90);
        BondLib.Bonds.eBond B29
          annotation (extent=[-80,-110; -60,-90], rotation=-90);
        BondLib.Sensors.De Tdev
          annotation (extent=[-86,40; -64,60], rotation=90);
        BondLib.Bonds.fBond B30
          annotation (extent=[-86,20; -64,40], rotation=90);
        BondLib.Spice.RS Rgb(
          Tnom=Tnom,
          R=1/GminDC,
          TR1=0,
          TR2=0,
          Area=1) annotation (extent=[90,98; 138,122],    rotation=0);
        BondLib.Bonds.Bond B31
                              annotation (extent=[70,100; 90,120],
                                                                 rotation=0);
        BondLib.Bonds.eBond B32
                               annotation (extent=[130,80; 150,100], rotation=
              -90);
        BondLib.Junctions.J0p5 J0p5_1
          annotation (extent=[-80,-90; -60,-70], rotation=90);
        BondLib.Junctions.J1p3 J1p3_6
          annotation (extent=[50,120; 70,140],   rotation=0);
        BondLib.Bonds.Bond B33
                              annotation (extent=[30,120; 50,140],
                                                                 rotation=0);
        BondLib.Bonds.Bond B34
                              annotation (extent=[70,120; 90,140],
                                                                 rotation=0);
        equation
        connect(J0p4_1.BondCon1, Gint)     annotation (points=[-130,0; -140,0],
            style(color=8, rgbcolor={192,192,192}));
        connect(J0p4_2.BondCon2, Bint)     annotation (points=[130,0; 140,0],
            style(color=8, rgbcolor={192,192,192}));
        connect(J0p4_3.BondCon4, Dint)     annotation (points=[0,130; 0,140],
            style(color=8, rgbcolor={192,192,192}));
        connect(J0p4_4.BondCon3, Sint)     annotation (points=[0,-130; 0,-140],
            style(color=8, rgbcolor={192,192,192}));
        connect(J1p3_1.BondCon3,B3. BondCon1) annotation (points=[60,50; 60.2,
              50], style(color=8, rgbcolor={192,192,192}));
        connect(B3.BondCon2,Ddb. BondCon1) annotation (points=[80,50; 80,50],
            style(color=8, rgbcolor={192,192,192}));
        connect(J1p3_2.BondCon3,B6. BondCon1) annotation (points=[60,-50; 60.2,
              -50], style(color=8, rgbcolor={192,192,192}));
        connect(B6.BondCon2,J0p3_1. BondCon1) annotation (points=[80,-50; 80,
              -50], style(color=8, rgbcolor={192,192,192}));
        connect(J0p3_1.BondCon2,B7. eBondCon1) annotation (points=[100,-50; 100,
              -50], style(color=8, rgbcolor={192,192,192}));
        connect(B7.fBondCon1,Vsb. BondCon1) annotation (points=[120,-50; 120,
              -50], style(color=8, rgbcolor={192,192,192}));
        connect(J0p3_1.BondCon3,B8. BondCon1) annotation (points=[90,-60; 90,
              -60.2], style(color=8, rgbcolor={192,192,192}));
        connect(B8.BondCon2,Dsb. BondCon1) annotation (points=[90,-80; 90,-80],
            style(color=8, rgbcolor={192,192,192}));
        connect(Ddb.BondCon2,B9. fBondCon1) annotation (points=[110,50; 110,50],
            style(color=8, rgbcolor={192,192,192}));
        connect(Dsb.BondCon2,B10. fBondCon1) annotation (points=[90,-110; 90,
              -110], style(color=8, rgbcolor={192,192,192}));
        connect(B13.BondCon1,J1p3_3. BondCon3) annotation (points=[9.8,
              -1.20011e-015; 2.9,-1.20011e-015; 2.9,-6.12303e-016; 10,
              -6.12303e-016], style(color=8, rgbcolor={192,192,192}));
        connect(J0p4_5.BondCon2,B13. BondCon2) annotation (points=[-10,0; -16,0;
              -16,1.22461e-015; -10,1.22461e-015], style(color=8, rgbcolor={192,
                192,192}));
        connect(B14.eBondCon1,J0p4_5. BondCon1) annotation (points=[-30,0; -30,
              0], style(color=8, rgbcolor={192,192,192}));
        connect(B15.eBondCon1,J0p4_5. BondCon4) annotation (points=[-20,10; -20,
              10], style(color=8, rgbcolor={192,192,192}));
        connect(Vsd.BondCon1,B15. fBondCon1) annotation (points=[-20,30; -20,30],
            style(color=8, rgbcolor={192,192,192}));
        connect(J0p4_5.BondCon3,B16. BondCon1) annotation (points=[-20,-10; -20,
              -10.1; -20,-10.2; -20,-10.2], style(color=8, rgbcolor={192,192,
                192}));
        connect(B16.BondCon2,Rsd. BondCon1) annotation (points=[-20,-30; -20,
              -30], style(color=8, rgbcolor={192,192,192}));
        connect(J1p3_4.BondCon2,B17. BondCon2) annotation (points=[-120,30;
              -110,30; -110,24],
                        style(color=8, rgbcolor={192,192,192}));
        connect(B18.BondCon1,J1p3_4. BondCon1) annotation (points=[-120,50.2;
              -120,50], style(color=8, rgbcolor={192,192,192}));
        connect(J1p3_4.BondCon3,B19. BondCon1) annotation (points=[-130,40;
              -140,40; -140,50.2], style(color=8, rgbcolor={192,192,192}));
        connect(Rgd.BondCon1,B19. BondCon2) annotation (points=[-140,70; -140,
              70], style(color=8, rgbcolor={192,192,192}));
        connect(B18.BondCon2,J0p4_3. BondCon1) annotation (points=[-120,70;
              -120,120; -10,120], style(color=8, rgbcolor={192,192,192}));
        connect(J1p3_5.BondCon3,B22. BondCon1) annotation (points=[-110,-40;
              -110,-40.2], style(color=8, rgbcolor={192,192,192}));
        connect(J0p3_2.BondCon1,B22. BondCon2) annotation (points=[-110,-60;
              -110,-60], style(color=8, rgbcolor={192,192,192}));
        connect(J0p3_2.BondCon2,B23. BondCon1) annotation (points=[-110,-80;
              -110,-80.2], style(color=8, rgbcolor={192,192,192}));
        connect(J0p3_2.BondCon3,B24. BondCon1) annotation (points=[-120,-70;
              -130,-70; -130,-70.2; -140,-70.2], style(color=8, rgbcolor={192,
                192,192}));
        connect(B24.BondCon2, Rsg.BondCon1)  annotation (points=[-140,-90; -140,
              -90], style(color=8, rgbcolor={192,192,192}));
        connect(B23.BondCon2,Vsg. BondCon1) annotation (points=[-110,-100; -110,
              -100], style(color=8, rgbcolor={192,192,192}));
        connect(Rsd.BondCon2,B25. fBondCon1) annotation (points=[-20,-80; -20,
              -80], style(color=8, rgbcolor={192,192,192}));
        connect(Rsg.BondCon2, B26.fBondCon1)  annotation (points=[-140,-140;
              -130,-140], style(color=8, rgbcolor={192,192,192}));
        connect(B27.fBondCon1,Rgd. BondCon2) annotation (points=[-140,120; -140,
              120], style(color=8, rgbcolor={192,192,192}));
        connect(IDS0.y, Gain1.u)   annotation (points=[-58.5,80; -50,80; -50,62],
            style(color=74, rgbcolor={0,0,127}));
        connect(IDS0.y1, y1)   annotation (points=[-58.5,73; -54,73; -54,120;
              -70,120; -70,140], style(color=74, rgbcolor={0,0,127}));
        connect(Vsd.OutPort1, IDS0.u1)   annotation (points=[-20,51; -20,112;
              -110,112; -110,88.4; -93,88.4], style(color=74, rgbcolor={0,0,127}));
        connect(Vsg.OutPort1, IDS0.u2)   annotation (points=[-110,-121; -110,
              -126; -96,-126; -96,40; -110,40; -110,71.6; -93,71.6], style(
              color=74, rgbcolor={0,0,127}));
        connect(ISD0.BondCon2,B14. fBondCon1) annotation (points=[-50,
              -1.77636e-015; -50,0], style(color=8, rgbcolor={192,192,192}));
        connect(ISD0.BondCon1,B28. fBondCon1) annotation (points=[-80,
              -1.77636e-015; -80,0], style(color=8, rgbcolor={192,192,192}));
        connect(J0p6_1.BondCon1, H)        annotation (points=[-70,-130; -70,
              -140], style(color=8, rgbcolor={192,192,192}));
        connect(B29.eBondCon1,J0p6_1. BondCon2) annotation (points=[-70,-110;
              -70,-110], style(color=8, rgbcolor={192,192,192}));
        connect(IDS0.Tdev, Tdev.OutPort1)   annotation (points=[-75,66; -75,61],
            style(color=74, rgbcolor={0,0,127}));
        connect(Tdev.BondCon1,B30. fBondCon1) annotation (points=[-75,40; -75,
              40], style(color=8, rgbcolor={192,192,192}));
        connect(B26.eBondCon1,J0p6_1. BondCon6) annotation (points=[-110,-140;
              -92,-140; -92,-116; -80,-116], style(color=8, rgbcolor={192,192,
                192}));
        connect(B27.eBondCon1,J0p6_1. BondCon5) annotation (points=[-140,140;
              -158,140; -158,-152; -88,-152; -88,-124; -80,-124], style(color=8,
              rgbcolor={192,192,192}));
        connect(J0p5_1.BondCon1,B29. fBondCon1) annotation (points=[-70,-90;
              -70,-90], style(color=8, rgbcolor={192,192,192}));
        connect(J0p5_1.BondCon5,B28. eBondCon1) annotation (points=[-80,-80;
              -92,-80; -92,-16; -104,-16; -104,0; -100,0], style(color=8,
              rgbcolor={192,192,192}));
        connect(J0p5_1.BondCon2,B30. eBondCon1) annotation (points=[-70,-70;
              -70,-26; -44,-26; -44,16; -75,16; -75,20], style(color=8,
              rgbcolor={192,192,192}));
        connect(Vsb.OutPort1, IDS0.u3)   annotation (points=[141,-50; 158,-50;
              158,160; -88,160; -88,108; -75,108; -75,94], style(color=74,
              rgbcolor={0,0,127}));
        connect(B31.BondCon2,Rgb. BondCon1) annotation (points=[90,110; 90,110],
            style(color=8, rgbcolor={192,192,192}));
        connect(Rgb.BondCon2,B32. fBondCon1) annotation (points=[138,110; 140,
              110; 140,100], style(color=8, rgbcolor={192,192,192}));
        connect(J1p3_6.BondCon3,B31. BondCon1) annotation (points=[60,120; 60,
              110; 70.2,110], style(color=8, rgbcolor={192,192,192}));
        connect(B33.BondCon2,J1p3_6. BondCon1) annotation (points=[50,130; 50,
              130], style(color=8, rgbcolor={192,192,192}));
        connect(J1p3_6.BondCon2,B34. BondCon1) annotation (points=[70,130; 70.2,
              130], style(color=8, rgbcolor={192,192,192}));
        connect(B34.BondCon2,J0p4_2. BondCon4) annotation (points=[90,130; 154,
              130; 154,30; 120,30; 120,10], style(color=8, rgbcolor={192,192,
                192}));
        connect(B17.BondCon1,J0p4_1. BondCon2) annotation (points=[-110,4.2;
              -110,0], style(color=8, rgbcolor={192,192,192}));
        connect(B33.BondCon1,J0p4_1. BondCon4) annotation (points=[30.2,130; 20,
              130; 20,164; -162,164; -162,20; -120,20; -120,10], style(color=8,
              rgbcolor={192,192,192}));
        connect(B25.eBondCon1,J0p5_1. BondCon4) annotation (points=[-20,-100;
              -20,-1